#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#


;==========================================================================
; File: tree-graphics.lisp     Version: 1.0             Created: 
;
; Locked by:  nobody.                                    Modified: 4/4/88
; 
; Purpose: These functions create and control the graphical tree 
;          representation of PRODIGY's problem solving trace.  
;
; Be careful when returning values using pg-xx forms. (eg. pg-with-window).
; In make-node-image, the return value should be outside the form.
;
;==========================================================================


(eval-when (compile)
   #+:coral (load-path *planner-path* "pg-mac")
   #-:coral (load-path *planner-path* "pg-x11")
   (load-path *PLANNER-PATH* "data-types")
)
;;
;; need prodigy-graphics to be loaded before this file is loaded.
;;

(require 'pg-system)		
#-:coral (use-package "PG")




(proclaim
 '(special  *TREE-WINDOW* 
            *TREE-X-OFFSET* *TREE-Y-OFFSET* *CLONE-OFFSET* *TREE-GRAPHICS*
            *TREE-X-OR* *TREE-Y-OR* *TREE-BACKGRD* *TREE-FOREGRD*
            *TREE-CHAR-WIDTH* *TREE-CHAR-HEIGHT* *TREE-FRAME* *TREE-NODES* 
            *TREE-FONT* *TREE-FONT-INFO* *TREE-ROOT* *PREVIOUS-TREE-ROOT*

	    *NODE-X-OR* *NODE-Y-OR* *BACKTRACK-WIDTH* *BACKTRACK-HEIGHT*
 	    *ARROW-ANGLE* *ARROW-SIDE-LEN* *ARROW-X-OFFSET*
	    *MAPPED* *LEVELS* *DISPLAY-TYPE* ))


(defvar *NODE-WIDTH* 65)   		        ;  width of a node window
(defvar *NODE-HEIGHT* 30)		        ;  height of a node window
(defvar *NODE-X-OFFSET* (+ *NODE-WIDTH* 10))  ;  x distance between nodes
(defvar *NODE-Y-OFFSET* (+ *NODE-HEIGHT* 40)) ;  y distance between nodes

(defvar *TREE-WIN-WIDTH* (+ 450 *NODE-X-OFFSET*));  width of tree window
(defvar *TREE-WIN-HEIGHT* (+ 430 *NODE-Y-OFFSET*));  height of tree window
(defvar *MAPPED* nil)

;**************************************************************************
;                       Rectangle Data Type
;**************************************************************************


;; Rectangles (node-rect's) are used to represent
;; the boxes for the graphic node images.

(defstruct (node-rect)   
 (left 0 :type fixnum)
 (top 0 :type fixnum)
 (right 0 :type fixnum)
 (bottom 0 :type fixnum)
 (level 0 :type fixnum)
 mapped-p  ; This may need initialization
 (backtrack-node nil)
 (current-path t)
)
;; The dumb compiler won't issue a pervasive proclimation for the
;; access funcitons for the defstruct. So I'll do it here.

(proclaim '(function node-rect-left (node-rect) fixnum))
(proclaim '(function node-rect-top (node-rect) fixnum))
(proclaim '(function node-rect-right (node-rect) fixnum))
(proclaim '(function node-rect-bottom (node-rect) fixnum))
(proclaim '(function node-rect-level (node-rect) fixnum))


(defun gcreaterect (x y w h l)
    (declare (fixnum x y w h l))
    (make-node-rect :left x 
		    :top y 
		    :right (+ x w) 
		    :bottom (+ y h)
		    :level l))


(defun gframerect (r)
  (declare (type node-rect r))
  "draws the rectangle"
  (pg-frame-rect *TREE-WINDOW* 
		(node-rect-left r) (node-rect-top r) 
                  (node-rect-right r) (node-rect-bottom r)))

(defun geraserect (r)
  (declare (type node-rect r))
  "clears a rectangular area on the screen"
  (pg-erase-rect *TREE-WINDOW*
	       (node-rect-left r) (node-rect-top r) 
                  (node-rect-right r) (node-rect-bottom r)))

(defun ginvertrect (r)
  (declare (type node-rect r))
  "prints the contents of a rectangle in reverse video"
  (pg-invert-rect *TREE-WINDOW*
		(node-rect-left r) (node-rect-top r) 
                  (node-rect-right r) (node-rect-bottom r)))

(defun gchangerect (r w h)
  (declare (type node-rect r)
	   (fixnum w h))
  "changes the width and height of a rectangle"
   (let ((bot (+ (node-rect-top r) h)) 
	 (rgt (+ (node-rect-left r) w)))
         (declare (fixnum bot rgt))
      (setf (node-rect-bottom r) bot)
      (setf (node-rect-right r) rgt)))


(defun gmoverect (r x y)
  (declare (type node-rect r))
  "changes the location of a rectangle"
  (let ((h (- (node-rect-bottom r) (node-rect-top r)))
        (w (- (node-rect-right r) (node-rect-left r))))
       (declare (fixnum h w))
    (setf (node-rect-left r) x)
    (setf (node-rect-top r) y)
    (setf (node-rect-right r) (+ x w))
    (setf (node-rect-bottom r) (+ y h))))

 
;**************************************************************************
;           Functions relating nodes to rectangles 
;**************************************************************************

#|
(defun get-rect-name (node)
  (declare (type node node))
 "Takes a node named n#  and returns a symbol of the form: rect-n#"
   (intern (concatenate 'string "RECT-" (nstring-upcase (princ-to-string 
							(node-name node))))))
|#
(defun get-rect-name (node)
  (declare (ignore node))
  nil)

#|
(defun get-rectangle (node)
  (declare (type node node))
  "Returns the rectangle associated with a node"
   (eval (get-rect-name node)))

; Note: The doc string seems incorrect--dkahn
(defun exists-rectangle (node)
  (declare (type node node))
  "Returns t if a rectangle has already been created for this node"
  (if (boundp (get-rect-name node)) (get-rectangle node)))


(defun get-level (node)
  (declare (type node node))
   (get (get-rect-name node) 'level))


(defun get-x (node)
  (declare (type node node))
   (get (get-rect-name node) 'x))


(defun get-y (node)
  (declare (type node node))
   (get (get-rect-name node) 'y))
|#

(defun get-rectangle (node)
  (declare (type node node))
  "Returns the rectangle associated with a node."
  (get (node-name node) 'rect))

(defun exists-rectangle (node)
   (declare (type node node))
  "Returns t if a rectangle has already been created for this node"
   (get (node-name node) 'rect))

(defmacro get-level (node)
   (declare (type node node))
   `(node-rect-level (get (node-name ,node) 'rect)))


;; The LEFT and TOP notations in the structure are confusing here and
;; elsewhere in the file. Perhaps they should be x and y
(defmacro get-x (node)
   (declare (type node node))
   `(node-rect-left (get (node-name ,node) 'rect)))

(defmacro get-y (node)
    (declare (type node node))
    `(node-rect-top (get (node-name ,node) 'rect)))

(defmacro node-rect (node)
    (declare (type node node))
    `(get (node-name ,node) 'rect))

;**************************************************************************
;               Functions to create a graphic node image
;**************************************************************************


(defun make-node-image (node width height top-node top-level backtrack-p)
  (declare (type node node))

  " MAKE-NODE-IMAGE sets all of the node's properties and creates an image 
    for the node. All of the properties MUST be set before an image can be
    created because the x and y coordinates of the image must be defined 
    and they are computed from the other node properties."

  (let* ((level-abs  (calculate-level node backtrack-p))
	 (level-rel(- level-abs top-level))
	 (rect (gcreaterect (calc-x node level-rel top-node)
			    (calc-y level-rel)
			    width 
			    height
			    level-abs
			    )))
        (declare (fixnum level-abs level-rel))
    
    (setf (node-rect node) rect) ; new way of storing nodes
   (if (not rect) (error "gcreaterect failed!"))
    (cond ((in-tree-window (get-x node) (get-y node))
           (map-node node)))

))



(defun in-tree-window (x y)
 "Returns t if the upper left-hand corner
  of the node is in the tree window"
   (declare (fixnum x y)
	    (special  *TREE-WINDOW-VIRTUAL-WIDTH* *TREE-X-OR*
		      *TREE-WINDOW-VIRTUAL-HEIGHT* *TREE-Y-OR*))
   (and (>= *TREE-WINDOW-VIRTUAL-WIDTH* x *TREE-X-OR*)
        (>= *TREE-WINDOW-VIRTUAL-HEIGHT* y *TREE-Y-OR*)))


(defun get-new-level (node top-level)
  (declare (type node node)
	   (fixnum top-level))
 "Calculates the new level of a node by shifting the"
  (- (get-level node) top-level))



;--------------------------------------------------------------------------  


(defun gdrawnode (rect node)
  (declare (type node node)
	   (type node-rect rect))
  "draws the image of a node"
  (pg-with-window *TREE-WINDOW*
    (geraserect rect)
    (gframerect rect)
    (write-in-node rect node)))



(defun write-in-node (rect node)
  (declare (type node node)
	   (type node-rect rect))
  "Prints the node information in the node rectangle"
  (let ((children (node-children node))
	(x (node-rect-left rect))
	(y (node-rect-top rect)))
    (pg-write-text *TREE-WINDOW* (+ x *NODE-X-OR*) (+ y *NODE-Y-OR* )
	(trim-for-rect (princ-to-string (node-name node)) rect))
    (if (and (not (backtrack-node-p node)) children)
	(let ((op (alt-op (node-generating-alt
			   (find-active-child children)))))
        (pg-write-text *TREE-WINDOW* (+ x *NODE-X-OR*)
		  (+ y *NODE-Y-OR* (pg-text-height *TREE-WINDOW* "A") 3)
		(trim-for-rect (princ-to-string op) rect))))))

(defun trim-for-rect (string rect)
  (declare (string string)
	   (type node-rect rect))
  "returns a subsequence of the string that will fit 
   inside the left and right bounds of the rect."
  (let* ((left-bound (node-rect-left rect))
	 (right-bound (node-rect-right rect))
	 (max-string-length (1- (floor (abs (- right-bound left-bound))
				   (pg-text-width *TREE-WINDOW* "A")))))
    (if (< (length string) max-string-length)
        string
        (subseq string 0 max-string-length))))

;--------------------------------------------------------------------------  


(defun highlight-node (node)
  (declare (type node node))
 "Changes the background of the node rectangle to be 
  black and writes the node information in white"
  (pg-with-window *TREE-WINDOW*
    (if (is-mapped node)
        (let ((rect (get-rectangle node)))
	  (declare (type node-rect rect))
          (gdrawnode rect node)
          (ginvertrect rect)
          (redraw-right-clones node)
	  (pg-refresh-window *TREE-WINDOW*)))))


(defun unhighlight-node (node)
  (declare (type node node))
  "Changes the background of the node rectangle to be 
   white and writes the node information in black"
  (pg-with-window *TREE-WINDOW*
    (if (is-mapped node)
        (let ((w (get-rectangle node)))
	  (declare (type node-rect w))
          (gdrawnode w node)
          (redraw-right-clones node)))))


(defun redraw-right-clones (node)
  (declare (type node node))
  "redraws the right clones of a node."
  (unless (backtrack-node-p node)
    (dolist (clone (get-right-clone-list node))
	    (declare (type node clone))
       (if (is-mapped clone)
           (gdrawnode (get-rectangle clone) clone)))))

;**************************************************************************
;                   Creating  The  Search  Tree
;**************************************************************************
; The calls to display-tree and make-tree-window had to be reversed to
; work with new refresh-graphics.  Display-tree now just does 
; initialization and no drawing.

; display-tree and initial-vars are now both in analyze and
; necessarily before the create-tree function is called.

(defun create-tree ()
  (declare (special *font-string*))
  "CREATE-TREE initializes the global tree variables, creates the tree
   window, and displays the tree structure in the window"
    (pg-init-graphics (current-host) *font-string*)
    (make-tree-window))
;--------------------------------------------------------------------------  

(proclaim '(fixnum  *TREE-X-OFFSET* *TREE-Y-OFFSET* *TREE-X-OR*
		    *TREE-Y-OR* *CLONE-OFFSET* *BACKTRACK-WIDTH*
		    *BACKTRACK-HEIGHT* *NODE-X-OR* *NODE-Y-OR*
		    *ARROW-ANGLE* *ARROW-SIDE-LEN* *TREE-WIN-WIDTH*
		    *TREE-WIN-HEIGHT*))


(defun initialize-vars ()
  "INITIALIZE-VARS initializes tree variables.
   When PRODIGY finishes, it will have created an extra node that it never
   used.  In this case, there will be an extra node in *ALL-NODES* with no 
   information in it.  If PRODIGY does not finish, however, all of the nodes
   will contain information.  So, I use the variable *TREE-NODES* to hold all 
   of the nodes that PRODIGY used.  That is, if PRODIGY has finished then 
   (cdr *ALL-NODES*) will remove the empty node since *ALL-NODES* is in
   reverse order (most recent node first).  I then reverse the list of nodes
   so that the start node (n1) will be in the front of the list."

    (setf *TREE-X-OFFSET* 0)                    ;x and y offsets of tree window
    (setf *TREE-Y-OFFSET* 0)                    ;  (see make-tree-geometry)
    (setf *TREE-X-OR* 5)          		;  x origin in tree window
    (setf *TREE-Y-OR* 5)			;  y origin in tree window
    (setf *CLONE-OFFSET* 30)      		;  x distance between clones
    (setf *BACKTRACK-WIDTH* 30)   		;  width of a backtrack node
    (setf *BACKTRACK-HEIGHT* 15)		;  height of a backtrack node
    (setf *NODE-X-OR* 3)			;  x origin in a node window
    (setf *NODE-Y-OR* 12) ; -rlj-		;  y origin in a node window
    (setf *ARROW-ANGLE* 20)			;  measured in degrees 
    (setf *ARROW-SIDE-LEN* 8)			;  (see draw-arrow)
    (setf *ARROW-X-OFFSET* 8)

;; x offset to align the arrow slightly to the right of a node
)

(defun initialize-tree (all-nodes)
;; Prodigy creates a dummy node at the beginning of *all-nodes* upon
;; finding a solution to a problem.  The dummy node must have a rect
;; property to prevent errors, but will never be printed.

;; The 'last-node-to-draw property is used by the draw routines to
;; determine where to stop drawing.  It prevents us from hitting the
;; last node.

    (dolist (nodes all-nodes)
      (setf nodes (reverse nodes))
      (cond ((and (cadr nodes)
		  (eq (node-current-op (cadr nodes)) '*FINISH*))
					;has extra empty node 
	     (setf *TREE-NODES* (reverse (cdr nodes)))
	     (setf (get (node-name (car nodes)) 'rect)
		   (make-node-rect :backtrack-node nil))
	     (setf (get (node-name (cadr nodes)) 'last-node-to-draw) t))
	    (t (setf (get (node-name (car nodes)) 'last-node-to-draw) t)
	       (setf *TREE-NODES* (reverse nodes)))))

      (setf *LEVELS* nil)
)

;--------------------------------------------------------------------------
; These functions are used by the analyze preset in analyze.lisp.
; They build a list of trees (from the abstraction system) that can
; each be viewed by using the function (the-abs-down-command) or
; (the-abs-up-command).

(defun build-tree-node-list (nodes heads)
  (cond ((null nodes) nil)
	(t (cons (sort (whole-tree (car nodes) heads) #'< :key #'node-sort-key)
	      (build-tree-node-list (member (node-abstract-child (car nodes))
					    nodes) heads)))))

(defun whole-tree (node heads)
  "Returns a list of all the nodes in the tree headed by node.  Heads
is used to mark the 'fictitious children' that need to be removed from
the children list."
  (cond ((null node) nil)
	(t (cons node (mapcan #'(lambda (x)
				  (set-difference (whole-tree x
								heads) heads))
				  (node-children node))))))

(defun node-sort-key (x)
  (parse-integer (symbol-name (node-name x)) :start 1))

(defun generate-heads (node)
  (cond ((null node) nil)
	(t (cons node (generate-heads (node-abstract-child node))))))

;--------------------------------------------------------------------------  

(defun make-tree-window ()
  "Creates the tree rectangle and initializes the global font variables"
  (setf *TREE-WINDOW* (pg-create-window 575 350 *TREE-WIN-WIDTH*
						*TREE-WIN-HEIGHT*
					:exposure-function #'tree-refresh
					:name "Tree"
					:config-function #'tree-config	))
  (pg-show-window *TREE-WINDOW*))

;--------------------------------------------------------------------------
;; TREE-REFRESH is called by the exposure routine in CLX to redraw the
;; tree window
;; *expl-node* used to be current node.

(defun tree-refresh (&rest x)
   (declare (ignore x)
	    (special *EXPL-NODE*))
   "Called by an Exposure event to redraw window."
	(refresh-tree *EXPL-NODE*)
)

;--------------------------------------------------------------------------  
;; TREE-CONFIG  is the routine that resets the width and height of the 
;; window when it is resized.


;; The virtual width and height trick some old tree code into thinking
;; that the window is larger then it actually is so that nodes near
;; the edge of the window get displayed, even though they don't fit
;; completely.
(defun tree-config (x-window x y width height)
      (declare (ignore x-window x y)
	       (special *TREE-WIN-WIDTH* *NODE-X-OFFSET*
			*TREE-WIN-HEIGHT* *NODE-Y-OFFSET*
			*TREE-WINDOW-VIRTUAL-WIDTH* 
			*TREE-WINDOW-VIRTUAL-HEIGHT*))
      (setf *TREE-WIN-WIDTH* width)
      (setf *TREE-WIN-HEIGHT* height)
      (setf *TREE-WINDOW-VIRTUAL-WIDTH* (+ width *NODE-X-OFFSET*))
      (setf *TREE-WINDOW-VIRTUAL-HEIGHT* (+ height *NODE-Y-OFFSET*))
)

;--------------------------------------------------------------------------  
;
;                    EVERYTHING YOU ALWAYS WANTED TO KNOW 
;
;                       ABOUT THE TREE DRAWING ALGORITHM, 
;
;                          BUT WERE AFRAID TO ASK...
;
;                       (and probably for good reason...)
;
;
;
;
; Chapter 1:  THE GENERAL TREE DRAWING ALGORITHM
;
;     The root of the tree is defined as the node which will appear at the 
; upper left-hand corner of the tree window.  Beginning with the root node,
; all of his children who are backtracking nodes are displayed one level
; below him.  One of his children, however, will not be a backtracking
; node and this node is labelled the "active" child. The active child is
; then displayed and the algorithm continues with the active child as
; the current node.
;     This drawing strategy guarantees that (1) a backtracking node
; will always be below his parent, and (2) an active node must be to the
; right of any backtracking siblings so that a path originating from this
; node can never get tangled up with a backtracking node.  
;     The algorithm continues in this fashion until either (1) it finds a 
; node with no children, or (2) it finds a rightmost clone of the root node 
; who has an applied child.  The fact that it has an applied child means that
; the path leaving this node continues above the parent node. If the parent
; node has no applied children then the path continues below the parent so we
; continue to draw the path.  
;

; Commented out some lines for new graphics -- dkahn
; display-tree no longer should draw any graphics, it is now
; just an initialization routine.  When the window comes up an Exposure
; event will be generated which will cause the screen to be drawn.
; display-tree is now called right after entering the analyze facility
; because the data structures it creates are needed for the node movement
; commands.  This commands should be independent of whether or not a tree 
; window exists.

;--------------------------------------------------------------------------
; Because we now (fall '90) need to be able to handle several trees at
; once for the abstraction system.  This function is part of that
; system and exists one call above display-tree.

(defun display-several-trees (trees node)
    (display-tree (find-tree node trees)))


(defun display-tree (tree)
  (declare (type node n1)
	   (special n1))
   "DISPLAY-TREE creates and displays the tree in the following manner:

    a) Beginning with node n1, a node rectangle is created and an arrow is 
       drawn from the node to its parent.  (except for n1 which has no parent)
    b) For each of the node's backtracking children, a backtrack node is
       created and an arrow is drawn from the child to its parent.
    c) Steps a & b are repeated for the node's active child to display the 
       subtree under this child until the active child is nil or we reach the 
       last node that PRODIGY has explored (last *TREE-NODES*)"
         
  (cond (tree ; perhaps this should be (car tree) ;(boundp 'n1)
;	   (initialize-tree tree)
	   (setf *NODE-LIST* (reverse tree))
           (setf *TREE-ROOT* (car tree))
           (setf *PREVIOUS-TREE-ROOT* *TREE-ROOT*)
           (do* ((lastnode nil node)
                 (node (car tree) (car (node-children node))))
                ((or (null node) (eq lastnode (car (last tree)))))
              (make-node-image node *NODE-WIDTH* *NODE-HEIGHT* (car tree) 0 nil)
              (if (cdr (node-children node))
		  (make-backtrack-nodes node)))

)))


;--------------------------------------------------------------------------  
;
; Chapter 2:  BACKTRACKING NODES
;
;     Each node has a 'backtrack-node property.  At any
; given time, a node may have at most one active child.  All of its other
; children are designated as backtracking nodes.  Initially, the
; active child will be the most recent child that PRODIGY has explored.  
; When the user switches backtracking paths in the explanation facility,
; however, a backtracking node is selected to become the new
; active child and the previously active child will become a backtracking node.
; The tree drawing algorithm will collapse the path originating from
; the previously active child and it will expand the path originating
; from the new active child. Only the *first* node on a backtracking path will 
; be a backtracking node!  The remaining nodes on the backtracking path will 
; never be explored because the tree drawing algorithm only looks at subtrees 
; of active children.  
; 
(defun backtrack-node-p (node)
  (declare (type node node))
  (node-rect-backtrack-node (get (node-name node) 'rect)))

(defun set-backtrack-node (node val)
  (declare (type node node))
  (setf (node-rect-backtrack-node (get (node-name node) 'rect)) val))

(defun make-backtrack-nodes (parent)
  (declare (type node parent n1)
	   (special n1))
  "Creates and displays the backtracking children of a node"
   (dolist (node (reverse (cdr (node-children parent))))
	 (declare (type node node)
		  (special *BACKTRACK-WIDTH* *BACKTRACK-HEIGHT*
			   *NODE-WIDTH* *NODE-HEIGHT*)
		  (fixnum *BACKTRACK-WIDTH* *BACKTRACK-HEIGHT*
			   *NODE-WIDTH* *NODE-HEIGHT*))
     (make-node-image node *BACKTRACK-WIDTH* *BACKTRACK-HEIGHT* n1 0 t)
     ; the child of a backtrack node must be init'ed and made
     ; unmapped because it will be used as the active node when the
     ; context command is printed for the backtracking node.
     (let ((active-child (car (node-children node))))
       (when active-child
	 (make-node-image active-child  *NODE-WIDTH*
			  *NODE-HEIGHT* n1 0 t)
	 (unmap-node active-child)))
     (set-backtrack-node node t)
))

(defun find-active-child (children)
  "Takes a list of children and returns the child on the current path"
   (cond ((null children) nil)
	 ((backtrack-node-p (car children))
	  (find-active-child (cdr children)))
	 (t (car children))))



(defun applied-child (children)
  "Returns t if one of the children is an applied node"
   (cond ((null children) nil)
	 ((node-applied-node (car children)) t)
	 (t (applied-child (cdr children)))))



;--------------------------------------------------------------------------
;
; Chapter 3:  THE CURRENT PATH
;
;    Each node has a 'current-path property.  The current path 
; consists of all nodes that are either active or backtracking nodes (but
; not nodes inside the backtracking paths).  This property is used only for
; error checking purposes inside the explanation facility to prevent
; the user from attempting to access a node inside a backtracking path.
;
;

;; If a node has no associated rectangle then it clearly must not be
;; on the current path. 
(defun on-current-path (node)
  "Returns t if node is on current path, NIL otherwise."
  (declare (type node node))
  (and (node-rect node)
       (node-rect-current-path (node-rect node))))

(defun set-current-path (node val)
  (declare (type node node))
  (setf (node-rect-current-path (get (node-name node) 'rect)) val))

(defun reset-current-path (new-child root-level root)
  (declare (type node node root))
 "RESET-CURRENT-PATH sets the nodes on the path originating from node 
  `new-child' to be on the current path.  Notice that the first nodes on 
  a backtracking path *will* be on the current path because they get printed 
  in the rectangle as smaller backtracking nodes."
   (initialize-new-path new-child root-level root)
   (do* ((node new-child next-node)
         (children (if node (node-children node) nil)
	           (if node (node-children node) nil))
         (next-node (find-active-child children) (find-active-child children)))
       ((null next-node))
       (set-current-path next-node t)
       (dolist (node (remove next-node children))  (set-current-path node t))))

(defun initialize-new-path (node root-level root)
   (declare (type node node root)
	    (special *NODE-WIDTH* *NODE-HEIGHT*))
   "Initializes, if needed, the nodes in the new current path."
   (cond ((null node) nil)
	 ((car (node-children node))
	  (if (not (node-rect (car (node-children node))))
	      (make-node-image (car (node-children node))
			       *NODE-WIDTH* *NODE-HEIGHT* 
			       root root-level nil))
	  (set-backtrack-node (car (node-children node)) nil)
	  (initialize-new-path (car (node-children node)) root-level root)
	  ; Now do the children.
	  (dolist (b-node (cdr (node-children node)))
		    (if (not (node-rect b-node))
			(make-node-image b-node 
					 *BACKTRACK-WIDTH*
					 *BACKTRACK-HEIGHT*
					 root root-level t))))))
	   


;--------------------------------------------------------------------------


(defun clear-current-path (old-child)
   "takes the nodes on the path originating from 
    node "old-child" off of the current path.  "
  (do* ((node old-child next-node)
         (children (if node (node-children node) nil)
	           (if node (node-children node) nil))
         (next-node (find-active-child children) (find-active-child children)))
       ((null next-node))
       (set-current-path next-node nil)
       (dolist (node (remove next-node children))
	    (set-current-path node nil))))
;	    (clear-current-path node))))


;--------------------------------------------------------------------------
;
; Chapter 4:  MAPPING
;
;     "Mapping" a rectangle draws it on the screen. Creating a rectangle
; does NOT map the rectangle and it will not appear on the screen until it is
; explicitly drawn with the function (gdrawnode).  The tree drawing
; algorithm maps a node only when its upper left-hand corner is inside the
; tree window. It also maintains a list, *MAPPED*, which keeps track of all
; nodes who are currently mapped. When the tree is redrawn, the previously
; mapped nodes are copied into a list of nodes to be unmapped and *MAPPED* is
; reset to nil.  *MAPPED* is updated dynamically as the new region of the tree
; is displayed and any node which remains in the tree-window is deleted
; from the list of nodes to unmap.  This strategy guarantees that all 
; previously mapped nodes will be unmapped by default.
;    
;     



; The function unmap-nodes had the good'ole unassigned delete bug,
; which I fixed, but the whole thing could be re-written as a simple
; do loop using with a POP inside, maybe I'll get to this.

(defun unmap-all-nodes ()
  (unmap-nodes (mapped-nodes)))

(defun unmap-nodes (nodes-to-unmap)
  (dolist (node nodes-to-unmap) 
     (declare (type node node))
     (unmap-node node)))

(defun unmap-node (node)
  (declare (type node node)
	   (special *MAPPED*))
  (setf (node-rect-mapped-p (node-rect node)) nil)
  (if *MAPPED* (setf *MAPPED* (delete node *MAPPED*))))

(defun map-node (node)
  (declare (type node node)
	   (special *MAPPED*))
;  (setf (get (get-rect-name node) 'mapped) t)
  (setf (node-rect-mapped-p (get (node-name node) 'rect)) t)
     (push node *MAPPED*))
 
(defun is-mapped (node)
  (if (node-p node) 
       (node-rect-mapped-p (node-rect node))))

(defun mapped-nodes ()
  (declare (special *MAPPED*))
  (and (boundp '*MAPPED*) *MAPPED*))


;**************************************************************************
;                           Exported   functions 
;**************************************************************************
;
;   USED  IN COMMANDS.LISP
;


(defun draw-nodes (root-level root oldmapped)  
  (declare (type node root))
   "DRAW-NODES draws all of the nodes on the current path beginning
    with the given root node.  Draws the root node first, then its 
    children, then the active child's children, etc."

   (do* ((node root next-node)
	 (nodes-to-unmap 
            (draw-node node root-level root oldmapped)
            (draw-children next-node children root-level root nodes-to-unmap))
         (children (if node (node-children node) nil)
	           (if node (node-children node) nil))
         (next-node (find-active-child children)
		    (find-active-child children)))
       ((last-node-to-draw node root) (unmap-nodes nodes-to-unmap))))



(defun draw-children (next-node children root-level root nodes-to-unmap)
  (declare (type node next-node root))
  "DRAW-CHILDREN draws the all of a node's children that aren't on the 
   current path as backtracking nodes first and then it draws the active 
   child. So, backtracking nodes will always be to the left of the active 
   child. Removes the active child and reverses the list so that the 
   children will be printed in increasing order"

   (dolist (node (reverse (remove next-node children)))
	   (declare (type node node))
      (set-backtrack-node node t)
      (setf nodes-to-unmap (draw-node node root-level root nodes-to-unmap)))
    (cond (next-node
	   (set-backtrack-node next-node nil)
           (setf nodes-to-unmap 
		    (draw-node next-node root-level root nodes-to-unmap))))
   nodes-to-unmap)

;; dkahn!! I don't think that the first clause of the COND in the let
;; is ever used now.  It probably can be eliminated.

(defun draw-node (node root-level root nodes-to-unmap)
  (declare (type node node))
 "Draws a node in the tree window.  If a node does not already have a
  rectangle associated with it then one will be created. "
  (if node
      (let* ((w (cond ((not (exists-rectangle node))
		       (make-node-image node *NODE-WIDTH* *NODE-HEIGHT* root
					 root-level nil))
		      (t (set-node-xy node (get-new-level node root-level) root)
			 (get-rectangle node))))
	     (node-x (get-x node))
	     (node-y (get-y node)))
	(declare (type node-rect w)
		 (fixnum node-x node-y))
	(cond ((in-tree-window node-x node-y)
               (if (backtrack-node-p node)
	           (gchangerect w *BACKTRACK-WIDTH* *BACKTRACK-HEIGHT*))
	       (gdrawnode w node)
	       (if (not (is-mapped node)) (map-node node))
	       (if (is-mapped (node-parent node)) (draw-arrow node))
	          (remove node nodes-to-unmap)) ; check this...
	      (t (if (eq node *EXPL-NODE*)
		     (format t "Error:  *Expl-node* not drawn~%"))
                 nodes-to-unmap))) 
      nodes-to-unmap))



(defun last-node-to-draw (node root)
"Returns t if the given node is the last one to be displayed:
 the node is last if it has no children and is not a backtracking node"
   (or (null node) 
       (get (node-name node) 'last-node-to-draw)
       (and (eq (leftmost-clone node) (leftmost-clone root))
            (applied-child (node-children node))
	    (not (eq node root)))
       (and (null (node-children node))
	    (not (backtrack-node-p node)))  ; -mpm 1/25/88

))


;--------------------------------------------------------------------------


(defun fully-in-tree-window (node)
  (declare (type node node))
  "Returns t if the lower right-hand corner
   of the node is in the tree window"
    (let ((x (get-x node))
	  (y (get-y node)))
      (declare (fixnum x y)
	       (special *NODE-WIDTH* *TREE-WIN-WIDTH*
			*NODE-HEIGHT* *TREE-WIN-HEIGHT*))
      (and (in-tree-window x y)
	   (<= (+ x *NODE-WIDTH*) *TREE-WIN-WIDTH*) 
	   (<= (+ y *NODE-HEIGHT*) *TREE-WIN-HEIGHT*))))


;--------------------------------------------------------------------------


(defun draw-backtrack-path (new-child old-node)
  (declare (special n1)
	   (type node new-child old-node n1))
  "DRAW-BACKTRACK-PATH redisplays the tree expanding the 
   backtracking path originating at node new-child."
  (cond ((not (and (node-rect new-child)
		   (backtrack-node-p new-child)))
	 (format t "Node ~A is not a backtrack node!" new-child) (terpri))
	(t ;(pg-clear-window *TREE-WINDOW*)
	   (let* ((children (node-children (node-parent new-child)))
		  (old-child (find-active-child children))
		  (root (if (v-pre-parent old-node) (v-pre-parent old-node) n1))
		  (root-level (1- (get-level root)))
		  (oldmapped (reverse (mapped-nodes))))
	     (declare (type node root old-child))
	     (unmap-all-nodes)
	     (clear-current-path old-child)
	     (set-backtrack-node old-child t)
	     (set-backtrack-node new-child nil)
	     (set-node-level new-child)  
	     (set-node-level old-child)
	     (reset-current-path new-child root-level root)
	     (setf *LEVELS* nil)
	     ; must reset the levels so that a prior backtrack node who is
	     ; a left-clone will be placed next to his clone instead of
	     ; below his parent (which is the default for all backtrack nodes)
	     ; Similarly, we must reset the old node which will now
	     ; become a backtrack node so that he will be drawn
	     ; below his parent even if he has a left-clone
             (gchangerect (get-rectangle new-child) *NODE-WIDTH* *NODE-HEIGHT*)
	     (gchangerect (get-rectangle old-child) *BACKTRACK-WIDTH* 
			    *BACKTRACK-HEIGHT*)
	     (draw-nodes root-level root oldmapped)))))



;--------------------------------------------------------------------------


; A couple of lines commented out for new graphics.

(defun refresh-tree (node)
  (declare (type node node))
  "Raises the tree rectangle and redisplays the tree at the given node."
  (cond (*TREE-GRAPHICS*
           (redisplay-tree node (boundp '*PREVIOUS-TREE-ROOT*))
           (highlight-node node))))


(defun redisplay-tree (node &optional (refresh-mode nil))
  (declare (type node node n1)
	   (special n1))
  "REDISPLAY-TREE redisplays the tree at the given node. 
   The root is set to be the virtual pre-parent of the 
   given node. The root defaults to be n1 since the only 
   nodes without a v-pre-parent should be right-clones of n1."

  (pg-with-window *TREE-WINDOW* ; -rlj-
    (pg-clear-window *TREE-WINDOW*)
    (let* ((root (cond (refresh-mode *PREVIOUS-TREE-ROOT*)
                       (t (if (v-pre-parent node) (v-pre-parent node) n1))))
           (root-level (1- (get-level root)))
           (oldmapped (reverse *MAPPED*)))
      (declare (type node root)
	       (fixnum root-level))
      ; puts nodes in increasing order to make search more efficient
      (unmap-all-nodes)
      (setf *LEVELS* nil)
      (setf *TREE-ROOT* root)
      (setf *PREVIOUS-TREE-ROOT* *TREE-ROOT*)
      (draw-nodes root-level root oldmapped)
      (pg-refresh-window *TREE-WINDOW*))))

;--------------------------------------------------------------------------


(defun kill-tree-graphics ()
  (declare (special *TREE-WINDOW*))
  "KILL-TREE-GRAPHICS destroys the tree rectangle and 
   sets the atoms associated with each node rectangle 
   to nil so that these windows cannot be accessed by 
   mistake in another run."

   (pg-kill-window *TREE-WINDOW*)
   (setf *TREE-GRAPHICS* nil)
)

(defun reset-display-tree ()
  (declare (special *TREE-NODES* *MAPPED*))
   (dolist (node *TREE-NODES*) 
        (declare (type node node))
	(remprop (node-name node) 'rect)
        (remprop (node-name node) 'last-node-to-draw)
	     ; This property will only be on a few nodes, but so what?
	(setf *TREE-NODES* nil)
	(setf *MAPPED* nil)
))    


;***********************************************************************
;                 Functions to draw the arrows between nodes   
;                       (a.k.a. BIOGRAPHY OF AN ARROW)
;***********************************************************************
;                  
;                        side1
;             tail ------->-------  head
;                        side2
;
;
; The algorithm for drawing an arrow:
;
;     1. The head is placed at the upper [lower] left corner of the node
;        if the arrow is pointing down [up].
;     2. The tail is placed at the upper [lower] left corner of the 
;        parent if the arrow is pointing up [down].       
;     3. The point where the two sides of the arrow meet (the 
;        "junction") is calculated as the midpoint of the arrow line. 
;     4. The length of each of the sides of the arrow is the constant 
;        *ARROW-SIDE-LEN*
;     5. The angle between each side and the line is the constant 
;        *ARROW-ANGLE*
;     6. The function "get-line-dist" computes the distance from the 
;        junction to the point on the arrow which lies on the same line
;        as the endpoints of the sides. (A line orthogonal to the arrow)
;     7. "Get-one-point-at-dist" determines the coordinates of this
;         point, ``line-point''.
;     8. The function "get-side-dist" computes the distance from this 
;        point to each of the endpoints of the arrow sides.
;     9. Since we know the slope "m" of the arrow line, we know that the
;        slope of the orthogonal line is -1/m. And we know a point on 
;        the orthogonal line, ``line-point'', so we now know the equation
;        of the orthogonal line. The two endpoints of the sides are the
;        points which lie on this line at distance ``side-dist'' from
;        `line-point'.
            								    

(defun draw-arrow (node)
  (declare (type node node))
  "Draws an arrow from a node's parent to itself"
   (let* ((x (get-x node))
          (y (get-y node))
	  (parent (node-parent node))
	  (parent-x (get-x parent))
	  (parent-y (get-y parent)))
     (declare (fixnum x y parent-x parent-y))
      (if (and parent (in-tree-window x y) 
                      (in-tree-window parent-x parent-y))
          (make-arrow node x y parent-x parent-y))))


;--------------------------------------------------------------------------  

(defstruct two-sides side1 side2)
(defstruct one-side
  (x 0 :type fixnum)
  (y 0 :type fixnum))

;(proclaim '(function one-side-x (one-side) fixnum))
;(proclaim '(function one-side-y (one-side) fixnum))

(defun make-arrow (node node-x node-y parent-x parent-y)
  (declare (type node node))
   (let* ((down (< parent-y node-y))
          (head-x (arrow-head-x node-x))
          (head-y (arrow-head-y down node-y))
 	  (tail-x (arrow-tail-x node parent-x))
  	  (tail-y (arrow-tail-y down parent-y))
	  (mid-x (midpoint head-x tail-x))
          (mid-y (midpoint head-y tail-y))
	  (m (slope mid-x mid-y tail-x tail-y))
          (arrow-sides (get-arrow-sides m mid-x mid-y down))
	  (side1 (two-sides-side1 arrow-sides))
	  (side2 (two-sides-side2 arrow-sides)))
     (declare (fixnum head-x head-y tail-x tail-y mid-x mid-y))
	 (pg-draw-line *TREE-WINDOW* head-x head-y tail-x tail-y)
	 (pg-draw-line *TREE-WINDOW* mid-x mid-y (one-side-x side1) 
		                                 (one-side-y side1))
	 (pg-draw-line *TREE-WINDOW* mid-x mid-y (one-side-x side2) 
		                                 (one-side-y side2))))

(defun midpoint (x1 x2)
   (declare (fixnum x1 x2))
   (+ x1 (truncate (- x2 x1) 2)))

(defun arrow-head-x (x)
  (declare (fixnum x))  
   (+ *ARROW-X-OFFSET* x))

(defun arrow-tail-x (node parent-x)
  (declare (type node node)
	   (fixnum parent-x))
   (let ((grandparent (node-parent (node-parent node))))
        (declare (type node grandparent))
	(cond ((and grandparent    
                    (eq (get-level grandparent) (get-level node)))
               (+ *ARROW-X-OFFSET* *ARROW-X-OFFSET* parent-x))
	      (t (+ *ARROW-X-OFFSET* parent-x)))))

	
(defun arrow-head-y (down y)
  (declare (fixnum y))
    (if down y (+ y *NODE-HEIGHT*)))


(defun arrow-tail-y (down y)
  (declare (fixnum y))
    (if down (+ y *NODE-HEIGHT*) y))

#|
(defun get-arrow-sides (m x1 y1 down)
 (declare (fixnum x1 y1)) 
   (let ((line-point 
            (get-one-point-at-dist (get-line-dist) m x1 y1 down)))
        (get-points-at-dist (get-side-dist) m (car line-point) 
	                                      (cdr line-point))))
|#
(defun get-arrow-sides (m x1 y1 down)
   (declare (fixnum x1 y1))
   (multiple-value-call #'get-points-at-dist (get-side-dist) m
			(get-one-point-at-dist (get-line-dist) m x1 y1 down)))

;--------------------------------------------------------------------------  

(defun degrees-to-radians (degrees)
   (/ (* degrees pi) 180))


(defun get-line-dist ()
   (values
    (round (* *ARROW-SIDE-LEN* (cos (degrees-to-radians *ARROW-ANGLE*))))))
   

(defun get-side-dist ()
  (values
   (round (* *ARROW-SIDE-LEN* (sin (degrees-to-radians *ARROW-ANGLE*))))))

;--------------------------------------------------------------------------  
;; Perhaps line-x1 and line-x2 should always be fixnums instead of floats.
(defun get-one-point-at-dist (dist m x1 y1 down)
  "Returns a ONE-POINT structure that indicates the position of the
   of the flange of the arrow head."
  (declare (fixnum dist x1 y1))
  (cond ((null m) 
	 (values x1 (if down (- y1 dist) (+ y1 dist))))
	(t (let* ((y-int (get-y-intercept x1 y1 m))
		  (a (A m))
		  (b (B x1 y1 m y-int))
		  (c (C x1 y1 dist y-int))
		  (line-x1 (round (quadratic-root1 a b c)))
		  (line-y1 (round (solve-for-y m line-x1 y-int))))
	          (declare (fixnum line-y1))
	     (cond ((between-end-points line-x1 line-y1 y1 down) 
		    (values line-x1 line-y1))
		   (t (let* ((line-x2 (quadratic-root2 a b c))
			     (line-y2 (round (solve-for-y m line-x2
							  y-int))))
			(declare (fixnum line-y2))
			(cond ((between-end-points (round line-x2) line-y2 y1 
						   down)
			       (values (round line-x2) line-y2))
			      (t (error "Both roots of quadratic are not in the
			      first quadrant!"))))))))))

;--------------------------------------------------------------------------  
  	 
(defun between-end-points (x y end-y down)
  (declare (fixnum x end-y))
   (and (>= x 0) (>= y 0) (if down (<= y end-y) (>= y end-y))))
    
;--------------------------------------------------------------------------  

(defun get-points-at-dist (dist m x1 y1)
   "Returns a TWO-SIDES structure that holds the points for the tips
    of the flanges of both sides of the arrow head."
   (declare (fixnum dist x1 y1))
   (cond ((or (null m) (zerop m))
          (make-two-sides :side1 (make-one-side :x (- x1 dist) :y y1) 
			  :side2 (make-one-side :x (+ x1 dist) :y y1)))
	 (t (let* ((orthogonal-m (- (/ m)))
	           (y-int (get-y-intercept x1 y1 orthogonal-m))
		   (a (A orthogonal-m))
		   (b (B x1 y1 orthogonal-m y-int))
		   (c (C x1 y1 dist y-int))
		   (line-x1 (quadratic-root1 a b c))
		   (line-x2 (quadratic-root2 a b c))
		   (line-y1 (round (+ (* orthogonal-m line-x1) y-int)))
		   (line-y2 (round (+ (* orthogonal-m line-x2) y-int))))
	      (declare (fixnum line-y1 line-y2))
		  (make-two-sides 
		   :side1 (make-one-side :x (round line-x1) :y line-y1) 
		   :side2 (make-one-side :x (round line-x2) :y line-y2))))))

;--------------------------------------------------------------------------  

(defun A (m)
   (1+ (* m m)))


(defun B (x y m b)
   (declare (fixnum x y))
   (+ (* -2 x) (* 2 m b)  (* -2 m y)))


(defun C (x y d b)
  (declare (fixnum x y d))
  (- (+ (* d d)  (- (* x x))  (- (* y y))  (* 2 b y)  (- (* b b)))))

;--------------------------------------------------------------------------  
; To get around math bugs in the bignum routines in CMU common lisp 
; I need to add abs to the square of b and use isqrt instead of sqrt.
; I also changed the test on root-of to be minusp, which should be
; more efficient than less-than 0.  --dkahn


(defun quadratic-root1 (a b c)
  (let ((root-of (truncate (- (* (abs b) (abs b)) (* 4 a c)))))
       (if (minusp root-of)
           (error "Attempt to take square root of negative number!"))
       (/ (+ (- b) (isqrt root-of)) (* 2 a))))


(defun quadratic-root2 (a b c)
  (let ((root-of (truncate (- (* (abs b) (abs b)) (* 4 a c)))))
       (if (minusp root-of) 
           (error "Attempt to take square root of negative number!"))
       (/ (- (- b) (isqrt root-of)) (* 2 a))))

;--------------------------------------------------------------------------  

(defun solve-for-y (m x b)
   (declare (fixnum x b))
   (+ (* m x) b))


(defun slope (x1 y1 x2 y2)
   (declare (fixnum x1 y1 x2 y2))
   (cond ((= x1 x2) nil)
         (t (/ (- y2 y1) (- x2 x1)))))

;; The returned value cannot be assumed to be a fixnum.
(defun get-y-intercept (x y m)
   (declare (fixnum x y))
   (if m (- y (* m x)) nil))

;--------------------------------------------------------------------------  

;; SET-NODE-LEVEL determines the level in the tree on which the node should 
;; be printed.  Beginning at level 1 (for node n1), each subgoalling node 
;; is printed one level down and each applied node is printed one level up.
;;
;; Ex.      n1   n3       n1 and n3 are at level 1
;;           \  /         n2 is at level 2
;;            n2
;; The level is computed in the following way:
;; If a node has a left-clone and it is not a backtracking node
;;   then it is printed on the same level as its left-clone
;; Else if a node has a parent  (all nodes except n1)
;;        then it is printed on level below its parent
;;        else (must be n1) it is printed at level 1

(defun set-node-level (node)
   (declare (type node node))
   (setf (get-level node) (calculate-level node (backtrack-node-p node))))

; Calculate level was modified so that if the nodes have not all been
; initialized the node is effectively assumed to have a
; bactracking-node-p of nil, which is kind of ugly.

(defun calculate-level (node bt-p)
   (declare (type node node))
   "This will calcuate what the level of a node should be."
   (cond ((and (not bt-p)
	       (node-left-clone node))
	   (get-level (node-left-clone node)))
	  ((node-parent node)
	   (1+ (get-level (node-parent node))))
	  (t 1))) ; node n1.



;--------------------------------------------------------------------------  

(defun set-node-xy (node level top-node)
  (declare (type node node top-node))
	   (declare (fixnum level))
  "Sets the x and y coordinates of a node"
     (let* ((r (node-rect node))
	    (h (- (node-rect-bottom r) (get-y node)))
	    (w (- (node-rect-right r) (get-x node))))
     (declare (fixnum h w)
	      (type node-rect r))
      (setf (get-x node) (calc-x node level top-node))
      (setf (get-y node) (calc-y level))
      (gchangerect r w h)))


;--------------------------------------------------------------------------  

;; SET-LEVELS updates the global variable *LEVELS* to include the new x 
;; value at the given level.  *LEVELS* is a list of the maximum x coordinate 
;; so far at a level.  That is, the Nth position in the list contains the 
;; x coordinate of the rightmost rectangle so far on level N.  For example, the
;;  list: (2 5 1) means that 
;;            the rightmost rectangle on level 1 is at x=2
;;            the rightmost rectangle on level 2 is at x=5
;;            the rightmost rectangle on level 3 is at x=1
;; 

(defun set-levels (level x)
  (declare (fixnum level x))
  (cond ((nthcdr (1- level) *LEVELS*); is there already a value at this level?
	 (setf (nth (1- level) *LEVELS*) x))
	(t (setf *LEVELS* (append *LEVELS* (list x)))))
  x)

;--------------------------------------------------------------------------  

;; PLACE-UNDER-PARENT  attempts to place a node directly under its parent.
;; Leftmost-x the first available x position on the node's level (i.e. there
;; are other nodes to the right of this position).  If this position is
;; greater than the parent's x position, then this is the best we can do and
;; so we return this x value. Otherwise, we return the parent's x coordinate.


(defun place-under-parent (node leftmost-x)
  (declare (type node node)
	   (fixnum leftmost-x))
   (let ((parent-x (get-x (node-parent node))))
     (max leftmost-x parent-x)))
;	(if (< leftmost-x parent-x) parent-x leftmost-x)))


;--------------------------------------------------------------------------  



(defun calc-x (node level top-node)
  (declare (type node node top-node))
 "CALC-X  Calculates the x coordinate for a node.  

 Cases: 
  1) the first node is placed at the window's origin
  2) a non-backtracking node with a left-clone is offset slightly to the
     right of its left-clone
  3) if there other nodes on the same level then it is placed either
     directly under its parent or as close to its parent as possible
  4) Default: placed directly under its parent"

  (if (> level 0)
      (cond  ((eq node top-node) (set-levels level *TREE-X-OR*))
	     ((and (node-left-clone node) 
		   (get (node-name node) 'rect) ; added by dkahn
		   (not (backtrack-node-p node)))
	      (set-levels level (+ (get-x (node-left-clone node))
				   *CLONE-OFFSET*)))
	     ((nth (1- level) *LEVELS*)
	      (set-levels 
	       level 
	       (place-under-parent 
		node
		(+ (nth (1- level) *LEVELS*) *NODE-X-OFFSET*))))
	     (t (set-levels level (get-x (node-parent node)))))
      -1))    ; -1 is to denote that it shouldn't be in window



(defun calc-y (level)
  (declare (fixnum level))
  "Calculates the y coordinate of a node."
  (let ((y  (+ *TREE-Y-OR* (* (1- level) *NODE-Y-OFFSET*))))

    y))

;=========================================================================
;                     End  of  tree-graphics.lisp
;=========================================================================
