#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#



; ======================================================================
; File:  matching.lisp		Version: 1-20 	     Created:  ....
;
; Locked by: nobody                                 Modified:  5/13/88
; 
; Purpose:    Contains the match routines for PRODIGY. 
;
; Changes:
;
; 5/13/88 - mpm - added meta-func/pred conditionals to (get-func-vals),
;                 because it signalled errors on the bindings for KNOWN, 
;                 and CANDIATE-BINDINGS.
;
; ======================================================================

;

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types"))


; note: I should combine with newtrack and make
; gen-values and negation match localf again - SNM

(proclaim 
    '(special *ACHIEVABLE-PREDS* *NEGATABLE-PREDS* *PREDICATES* *CURRENT-NODE*
              *FUNCTION-PREDS* *CLOSED-PREDS* *STATIC-PREDS* *MATCH-COUNT* 
	      *SCR-TRACING* *OP-TRACING* *PRINT-TRACING* *META-FUNCTIONS*
	      *EBL-FLAG* *MATCHER-TRACING* *MATCHER-TRACE* *FAILURE-RECORDING* 
	      *FAILURE-RECORD*  *STATE* *INFERENCES-ONLY* *MATCH-EXPLANATION* 
	      *SUCCESS-BINDINGS* *EXPLAIN-MATCH-FAILURES* *ABSTRACTION-LEVEL*))

 
; --------  Syntax of Expressions, data-structures ------------
; 		(See file data-types.l also)

;  A bindings-list looks like ((var val)(var val)...(nil nil))
;  The last pair in the list is usually (nil nil), by convention,
;  but this is optional.
; 
; Conventions: a bindings list will never be nconced, but
; a list of binding lists may. (Functions return a
; list of binding lists, which should be mungable on the top level).
; 
 
; EXP-MATCH is the top-level call to the matcher. After
; matching an expression, *FAILURE-RECORD* contains a list of unmatched 
; literals that the matcher failed on.
; If a match is found, EXP-MATCH returns non-nil value - a
; list of bindings-lists that the expression matches
; with. (NOTE: only bindings for existentially quantified variables are 
; returned). 

(defun exp-match (exp in-bindings state)
  (setq *STATE* state
	*FAILURE-RECORD* nil
	*FAILURE-RECORDING* t)
  (cond (*OP-TRACING* 
	 (setq *MATCHER-TRACING* t)
	 (set-up-matcher-trace exp))
	(t (setq *MATCHER-TRACING* nil)))
  (descend-match exp (or in-bindings (list '(nil nil))) 'all))


; also a top level match but
;  we don't bother doing failure recording, and don't need state,
; since we are really matching agains the state of the planner.

(defun scr-match (rule-nm exp in-bindings)
    (cond ((and *ABSTRACTION-LEVEL*
		(scr-abs-level rule-nm)
		(not (eq (scr-abs-level rule-nm) *ABSTRACTION-LEVEL*)))
	   nil)
	  ((and in-bindings exp)
	   (setq *STATE* nil
		 *FAILURE-RECORD* nil
		 *FAILURE-RECORDING* nil)
	      (cond (*SCR-TRACING* 
			(setq *MATCHER-TRACING* t)
			(set-up-matcher-trace exp))
		    (t (setq *MATCHER-TRACING* nil)))
	   (descend-match exp in-bindings 'all))))

(defun set-up-matcher-trace (exp)
    (push (list 'expression exp) *MATCHER-TRACE*)
    (cond ((eq *PRINT-TRACING* t)
	   (terpri)
	   (terpri)
	   (format t "Matching expression:  ~A" exp))))



; Also a top-level-function, a quicker form of exp-match used when the exp is
; an atomic-formula.

(defun atomic-formula-match (exp state)
    (setq *STATE* state)
    (setq *MATCH-COUNT* 0)
    (let ((values (gen-values exp nil)))
	 (cond ((eq values 'no-match-attempted) nil)
	       ((eq values t) (list '((nil nil))))
	       (values))))

; Also a top-level-function, a quicker form of exp-match used when the exp is
; an instantiated literal, returns t or nil

(defun lit-not-true (lit state)
    (setq *STATE* state)
    (cond ((negated-p lit)
	   (cond ((function-p (cadr lit))
		  (get-func-vals (cadr lit) '((nil nil)) t))
		 ((member-state-inst (cadr lit) 
				     (state-true-assertions *STATE*)
				     nil))))
	  ((function-p lit)
	   (not (get-func-vals lit '((nil nil)) t))) ; a bit wasteful
	  ; if no-match-attempted, lit assumed to be true 
	  (t (not (member-state-inst lit 
				     (state-true-assertions *STATE*)
				     t)))))


; The *FAILURE-RECORD* contains a list of predicates that caused
; and expression to fail to match. For example, if the
; expression is: (exists (<sq>)(is-square <sq>)(empty <sq>)
; and there is no empty square, then (empty <sq>) will
; *FAILURE-RECORD* contain (empty <sq>) as a member.
; Note: Sometimes an expression will fail to match because 
; a predicate is true - In this case, FAILURE record
; contains the negated predicate.

;  RECORD-FAILURE updates the *FAILURE-RECORD*, assuming the flag
;  *FAILURE-RECORDING* is set and the lit is possible to achieve

(defun record-failure (lit bindings) 
    (cond (*MATCHER-TRACING* 
	      (cond ((eq *PRINT-TRACING* t)
		     (terpri) 
		     (format t "  Matcher Failed on: ~A" lit)
		     (terpri)
		     (format t "  bindings: ~A" bindings)))
	      (push (list lit bindings) *MATCHER-TRACE*)))
    (cond (*FAILURE-RECORDING* 
	      (cond ((and *INFERENCES-ONLY*      ;should put this somewhere in
			  (member (car lit) *CLOSED-PREDS*)))         ; engine
		    ((and *EXPLAIN-MATCH-FAILURES*
			  (push (list lit bindings)
				*MATCH-EXPLANATION*)
                          nil))
		    ((impossible-to-achieve-lit lit))
		    ((push (list lit bindings) *FAILURE-RECORD*)))))
    nil)


;  This is the recursive match routine that does all the work. 
;  (Can make more efficient by remmoving atomic formula test & moving to rear)


(defun descend-match (exp bindings mode)
;    (and *EBL-FLAG* (setq *MATCH-COUNT* (1+ *MATCH-COUNT*)))
    (cond ((eq exp t) (list bindings))
	  ((eq (car exp) 'and)
	   (and-match (cdr exp) bindings mode))
	  ((eq (car exp) 'exists)	       
	   (exists-match bindings (get-gen-exp exp) (get-exp exp) mode))
	  ((eq (car exp) 'forall)
	   (forall-match bindings (get-gen-exp exp) (get-exp exp)))
	  ((eq (car exp) '~)
	   (cond ((negation-match (cadr exp) bindings))
		 (t (record-failure exp bindings))))
	  ((eq (car exp) 'or)
	   (or-match (cdr exp) bindings mode))
	  ((null exp) (list bindings))
	  ; must be an atomic formula
	  ((let ((vals (gen-values exp bindings)))
		(cond ((null vals)
		       (record-failure exp bindings))
		      ((eq vals 'no-match-attempted)
		       nil)
		      ((eq vals t)
		       (list bindings))
		      ((mapcar #'(lambda (b) (append b bindings))
			       vals))))))) ; should replace with nconc 


; at present time, exp must be an non-negated literal with all vars bound
; or an expression of the form (exists ...)
; Don't need mode flage, since we know we only have to look for a single
; bindings

(defun negation-match (exp bindings)
    (cond ((eq exp t) nil)
	  ((eq (car exp) 'exists)
	   (negated-exists-match exp bindings))
	  ((member (car exp) *PREDICATES*)
	   (negated-pred-match exp bindings))
	  (t (error "NEGATION-MATCH: bad expression ~A" exp))))


(defun negated-exists-match (exp bindings)
    (prog (new-bindings matched-lits)
	  (setq *FAILURE-RECORDING* nil)
	  (setq new-bindings (car (descend-match exp bindings 'one)))
	  (setq *FAILURE-RECORDING* t)
	  (cond ((not new-bindings) (return (list bindings)))
		((setq matched-lits 
		       (mapcan #'(lambda (lit) 
				   (cond ((not (impossible-to-achieve-lit 
						   `(~  ,lit)))
					  (list lit))
					 (t nil)))
			       (get-lits (subst-bindings (get-exp exp) 
					     new-bindings))))
		 (dolist (matched-lit matched-lits nil)
		     (record-failure `(~ ,matched-lit) new-bindings)))
		(t (record-failure 
		       `(~ ,(subst-bindings (get-gen-exp exp) new-bindings))
		       new-bindings)))))
								 
		 
; exp = (P x y...)
; Returns bindings if exp is known to be  False.

; should do a test to make sure all vars are bound!

(defun negated-pred-match (exp bindings)
    (cond ((function-p exp)
	   (and (not (get-func-vals exp bindings 'no-match-attempted))
		(list bindings)))
	  ((member (car exp) *CLOSED-PREDS*)
	   (cond ((has-vars exp) 
		  (and (not (member-state 
			     exp (state-closed-world *STATE*) bindings nil))
		       (list bindings)))
		 ((not (member-state-inst exp (state-closed-world *STATE*) nil))
		  (list bindings))))
	  ((member-state exp (state-false-assertions *STATE*) bindings t)
	   (list bindings))))

; should make the independence test in and-match
; more sophisticated, ack-- cant do independent cause
; record-failure loses the bindings from the other clause.
; leads to funny behavior....

; Exps = (sub-exp sub-exp ....)
; if all sub-exps in exps match, then their bindings are combined and 
; returned. Better not be two variables with the same name in exp.


(defun and-match (exps bindings mode)
    (cond ((null exps) (list bindings))
	  ((null (cdr exps))
	   (nreverse (descend-match (car exps) bindings mode)))
	  ((g-loop (init new-bls (descend-match (car exps) bindings mode)
			  ret-val nil)
	            (while new-bls)
		    (do (setq ret-val 
			      (nconc (and-match (cdr exps) (pop new-bls) mode)
				     ret-val)))
		    (until (and ret-val (eq mode 'one)))
		    (result ret-val)))))

;	  ((let ((new-bls (descend-match (car exps) bindings mode)))
;		(mapcan #'(lambda (new-bindings)
;				  (and-match (cdr exps) new-bindings mode))
;			new-bls)))))

; 
;(defun and-match (exps bindings mode)
;    (cond ((null exps) (list bindings))
;	  ((null (cdr exps))
;	   (descend-match (car exps) bindings mode))
;	  ((let ((new-bls (descend-match (car exps) bindings mode)))
;		(and new-bls
;		     (cond ((atomic-formula-p (car exps))
;			    (mapcan #'(lambda (new-bindings)
;					      (and-match (cdr exps)
;						  new-bindings mode))
;				    new-bls))
;			   ((let ((second-bls 
;				      (and-match (cdr exps) bindings mode)))
;				 (and second-bls
;				      (g-loop (init new nil ret-val nil)
;					       (while (setq new (pop new-bls)))
;   					       (do (setq new (ldiff new 
;								    bindings))
;						   (dolist (second bls)
;							   (push (nconc new
;									second)
;								 ret-val)))
;                                               (result ret-val)))))))))))
 

; OR-MATCH returns the bindings of the 1st sub-exp to match.

(defun or-match (exps bindings mode)
    (g-loop (init subresult nil exp nil)
	     (while (setq exp (pop exps)))
	     (do (setq subresult (remove-duplicates
			(nconc (descend-match exp bindings mode)
					subresult) :test #'equal)))
	     (until (and (eq mode 'one) subresult))
	     (result subresult)))


; Matches a universally quantified expression. Returns start-bindings
; on success (unless no variables are bound,in which case lower-level bindings
; are also returned.
; 
(defun forall-match (start-bindings gen-exp exp)
    (g-loop (init values (gen-values gen-exp start-bindings) bindings nil)
	     (before-starting           ; make sure values are good
		 (cond ((null values) (return (list start-bindings)))
		       ((eq values 'no-match-attempted) (return nil))
		       ((eq values t)
			(return (descend-match exp start-bindings 'one)))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do (cond ((not (descend-match exp bindings 'one))
			(or (static-p gen-exp)
			    (record-failure (list '~ gen-exp) bindings))
			(return))))
	     (result (list start-bindings))))


; Matches an existentially quantified expression. Returns start-bindings,
; with the new var-value pairs appended on front. 

(defun exists-match (start-bindings gen-exp exp mode)
    (g-loop (init values (gen-values gen-exp start-bindings)
		   subresults nil bindings nil)
	     (before-starting 
		 (cond ((null values)
			(record-failure gen-exp start-bindings) (return))
		       ((eq values 'no-match-attempted) (return))
		       ((eq values t)
			(return (descend-match exp start-bindings mode)))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do (setq subresults (nconc (descend-match exp bindings mode)
					 subresults)))
	     (until (and subresults (eq mode 'one)))
	     (result subresults)))



; Returns  a non-nil value if lit is definitely NOT true.
; (Lit can contain variables).

(defun not-in-database (lit bindings)
    (cond ((negated-p lit)
	   (cond ((and (closed-predicate (cadr lit)) (has-vars (cadr lit)))
		  nil)
		 (t (not (in-database (cadr lit) bindings)))))
	  ((function-p lit)
	   (not (get-func-vals lit bindings t)))         ; a bit wasteful
                   	  ; if no-match-attempted, lit assumed to be true 
	  ((has-vars lit)
	   (not (member-state lit (state-true-assertions *STATE*)
			      bindings nil)))
	  (t (not (member-state-inst lit (state-true-assertions
					  *STATE*) nil)))))
 	  
;  tests a literal, which may or may not be completely instantiated
;  to see if it is either in the database, or if a function, whether
;  it has a value. Returns a non-nil value if the literal is definitely
;  true.


(defun in-database (lit bindings)
    (cond ((function-p lit)
	   (get-func-vals lit bindings nil))         ; a bit wasteful
	  ((negated-p lit)
	   (negation-match (cadr lit) bindings))
	  ((has-vars lit)
	   (member-state lit (state-true-assertions *STATE*) bindings t))
	  ((member-state-inst lit (state-true-assertions *STATE*) t))))


;  GEN-VALUES generates the values for a generator expression
;  If there are variables in the expression, a list of bindings
;  lists for the new variables is returned. Otherwise, t or nil
;  is returned, depending on whether the expression matches.

; For now, we'll make some explicit assumptions about gen-exp,
; if we expect the failure recording stuff to work right then
; it should be a non-negated literal. (Only important when
; matcher is called by engine).

; Note: If var is actually a constant, then gen-exp has already been
; at least partially instantiated
; Note: If no-match-attempted on a function-pred, then this 
; returns 'no-match-attempted

;  ASSUMEs only (meta-level) functions can have internal variables....

(defun gen-values (gen-exp bindings)
    (cond ((function-p gen-exp)
	   (get-func-vals gen-exp bindings 'no-match-attempted))
	  ((not (atomic-formula-p gen-exp))
	   (remove-extra-ts
	       (mapcar #'(lambda (b) (diff-bindings b bindings))
		       (descend-match gen-exp bindings 'all))))
	  ((and *ABSTRACTION-LEVEL*
		(not (member (extract (strip-negation gen-exp))
			     *ABSTRACTION-LEVEL* 
			     :test #'equal))))
	  ((has-unbound-vars gen-exp bindings)
	   (get-pred-values gen-exp bindings))
	  ((member-state-inst (subst-bindings gen-exp bindings)
			      (state-true-assertions *STATE*)
			      t)
	   t)
	  (t nil)))

; removes all t's from list of binding-lists, but if they're all t, 
; returns t

(defun remove-extra-ts (bls)
    (and bls
	 (let ((new-bls (mapcan #'(lambda (b) (cond ((not (eq b t)) (list b))
						    (t nil)))
				bls)))
	      (cond ((null new-bls) t)
		    (new-bls)))))
    
;  assumes newbindings have been consed on front.
;  NOTE: if oldb is not a tail of newb then ldiff returns newb.
;  In franz lisp, ldiff will produce an error.

(defun diff-bindings (newb oldb)
    (cond ((equal newb oldb) t)
	  ((ldiff newb oldb))))			 

;  Call Craig's code for function preds. The No-match-attempted-flag
;  tell what to return if there was no match attempted.

(defun get-func-vals (exp bindings no-match-attempted-flag)
  (let ((bls (cond ((has-any-unbound-vars exp nil)
		    (apply (car exp) (subst-bindings (cdr exp) bindings)))
		   ((apply (car exp) (cdr exp))))))
    (cond ((eq bls 'no-match-attempted) no-match-attempted-flag)
	  ((list-of-binding-lists-p bls)  bls)
	  ((predicate-value bls) bls)
	  ((member (car exp) *META-FUNCTIONS*)
	   bls)
	  (t (error 
	      "~3% Error: The function `~@(~A~)' does not return a~
	      ~% proper list of binding lists, or predicate value.~%"
	      (car exp))))))


(defun predicate-value (exp)
  (or (null exp) (eql exp t)))


(defun list-of-binding-lists-p (exp)
; is expression of the form (((<var> val) ...)))
  (and (listp exp) (every #'binding-list-p  exp) t))


(defun binding-list-p (exp) 
  (and (listp exp) (every #'binding-p exp) t))


(defun binding-p (x)
  (and (listp x) 
       (symbolp  (car x))
       (cadr x)
       (or (atom (cadr x))
	   (listp (cadr x)))
       (null (cddr x))
       t))




(defun get-pred-values (literal bindings)
; assumes that there are unbound vars in literal.
    (setq literal (subst-bindings literal bindings))
    (mapcan #'(lambda (assertion)
		      (let ((binding (lit-match3 literal assertion)))
			   (cond (binding (list binding))
				 (t nil))))
	    (state-true-assertions *STATE*)))


;  Tests whether spec matches any member of state. Returns original bindings.

(defun member-state (spec state bindings success)
  (cond (*ABSTRACTION-LEVEL*
	 (let ((ispec (subst-bindings spec bindings)))
	   (cond ((not (has-vars ispec))
		  (member-state-inst ispec state success))
		 ((not (member (extract (strip-negation spec))
			   *ABSTRACTION-LEVEL* 
			   :test #'equal))
		  success)
		 (t (g-loop (while state)
			    (do (and (pred-match-p spec (car state) bindings) 
				     (return bindings)))
			    (next state (cdr state))
			    (result nil))))))
	 (t (g-loop (while state)
		    (do (and (pred-match-p spec (car state) bindings) 
			     (return bindings)))
		    (next state (cdr state))
		    (result nil)))))

; Same as member-state except it takes a fully instantiated literal.
(defun member-state-inst (literal state success)
  (cond ((and *ABSTRACTION-LEVEL*
	      (not (member (extract-instance
			    (strip-negation literal))
			   *ABSTRACTION-LEVEL* 
			   :test #'equal)))
	 success)
	(t (member literal state :test #'equal))))


; matches predicates, eg. exps such as (AT RMG <y>), ie. literals
; that are not negated.
; All values must be variables or constants.
; you have to call match with a non-nil bindings list, otherwise it'll die

(defun pred-match (spec obj bindings)
    (cond ((eq (car spec) (car obj))
	   (setq spec (cdr spec))
	   (setq obj (cdr obj))
	   (g-loop (while spec)
	            (do (setq bindings 
			      (value-match (car spec) (car obj) bindings)))
		    (until (null bindings))
		    (next spec (cdr spec) obj (cdr obj))
		    (result bindings)))
	  (t nil)))

; returns t if spec (a predicate, can have vars) matches obj.


; The pred-match-p function takes a spec (a literal with its vars)
; and obj (fully instantiated) and compares them using the bindings 
; to instantiate spec.  If spec turns out to be partially instantiated
; (i.e. the binding list doesn't have a value for a particular
; variable in spec)  then that variable will match any value in the
; corresponding position in obj.  This "wild card" feature was fixed
; by dkahn in May '90 by using tree-equal instead of equal.  This
; should permit the use of negated partially instantiated predicates.

;A variable in spec will match any value for its corresponding argument in obj.



(defun pred-match-p (spec obj bindings)     ;; 1/31/88 -snm
    ; first check that they are the same predicates.
    (cond ((eq (car spec) (car obj))
	   (setq spec (cdr spec))
	   (setq obj (cdr obj))
	   ; now begin checking arguments...
	   (g-loop (init binding-pair nil)
		   (while spec)
		   ; Are they equal? Return t.
		   (do (cond ((equal (car spec) (car obj)) t)
			     ; Is one a var? Save its binding.
			     ((is-variable (car spec))
			      (setq binding-pair
				    (assoc (car spec) bindings))
			      ; If no binding found.  Make one
			      (cond ((null binding-pair)
				     (push (list (car spec) (car obj)) 
					   bindings)) ;return bindings
				    ((is-variable (cadr binding-pair))
				     ; complex bindings, so expensive
				     ; check.  dkahn changed this to
				     ; tree-equal from equal.
				     (return 
				      (tree-equal (subst-bindings 
							spec bindings)
						    (subst-bindings obj
							bindings)
						    :test
						    #'var-or-match)  
				        )); full match so return t
				    ((equal (cadr binding-pair) 
					    (car obj)) 
				     t)
				    ; no match so exit with nil.
				    (t (return nil))))
			     ; unequal arguments so return nil.
			     ((return nil))))
		   (next spec (cdr spec) obj (cdr obj))
		   (result t)))))

(defun var-or-match (obj1 obj2)
  "If obj1 and obj2 are equal or if one is a variable then the test
   returns true."
   (or (is-variable obj1)
       (is-variable obj2)
       (equal obj1 obj2)))




(defun var-or-match (obj1 obj2)
  "If obj1 and obj2 are equal or if one is a variable then the test
   returns true."
   (or (is-variable obj1)
       (is-variable obj2)
       (equal obj1 obj2)))



#| ; obsolite version saved for posterity
(defun pred-match-p (spec obj bindings)     ;; 1/31/88 -snm
    (cond ((eq (car spec) (car obj))
	   (setq spec (cdr spec))
	   (setq obj (cdr obj))
	   (g-loop (init binding-pair nil)
		   (while spec)
		   (do (cond ((equal (car spec) (car obj)))
			     ((is-variable (car spec))
			      (setq binding-pair
				    (assoc (car spec) bindings))
			      (cond ((null binding-pair)
				     (push (list (car spec) (car obj)) 
					   bindings))
				    ((is-variable (cadr binding-pair))
				     ; complex bindings, so expensive check
				     (return (equal (subst-bindings 
							spec bindings)
						    (subst-bindings obj
							bindings))))
				    ((equal (cadr binding-pair) 
					    (car obj))
				     t)
				    (t (return nil))))
			     ((return nil))))
		   (next spec (cdr spec) obj (cdr obj))
		   (result t)))))


|#

(defun variable-match (cond-variable obj-val bindings)    ;; 1/31/88 -snm
    (let ((binding (assoc cond-variable bindings)))
	 (cond (binding 
		   (cond ((is-variable (cadr binding))
			  (variable-match (cadr binding) obj-val bindings))
			 ((equal obj-val (cadr binding))
			  bindings)
			 (t nil)))
	       ((cons (list cond-variable obj-val) bindings)))))


(defun value-match (spec-val obj-val bindings)
    (cond ((is-variable spec-val)
	   (variable-match spec-val obj-val bindings))
	  ((equal spec-val obj-val)
	   bindings)))

; matches literals against each other. Returns nil or bindings.

(defun lit-match (spec obj)
    (cond ((eq (car spec) '~)
	   (and (eq (car obj) '~)
		(pred-match (cadr spec) (cadr obj) '((nil nil)))))
	  ((pred-match spec obj '((nil nil))))))
 
; matches literals against each other. Returns nil or bindings.
; Same as above but takes bindings

(defun lit-match2 (spec obj bindings)
    (cond ((eq (car spec) '~)
	   (and (eq (car obj) '~)
		(pred-match (cadr spec) (cadr obj) bindings)))
	  ((pred-match spec obj bindings))))

;  Same as lit-match1, but assumes that there are vars in spec, so
;  (nil nil) isn't used.	

(defun lit-match3 (spec obj)
    (cond ((eq (car spec) '~)
	   (and (eq (car obj) '~)
		(lit-match3 (cadr spec) (cadr obj))))
	  ((eq (car spec) (car obj))
	   (setq spec (cdr spec))
	   (setq obj (cdr obj))
	   (g-loop (init bindings nil bound-pair nil)
	            (while spec)
		    (do (cond ((and (node-p (car spec)) (node-p (car obj)))
		               (equal (node-name (car spec)) 
			              (node-name (car obj))))
		              ((equal (car spec) (car obj)))
			      ((setq bound-pair (assoc (car spec) bindings))
			       (or (equal (cadr bound-pair) (car obj))
				   (return)))
			      ((is-variable (car spec))
			       (push (list (car spec) (car obj)) bindings))
			      ((return))))
		    (next spec (cdr spec) obj (cdr obj))
		    (result bindings)))))


; used for matching the postconditions of an operator to a goal.
; goal can have variables in it.

(defun rhs-match (post goal)
    (cond ((negated-p goal)
	   (and (negated-p post)
		(rhs-match (cadr post) (cadr goal))))
	  ((eq (car post)(car goal))
	   (setq post (cdr post))
	   (setq goal (cdr goal))
	   (g-loop (init binding-pair nil bindings '((nil nil)))
	            (while post)
		    (do (cond ((equal (car post) (car goal)))
			      ((is-variable (car post))
			       (cond ((setq binding-pair
					    (assoc (car post) bindings))
				      (or (equal (cadr binding-pair)(car goal))
					  (return)))
				     (t (push (list (car post)(car goal)) 
					      bindings))))
			      ((is-variable (car goal))
			       (setq goal (subst (car post) (car goal) goal)))
			      ((return))))
		    (next post (cdr post) goal (cdr goal))
		    (result bindings)))))


; matches a sequence of literals.

(defun sequence-match (spec obj bindings)
    (g-loop (while (and spec bindings))
	     (do (setq bindings (lit-match2 (car spec) (car obj) bindings)))
	     (next spec (cdr spec) obj (cdr obj))
	     (result bindings)))

 
; Takes an expression and returns a list of the literals
;  Any literals inside "Negated exists" are not returned.

(defun get-lits (exp)
    (cond ((null exp) nil)
	  ((or (eq (car exp) 'exists)
	       (eq (car exp) 'forall))
	   (cons (get-gen-exp exp)
		 (get-lits (get-exp exp))))
	  ((atomic-formula-p exp)
	   (list exp))
	  ((negated-p exp)
	   (cond ((eq (caadr exp) 'exists) nil)
		 (t (list exp))))
	  ((or (eq (car exp) 'and)
	       (eq (car exp) 'or))
	   (mapcan #'get-lits (cdr exp)))
	  (t (error "~A" exp))))

; Checks whether any member of a list of literals is impossible to achieve.
; (A literal is impossible to achieve it matches an impossible goal,
;  or its predicate).

; Both impossible-to-achieve-lits and impossible-to-achieve-lit are
; NOT used by the EBL subsystem.

(defun impossible-to-achieve-lits (lits)
  (if (not *EBL-FLAG*)
      (g-loop (init lit nil)
	      (while (setq lit (pop lits)))
	      (do (cond ((negated-p lit)
			 (or (member (caadr lit) *NEGATABLE-PREDS*)
			     (return t)))
			((not (member (car lit) *ACHIEVABLE-PREDS*))
			 (return t)))))))

; Checks whether a literal is impossible to achieve, because the predicate
; is static!    				

(defun impossible-to-achieve-lit (lit)
    (cond (*EBL-FLAG* nil)
          ((negated-p lit)
	   (not (member (caadr lit) *NEGATABLE-PREDS*)))
	  ((not (member (car lit) *ACHIEVABLE-PREDS*)))))


;  returns t if a list has a variable anywhere inside it.

(defun has-any-unbound-vars (l bindings)
    (cond ((atom l) (and (is-variable l) (not (assoc l bindings))))
	  ((has-any-unbound-vars (car l) bindings))
	  ((has-any-unbound-vars (cdr l) bindings))))



;  -------  TRACKING ---------------------

; Below are the tracking routines for PRODIGY. (Finds the justfications
; of an expression that has been succesfully matched).
; This code is a slight variation of the matching code. (See above)

; Exp-track, unlike Exp-match, expects bindings to
; be complete, so that only one match is generated. 

(defun exp-track (exp bindings state)
    (setq *STATE* state)
    (setq *FAILURE-RECORD* nil)
    (setq *FAILURE-RECORDING* nil)         ; used during negation match.
    (just-filter (cond (bindings 
			   (descend-track exp bindings))
		       (t  (descend-track exp (list '(nil nil)))))))

;  removes duplicates, extraneous t's, function-preds, etc. from a list
;  of justifications

(defun just-filter (l)
    (cond ((null l) nil)
	  ((or (eq (car l) t)
	       (static-p (car l))
	       (and (negated-p (car l))
		    (static-p (cadar l)))
	       (member (car l) (cdr l) :test #'equal))
	   (just-filter (cdr l)))
	  ((cons (car l) (just-filter (cdr l))))))


;  This is the recursive match routine that does all the work. 

(defun descend-track (exp bindings)
    (cond ((eq (car exp) 'forall)
	   (forall-track bindings (get-gen-exp exp) (get-exp exp)))
	  ((eq (car exp) 'exists)	       
	   (exists-track bindings (get-gen-exp exp) (get-exp exp)))
	  ((atomic-formula-p exp)
	   (cond ((has-any-unbound-vars exp bindings)
		  (exists-track bindings exp nil))
		 ((function-p exp)
		  (and (get-func-vals exp bindings nil) (list t)))
		 ((has-vars exp)
		  (and (member-state exp (state-true-assertions *STATE*) 
			   bindings t)
		       (list (subst-bindings exp bindings))))
		 ((member-state-inst exp (state-true-assertions *STATE*) t)
		  (list exp))))
	  ((eq (car exp) '~)
	   (cond ((eq (caadr exp) 'exists)
		  (negated-exists-track (cadr exp) bindings))
		 (t (cond ((negated-pred-match (cadr exp) bindings) 
			   (cond ((static-p (cadr exp)) (list t))
				 (t (list exp))))))))
	  ((eq (car exp) 'and)
	   (and-track (cdr exp) bindings))
	  ((eq (car exp) 'or)
           (or-track (cdr exp) bindings))
	  ((null exp) (list t))
	  (t (error "DESCEND-TRACK - bad expression: ~A" exp))))

; returns generator without bindings as justification, along with
; unmatched lit. Note that if the generator was unmatched, we may
; get not only the generator but an instantiation of it as well.

(defun negated-exists-track (exp bindings)
    (prog (subresult unmatched-lits)
	  (setq *FAILURE-RECORDING* t)
	  (setq subresult (descend-match exp bindings 'one))
	  (setq *FAILURE-RECORDING* nil)
	  (setq unmatched-lits 
		(mapcar #'(lambda (pair)
                             (cond ((has-vars (car pair))
				    (subst-bindings (car pair) (cadr pair)))
				   (t (car pair))))
			*FAILURE-RECORD*))
	  (setq *FAILURE-RECORD* nil)
	  (return (cond (subresult (return nil))
			((static-p (get-gen-exp exp))
			 (mapcar #'(lambda (lit) (negate lit)) unmatched-lits))
			(t (cons (list '~ (get-gen-exp exp)) 
				 (mapcar #'(lambda (lit) (negate lit))
					 unmatched-lits)))))))

								 
(defun or-track (exps bindings)
    (g-loop (init subresult nil exp nil)
	     (while (setq exp (pop exps)))
	     (do (cond ((and (atomic-formula-p exp)
			     (has-any-unbound-vars exp bindings))
			(setq subresult (exists-track bindings exp nil)))
		       (t (setq subresult (descend-track exp bindings)))))
	     (until subresult)
	     (result subresult)))

; Exp = (and sub-exp sub-exp ....)

(defun and-track (exps bindings)
    (cond ((and (atomic-formula-p (car exps))
		(has-any-unbound-vars (car exps) bindings))
	   (cond ((cddr exps)
		  (exists-track bindings (car exps) (cons 'and (cdr exps))))
		 ((exists-track bindings (car exps) (cadr exps)))))
	  ((let ((first-result (descend-track (car exps) bindings))
		 second-result)
		(cond ((not first-result) nil)
		      ((null (cdr exps)) first-result)
		      ((setq second-result
			     (and-track (cdr exps) bindings))
		       (append first-result second-result)))))))



; Matches a universally quantified expression. If matched, returns the
; justifications for exp, under all generated bindings, plus the
; generator justification.

(defun forall-track (start-bindings gen-exp exp)
    (g-loop (init values (gen-values gen-exp start-bindings)
	           subresult nil bindings nil ret-val nil)
	     (before-starting  
		 (cond ((null values) (setq ret-val (list t)))
		       ((eq values 'no-match-attempted) (return nil))
		       ((eq values t) 
			(setq ret-val (descend-track exp start-bindings))
			(setq values nil))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do (setq subresult (descend-track exp bindings))
		 (or subresult (return))
		 (setq ret-val (nconc subresult ret-val)))
	     (result (cond ((static-p gen-exp) ret-val)
			   (t (cons (list '~ gen-exp) ret-val))))))
 


; tracks an existentially quantified expression. 
; (Only returns justifications for the 1st match).

(defun exists-track (start-bindings gen-exp exp)
    (g-loop (init values (gen-values gen-exp start-bindings)
		   subresult nil bindings nil)
	     (before-starting                     ; make sure values are good
		 (cond ((null values) (return))
		       ((eq values 'no-match-attempted) (return))
		       ((eq values t) 
			(setq subresult (descend-track exp start-bindings))
			(setq values nil))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do  (setq subresult (descend-track exp bindings)))
	     (until subresult)
	     (result (cond ((null subresult) nil)
			   ((static-p gen-exp) subresult)
			   (t (cons (subst-bindings gen-exp 
					(append bindings start-bindings))
				    subresult))))))

