#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#


;==========================================================================
; File:   graphics.lisp     Version: 0.0                  Created: 3/2/88
;
; Locked by:  none.                                      Modified: 4/4/88
; 
; Purpose:    To standardize domain graphics for Prodigy.
;
;==========================================================================


(eval-when (compile) 
  #+:coral (load-path *planner-path* "pg-mac")
  #-:coral (load-path *planner-path* "pg-x11")
  (load-path *planner-path* "data-types")
)



(require 'pg-system)
(use-package (symbol-name 'pg))


(proclaim
 '(special *DOMAIN-WINDOW* *FINISH*   *NODE-MSG-X*    *NODE-MSG-Y*
	   *STATE-MSG-X*   *STATE-MSG-Y*   *START-STATE*   *WIDTH*
	   *DOMAIN-STATE*  *DOMAIN-NODE*  *DOMAIN-GRAPHICS*))


;==========================================================================
;                      Domain  Dependent  Functions
;==========================================================================

;; RESET-DOMAIN-GRAPHICS-PARAMETERS should be called when a new problem is 
;; loaded. This function sets graphics variables to null. In particular, 
;; the variables  *NODE-MSG-X*, *NODE-MSG-Y*, which determine the location
;; of the node message in the graphics window and *STATE-MSG-X*, and 
;; *STATE-MSG-Y* which determine the location of the state message in the
;; graphics window should be set to null in this function.
;;

(defun reset-domain-graphics-parameters ()
  (psetq *NODE-MSG-X*  NIL 
	 *NODE-MSG-Y*  NIL
	 *STATE-MSG-X* NIL 
	 *STATE-MSG-Y* NIL
	 *WIDTH*       NIL))     ; maximum width of an object

;--------------------------------------------------------------------------

;; DETERMINE-DOMAIN-GRAPHICS-PARAMETERS sets the variables used in drawing 
;; the domain graphics for a given problem. This routine should also 
;; determine the location of the node and state messages: i.e., set the
;; variables (*NODE-MSG-Y*, *NODE-MSG-X*, *STATE-MSG-X*, *STATE-MSG-Y*).
;;

(defun determine-domain-graphics-parameters (problem)
  (setq *WIDTH* 0
	*NODE-MSG-X*  10
	*NODE-MSG-Y*  30
	*STATE-MSG-X* 35
	*STATE-MSG-Y* 30))


;--------------------------------------------------------------------------

;; DRAW-DOMAIN-BACKGROUND uses domain graphics parameters to draw the
;; rear plane for the given domain -- drawn before any domain objects
;; are added to the domain graphics window. 
;;

(defun draw-domain-background ())


;--------------------------------------------------------------------------

;; DELETE-DOMAIN-GRAPHIC-OBJECTS erases objects from the domain window. 
;; The argument to this function is a list containing predicates to delete.
;; The function must parse the relevant graphics predicates and devise some
;; method for removing them from the domain window.
;;

(defun delete-domain-graphic-objects (state-predicates))


;--------------------------------------------------------------------------

;; ADD-DOMAIN-GRAPHIC-OBJECTS adds objects to the domain window. This 
;; function is the complement of (delete-domain-graphic-objects). The 
;; argument to this function may contain predicates irrelevant to graphics, 
;; therefore the function must devise a method to parse relevant predicates
;; and add the appropriate objects to the domain window. 
;; 
;;  
;; 

(defun add-domain-graphic-objects (state-predicates))


;--------------------------------------------------------------------------

;; DRAW-DOMAIN-FOREGROUND uses domain graphics parameters to draw the 
;; foremost plane of graphics -- drawn after the domain objects.
;;

 (defun draw-domain-foreground ())

;; DOMAIN-CONFIG is called every time the window is resized.  It normally
;; sets the variables *DOMAIN-DIMENSION-X* and *DOMAIN-DIMENSION-Y* only, but
;; the domain author can added other calculations based on these.

(defun domain-dependent-configure (new-x new-y)
	(declare (special *DOMAIN-DIMENSION-X* *DOMAIN-DIMENSION-Y*))

	(setf *DOMAIN-DIMENSION-X* new-x *DOMAIN-DIMENSION-Y* new-y)

)

;==========================================================================
;                     Domain  General  Functions
;==========================================================================

;; DISPLAY-DOMAIN-GRAPHICS-START-STATE is called from the output facility to 
;; display the initial state in the domain graphics window.  This is done 
;; when graphics are turned on (using (toggle-domain-graphics)) before a
;; problem solving execution (using (run)).


(defun display-domain-graphics-start-state ()
  (determine-domain-graphics-parameters
   (cons (op-preconds '*FINISH*) *START-STATE*))
  (pg-clear-window *DOMAIN-WINDOW*)
  (draw-domain-background)
  (add-domain-graphic-objects *START-STATE*)
  (draw-domain-foreground)
  (display-state-msg "s1-1"))


;-------------------------------------------------------------------------

;; DISPLAY-DOMAIN-GRAPHICS is called from the analysis facility to display
;; a single state.  This function determines the parameters for the domain,
;; draws the appropriate graphics, then resets the paramters.  


(defun display-domain-graphics (node)
  (cond ((or (not (boundp '*WIDTH*)) (null *WIDTH*))
	 (determine-domain-graphics-parameters 
	  (cons (op-preconds '*FINISH*) 
		*START-STATE*))))
  (refresh-domain-graphics  node))



;-------------------------------------------------------------------------

;; DISPLAY-DOMAIN-GRAPHICS-STATE-CHANGE is called from the output facility 
;; to update the domain graphics during planning.  This function draws the 
;; differences between the graphics for the prior node and graphics for a 
;; new node. 


(defun display-domain-graphics-state-change (new-node)
  (let ((old-state *DOMAIN-STATE*)
	(old-node *DOMAIN-NODE*)
	(new-state (node-state new-node)))
	(declare (special n1))
    (setq *DOMAIN-NODE* new-node)
    (cond ((and *DOMAIN-GRAPHICS* 
		(not (eq new-node n1))
		(<= (node-to-number (node-name new-node)) 
		    (node-to-number (node-name old-node))))
	   (refresh-domain-graphics new-node)))
    (cond ((not (eql (state-num old-state) (state-num new-state)))
	   (setq *DOMAIN-STATE* new-state)
	   (let ((old-closed-world (state-closed-world old-state))
		 (new-closed-world (state-closed-world new-state)))
	     (draw-domain-background)
	     (delete-domain-graphic-objects 
	      (calc-state-dels  old-closed-world  new-closed-world))
	     (add-domain-graphic-objects
	      (calc-state-adds  old-closed-world  new-closed-world))
	     (draw-domain-foreground)
	     (display-node-msg  new-node)
	     (display-state-msg  new-state)
	     (pg-refresh-window *DOMAIN-WINDOW*))))))



(defun calc-state-dels (old new)
  (cond ((or (null new)(null old)) old)
	((equal (car new)(car old))
	 (calc-state-dels (cdr old)(cdr new)))
	((alphalessp-f (car new)(car old))
	 (calc-state-dels old (cdr new)))
	(t (cons (car old)(calc-state-dels (cdr old) new)))))



(defun calc-state-adds (old new)
  (cond ((or (null old)(null new)) new)
	((equal (car new)(car old))
	 (calc-state-adds (cdr old)(cdr new)))
	((alphalessp-f (car old)(car new))
	 (calc-state-adds (cdr old) new))
	(t (cons (car new)(calc-state-adds old (cdr new))))))


;-------------------------------------------------------------------------

;; REFRESH-DOMAIN-GRAPHICS draws all the objects for the given node in the 
;; domain window. 


(defun refresh-domain-graphics (node)
  (pg-clear-window *DOMAIN-WINDOW*)
  (draw-domain-background)
  (add-domain-graphic-objects
   (state-closed-world (node-state node)))
  (draw-domain-foreground)
  (display-node-msg  node)
  (display-state-msg (state-name (node-state node)))
  (pg-refresh-window *DOMAIN-WINDOW*))

;-------------------------------------------------------------------------
;; domain-config is called when the domain window is resized.  It calls 
;; the domain dependent configure routine and then sets some values in 
;; the win structure.

(defun domain-config (window x y width height)
	(declare (ignore window x y))
	(domain-dependent-configure width height)
)

	

;-------------------------------------------------------------------------

;; DISPLAY-NODE-MSG and DISPLAY-STATE-MSG are used to display the current
;; node and state for the particular problem.  These functions require 
;; the domain dependent functions to RESET and DETERMINE the abscissa and 
;; ordinate locations of each message (using the parameters *NODE-MSG-X*,
;; *NODE-MSG-Y*, *STATE-MSG-X*, *STATE-MSG-Y*). 


(defun display-node-msg (node)
  (write-in-domain-window 
   (concatenate 'string "Node: " (princ-to-string (node-name node)))
   *NODE-MSG-X*
   *NODE-MSG-Y*))


(defun display-state-msg (state)
  (write-in-domain-window
   (concatenate 
    'string 
    "State: " 
    (princ-to-string 
     (if (state-p state) (state-name state) state)))
   *STATE-MSG-X*
   *STATE-MSG-Y*))



;=========================================================================
;                      Domain  Window  Functions
;=========================================================================




(defun set-up-domain-window ()
 (declare (special *FONT-STRING* 
		   *INIT-DOMAIN-X* 
		   *INIT-DOMAIN-Y*
		   *INIT-DOMAIN-WIDTH*
		   *INIT-DOMAIN-HEIGHT*))
 "creates the domain window and initializes the global 
  domain parameters."
 (pg-init-graphics (current-host) *FONT-STRING*)
  (setq *DOMAIN-WINDOW*
        (pg-create-window *INIT-DOMAIN-X*
			  *INIT-DOMAIN-Y*
			  *INIT-DOMAIN-WIDTH*
			  *INIT-DOMAIN-HEIGHT*
			  :uspp t
			  :exposure-function #'domain-refresh 
			  :name "Domain Graphics"
			  :config-function #'domain-config))
  (pg-show-window *DOMAIN-WINDOW*))

;-------------------------------------------------------------------------

;; DOMAIN-REFRESH is called by the exposure event handler when the
;; window needs updated.
(defun domain-refresh (&rest x)
  (declare (special *EXPL-NODE*) (ignore x))
  "Domain-refresh is called by the exposure event handler when the
   domain window needs updated."
   (and (boundp '*EXPL-NODE*) *EXPL-NODE*
        (display-domain-graphics *EXPL-NODE*))
)
;-------------------------------------------------------------------------


;; WRITE-IN-DOMAIN-WINDOW prints a string in the domain window beginning
;; at position (x,y). 


(defun write-in-domain-window (string x y)
  (pg-write-text *DOMAIN-WINDOW* x y string))


;-------------------------------------------------------------------------


(defun kill-domain-graphics () 
  (pg-hide-window *DOMAIN-WINDOW*)
  (pg-kill-window *DOMAIN-WINDOW*)
  (setq *DOMAIN-WINDOW* nil))

;====================================================================
;  Code to implement a "movie" of the solution.
;====================================================================



(defun show-movie (&optional (start-node *EXPL-NODE*) 
			     (end-node nil) 
			     (delay *DEFAULT-DELAY*))
   (declare (special *EXPL-NODE* *DOMAIN-GRAPHICS*))
   "Checks to be sure tree and domain graphics are on, checks that the 
    starting node is on the current path and the the end node is after
    the start.  Then do-movie is called."
	(cond ((not (and *DOMAIN-GRAPHICS*)) nil) 
             ((not (on-current-path start-node))
		  (format *QUERY-IO* "Start node ~A is not on current path.~%"
				start-node))
	      ((and end-node (not (on-current-path end-node)))
		  (format *QUERY-IO* "End node ~A is not on current path.~%"
				end-node))
	      ((and end-node 
		   (> (node-to-number start-node) (node-to-number end-node)))
			(format *QUERY-IO* "The end node ~A is before the start node ~A.~%" end-node start-node))
	      (t (refresh-domain-graphics start-node)
		 (format t "Use 'q' to return to Analysis.~%")
		 (do-movie (find-active-child (node-children start-node))
			    end-node delay start-node))))


(defun do-movie (new-node end-node delay old-node)
  "For each different state in the solution path do-movie will cause
   that state to be displayed on the screen with a delay between states
   determined by *DEFAULT-DELAY*."

    (let* ((next-node (find-active-child (node-children new-node)))
 	   (new-state (state-num (node-state new-node)))
	   (old-state (state-num (node-state old-node))))


   	    (when (not (eql old-state new-state))
			(display-next-frame-change new-node old-node)
	 	(wait delay))
	    (if (and (node-children new-node) 
		    (not (check-for-interesting-input))
		    (not (and (eq '*FINISH* (node-current-op new-node))
			      (eq new-node (second *ALL-NODES*)))))
		    
		    
		(do-movie next-node end-node delay new-node)
	        nil)
    ))


(defun display-next-frame-change (new-node old-node)
   "This function calculates the difference between the last state
    and the current state and then draws the changes."
   (let ((old-closed-world (state-closed-world (node-state old-node)))
	 (new-closed-world (state-closed-world (node-state new-node))))

     (draw-domain-background)
     (delete-domain-graphic-objects 
      (calc-state-dels  old-closed-world  new-closed-world))
     (add-domain-graphic-objects
      (calc-state-adds  old-closed-world  new-closed-world))
     (draw-domain-foreground)
     (display-node-msg  new-node)
     (display-state-msg  (node-state new-node))
     (pg-refresh-window *DOMAIN-WINDOW*)))

(defun wait (delay)
   "WAIT will sleep for delay seconds or pause if delay is :pause"
   (cond ((eq :pause delay)
		(format *QUERY-IO* "~%Paused...Press ENTER to continue.")
		(read-line *QUERY-IO*)
	 )
	 ((and (numberp delay) (not (minusp delay)))
	 	(sleep delay))
   )
)

(provide 'pg-state)

