#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#



(proclaim 
    '(special *EBL-FLAG* *ABSTRACTION-LEVEL* *PREV-ABS-LEVEL*))

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types"))


; Executes an operator (or inference rule). ALT must not have
; any unmatched conditions.
; 
; This code includes the reasoning-maintenance system which is used to
; maintain a correct set of inferences between states.  This is done by 
; keeping a table with a literal and a set of justifications on which the
; literal depends.  
;                    ((lit1 (jus1 jus2 jus3...))
;		      (lit2 (jus4 ...))
;		      .
;		      .
;		      .                  )
; 

; ADJUST-STATE returns new-state (or old-state, appropriately altered)

(defun adjust-state (alt state)
    (let ((op (alt-op alt))
	  (vars (alt-vars alt)))
	 (cond ((operator-p op)
		(cond (*EBL-FLAG* (simple-do-op op vars state))
		      (t (do-op op vars state))))
	       (t (do-inference op vars state)))))

; DO-OP assumes all adds and deletes are closed state. 
; State = (true-assertions closed-world false-assertions justification-table)
;
; For each operator invocation the true and false assertions must be updated.
; This is done using the justification table to find the dependencies.  The
;  justification table is updated as well by removing dependencies that are
;  no longer in effect.
 
(defun do-op (op ivars old-state)
   (let ((idels (sort (remove-abstracted
		       (expand-conditional-dels old-state
			   (subpair (op-vars op) ivars 
                                    (op-del-list op))))
		       'alphalessp-f))
         (iadds (sort (remove-abstracted
		       (expand-conditional-adds old-state
			   (subpair (op-vars op) ivars 
			            (op-add-list op))))
		       'alphalessp-f))
	 (new-state (make-new-state-nm)))
	 (set-closed-world new-state 
	     (change-state iadds idels (state-closed-world old-state)))
	 (set-true-assertions new-state 
	     (change-state iadds idels (state-true-assertions old-state)))
	 (set-false-assertions new-state (state-false-assertions old-state))
	 (set-justification-table new-state 
	     (state-justification-table old-state))
	 (set-justification-table new-state 
	     (update-just-table (state-justification-table new-state)
		 (update-assertions-idels new-state idels iadds)))
	 new-state))


(defun remove-abstracted (lits)
  (cond ((null *ABSTRACTION-LEVEL*) lits)
	(t (let ((new-lits (remove-each-abstracted lits)))
	     (cond ((or (null *PREV-ABS-LEVEL*)
			(generating-subproblem *CURRENT-NODE*))
		    new-lits)
		   ((check-monotonicity-violation lits))
		   (t new-lits))))))

(defun remove-each-abstracted (lits)
  (cond ((null lits) nil)
	((not (member (extract (car lits)) *ABSTRACTION-LEVEL* 
		      :test #'equal))
	 (remove-each-abstracted (cdr lits)))
	(t (cons (car lits)
		 (remove-each-abstracted (cdr lits))))))


(defun check-monotonicity-violation (lits)
  (cond ((null lits) nil)
	((member (extract-instance (car lits)) *PREV-ABS-LEVEL*
		 :test #'equal)
	 (throw 'omv (car lits)))
	(t (check-monotonicity-violation (cdr lits)))))


(defun generating-subproblem (node)
  (cond ((null (node-left-clone node))
	 (if (eq 'start-goal (alt-goal (node-generating-alt node)))
	     t))
	(t (generating-subproblem (node-left-clone node)))))

; this is only used during EBL (maybe a temporary measure) because the EBL
; module is not prepared to deal with the complexity added by the RMS 
; subsystem. So we evectively shutoff the RMS system when EBL is going
; to be used.

(defun simple-do-op (op ivars old-state)
    (let ((idels (sort (expand-conditional-dels old-state
			   (subpair (op-vars op) ivars (op-del-list op)))
		       'alphalessp-f))
	  (iadds (sort (expand-conditional-adds old-state
			   (subpair (op-vars op) ivars (op-add-list op)))
		       'alphalessp-f))
	  (new-state (make-new-state-nm)))
	 (set-closed-world new-state 
	     (change-state iadds idels (state-closed-world old-state)))
	 (set-true-assertions new-state (state-closed-world new-state))
	 (set-false-assertions new-state nil)
	 (set-justification-table new-state nil)
	 new-state))

; Change-state does the actual deletions and additions. 
; Assumes adds, dels and state-lits are sorted.
; Duplicate dels are allowed.

(defun change-state (adds dels state-lits)
    (cond ((null state-lits) 
	   (if dels 
               (format t "~%WARNING: Can't find deletions in state ~A" dels))
	   adds)
	  ((and (null adds) (null dels)) state-lits)
	  ((equal (car dels) (car state-lits)) 
           (cond ((equal (car dels) (cadr dels))
		  (change-state adds (cdr dels) state-lits))
	         (t (change-state adds (cdr dels) (cdr state-lits)))))
	  ((or (null adds)
	       (alphalessp-f (car state-lits) (car adds)))
	   (cons (car state-lits)
		 (change-state adds dels (cdr state-lits))))
	  (t (cons (car adds)
		   (change-state (cdr adds) dels state-lits)))))

; This function takes either the add list or delete list and tests all 
; the conditionals in the list to see if the consequent should actually
; be added or deleted.  This function should also handle the case where the
; antecedent is actually a generator.  
; For example the given list might be:
;     ((foo 1)
;      (if (blah) (bar <y>))
;      (if (tribble <x>)(frob <x>)))
; If blah is true and tribble x returns 1, 2 and 3 for bindings of x, then
; expand-conditionals should return:
;  	((foo 1)
;	 (bar <y>)
;	 (frob 1)
;	 (frob 2)
;	 (frob 3))
 
(defun expand-conditional-adds (old-state adds)
    (mapcan #'(lambda (add)
		      (cond ((eq (car add) 'if) 
			     (expand-conditional-add old-state add))
			    ((list add))))
	    adds))

(defun expand-conditional-add (old-state conditional)
    (g-loop (init test (cadr conditional)
		   bindings-lists
		      (cond ((atomic-formula-p test)
			     (atomic-formula-match test old-state))
			    ((exp-match test nil old-state)))
		   adds (cddr conditional)
		   ret-val nil bindings nil)
	     (while (setq bindings (pop bindings-lists)))
	     (do (setq ret-val (append (subst-bindings adds bindings) 
				       ret-val)))
	     (result ret-val)))


(defun expand-conditional-dels (old-state dels)
    (mapcan #'(lambda (del)
		      (cond ((eq (car del) 'if) 
			     (expand-conditional-del old-state del))
			    ((has-vars del)             ; wildcard
			     (expand-wildcard-del del old-state))
			    ((list del))))
	    dels))

(defun expand-conditional-del (old-state conditional)
    (g-loop (init test (cadr conditional)
		   bindings-lists (cond ((atomic-formula-p test)
					 (atomic-formula-match test old-state))
					((exp-match test nil old-state)))
		   dels (cddr conditional)
		   ret-val nil bindings nil)
	     (while (setq bindings (pop bindings-lists)))
	     (do (dolist (idel (subst-bindings dels bindings))
			 (cond ((has-vars idel)
				(setq ret-val (append (expand-wildcard-del
							  idel old-state)
						      ret-val)))
			       ((push idel ret-val)))))
	     (result ret-val)))

(defun expand-wildcard-del (del old-state)
    (mapcar #'(lambda (bindings) (subst-bindings del bindings))
	    (atomic-formula-match del old-state)))

; ----------------------------------------------------------------

; Dependencies no longer in effect in the justification table are removed.
; The list of things to delete is built up by update-assertions-idels.
 
(defun update-just-table (justification-table deletes)
    (cond ((null justification-table) nil)
	  ((member (car justification-table) deletes)
	   (update-just-table (cdr justification-table) deletes))
	  (t (cons (car justification-table)
		   (update-just-table (cdr justification-table) deletes)))))


; The idels are checked for any assertions that might depend on them.
 
(defun update-assertions-idels (state idels iadds)
    (cond ((null idels) (update-assertions-iadds state iadds))
	  (t (append (just-member state
			 (car idels)
			 (state-justification-table state))
		     (update-assertions-idels state (cdr idels) iadds)))))



; The iadds are checked for any assertions that might depend on the 
; adds not being true.
; 
(defun update-assertions-iadds (state iadds)
    (cond ((null iadds) nil)
	  (t (append (just-member state
			 (list '~ (car iadds))
			 (state-justification-table state))
		     (update-assertions-iadds state (cdr iadds))))))


; Checks to see if a given literal is used as a justification anywhere.  
; If so the things that depend on it are deleted.
 
(defun just-member (state literal justification-table)
    (cond ((null justification-table) nil)
	  ((just-match literal (cadar justification-table))
	   (delete-assertion state (caar justification-table))
	   (cons (car justification-table)
		 (append (just-member state literal 
			     (cdr justification-table))
			 (just-member state
        		     (caar justification-table)
			     (state-justification-table state)))))
	  (t (just-member state literal (cdr justification-table)))))


; Checks for the actual match betweenthe literal and a set of justifications
; for a given inference.
 
(defun just-match (literal justifications)
    (cond ((null justifications) nil)
	  ((lit-match (car justifications) literal) t)
	  (t (just-match literal (cdr justifications)))))


; Deletes an assertion from the appropriate list.  If there is a 
; negation in front of the assertion it is deleted from the set
; of false assertions and otherwise from the set of true assertions.
 
(defun delete-assertion (state assertion)
    (cond ((eq (car assertion) '~)
	   (set-false-assertions state 
	       (remove (cadr assertion) (state-false-assertions state) 
		       :test #'equal)))
	  (t (set-true-assertions state
		 (remove assertion (state-true-assertions state) 
			 :test #'equal)))))


; Adds an inference to the appropriate inference set.  It also finds 
; all the justifications for a given inference and saves the info in 
; the justification table.
 
(defun do-inference (op ivars state)
    (let ((new-state (make-sub-state state))
	  (justifications 
	      (exp-track 
		  (op-preconds op)
		  (make-binding-list (op-vars op) ivars)
		  state))
	  (justificands (subpair (op-vars op) ivars (op-add-list op))))
	 (dolist (cand justificands)
		 (cond ((not (null justifications))
			(set-justification-table new-state
			    (cons (list cand justifications)
				  (state-justification-table new-state)))))
		 (cond ((eq '~ (car cand))
			(set-false-assertions new-state
			    (change-state (list (cadr cand)) nil 
					  ; changed `cond' to `cand' in the 
					  ; preceeding line. -MPM
				(state-false-assertions new-state))))
		       (t (set-true-assertions new-state
			      (change-state (list cand) nil
				  (state-true-assertions new-state))))))
	 new-state))


