#|
*******************************************************************************
PRODIGY/EBL Module Version 2.0  
Copyright 1989 by Steven Minton.

The PRODIGY/EBL module was designed and built by Steven Minton. Thanks
to Jaime Carbonell and Craig Knoblock for their helpful advice. Andy
Philips contributed to the version 2.0 modifications.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#

;;;  Host: bacon.arc.nasa.gov
;;;  File: /home/copernicus/abp/prodigy/ebl/patches.lisp
;;;  Contains: Modifies operator names and Warns user of errors in PDL exp
;;;  Author:  Andy Philips (abp@bacon.arc.nasa.gov)
;;;  Created: Sun May 12 19:07:42 1989
;;;  Updated: Fri Jul  7 18:17:06 1989

(proclaim '(special *EVALUABLE-SIGS* *CACHABLE-SIGS* *INFERENCE-RULES*
		*OPERATORS* *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
		*SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
		*SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* *UNIQUE-WARN*
		*SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
		*SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
		*SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*
		*STATIC-PREDS* *CLOSED-PREDS* *META-FUNCTIONS-EBL-CAN-HANDLE*))

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types")
	(load-path *EBL-PATH*     "ebl-data-types"))

; EBL-UNIQUEFY-RULES will modify the operator and inference rule lists
; by searching through them for variables with the same name (in
; different rules).  If a variable is mentioned in two different rules,
; it is assigned a new name (oldname + name of the operator) which is
; then substituted into the expression.  Then the uniquefied OPERATORS
; and INFERENCE-RULES are stored.

; This function relies on the fact that var-lst will contain duplicate
; variables iff the variable appears in more than one rule.

(defun ebl-uniquefy-rules ()
  (let ((var-lst (do ((x (append *OPERATORS* *INFERENCE-RULES*) (cdr x))
		      (y nil (append y (find-all-vars (cdar x)))))
		     ((null x) y))))
    (setq *UNIQUE-WARN* nil)
    (setq *OPERATORS*
	  (mapcar #'(lambda (x) (help-uniquefy-exp
				 (car x) x var-lst)) *OPERATORS*))
    (setq *INFERENCE-RULES*
	  (mapcar #'(lambda (x) (help-uniquefy-exp
				 (car x) x var-lst)) *INFERENCE-RULES*))
    (if *UNIQUE-WARN*
	(format t "~%  Warning:  You may wish to rename the above variables~
~%            in domain definitions yourself.~%"))))


; HELP-UNIQUEFY-EXP checks the variables mentioned in the expression against
; the variables in var-list, and substitutes a new variable name in the
; expression for each variable found at least twice in var-list.

; This function is used by EBL-UNIQUEFY-RULES above, and also by
; CHECK-EBL-SIMP-RULES in axioms-check.lisp

(defun help-uniquefy-exp (func-name exp var-lst)
  (dolist (x (find-all-vars (cdr exp)) exp)
	  (if (member x (cdr (member x var-lst)))
	      (let ((new-name (new-var-name func-name x)))
		(format t "EBL: Renamed ~a to ~a in ~a~%"
			x new-name func-name)
		(setq *UNIQUE-WARN* t)
		(setq exp (subst new-name x exp))))))


; NEW-VAR-NAME creates a new variable name by appending the fn (function
; name) onto the old variable name.

(defun new-var-name (fn x)
  (intern (concatenate
	   'string "<" (string-trim "<>" (string x)) "-" (string fn) ">")))

; EBL-DOMAIN-PROCESSING several some things.  It checks each
; of the operators and inference rules for expressions that it cannot
; handle.  Second, it checks the operators and inference-rules for
; conditional effects.  Finally, it checks the search control rules for
; meta-functions it can't handle.

(defun ebl-domain-processing ()
  (let ((op-or-inf-lst (append *OPERATORS* *INFERENCE-RULES*)))
    (dolist (op-or-inf op-or-inf-lst)
	    (warning-search (get (car op-or-inf) 'preconds) nil)
	    (warn-about-potential-negated-goal 
                  (get (car op-or-inf) 'preconds)))
    (warning-search-for-if op-or-inf-lst)
    (ebl-check-all-scr)
    (add-info-to-scrs)
    (add-info-to-ops)
    (process-ebl-axioms)))


;  Checks a precondition expression to see if there are any formulas
;  that might become negated goals. (Negations and forall-generators.)
;  Prints warnings to user.

(defun warn-about-potential-negated-goal (exp)
  (cond ((eq exp t) nil)
	((null exp) nil)
	((atomic-formula-p exp) nil)
	((or (eq (car exp) 'and) (eq (car exp) 'or))
         (dolist (sub (cdr exp)) 
		 (warn-about-potential-negated-goal sub)))
	((eq (car exp) 'exists)
         (warn-about-potential-negated-goal (get-gen-exp exp))
         (warn-about-potential-negated-goal (get-exp exp)))
	((eq (car exp) 'forall)
         (warn-about-potential-negated-goal (get-exp exp))
         (if  (and (atomic-formula-p (get-gen-exp exp))
		   (member (car exp) *NEGATABLE-PREDS*))
	     (print-ebl-warning "Potential to become negated goal: "
		 (get-gen-exp exp))))
	((eq (car exp) '~) 	;  do nothing with negated existentials
	 (if (and (atomic-formula-p (cadr exp))
	          (member (car (cadr exp)) *NEGATABLE-PREDS*))
	     (print-ebl-warning "Potential to become negated goal: " exp)))
	(t (list (list "Warning: Unknown expression found" exp)))))


; WARNING-SEARCH takes an expression and a variable-list.  The expression
; is expected to be a syntactically correct PDL expression.  The variable
; list is a list of variables that have been instantiated outside the scope
; of the expression.  It parses the PDL expression searching for sources of
; potential error, and reports those to the user.  The function will then
; return any newly instantiated variables when it ends.
;
; The following warnings are searched for:
;     Variables generated within an OR
;     Variables generated within a NOT
;     Negated existentials
;

(defun warning-search (exp var-lst)
  (cond ((eq exp t) nil)
	((null exp) nil)
	((atomic-formula-p exp)
	 (set-difference
	  (apply 'append (mapcar #'(lambda (x)
				     (if (is-variable x) (list x))) (cdr exp)))
	  var-lst))
	((eq (car exp) 'and)
	 (do* ((next exp (cdr next))
	       (data nil (warning-search
			  (car next) (append var-lst just-declared)))
	       (just-declared nil (append just-declared data)))
	      ((null next) just-declared)))
	((eq (car exp) 'exists)
	 (do* ((next (list (get-gen-exp exp) (get-exp exp)) (cdr next))
	       (new-var-lst (append var-lst (get-vars-lst exp)) new-var-lst)
	       (data nil (warning-search
			  (car next) (append new-var-lst just-declared)))
	       (just-declared nil (append just-declared data)))
	      ((null next) just-declared)))
	((eq (car exp) 'forall)
	 (do* ((next (list (get-gen-exp exp) (get-exp exp)) (cdr next))
	       (new-var-lst (append var-lst (get-vars-lst exp)) new-var-lst)
	       (data nil (warning-search
			  (car next) (append new-var-lst just-declared)))
	       (just-declared nil (append just-declared data)))
	      ((null next) just-declared)))
	((eq (car exp) '~)
	 (if (eq (caadr exp) 'exists)
	     (print-ebl-warning "Negated existentials are not allowed in EBL"
				exp))
	 (let ((data (warning-search (cadr exp) var-lst)))
	   (if data (print-ebl-warning "Variables generated within a NOT -- implicit negated existential: " exp :app data))))
	((eq (car exp) 'or)
	 (do* ((next exp (cdr next))
	       (data nil (warning-search (car next) var-lst))
	       (bad-vars nil (append bad-vars (set-difference data bad-vars))))
	      ((null next)
	       (if bad-vars (print-ebl-warning
			     "Variables generated within an OR: "
			     exp :app bad-vars))
	       nil)))
	(t (list (list "Warning: Unknown expression found" exp)))))


; WARNING-SEARCH-FOR-IF will search for all instances of IF within
; the effects-list of the operator or inference list, printing EBL
; warnings if it finds any.

(defun warning-search-for-if (op-or-inf-lst)
  (dolist (op-rule op-or-inf-lst)
	  (dolist (x (cadr (assoc 'effects (cdr op-rule))))
		  (if (eql 'if (car x))
		      (print-ebl-warning
		       "Cannot handle IF conditional found in operator "
		       x :app (car op-rule))))))


; PRINT-EBL-WARNING does just that.

(defun print-ebl-warning (warn-str exp &key ((:app app) nil))
  (format t "~2%EBL Warning: ~a" warn-str)
  (if app (format t "~a" app))
  (if exp (format t "~%Expression:~%~a" exp)))


; EBL-CHECK-SCR-EXP parses through SCR and passing off expressions to
; EBL-CHECK-META-PRED to be checked for unknown meta-functions.

(defun ebl-check-scr-exp (exp scr-nm)
  (cond ((null exp) nil)
	((member (car exp) '(exists forall))
	 (ebl-check-meta-pred (get-gen-exp exp) scr-nm)
	 (ebl-check-scr-exp (get-exp exp) scr-nm))
	((eq (car exp) '~)
	 (cond ((equal (caadr exp) 'exists)
		(ebl-check-scr-exp (cadr exp) scr-nm))
	       (t (ebl-check-meta-pred (cadr exp) scr-nm))))
	((member (car exp) '(or and))
         (dolist (sub (cdr exp))
                 (ebl-check-scr-exp sub scr-nm)))
	(t (ebl-check-meta-pred exp scr-nm))))


; EBL-CHECK-META-PRED will examine all functions and print warnings when
; it encounters unknown meta-predicates.

(defun ebl-check-meta-pred (exp scr-nm)
  (cond ((not (member (car exp) *META-FUNCTIONS-EBL-CAN-HANDLE*))
	 (terpri) 
	 (print-ebl-warning "Unknown META-FUNCTION in control rule "
			    (car exp) :app scr-nm))
	((eq (car exp) 'known)
	 (check-exp-syntax (caddr exp)))))
  

(setq *META-FUNCTIONS-EBL-CAN-HANDLE*
      '(primary-candidate-goal known current-goal current-goal
	is-equal not-equal high-on-goal-stack previous-state-diff
	on-goal-stack alt-on-deck was-added was-deleted protected-goal
	candidate-goal has-bound-vars not-top-level-node in-goal-exp
	is-top-level-goal candidate-node current-node current-op
	adjunct-goal candidate-bindings candidate-op direct-supergoal-of))





; EBL-CHECK-ALL-SCR examines the preconditions of the SCRs to determine
; if they have any offending meta-functions within them.  If they do
; then a warning is printed. (Note: preference rules do not have to be
; checked, since the ebl-system doesn't care about them.)

(defun ebl-check-all-scr ()
  (if *SCR-NODE-SELECT-RULES*
      (print-ebl-warning
       "There can NOT be any node selection control rules defined"
       *SCR-NODE-SELECT-RULES*))
  (if *SCR-NODE-PREFERENCE-RULES*
      (print-ebl-warning
       "There can NOT be any node preference control rules defined"
       *SCR-NODE-PREFERENCE-RULES*))
  (if *SCR-BINDINGS-SELECT-RULES*
      (print-ebl-warning
       "There can NOT be any bindings selection control rules defined"
       *SCR-BINDINGS-SELECT-RULES*))
  (if (not (and (= (length *SCR-GOAL-SELECT-RULES*) 1)
		(equal (caar *SCR-GOAL-SELECT-RULES*) 'SELECT-FIRST-GOAL)))
      (print-ebl-warning
       "There can be only one GOAL-SELECT-RULE:  SELECT-FIRST-GOAL"
       *SCR-GOAL-SELECT-RULES*))
  (let ((scr-lst (apply 'append (mapcar #'eval
	     '(*SCR-BINDINGS-REJECT-RULES* *SCR-OP-REJECT-RULES*
	       *SCR-GOAL-REJECT-RULES* *SCR-NODE-REJECT-RULES*
	       *SCR-OP-SELECT-RULES*)))))
    (dolist (scr scr-lst)
	    (ebl-check-scr-exp (get (car scr) 'lhs) (car scr)))))

; ADD-INFO-TO-OPS adds a new property to an operator,
; "all-rhs-lhs-vars", which lists all variables found both in the
; effects-list and the preconditions.

(defun add-info-to-ops ()
    (g-loop (init ops (append *OPERATORS* *INFERENCE-RULES*) op nil)
	    (while (setq op (car (pop ops))))
	    (do (setf (get op 'all-rhs-lhs-vars)
		      (g-map (v in (get op 'all-vars-in-effects))
			     (when (r-memq v (get op 'preconds)))
			     (save v))))))



; ADD-INFO-TO-SCRS will add important information into the SCRs in the
; domain.

(defun add-info-to-scrs ()
  (dolist (rule-body
	   (append *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
		   *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
		   *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		   *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
		   *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
		   *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*))
	  (normalize-scr-for-ebs (car rule-body))))


; on all of these should really use "get-form-from-exp"
; not get-forms-from-exp

(defun normalize-scr-for-ebs (rule-nm)
  (cond ((get rule-nm 'was-learned)
	 (or (get rule-nm 'lhs-for-ebs)
	     (error "normalize-scr: learned rule incomplete"))
	 (setf (get rule-nm 'sig-for-ebs) (get rule-nm 'unique-sig)))
	((member (get rule-nm 'rule-type) '(node-reject node-select))
	 (normalize-node-filter-rule rule-nm))
	((member (get rule-nm 'rule-type) '(goal-reject goal-select))
	 (normalize-goal-filter-rule rule-nm))
	((member (get rule-nm 'rule-type) '(op-reject op-select))
	 (normalize-op-filter-rule rule-nm))
	((member (get rule-nm 'rule-type) '(bindings-reject bindings-select))
	 (normalize-bindings-filter-rule rule-nm))
	((member (get rule-nm 'rule-type) 
		 '(node-pref goal-pref op-pref bindings-pref))
	 nil)
	(t (break))))


; a filter rule is a reject or select


(defun normalize-node-filter-rule (rule)
  (let ((lhs (get rule 'lhs))
	sig lits node)
    (setq lits (get-forms-from-exp 'candidate-node lhs))
    (setq lhs (replace-lits-with-t lits lhs)) 
    (setq node (caddr (get rule 'rhs)))
    (push node sig)
    (and (TF-trimable lhs)
	 (setq lhs (TF-trim lhs)))
    (setf (get rule 'lhs-for-ebs)  lhs)
    (setf (get rule 'sig-for-ebs) (cons 'node-sig sig) )))


(defun normalize-goal-filter-rule (rule)
  (let ((lhs (get rule 'lhs))
	sig lits goal node)
    (setq lits (get-forms-from-exp 'candidate-goal lhs))
    (setq goal (caddr (car lits)))
    (push goal sig)
    (setq lhs (replace-lits-with-t lits lhs))
    (setq lits (get-forms-from-exp 'current-node lhs))
    (setq node (cadr (car lits)))
    (push node sig)
    (or (null (cdr lits))(error "mult nodes"))
    (setq lhs (replace-lits-with-t lits lhs))
    (and (TF-trimable lhs)
	 (setq lhs (TF-trim lhs)))
    (setf (get rule 'lhs-for-ebs)  lhs)
    (setf (get rule 'sig-for-ebs) (cons 'goal-sig sig))))


(defun normalize-op-filter-rule (rule)
  (let ((lhs (get rule 'lhs))
	sig lits op goal node)
    (setq lits (get-forms-from-exp 'candidate-op lhs))
    (setq lhs (replace-lits-with-t lits lhs)) 
    (setq op (caddr (get rule 'rhs)))
    (push op sig)
    (setq lits (get-forms-from-exp 'current-goal lhs))
    (setq goal (caddr (car lits)))
    (push goal sig)
    (or (null (cdr lits))(error "mult goals"))
    (setq lhs (replace-lits-with-t lits lhs))
    (setq lits (get-forms-from-exp 'current-node lhs))
    (setq node (cadr (car lits)))
    (push node sig)
    (or (null (cdr lits))(error "mult nodes"))
    (setq lhs (replace-lits-with-t lits lhs))
    (and (TF-trimable lhs)
	 (setq lhs (TF-trim lhs)))
    (setf (get rule 'lhs-for-ebs) lhs)
    (setf (get rule 'sig-for-ebs) (cons 'op-sig sig) )))


(defun normalize-bindings-filter-rule (rule)
  (let ((lhs (get rule 'lhs))
	sig lits op goal node bindings)
    (setq lits (get-forms-from-exp 'candidate-bindings lhs))
    (setq lhs (replace-lits-with-t lits lhs)) 
    (setq bindings (caddr (get rule 'rhs)))
    (push bindings sig)
    (setq lits (get-forms-from-exp 'current-op lhs))
    (setq lhs (replace-lits-with-t lits lhs)) 
    (setq op (caddr (car lits)))
    (push op sig)
    (or (null (cdr lits))(error "mult ops"))
    (setq lits (get-forms-from-exp 'current-goal lhs))
    (setq goal (caddr (car lits)))
    (push goal sig)
    (or (null (cdr lits))(error "mult goals"))
    (setq lhs (replace-lits-with-t lits lhs))
    (setq lits (get-forms-from-exp 'current-node lhs))
    (setq node (cadr (car lits)))
    (push node sig)
    (or (null (cdr lits))(error "mult nodes"))
    (setq lhs (replace-lits-with-t lits lhs))
    (and (TF-trimable lhs)
	 (setq lhs (TF-trim lhs)))
    (setf (get rule 'lhs-for-ebs) lhs)
    (setf (get rule 'sig-for-ebs) (cons 'bindings-sig sig)))) 




(defun replace-lits-with-t (lits exp)
  (dolist (l lits) 
	  (setq exp (subst t l exp :test #'equal)))
  exp)
