(use-package 'pg)

(defun reset-domain-graphics-parameters ()
)

(defmacro get-nth (n lst sym)
  `(get (nth ,n ,lst) ,sym))

(defun determine-domain-graphics-parameters (problem)
   (declare (special *FONT* *SCALE-FACTOR* *DISKS* *PEGS* *DOMAIN-WINDOW*))
   (setf *SCALE-FACTOR* (* 3 (pg-font-height *FONT*)))
   (setf *Y-OFFSET* (- (pg-window-height *DOMAIN-WINDOW*) *SCALE-FACTOR*))
   (setf *NODE-MSG-X*  10
	 *NODE-MSG-Y*  (truncate *SCALE-FACTOR* (/ 2.66666))
	 *STATE-MSG-X* 10
	 *STATE-MSG-Y* (truncate *SCALE-FACTOR* (/ 3)))

   (setf *DISKS* (mapcar #'second (remove 'on problem :test-not #'eql
				       :key #'car)))
   
   (let* ((relations (remove 'smaller (cdr problem) :test-not #'eql
			     :key #'car))
	  (disk-height (truncate *SCALE-FACTOR* 2))
	  (total-disk-height (* disk-height (length *DISKS*))))

     ;; organize disks.
     (setf *DISKS* (sort *DISKS* #'(lambda (x y) (sort-test relations
							    x y))))
     
     (dotimes (n (length *DISKS*))
       (setf (get-nth n *DISKS* 'disk-data)
	     (disk-data n *SCALE-FACTOR* disk-height))
       (setf (get-nth n *DISKS* 'disk-size) n))


     ;; organize pegs.
     (setf *PEGS* (mapcar #'second (remove 'is-peg (cdr problem) :test-not #'eql
					   :key #'car)))
   
     (setf *PEGS* (sort *PEGS* #'string<))
     (dotimes (n (length *PEGS*))
       (setf (get-nth n *PEGS* 'peg-data)
	     (peg-data n *SCALE-FACTOR* total-disk-height))
       (setf (get-nth n *PEGS* 'disks) nil))))

(defun sort-test (preds p1 p2)
  (find `(smaller ,p1 ,p2) preds :test #'equal))

(defun disk-data (n scale disk-height)
  (declare (fixnum n scale))
  "This function returns a list containing the graphic data for
each disk in the for (left top right base)"
  (list (* (+ 2 n) (truncate scale 5)) ;width
        disk-height
	(* n scale) ;right
	))

(defun peg-data (n scale total-height)
  (declare (fixnum n scale) (special *Y-OFFSET*))
    "This function returns a list containing the graphic data for
each peg in the for (left top right base)"
    (let ((x-offset 100))
      (list (+ (* n scale 3) x-offset) ;left
	    (- *Y-OFFSET* (* 3 scale) total-height) ;top
	    (+ (* n scale 3) x-offset) ;right-- add a little more for width
	    (- *Y-OFFSET* (* 3 scale)) ;base
	    )))
	
(defun delete-domain-graphic-objects (state-preds)
  (dolist (peg (sort-by-pegs state-preds))
          (setf (get (car peg) 'disks) (delete (caadr peg)
					       (get (car peg) 'disks)))
	  (dolist (disk (cadr peg))
		  (erase-disk disk))))

(defun add-domain-graphic-objects (state-preds)
  (dolist (peg (sort-by-pegs state-preds))
    (let ((disks (delete-duplicates 
	   (merge 'list (cadr peg) (get (car peg) 'disks) #'string<))))
      (setf (get (car peg) 'disks) disks)
      (dotimes (n (length disks))
	(setf (get (nth n disks) 'peg) (car peg))
	(draw-disk-on-peg (nth n disks) (car peg) n)))))
)


(defun sort-by-pegs (state)
  "Sort-by-pegs sorts by the peg names, This could get yukky with more
then 10 pegs.  It creats lists that contain a cons of a peg name and
all the disks (sorted by size) on the peg.  For example:

   ((peg1 disk2 disk3) (peg2 disk1))"
  
  (sort (sort-inner-lists state) #'string< :key #'car))

(defun sort-inner-lists (state)
  "Sort-inner-lists takes an a-list with the peg as the key and
a list of disks as data and returns an a-list where each data list
has been sorted according to ."
  (mapcar #'(lambda (x) (list (car x) (sort (cdr x) #'>
					    :key #'(lambda (y)
						     (get y 'disk-size)))))
	  (catagorize-ons state)))
    

(defun catagorize-ons (state)
  "This function takes a list of on preds and returns a list of lists,
where each sub list is an a-list with the key as the peg name and data is
a list of disks"
  (let ((a-list nil))
    (dolist (pred (on-preds state))
       (pushnew (list (third pred)) a-list :key #'car :test #'equal)
       (push (second pred)
	     (cdr (nth (position (third pred) a-list :key #'car) a-list))))
    a-list))

(defun on-preds (state)
  "This function takes the state an returns only those lists
which are ON predicates.  It destructively modifies the list."
  (remove 'on state :test-not #'eq :key #'car))
  

(defun draw-disk (disk disk-data peg-data position)
  (declare (special *DOMAIN-WINDOW* *Y-OFFSET*))
   (let* ((peg-center (truncate (+ (first peg-data) (third peg-data)) 2))
	  (peg-base (fourth peg-data))
	  (new-disk-position (list (- peg-center (first disk-data))
				   (- peg-base (second disk-data) position)
				   (+ peg-center (first disk-data))
				   (- peg-base position))))
     ; remember that the base of the disk depends only on the position
     ; of the disk in the stack, not on its own disk parameters.
    (apply '(lambda (l to r b) (pg-frame-rect *DOMAIN-WINDOW* l to r b))
	   new-disk-position)
;    (pg-write-text *DOMAIN-WINDOW* (first new-disk-position)
;		                   (second new-disk-position)
;				   (symbol-name disk))
    ; store position for the erase function.
    (setf (get disk 'disk-position) new-disk-position)))

(defun erase-disk (disk)
  (declare (special *DOMAIN-WINDOW*))
  (let ((peg (get disk 'peg)))
    (setf (get peg 'disks) (delete disk (get peg 'disks)))
    (setf (get disk 'peg) nil) ;this line is unnecessary

    (apply #'pg-erase-rect (cons *DOMAIN-WINDOW*
				 (get disk 'disk-position)))
    (draw-peg peg))
;    (pg-write-text *DOMAIN-WINDOW* (first (get disk 'disk-position))
;		   (second (get disk 'disk-position))
;		   "      ")
)

(defun draw-domain-background ()
  (declare (special *PEGS*))
  (dolist (peg *PEGS*)
    (draw-peg peg))
  (let ((left-peg-x (first (get (car *PEGS*) 'peg-data)))
	(right-peg-x (first (get (car (last *PEGS*)) 'peg-data)))
	(peg-y (+ 3 (fourth (get (car *PEGS*) 'peg-data)))))

    (pg-frame-rect *DOMAIN-WINDOW* left-peg-x peg-y
		                   right-peg-x (+ 3 peg-y))))

(defun draw-peg (name)
    (declare (special  *DOMAIN-WINDOW*))
    (let* ((peg-data (get name 'peg-data))
	   (left (first peg-data))
	   (top  (second peg-data))
	   (right (third peg-data))
	   (base (+ 3 (fourth peg-data))))

      (pg-frame-rect *DOMAIN-WINDOW* left top right base)))

	  


(defun draw-disk-on-peg (disk peg n)
  (let ((disk-data (get disk 'disk-data)))
    (draw-disk disk disk-data (get peg 'peg-data) (* n (second disk-data)))))

(defun draw-domain-foreground ())
