;;;
;;; VOICELEADING
;;;
;;; Voiceleading is the process of planning a succession of chords in
;;; music such that they are aesthetically pleasing when played or sung.
;;; Each voice (e.g. soprano, alto, tenor, bass) has a particular note
;;; from each chord in the progression.  There are rules which govern
;;; the position of voices in the chord (e.g. they cannot be pitched
;;; too high or too low), relationship of voices to each other in the
;;; chord (e.g. they cannot cross each other), relationship of voices
;;; to each other in successive chords (e.g. they cannot move in paral-
;;; lel if they are certain distances apart), and relationships of like
;;; voices in successive chords (e.g. the tenor note cannot skip to a
;;; note more than a given distance away.) Given a key, the appropriate
;;; valid ranges for the voices, and a chord progression, this domain
;;; plans out the assignment of notes in the chords to the various
;;; voices so that the rules of voiceleading are maintained.  By speci-
;;; fying in the goal what note is to be found for the soprano of each
;;; chord, it is possible to harmonize a melody given its chord progres-
;;; sion.
;;;
;;; After experimenting with several possibilities in representing this
;;; domain, the cleanest approach seemed to be to have only one non-static
;;; closed world predicate called "pitch".  This maps the voice in a
;;; chord of the progression to the note and octave for it.  The many
;;; other non-static predicates are open world, produced by several
;;; inference rules and specify the inferred relationships between
;;; chords in the progression, notes in the chord, voices in one chord
;;; or successive chords, etc.
;;;
;;; The principle static predicates are progchord, range, and key.
;;; Progchord is the relationship between a chord in the progression
;;; and that chord's chord value.  To alleviate confusion, the former
;;; is called something such as <pchord> when used as a variable and
;;; the latter is simply something like <chord>.  Each voice has a
;;; particular range of notes that are valid and that is specified
;;; with the static predicate "range".  To indicate what key voice-
;;; leading is to be performed in, the static predicate "key" must be
;;; in the initial state.
;;;
;;; The following are generators written in LISP:
;;;      (voice <voice>)
;;;      (octave <octave>)
;;;      (tone <tone>)
;;;      (note-in-chord <note> <chord> <key>) - only <note> is generated
;;;      (note-between <note> <octave>
;;;                    <low-note> <low-octave>
;;;                    <high-note> <high-octave>) - only <octave> is
;;;                                                 generated
;;;      (is-tone <note> <part> <chord> <key>) - only <note> is generated
;;;      (common-note <note> <chord1> <chord2> <key>) - only <note> is
;;;                                                     generated
;;;      (nearest-note <note1> <octave1>
;;;                    <note2> <octave2>
;;;                    <pchord2> <key>) - only <note2> and <octave2> are
;;;                                       generated
;;;      (higher-pitch <note1> <octave1>       generate <hi-note>/<hi-octave>
;;;                    <note2> <octave2>
;;;                    <hi-note> <hi-octave>)
;;;      (lower-pitch <note1> <octave1>        generate <lo-note>/<lo-octave>
;;;                    <note2> <octave2>
;;;                    <lo-note> <lo-octave>)
;;;      (match <obj1> <obj2>)
;;;      (diff <minuend> <subtrahend> <difference>)
;;;
;;; A BRIEF INTRODUCTION TO MUSIC THEORY
;;; 
;;; Music consists of pitches (audio frequencies) sounded in succession
;;; (called "melody") and/or sounded simultaneously (called "harmony").
;;; In western music, we have chosen a subset of all possible pitches on
;;; which to base our music.  The relationship between any of these
;;; pitches and the next higher or lower pitch on either side of it is
;;; called a "semitone". The relationship to the second pitch on either
;;; side is called a "whole tone".
;;; 
;;; In terms of physics, the ratio of the frequency of any pitch to the
;;; semitone below it is the 12th root of 2.  Therefore, there are
;;; twelve semitones between any pitch and the pitch which is twice its
;;; frequency (which is defined to be an "octave" higher) or half its
;;; frequency (an octave lower).  Various subsets of these twelve pitches,
;;; arranged in either ascending or descending order, comprise what are
;;; called "scales".  One of the most popular is called the "major scale"
;;; which has seven tones (called scale "degrees"). It has a particular
;;; arrangement of whole tones and semitones between successive pitches.
;;; These seven degrees of the scale are designated by the Roman numbers
;;; I, II,...,VII.  Each also has a special name, but the most important
;;; are the tonic (I), mediant (III), subdominant (IV), dominant (V),
;;; and leading tone (VII).  Remember that these refer to notes indepen-
;;; dent of the octave in which they are found.  Therefore, the note which
;;; is III in one octave is also III when the pitch is one octave higher
;;; or lower.
;;; 
;;; Two notes form an "interval" (either melodically or harmonically.)
;;; Intervals are specified by the distance between the two notes.  For
;;; example, from I to III is a third and from IV to VII is a fourth.
;;; Three notes form a "triad" and three or more notes sounded together
;;; form a "chord". There are certain triads that are of special interest.
;;; They are formed by taking one of the scale degrees and adding to it
;;; the second and fourth degrees higher.  For example, I-III-V form one
;;; of these special triads.  So does V-VII-II (note the "wrap-around").
;;; The first note in these triads is called the "root" and the other two
;;; in order are called the "third" and the "fifth" (because of the inter-
;;; vals formed with the root).   Rather than explicitly naming the notes
;;; in each of these triads, they are specified simply by their root (e.g.
;;; I instead if I-III-V).  It is not necessary that the tones in a triad
;;; be sounded such that the pitch of the root is lower than that of the
;;; third and fifth.  For example, with an IV triad, the lowest pitch may
;;; be VI, followed by IV, followed by I.  This rearrangement is called
;;; an "inversion".  If the third is lowest, it is a "first inversion".
;;; If the fifth is lowest, it is a "second inversion".
;;; 
;;; Up to this point, nothing has been said about what scale degrees map
;;; to what specific audio frequencies.  This is because the two are
;;; independent.  Specific audio frequencies within an octave are given
;;; the letters A through G, possibly modified by a "sharp" or "flat".
;;; For a specific "key", we could map I (the tonic) to C.  For a major
;;; scale, this would in turn map II to D, III to E, and so on up to map-
;;; ping VII to B (note the "wrap-around" again.)  This particular mapping
;;; is called the "key of C major" and there are no sharps or flats in
;;; this key.
;;; 
;;; Voiceleading is the process of planning a succession of chords such
;;; that they are aesthetically pleasing when played.  In the eighteenth
;;; and nineteenth centuries, composers developed a set of rules to guide
;;; voiceleading.  This has been referred to as "common practice" voice-
;;; leading.  In the twentieth century, composers began breaking the
;;; rules, but common practice voiceleading is the foundation for the
;;; vast majority of modern mainstream music.
;;; 
;;; Suppose, for example, that a III chord is being sung by a choir. 
;;; The sopranos may sing the V, the altos the III, the tenors the VII,
;;; and the basses "doubling" the altos an octave lower, also on a III.
;;; If the next chord in the song is a V, the rules of voiceleading
;;; dictate the possibilities of who will sing the V, the VII, and the
;;; II and what note will be doubled.  This is also subject to other
;;; constraints on the notes of the chord itself.  For example, giving
;;; the II to the sopranos may mean that the note is too high to be sung
;;; in the particular key given or if it sung an octave lower, it will
;;; be lower than the altos (both situations are undesirable.)
;;;

;;;
;;; Operators
;;;

(setq *OPERATORS* '(

;;;
;;; ADD-VOICE
;;; Add a soprano, alto, tenor, or bass note to a chord in the progression.
;;; Only static generators occur in the preconditions; none of them will
;;; be subgoaled upon.
;;;

(ADD-VOICE
 (params (<pchord> <voice> <note> <octave>))
 (preconds (and (progchord <pchord> <chord>)
	        (voice <voice>)
	        (key <key>)
	        (note-in-chord <note> <chord> <key>) ; static gen for <note>
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(octave <octave>)
		(note-between <note> <octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)))
 (effects ((add (pitch <pchord> <voice> <note> <octave>)))))

;;;
;;; DOUBLE-VOICE
;;; Take an existing note in a chord and double it in another voice.
;;;

(DOUBLE-VOICE
 (params (<pchord> <voice1> <voice2> <note> <octave> <new-octave>))
 (preconds (and (progchord <pchord> <chord>)
	        (voice <voice1>)
	        (voice <voice2>)
		(~ (match <voice1> <voice2>))        ; voices not equal
	        (key <key>)
	        (note-in-chord <note> <chord> <key>) ; static gen for <note>
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(octave <new-octave>)
		(note-between <note> <new-octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(octave <octave>)
		(pitch <pchord> <voice1> <note> <octave>)))
 (effects ((add (pitch <pchord> <voice2> <note> <new-octave>)))))

;;;
;;; DELETE-VOICE
;;; Remove the note for a voice from a chord in the progression.
;;;

(DELETE-VOICE
 (params (<pchord> <voice> <note> <octave>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice>)
		(key <key>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
		(octave <octave>)
		(pitch <pchord> <voice> <note> <octave>)))
 (effects ((del (pitch <pchord> <voice> <note> <octave>)))))

;;;
;;; COMMON-TONE-COPY
;;; For two successive chords in a the progression, copy a note which is in
;;; the first chord and common to the second chord to the same voice in the
;;; second chord.  This is a rule-of-thumb operation for helping to generate
;;; progressions of chords that follow the voiceleading rules.
;;;

(COMMON-TONE-COPY
 (params (<pchord1> <pchord2> <note>))
 (preconds (and (progchord <pchord1> <chord1>)
	        (progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(key <key>)
		(common-note <note> <pchord1> <pchord2> <key>) ; static gen
		(voice <voice>)
		(or (~ (match <voice> soprano)) ; special case: don't copy
		    (~ (match <chord1> II))     ; the soprano of a II-V
		    (~ (match <chord2> V)))     ; progression
		(octave <octave>)
		(pitch <pchord1> <voice> <note> <octave>)))
 (effects ((add (pitch <pchord2> <voice> <note> <octave>)))))

;;;
;;; NEAREST-TONE-COPY
;;; For two successive chords in a the progression, create in the second
;;; chord the note nearest the note in the first chord and in the same
;;; voice.  This is another rule-of-thumb operation for helping to generate
;;; progressions of chords that follow the voiceleading rules.
;;;

(NEAREST-TONE-COPY
 (params (<pchord1> <pchord2> <note>))
 (preconds (and (progchord <pchord1> <chord1>)
	        (progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(or (~ (match <chord1> V))     ; special case: don't do this
		    (~ (match <chord2> VI)))   ; for a V-VI progression
		(key <key>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
		(nearest-note <note1> <octave1> <note2> <octave2>
                              <chord2> <key>)   ; static gen: <note2> <octave2>
		(voice <voice>)
		(pitch <pchord1> <voice> <note1> <octave1>)))
 (effects ((add (pitch <pchord2> <voice> <note2> <octave2>)))))

;;;
;;; TRANSPOSE
;;; Move the pitch for a voice in a chord into the valid range for that
;;; voice.
;;;

(TRANSPOSE
 (params (<pchord> <voice> <note> <new-octave>))
 (preconds (and (progchord <pchord> <chord>)
		(key <key>)
		(voice <voice>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(octave <new-octave>)
		(note-between <note> <new-octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(octave <old-octave>)
		(pitch <pchord> <voice> <note> <old-octave>)))
 (effects ((add (pitch <pchord> <voice> <note> <new-octave>))
           (del (pitch <pchord> <voice> <note> <old-octave>)))))

;;;
;;; ADD-TONE
;;; Add a specific tone of the chord to a chord in the progression (i.e.
;;; root, third, fifth).
;;;

(ADD-TONE
 (params (<pchord> <voice> <tone> <octave>))
 (preconds (and (progchord <pchord> <chord>)
	        (voice <voice>)
	        (key <key>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
		(tone <tone>)
	        (is-tone <note> <tone> <chord> <key>)
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(octave <octave>)
	        (note-between <note> <octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)))
 (effects ((add (pitch <pchord> <voice> <note> <octave>)))))

;;;
;;; EXCHANGE-NOTES
;;; Take two notes in a chord and exchange them between the two voices
;;; to which they are assigned.  This is valuable when voices have crossed
;;; (i.e. alto is higher than soprano).
;;; 

(EXCHANGE-NOTES
 (params (<pchord> <voice1> <voice2>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice1>)
		(voice <voice2>)
		(~ (match <voice1> <voice2>)) ; voices can't be the same
		(key <key>)
		(note-in-chord <note1> <chord> <key>) ; static gen for <note1>
		(octave <octave2>)
		(pitch <pchord> <voice1> <note1> <octave1>)
		(note-in-chord <note2> <chord> <key>) ; static gen for <note2>
		(octave <octave1>)
		(pitch <pchord> <voice2> <note2> <octave2>)))
 (effects ((del (pitch <pchord> <voice1> <note1> <octave1>))
           (del (pitch <pchord> <voice2> <note2> <octave2>))
	   (add (pitch <pchord> <voice1> <note2> <octave2>))
           (add (pitch <pchord> <voice2> <note1> <octave1>)))))

;;;
;;; INVERT-CHORD-UP
;;; For a chord in the progression, move the soprano note to bass, bass to
;;; tenor, tenor to alto, and alto to soprano.  In the process, put each
;;; into an octave in its range.
;;;

(INVERT-CHORD-UP
 (params (<pchord>))
 (preconds (and (progchord <pchord> <chord>)
		(key <key>)
		(note-in-chord <s-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <a-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <t-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <b-note> <chord> <key>) ; static gen for <note>
                (range <voice> <s-low> <s-low-oct> <s-hi> <s-hi-oct>)
	        (note-between <a-note> <s-octave-new>  ; static gen
			      <s-low> <s-low-oct>
			      <s-hi> <s-hi-oct>)
                (range <voice> <a-low> <a-low-oct> <a-hi> <a-hi-oct>)
	        (note-between <t-note> <a-octave-new>  ; static gen
			      <a-low> <a-low-oct>
			      <a-hi> <a-hi-oct>)
                (range <voice> <s-low> <s-low-oct> <s-hi> <s-hi-oct>)
	        (note-between <b-note> <t-octave-new>  ; static gen
			      <t-low> <t-low-oct>
			      <t-hi> <t-hi-oct>)
                (range <voice> <b-low> <b-low-oct> <b-hi> <b-hi-oct>)
	        (note-between <s-note> <b-octave-new>  ; static gen
			      <b-low> <b-low-oct>
			      <b-hi> <b-hi-oct>)
		(octave <s-octave>)
		(octave <a-octave>)
		(octave <t-octave>)
		(octave <b-octave>)
		(pitch <pchord> soprano <s-note> <s-octave>)
		(pitch <pchord> alto <a-note> <a-octave>)
		(pitch <pchord> tenor <t-note> <t-octave>)
		(pitch <pchord> bass <b-note> <b-octave>)))
 (effects ((add (pitch <pchord> soprano <a-note> <s-octave-new>))
           (add (pitch <pchord> alto <t-note> <a-octave-new>))
           (add (pitch <pchord> tenor <b-note> <t-octave-new>))
           (add (pitch <pchord> bass <s-note> <b-octave-new>))
           (del (pitch <pchord> soprano <s-note> <s-octave>))
           (del (pitch <pchord> alto <a-note> <a-octave>))
           (del (pitch <pchord> tenor <t-note> <t-octave>))
           (del (pitch <pchord> bass <b-note> <b-octave>)))))

;;;
;;; INVERT-CHORD-DOWN
;;; For a chord in the progression, move the bass note to soprano, tenor
;;; to bass, alto to tenor, and soprano to alto.  In the process, put each
;;; into an octave in its range.
;;;

(INVERT-CHORD-DOWN
 (params (<pchord>))
 (preconds (and (progchord <pchord> <chord>)
		(key <key>)
		(note-in-chord <s-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <a-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <t-note> <chord> <key>) ; static gen for <note>
		(note-in-chord <b-note> <chord> <key>) ; static gen for <note>
                (range <voice> <s-low> <s-low-oct> <s-hi> <s-hi-oct>)
	        (note-between <b-note> <s-octave-new> ; static gen
			      <s-low> <s-low-oct>
			      <s-hi> <s-hi-oct>)
                (range <voice> <a-low> <a-low-oct> <a-hi> <a-hi-oct>)
	        (note-between <s-note> <a-octave-new> ; static gen
			      <a-low> <a-low-oct>
			      <a-hi> <a-hi-oct>)
                (range <voice> <s-low> <s-low-oct> <s-hi> <s-hi-oct>)
	        (note-between <t-note> <t-octave-new> ; static gen
			      <t-low> <t-low-oct>
			      <t-hi> <t-hi-oct>)
                (range <voice> <b-low> <b-low-oct> <b-hi> <b-hi-oct>)
	        (note-between <a-note> <b-octave-new> ; static gen
			      <b-low> <b-low-oct>
			      <b-hi> <b-hi-oct>)
		(octave <s-octave>)
		(octave <a-octave>)
		(octave <t-octave>)
		(octave <b-octave>)
		(pitch <pchord> soprano <s-note> <s-octave>)
		(pitch <pchord> alto <a-note> <a-octave>)
		(pitch <pchord> tenor <t-note> <t-octave>)
		(pitch <pchord> bass <b-note> <b-octave>)))
 (effects ((add (pitch <pchord> soprano <b-note> <s-octave-new>))
           (add (pitch <pchord> alto <s-note> <a-octave-new>))
           (add (pitch <pchord> tenor <t-note> <t-octave-new>))
           (add (pitch <pchord> bass <a-note> <b-octave-new>))
           (del (pitch <pchord> soprano <s-note> <s-octave>))
           (del (pitch <pchord> alto <a-note> <a-octave>))
           (del (pitch <pchord> tenor <t-note> <t-octave>))
           (del (pitch <pchord> bass <b-note> <b-octave>)))))

))

;;;
;;; Inference rules
;;;

(setq *INFERENCE-RULES* '(

;;;
;;; INFER-PART
;;; Infer that there is a particular voice (ie. soprano, alto, tenor, or
;;; bass) in a particular chord in the progression.
;;;

(INFER-PART
 (params (<pchord> <voice> <note>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice>)
		(key <key>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
		(octave <octave>)
                (range <voice> <low-note> <low-octave> <high-note> <high-octave>)
	        (note-between <note> <octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord> <voice> <note> <octave>)))
 (effects ((add (part <pchord> <voice>)))))

;;;
;;; INFER-VALID-CHORD-PROGRESSION
;;; Infer that the succession of chords in the state is valid.  There are
;;; two parts to this: all the notes in each chord must be correct and
;;; all the relationships between successive chords must be correct.
;;;

(INFER-VALID-CHORD-PROGRESSION
 (params nil)
 (preconds (and (forall (<pchord>)
		        (progchord <pchord> <chord>)
		        (valid-chord <pchord>))
	        (forall (<pchord1> <pchord2>)
		        (follows <pchord1> <pchord2>)
		        (valid-progression <pchord1> <pchord2>))))
 (effects ((add (valid-chord-progression)))))

;;;
;;; INFER-VALID-CHORD
;;; Infer that the notes in a chord in the progression are correct.  There
;;; are several requirements: (1) each voice must have a pitch, (2) no
;;; voice may have more than one pitch, (3) each voice in the chord must
;;; be within its valid range, (4) the chord must have a root, (5) the
;;; chord must have a third, (6) and the voices must be ordered highest to
;;; lowest as soprano, alto, tenor, and bass.
;;;

(INFER-VALID-CHORD
 (params (<pchord>))
 (preconds (and (progchord <pchord> <chord>)
		(full-chord <pchord>)
		(forall (<voice>)
			(voice <voice>)
			(and (single-part <pchord> <voice>)
			     (range-valid <pchord> <voice>)))
		(has-tone <pchord> 1)
		(has-tone <pchord> 3)
		(voices-ordered <pchord> soprano alto)
		(voices-ordered <pchord> alto tenor)
		(voices-ordered <pchord> tenor bass)))
 (effects ((add (valid-chord <pchord>)))))

;;;
;;; INFER-SINGLE-PART
;;; Infer that a voice of a chord in the progression has only one
;;; note assigned to it.
;;;

(INFER-SINGLE-PART
 (params (<pchord> <voice> <note1> <octave1>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice>)
		(key <key>)
		(note-in-chord <note1> <chord> <key>) ; static gen for <note1>
		(octave <octave1>)
		(pitch <pchord> <voice> <note1> <octave1>)
		(forall (<note2>)
		        (note-in-chord <note2> <chord> <key>) ; static gen
			(forall (<octave2>)
				(octave <octave2>)
				(or (and (match <note1> <note2>)
		                         (match <octave1> <octave2>))
		                    (~ (pitch <pchord> <voice>
					      <note2> <octave2>)))))))
 (effects ((add (single-part <pchord> <voice>)))))

;;;
;;; INFER-FULL-CHORD
;;; Infer that all voices in a chord in the progression have notes.
;;;

(INFER-FULL-CHORD
 (params (<pchord>))
 (preconds (and (progchord <pchord> <chord>)
	        (forall (<voice>) (voice <voice>) (part <pchord> <voice>))))
 (effects ((add (full-chord <pchord>)))))

;;;
;;; INFER-RANGE-VALID
;;; Infer that the note assigned to voice in a chord in the progression has
;;; a pitch within its valid range.
;;;

(INFER-RANGE-VALID
 (params (<pchord> <voice>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice>)
                (range <voice> <low-note> <low-octave> <high-note> <high-octave>)
		(key <key>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
		(octave <octave>)
	        (note-between <note> <octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord> <voice> <note> <octave>)))
 (effects ((add (range-valid <pchord> <voice>)))))

;;;
;;; INFER-HAS-TONE
;;; Infer that a chord in the progression has a particular tone in the
;;; chord (ie. root, third, or fifth).
;;;

(INFER-HAS-TONE
 (params (<pchord> <tone>))
 (preconds (and (progchord <pchord> <chord>)
		(tone <tone>)
		(key <key>)
		(note-in-chord <note> <chord> <key>) ; static gen for <note>
	        (is-tone <note> <tone> <chord> <key>)
		(voice <voice>)
		(octave <octave>)
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(note-between <note> <octave>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord> <voice> <note> <octave>)))
 (effects ((add (has-tone <pchord> <tone>)))))

;;;
;;; INFER-VOICES-ORDERED
;;; Infer that two voices in a chord in the progression are have pitches
;;; such that the first is higher than the second.
;;;

(INFER-VOICES-ORDERED
 (params (<pchord> <voice1> <note1> <octave1> <voice2> <note2> <octave2>))
 (preconds (and (progchord <pchord> <chord>)
		(voice <voice1>)
		(voice <voice2>)
		(key <key>)
	        (note-in-chord <note1> <chord> <key>) ; static gen for <note>
	        (note-in-chord <note2> <chord> <key>) ; static gen for <note>
		(octave <octave1>)
		(octave <octave2>)
		(higher-pitch <note1> <octave1>       ; gen hi-note/hi-octave
			      <note2> <octave2>
			      <hi-note> <hi-octave>)
		(lower-pitch <note1> <octave1>        ; gen lo-note/lo-octave
		             <note2> <octave2>
			     <lo-note> <lo-octave>)
	        (pitch <pchord> <voice1> <hi-note> <hi-octave>)
	        (pitch <pchord> <voice2> <lo-note> <lo-octave>)))
 (effects ((add (voices-ordered <pchord> <voice1> <voice2>)))))

;;;
;;; INFER-VALID-PROGRESSION
;;; Infer that the all relationships between two successive chords are
;;; correct.  This involves: (1) there can be no parallel octaves, (2)
;;; there can be no parallel fifths, (3) the tenor and alto voices cannot
;;; change by more than a fourth, (4) all voices in the first chord cannot
;;; move the same direction (up or down) to get to the second chord,
;;; (5) there can be no direct octaves or fifths, (6) voices cannot
;;; overlap, (7) certain constraints on II-V and V-VI progressions must
;;; be met.
;;;

(INFER-VALID-PROGRESSION
 (params (<pchord1> <pchord2>))
 (preconds (and (progchord <pchord1> <chord1>)
                (progchord <pchord2> <chord2>)
                (follows <pchord1> <pchord2>)
		(forall (<voice1>)
			(voice <voice1>)
			(forall (<voice2>)
				(voice <voice2>)
				(or (match <voice1> <voice2>)
				    (and (no-parallel-octave <pchord1>
							     <pchord2>
					                     <voice1>
							     <voice2>)
				         (no-parallel-fifth <pchord1>
							    <pchord2>
					                    <voice1>
							    <voice2>)
				         (no-direct-octave <pchord1>
							   <pchord2>
					                   <voice1>
							   <voice2>)
				         (no-direct-fifth <pchord1>
							  <pchord2>
					                  <voice1>
							  <voice2>)))))
		(small-skip <pchord1> <pchord2> tenor)
		(small-skip <pchord1> <pchord2> alto)
		(dissimilar-motion <pchord1> <pchord2>)
	        (voices-not-overlapped <pchord1> <pchord2> soprano alto)
	        (voices-not-overlapped <pchord1> <pchord2> alto tenor)
	        (voices-not-overlapped <pchord1> <pchord2> tenor bass)
	        (voices-not-overlapped <pchord2> <pchord1> soprano alto)
	        (voices-not-overlapped <pchord2> <pchord1> alto tenor)
	        (voices-not-overlapped <pchord2> <pchord1> tenor bass)
		(or (valid-II-V <pchord1> <pchord2>)
		    (~ (match <pchord1> II))
		    (~ (match <pchord2> V)))
		(or (valid-V-VI <pchord1> <pchord2>)
		    (~ (match <pchord1> V))
		    (~ (match <pchord2> VI)))))
 (effects ((add (valid-progression <pchord1> <pchord2>)))))

;;;
;;; INFER-NO-PARALLEL-OCTAVE
;;; Infer that two voices in successive chords are not an octave apart
;;; in both chords. (This also includes multiples of octaves.)
;;;

(INFER-NO-PARALLEL-OCTAVE
 (params (<pchord1> <pchord2>
	  <voice1> <note1> <octave1>
	  <voice2> <note2> <octave2>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(voice <voice1>)
                (range <voice1> <low-note1> <low-octave1>
                       <high-note1> <high-octave1>)
		(voice <voice2>)
                (range <voice2> <low-note2> <low-octave2>
                       <high-note2> <high-octave2>)
                (key <key>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
		(note-between <note1> <octave1>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord1> <voice1> <note1> <octave1>)
		(note-in-chord <note2> <chord1> <key>) ; static gen for <note2>
		(octave <octave2>)
		(note-between <note2> <octave2>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord1> <voice2> <note2> <octave2>)
		(note-in-chord <note3> <chord2> <key>) ; static gen for <note3>
		(octave <octave3>)
		(note-between <note3> <octave3>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord2> <voice1> <note3> <octave3>)
		(note-in-chord <note4> <chord2> <key>) ; static gen for <note4>
		(octave <octave4>)
		(note-between <note4> <octave4>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord2> <voice2> <note4> <octave4>)
		(diff <octave1> <octave2> <diff1>)
		(diff <octave3> <octave4> <diff2>)
		(or (~ (match <note1> <note2>))
		    (~ (match <note3> <note4>))
		    (~ (match <diff1> <diff2>)))))
 (effects ((add (no-parallel-octave <pchord1> <pchord2> <voice1> <voice2>)))))

;;;
;;; INFER-NO-PARALLEL-FIFTH
;;; Infer that two voices in successive chords are not a fifth apart
;;; in both chords. (This also includes variations on fifths such
;;; as a 12th.)  This differs from INFER-NO-PARALLEL-OCTAVE in that by
;;; the nature of the triads, the chord tones are specific.
;;;

(INFER-NO-PARALLEL-FIFTH
 (params (<pchord1> <pchord2>
	  <voice1> <note1> <octave1>
	  <voice2> <note2> <octave2>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(voice <voice1>)
                (range <voice1> <low-note1> <low-octave1>
                       <high-note1> <high-octave1>)
		(voice <voice2>)
                (range <voice2> <low-note2> <low-octave2>
                       <high-note2> <high-octave2>)
                (key <key>)
	        (is-tone <root1> 1 <chord1> <key>)     ; static gen for <root1>
	        (is-tone <fifth1> 5 <chord1> <key>)    ; static gen for <fifth1>
	        (is-tone <root2> 1 <chord2> <key>)     ; static gen for <root2>
	        (is-tone <fifth2> 5 <chord2> <key>)    ; static gen for <fifth2>
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
		(note-between <note1> <octave1>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord1> <voice1> <note1> <octave1>)
		(note-in-chord <note2> <chord1> <key>) ; static gen for <note2>
		(octave <octave2>)
		(note-between <note2> <octave2>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord1> <voice2> <note2> <octave2>)
		(note-in-chord <note3> <chord2> <key>) ; static gen for <note3>
		(octave <octave3>)
		(note-between <note3> <octave3>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord2> <voice1> <note3> <octave3>)
		(note-in-chord <note4> <chord2> <key>) ; static gen for <note4>
		(octave <octave4>)
		(note-between <note4> <octave4>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord2> <voice2> <note4> <octave4>)
		(diff <octave1> <octave2> <diff1>)
		(diff <octave3> <octave4> <diff2>)
		(or (~ (match <note1> <root1>))
		    (~ (match <note2> <fifth1>))
		    (~ (match <note3> <root2>))
		    (~ (match <note4> <fifth2>))
		    (~ (match <diff1> <diff2>)))))
 (effects ((add (no-parallel-fifth <pchord1> <pchord2> <voice1> <voice2>)))))

;;;
;;; INFER-SMALL-SKIP
;;; Infer that a particular voice in two successive chords moves no
;;; farther than a fourth (five semitones) from one to the next.
;;;

(INFER-SMALL-SKIP
 (params (<pchord1> <voice> <note1> <octave1>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(voice <voice>)
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(key <key>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note>
		(octave <octave1>)
		(note-between <note1> <octave1>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord1> <voice> <note1> <octave1>)
		(note-in-chord <note2> <chord2> <key>) ; static gen for <note>
		(octave <octave2>)
		(note-between <note2> <octave2>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord2> <voice> <note2> <octave2>)
		(smaller-interval <note1> <octave1> <note2> <octave2> 5)))
 (effects ((add (small-skip <pchord1> <pchord2> <voice>)))))

;;;
;;; INFER-DISSIMILAR-MOTION
;;; Infer that not all the voices between two successive chords either all
;;; move up or all move down to get from the first to the second.
;;;

(INFER-DISSIMILAR-MOTION
 (params (<pchord1> <note1> <octave1> <pchord2> <note3> <octave3>))
 (preconds (and (progchord <pchord1> <chord1>)
	        (progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(key <key>)
                (range soprano <s-low-note> <s-low-octave>
                       <s-high-note> <s-high-octave>)
		(voice <voice>)
		(~ (match soprano <voice>))
                (range <voice> <low-note> <low-octave>
                       <high-note> <high-octave>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
	        (note-between <note1> <octave1>
			      <s-low-note> <s-low-octave>
			      <s-high-note> <s-high-octave>)
		(pitch <pchord1> soprano <note1> <octave1>)
		(note-in-chord <note2> <chord1> <key>) ; static gen for <note2>
		(octave <octave2>)
	        (note-between <note2> <octave2>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord1> <voice> <note2> <octave2>)
		(note-in-chord <note3> <chord2> <key>) ; static gen for <note3>
		(octave <octave3>)
	        (note-between <note3> <octave3>
			      <s-low-note> <s-low-octave>
			      <s-high-note> <s-high-octave>)
		(pitch <pchord2> soprano <note3> <octave3>)
		(note-in-chord <note4> <chord2> <key>) ; static gen for <note4>
		(octave <octave4>)
	        (note-between <note4> <octave4>
			      <low-note> <low-octave>
			      <high-note> <high-octave>)
		(pitch <pchord2> <voice> <note4> <octave4>)
		(or (and (note-above <note1> <octave1> <note3> <octave3>)
		         (note-above <note4> <octave4> <note2> <octave2>))
		    (and (note-above <note3> <octave3> <note1> <octave1>)
		         (note-above <note2> <octave2> <note4> <octave4>)))))
 (effects ((add (dissimilar-motion <pchord1> <pchord2>)))))

;;;
;;; INFER-NO-DIRECT-OCTAVE
;;; Infer that a direct octave exists between two chords.  Given two voices,
;;; it is not valid for both to move in the same direction from the first
;;; chord to the second by an interval greater than a whole tone if their
;;; spacing in the second chord is an octave (or multiple of an octave).
;;;

(INFER-NO-DIRECT-OCTAVE
 (params (<pchord1> <pchord2>
	  <voice1> <note1> <octave1>
	  <voice2> <note2> <octave2>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(voice <voice1>)
                (range <voice1> <low-note1> <low-octave1>
                       <high-note1> <high-octave1>)
		(voice <voice2>)
                (range <voice2> <low-note2> <low-octave2>
                       <high-note2> <high-octave2>)
                (key <key>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
		(note-between <note1> <octave1>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord1> <voice1> <note1> <octave1>)
		(note-in-chord <note2> <chord1> <key>) ; static gen for <note2>
		(octave <octave2>)
		(note-between <note2> <octave2>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord1> <voice2> <note2> <octave2>)
		(note-in-chord <note3> <chord2> <key>) ; static gen for <note3>
		(octave <octave3>)
		(note-between <note3> <octave3>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord2> <voice1> <note3> <octave3>)
		(note-in-chord <note4> <chord2> <key>) ; static gen for <note4>
		(octave <octave4>)
		(note-between <note4> <octave4>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord2> <voice2> <note4> <octave4>)
		(or (~ (match <note3> <note4>))
		    (and (note-above <note1> <octave1> <note3> <octave3>)
		         (note-above <note4> <octave4> <note2> <octave2>))
		    (and (note-above <note3> <octave3> <note1> <octave1>)
		         (note-above <note2> <octave2> <note4> <octave4>))
		    (smaller-interval <note1> <octave1> <note3> <octave3> 2)
		    (smaller-interval <note2> <octave2> <note4> <octave4> 2))))
 (effects ((add (no-direct-octave <pchord1> <pchord2> <voice1> <voice2>)))))

;;;
;;; INFER-DIRECT-FIFTH
;;; Infer that a direct fifth exists between two chords.  Given two voices,
;;; it is not valid for both to move in the same direction from the first
;;; chord to the second by an interval greater than a whole tone if their
;;; spacing in the second chord is an fifth (or a variation on a fifth
;;; such as a 12th.)  This differs from INFER-DIRECT-OCTAVE in that by the
;;; nature of the triads, the chord tones are specific.
;;;

(INFER-NO-DIRECT-FIFTH
 (params (<pchord1> <pchord2>
	  <voice1> <note1> <octave1>
	  <voice2> <note2> <octave2>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(follows <pchord1> <pchord2>)
		(voice <voice1>)
                (range <voice1> <low-note1> <low-octave1>
                       <high-note1> <high-octave1>)
		(voice <voice2>)
                (range <voice2> <low-note2> <low-octave2>
                       <high-note2> <high-octave2>)
                (key <key>)
	        (is-tone <root> 1 <chord2> <key>)      ; static gen for <root>
	        (is-tone <fifth> 5 <chord2> <key>)     ; static gen for <fifth>
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
		(note-between <note1> <octave1>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord1> <voice1> <note1> <octave1>)
		(note-in-chord <note2> <chord1> <key>) ; static gen for <note2>
		(octave <octave2>)
		(note-between <note2> <octave2>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord1> <voice2> <note2> <octave2>)
		(note-in-chord <note3> <chord2> <key>) ; static gen for <note3>
		(octave <octave3>)
		(note-between <note3> <octave3>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord2> <voice1> <note3> <octave3>)
		(note-in-chord <note4> <chord2> <key>) ; static gen for <note4>
		(octave <octave4>)
		(note-between <note4> <octave4>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord2> <voice2> <note4> <octave4>)
		(or (~ (match <note3> <root>))
		    (~ (match <note4> <fifth>))
		    (and (note-above <note1> <octave1> <note3> <octave3>)
		         (note-above <note4> <octave4> <note2> <octave2>))
		    (and (note-above <note3> <octave3> <note1> <octave1>)
		         (note-above <note2> <octave2> <note4> <octave4>))
		    (smaller-interval <note1> <octave1> <note3> <octave3> 2)
		    (smaller-interval <note2> <octave2> <note4> <octave4> 2))))
 (effects ((add (no-direct-fifth <pchord1> <pchord2> <voice1> <voice2>)))))

;;;
;;; INFER-VOICES-NOT-OVERLAPPED
;;; Infer that two voices do not overlap between two chords.  For
;;; example, it is not valid for the soprano in the second chord to
;;; be lower than the alto in the first chord.
;;;

(INFER-VOICES-NOT-OVERLAPPED
 (params (<pchord1> <pchord2> <voice1> <voice2> <note1> <octave1>))
 (preconds (and (progchord <pchord1> <chord1>)
		(progchord <pchord2> <chord2>)
		(key <key>)
                (voice <voice1>)
                (range <voice1> <low-note1> <low-octave1>
                       <high-note1> <high-octave1>)
		(note-in-chord <note1> <chord1> <key>) ; static gen for <note1>
		(octave <octave1>)
	        (note-between <note1> <octave1>
			      <low-note1> <low-octave1>
			      <high-note1> <high-octave1>)
		(pitch <pchord1> <voice1> <note1> <octave1>)
                (voice <voice2>)
                (range <voice2> <low-note2> <low-octave2>
                       <high-note2> <high-octave2>)
		(note-in-chord <note2> <chord2> <key>) ; static gen for <note2>
		(octave <octave2>)
	        (note-between <note2> <octave2>
			      <low-note2> <low-octave2>
			      <high-note2> <high-octave2>)
		(pitch <pchord2> <voice2> <note2> <octave2>)
		(note-above <note1> <octave1> <note2> <octave2>)))
 (effects ((add (voices-not-overlapped <pchord1> <pchord2> <voice1> <voice2>)))))

;;;
;;; INFER-VALID-II-V
;;; Infer a valid II-V chord progression.  For this special case, the
;;; common tone between the two chords (ie. II) is not duplicated in
;;; the same voice.
;;;

(INFER-VALID-II-V
 (params (<pchord1> <pchord2>))
 (preconds (and (progchord <pchord1> II)
		(progchord <pchord2> V)
		(follows <pchord1> <pchord2>)
		(key <key>)
		(is-tone <note> 1 II <key>)
		(voice <voice>)
		(octave <octave>)
		(pitch <pchord1> <voice> <note> <octave>)
		(~ (pitch <pchord2> <voice> <note> <octave>))))
 (effects ((add (valid-II-V <pchord1> <pchord2>)))))

;;;
;;; INFER-VALID-V-VI
;;; Infer a valid V-VI progression.  For this special case, if the
;;; leading tone of V is in the soprano, it moves up to the tonic
;;; in VI.
;;;
 
(INFER-VALID-V-VI
 (params (<pchord1> <pchord2>))
 (preconds (and (progchord <pchord1> V)
		(progchord <pchord2> VI)
		(follows <pchord1> <pchord2>)
		(key <key>)
		(is-tone <note1> 3 V <key>)      ; static gen for <note1>
		(is-tone <note2> 3 VI <key>)     ; static gen for <note2>
		(octave <octave1>)
		(octave <octave2>)
		(or (and (pitch <pchord1> soprano <note1> <octave1>)
		         (pitch <pchord2> soprano <note2> <octave2>))
		    (and (~ (pitch <pchord1> soprano <note1> <octave1>))
		         (~ (pitch <pchord2> soprano <note2> <octave2>))))))
 (effects ((add (valid-V-VI <pchord1> <pchord2>)))))
 
))
