;;; ~/my-domains/jobsite/domain.lisp
;;;
;;; AUTHOR : Bruce LeSourd
;;;          c473ak
;;;
;;; DATE   : 5/12/91
;;;
;;; PROJECT - domain definition

;;; ******************************************
;;; OPERATORS
;;; ******************************************

"
An example problem space:

(project-space '(3 3 1 1))
    FLOORS
 -1   0   1
 012 012 012
0BBB SSS BBB
1BBB SBS BBB
2BBB SSS BBB

B - building location - these locations  can be excavated or (if they were implemented)
                        framed and poured on.
S - staging location  - these locations are adjacent to offsite.

Floor 0 is at ground level.  All locations on Floor 0 start out as ground.
Floor -1 is below ground level.  All locations on Floor -1 start out as solid.
Floor 1 is above ground level.  All locations on Floor 1 start out as air.
"



(setq *OPERATORS*
  '(
    ;; IDLE-RESOURCE
    ;; Allows resources (equipment and subcontractors) to "wait" a time unit
    ;; at their current location.  Preconds for this operator are not as
    ;; stringent as those for MOVE-RESOURCE.  To fake an idle using MOVE-RESOURCE,
    ;; Prodigy must move the resource off its current location, then back on,
    ;; since a location is not adjacent to itself.
    (IDLE-RESOURCE
     (PARAMS (<r-id> <at> <time-i> <time-e>))
     (PRECONDS
      (and
       (project-deadline <deadline>)
       (project-budget   <budget>)
       (project-space    <space-dsc>)

       ;;MATERIEL
       (resource <r-id> <r-dsc>)
       (resource-type <r-dsc> <r-type>)

       ;;TIME/SPACE GENERAL
       (schedule-operation 1 0 <time-i> <time-e> <deadline>)
       (location <at> <space-dsc>)
       (at-loc <r-id> <at> <time-i>)
       (forall (<time>) (between-include-high <time-i> <time> <deadline>)
	       (forall (<any-coords>) (location <any-coords> <space-dsc>)
		       (~ (at-loc <r-id> <any-coords> <time>))))))

     (EFFECTS
      ((ADD (at-loc <r-id> <at> <time-e>)))))

    ;; MOVE-RESOURCE
    ;; Moves a resource and subcontractor between two adjacent locations
    ;; with the proper surface characteristics (ground or slab).  OFFSITE
    ;; is adjacent to all staging locations.  Dump-trucks are unloaded
    ;; when they move OFFSITE (see BUCKET-EXCAVATE).
    (MOVE-RESOURCE
     (PARAMS (<r-id> <from> <to> <time-i> <time-e>))
     (PRECONDS
      (and
       (project-deadline <deadline>)
       (project-budget   <budget>)
       (project-space    <space-dsc>)

       ;;MATERIEL
       (resource <r-id> <r-dsc>)
       (resource-type <r-dsc> <r-type>)
       (not-equal <r-type> (equipment crane))
       (not-equal <r-type> (equipment concrete-pumper))

       ;;TIME/SPACE GENERAL

       ;; pick a time for operation
       (schedule-operation 1 0 <time-i> <time-e> <deadline>)

       ;; make sure resource is at "from" coords at this time
       (location <from> <space-dsc>)
       (at-loc <r-id> <from> <time-i>)

       ;; make sure resource has not already been scheduled by some
       ;; operation "in the future"
       (forall (<time>) (between-include-high <time-i> <time> <deadline>)
	       (forall (<any-coords>) (location <any-coords> <space-dsc>)
		       (~ (at-loc <r-id> <any-coords> <time>))))

       ;;SPACE
       (location <to>   <space-dsc>)
       (adjacent <from> <to> <space-dsc>)
       (or (ground <to>)
	   (slab <to>))))

     (EFFECTS
      ((if (and
	    (is-equal <to> offsite)
	    (is-equal <r-type> (equipment dump-truck))
	    (loaded <r-id>))
	   (DEL (loaded <r-id>)))
       (ADD (at-loc <r-id> <to> <time-e>)))))
     

    ;; POSITION-SUBCONTRACTOR
    ;; A short-cut move operator for subcontractors only (they're just
    ;; people, so they can go anywhere they can breath).
    (POSITION-SUBCONTRACTOR
     (PARAMS (<sub-id> <from> <to> <time-i> <time-e>))
     (PRECONDS
      (and
       (project-deadline <deadline>)
       (project-budget   <budget>)
       (project-space    <space-dsc>)

       ;;MATERIEL
       (resource <sub-id> <sub-dsc>)
       (resource-type <sub-dsc> <sub-type>)
       (is-sub <sub-type>)

       ;;TIME/SPACE GENERAL
       (schedule-operation 1 0 <time-i> <time-e> <deadline>)
       (location <from> <space-dsc>)
       (at-loc <sub-id> <from> <time-i>)
       (forall (<time>) (between-include-high <time-i> <time> <deadline>)
	       (forall (<any-coords>) (location <any-coords> <space-dsc>)
		       (~ (at-loc <sub-id> <any-coords> <time>))))

       ;;SPACE
       (location <to>   <space-dsc>)
       (~ (solid <to>))))

     (EFFECTS
      ((ADD (at-loc <sub-id> <to> <time-e>)))))

    ;; BUCKET-EXCAVATE
    ;; The real meat of this toy domain.  Bucket-excavators can excavate a hole of
    ;; arbitrary depth from any ground-level location with surface characteristic
    ;; ground or slab.  Bucket-excavators transfer the dirt they remove to a dump-truck.
    ;; Currently, the dump-truck need not be empty for this to work.  In the real
    ;; domain, a precond for this operator would be (~ (loaded <truck-id>)).  There
    ;; would need to be more than one Dump-truck for this to work with reasonable
    ;; efficiency.  Dump-trucks are unloaded when they move offsite.  In the real
    ;; doman, bucket-excavate would take several time units.
    (BUCKET-EXCAVATE
     (PARAMS (<coords> <excavator-coords> <time-i> <excavator-id> <bucket-id> <truck-id>))
     (PRECONDS
      (and
       (project-deadline <deadline>)
       (project-budget   <budget>)
       (project-space    <space-dsc>)

       ;;MATERIEL
       (resource <excavator-id> <excavator-dsc>)
       (resource-type <excavator-dsc> (subcontractor excavation))
       (resource <bucket-id> <bucket-dsc>)
       (resource-type <bucket-dsc> (equipment bucket-excavator))
       (resource <truck-id> <truck-dsc>)
       (resource-type <truck-dsc> (equipment dump-truck))

       ;;TIME/SPACE GENERAL
       (schedule-operation 1 0 <time-i> <time-e> <deadline>)
       (forall (<time1>) (between-include-high <time-i> <time1> <deadline>)
	       (forall (<coords1>) (location <coords1> <space-dsc>)
			(~ (at-loc <excavator-id> <coords1> <time1>))))
       (forall (<time2>) (between-include-high <time-i> <time2> <deadline>)
	       (forall (<coords2>) (location <coords2> <space-dsc>)
			(~ (at-loc <bucket-id>    <coords2> <time2>))))
       (forall (<time3>) (between-include-high <time-i> <time3> <deadline>)
	       (forall (<coords3>) (location <coords3> <space-dsc>)
			(~ (at-loc <truck-id>     <coords3> <time3>))))

       ;;SPACE
       (building-location <coords> <space-dsc>)
       (is-below-ground-level <coords>)
       (solid <coords>)
       (below <coords> <above-coords>)
       (ground <above-coords>)
       (xy-adjacent <excavator-coords> <coords> <space-dsc>)
       (at-loc <truck-id>     <excavator-coords> <time-i>)
       (at-loc <bucket-id>    <excavator-coords> <time-i>)
       (at-loc <excavator-id> <excavator-coords> <time-i>)))

     (EFFECTS
      ((ADD (at-loc <excavator-id> <excavator-coords> <time-e>))
       (ADD (at-loc <bucket-id>    <excavator-coords> <time-e>))
       (ADD (at-loc <truck-id>     <excavator-coords> <time-e>))
       (DEL (solid <coords>))
       (DEL (ground <above-coords>))
       (ADD (ground <coords>))
       (ADD (air <above-coords>))
       (ADD (loaded <truck-id>)))))

     ))


      
     
;;; ******************************************
;;; INFERENCE RULES
;;; ******************************************
(setq *INFERENCE-RULES*
  '(
    (INFER-OFFSITE-AT-TIME-ZERO
     (PARAMS (<id>))
     (PRECONDS
      (or (resource <id> <dsc>)
	  (material <id> <dsc>)
	  (concrete-batch <id> <dsc>)))
     (EFFECTS
      ((ADD (at-loc <id> offsite 0)))))
      
       
	     
    (INFER-SOLID
     (PARAMS (<coords>))
     (PRECONDS
      (and
       (project-space <space-dsc>)
       (location <coords> <space-dsc>)
       (is-below-ground-level <coords>)
       (~ (ground <coords>))
       (~ (framed <coords>))
       (~ (air <coords>))
       (~ (slab <coords>))))
      (EFFECTS
       ((ADD (solid <coords>)))))

    (INFER-GROUND
     (PARAMS (<coords>))
     (PRECONDS
      (and
       (project-space <space-dsc>)
       (location <coords> <space-dsc>)
       (is-at-ground-level <coords>)
       (~ (air <coords>))
       (~ (slab <coords>))
       (~ (framed <coords>))))
     (EFFECTS
      ((ADD (ground <coords>)))))
      
    (INFER-AIR
     (PARAMS (<coords>))
     (PRECONDS
      (and
       (project-space <space-dsc>)
       (location <coords> <space-dsc>)
       (is-above-ground-level <coords>)
       (~ (ground <coords>))
       (~ (framed <coords>))
       (~ (slab <coords>))))
      (EFFECTS
       ((ADD (air <coords>)))))
    ))
