;;;
;;; Sean T. Lamont
;;; CSE 473
;;; Problem domain specification:
;;; "Farmworld"
;;; 
;;; Lisp functions file
;;;
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; OVERVIEW:
;;;
;;; the whole plant/grow/harvest procedure revolves around a plant
;;; database, which contains things such as when they like to be
;;; planted in, how long they grow, etc.  This is contained 
;;; in the *PLANT-ATTRIBUTES* global.  There are also some other
;;; Important globals which are used in functions which return all
;;; possible values for the domain (such as maximum growth time)
;;; 
;;; this method makes it easy to modify the database, which should
;;; be done for large-scale planning.
;;;
;;; Below is a description of each of the fields in order and description
;;;
;;; Plantname:  the name of the plant: {corn,beans}
;;; Water-need:  the optimal amount of water for the plant to grow 
;;; water-tolerance: a value 0 (tolerant) or 1 (intolerant) describing
;;;                  the plant's reaction to imperfect water conditions
;;; Plant-height:    the height of the plant.  Used in neighbor calculation
;;;                  During the growth due to light inferences.
;;; Light-need:      basically the light tolerance. used in neighbor calcs.
;;; Growth-time:     The number of growth cycles it takes for this plant
;;;                  before harvesting.
;;; min-planting:    The earliest something can be planted 
;;; max-planting:    the latest something can be planted
;;; fertilizer-need: The optimal amount of fertilizer on the crop
;;; Fertilizer-tol:  The reaction (0/1 as above) of a plant to imperfect
;;;                  fertilizer conditions
;;;
;;; plant attributes is an association list by the first element
;;; format for a cell overview:
;;; plantname      water-need        water-tolerance        plant-height
;;; light-need     growth-time       min-planting-time      max-planting-time
;;; fertilizer-need                  fertilizer-tolerance
;;;
;;; all tolerances are 0=tolerant 1=intolerant
;;;
;;; OTHER GLOBALS:
;;; 
;;; MAX-X-VALUE/MAX-Y-VALUE :  the maximum size of our farm grid
;;; MAX-DOMAIN-TIME         :  maximum time represntable in our domain
;;; MAX-LIGHT/MAX-WATER/MAX-HEIGHT:  Maxima for plants at a particular time
;;; MAX-GROWTH:              The longest growth cycle a plant can have
;;;

(setf *MAX-X-VALUE* 2)
(setf *MAX-Y-VALUE* 2)
(setf *MAX-DOMAIN-TIME* 5)
(setf *MAX-FERT* 5)
(setf *MAX-LIGHT* 10)
(setf *MAX-WATER* 5)
(setf *MAX-HEIGHT* 5)  
(setf *MAX-GROWTH* 5)  

(setf *PLANT-ATTRIBUTES*
      '(
	(     garlic     2 1 1 0 2 1 4 2 1)
	(     tomatoes   3 0 2 0 2 1 3 5 0)
	(     corn       4 0 3 1 4 1 2 5 0)
	))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The functions themselves:
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Earlier:
;;;
;;; tells whether a time is earlier than another time, or no match.


(defun earlier (time than-time)
  (cond 
    ((is-variable time)      'no-match-attempted)
    ((is-variable than-time) 'no-match-attempted)
    (T (< time than-time))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; CROP
;;;
;;; Crop is a predicate that determines whether a variable
;;; is a valid plant within our domain, or generates a list of them.

(defun crop (plant)
  (cond ((is-variable plant)
	 (binding-list
	  plant (mapcar  #'(lambda(x) (nth 0 x)) *PLANT-ATTRIBUTES*)))
	(T 
	 (equal  plant (nth 0 (assoc plant *PLANT-ATTRIBUTES*))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; WATER-TOLERANCE:
;;;
;;; This is a predicate which tells whether the tolerance
;;; of crop "plant" is "tolerance", or generates a valid
;;; tolerance for that plant.
;;;

(defun water-tolerance (plant tolerance)
  (let* ((plant-attribute (assoc plant *plant-attributes*))
	 (plant-water-tolerance (nth 2 plant-attribute)))
    (cond ((is-variable plant) 'no-match-attempted)
	  ((is-variable tolerance)
	   (binding-list tolerance (list plant-water-tolerance)))
	  (T (equal plant-water-tolerance tolerance)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; NEEDS-FERT
;;;
;;; this is a predicate which tells whether the plant "plant"
;;; needs fertilizer amount "fert", or generates values for fert.
;;;

(defun needs-fert (plant fert)
    (let* ((plant-attribute (assoc plant *plant-attributes*))
	   (fert-needed (nth 8 plant-attribute)))
      (cond ((is-variable plant) 'no-match-attempted)
	    ((is-variable fert) 
	     (binding-list fert (list fert-needed)))
	    (T (equal fert fert-needed)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; FERT-TOLERANCE
;;;
;;; this is a predicate which tells whether the plant "plant"
;;; has fertilizer tolerance "tolerance", or generates a list
;;; of possible values for tolerance.

(defun fert-tolerance (plant tolerance)
  (let* ((plant-attribute (assoc plant *plant-attributes*))
	 (plant-fert-tolerance (nth 9 plant-attribute)))
    (cond ((is-variable plant) 'no-match-attempted)
	  ((is-variable tolerance)
	   (binding-list tolerance (list plant-fert-tolerance)))
	  (T (equal plant-fert-tolerance tolerance)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; NEEDS-WATER
;;;
;;; this is a predicate which determines whether plant "plant' has need
;;; of water amount "water", or generates a valid value for water.
;;;

(defun needs-water (plant water)
    (let* ((plant-attribute (assoc plant *plant-attributes*))
	   (water-needed (nth 1 plant-attribute)))
      (cond ((is-variable plant) 'no-match-attempted)
	    ((is-variable water) 
	     (binding-list water (list water-needed)))
	    (T (equal water water-needed)))))
	   
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; HEIGHT:
;;;
;;; This is a predicate that tells whether the height of plant "plant"
;;; is "height", or generates a value for height from the database.
;;;


(defun height (plant height)
  (cond ((is-variable plant) 'no-match-attempted)
	((is-variable height) 
	 (binding-list height (list (nth 3 (assoc plant *plant-attributes*)))))
	(T (equal (nth 3 (assoc plant *plant-attributes*)) height))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; GROWTH
;;;
;;; this is a predicate which tests whether the growth of plant
;;; is actually "growth", or generates a valid value for growth.
;;;


(defun growth (plant growth)
  (cond ((is-variable plant) 'no-match-attempted)
	((is-variable growth) 
	 (binding-list growth (list (nth 5 (assoc plant *plant-attributes*)))))
	(T (equal (nth 5 (assoc plant *plant-attributes*)) growth))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; NEEDS-LIGHT
;;;
;;; this is a predicate which tells whether the light tolerance of
;;; plant "plant" is "light", or returns a valid value of light.
;;;

(defun needs-light (plant light)
  (cond ((is-variable plant) 'no-match-attempted)
	((is-variable light )
	 (binding-list light (list (nth 4 (assoc plant *plant-attributes*)))))
	(T (equal (nth 4 (assoc plant *plant-attributes*)) light))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; PLANT-TIME-MIN / PLANT-TIME-MAX:
;;;
;;; these functions tell whether the max planting time of a crop
;;; is time, or return a valid list for time.
;;;

(defun plant-time-min (plant time)
  (cond ((is-variable plant ) 'no-match-attempted)
	((is-variable time) 
	 (binding-list time (list (nth 6 (assoc plant *plant-attributes*)))))
	(T (equal (nth 6 (assoc plant *plant-attributes*)) time))))

(defun plant-time-max (plant time)
  (cond ((is-variable plant ) 'no-match-attempted)
	((is-variable time) 
	 (binding-list time (list (nth 7 (assoc plant *plant-attributes*)))))
	(T (equal (nth 7 (assoc plant *plant-attributes*)) time))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; IN-XRANGE /IN-YRANGE:
;;;
;;; Tells whether the value is within the x-y grid of our
;;; domain, or returns a list of valid values.

(defun in-xrange (xvalue)
  (if (is-variable xvalue)                ;; then return
      (binding-list xvalue (1-to-value *max-x-value*)) ;; else return
      (and
       (<= xvalue *max-x-value* )
       (>  xvalue 0))))

(defun in-yrange (yvalue)
  (if (is-variable yvalue)                ;; then return
      (binding-list yvalue (1-to-value *max-y-value*)) ;; else return
      (and
       (<= yvalue *max-y-value* )
       (> yvalue 0))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; VALID-TIME
;;;
;;; This is a predicate that checks whether a time is within the
;;; range 0- max_time (*max-domain-time*), or generates a list
;;; of these possible values.
;;;

(defun valid-time (time)
  (if (is-variable time)                ;; then return
      (binding-list time  (1-to-value *max-domain-time*)) ;; else return
      (and
       (<= time *max-domain-time* ) (>= time 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; VALID-HEIGHT:
;;;
;;; This is a function that determines whether "height" is a possible
;;; height, or return a list of values.  It does not bind a
;;; value of height per crop, that is done with another function.



(defun valid-height (height)
  (if (is-variable height)                ;; then return
      (binding-list height (cons 0 (1-to-value *max-height*))) ;; else return
      (and
       (<= height *max-height* ) (>= height 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; VALID-GROWTH-AMOUNT
;;;
;;; Tells how much something can grow in one growth cycle, or returns
;;; a list of possibilities, 0 or 1.   Not to be confused with valid-
;;; growth, which tells how much something can grow (IE how many
;;; times GROW can be applied between plant and harvest for the
;;; longest growing plant in the domain. )
;;;


(defun valid-growth-amount (growth)
  (if (is-variable growth)
      (binding-list growth '(1 0 ))
      (or (= growth 0) (= growth 1))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; VALID-GROWTH:
;;;
;;; a predicate telling whether "growth" is a valid growth cycle value
;;; within our domain.  (IE the distance between plant & harvest, or the
;;; max number of possible times grow can be applied.
;;;

(defun valid-growth (growth)
  (if (is-variable growth)                ;; then return
      (binding-list growth (cons 0 (1-to-value *max-growth*))) ;; else return
      (and
       (<= growth *max-growth* ) (>= growth 0))))

(defun valid-water (water)
  (if (is-variable water)                ;; then return
      (binding-list water (cons 0 (1-to-value *max-water*))) ;; else return
      (and
       (<= water *max-water* ) (>= water 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; VALID-FERT
;;;
;;; Tells whether the fertilizer amount FERT is valid within our domain,
;;; or returns a list of possible values.
;;; 
;;; not to be confused with needs-fert, which is specific to a plants
;;; fertilizer needs.

(defun valid-fert (fert)
  (if (is-variable fert)                ;; then return
      (binding-list fert (1-to-value *max-fert*)) ;; else return
      (and
       (<= fert *max-fert* ) (> fert 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; HEIGHT-BASED-ON-GROWTH
;;;
;;; this function is used to calculate the effective height
;;; based on the growth cycle.  Since the height and the growth
;;; cycle are seen as growing proportionately, one can be generated
;;; from the other.  This either tells whether height based on
;;; growth for plant "plant" is true, or generates values
;;; for height only.  if growth or plant are unbound, this
;;; doesn't match.

(defun height-based-on-growth (plant growth height)
  (let* ((plant-attribute (assoc plant *plant-attributes* ))
	 (plant-height (nth 3 plant-attribute))
	 (plant-growth (nth 5 plant-attribute))
	 (scale-factor (/ plant-height plant-growth)))
  (cond ((is-variable plant)  'no-match-attempted)
	((is-variable growth) 'no-match-attempted)
	((is-variable height) 
	 (binding-list height (list (* growth scale-factor))))
	(T (equal height      (* growth scale-factor))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; One-greater 
;;; adds one to the original value and returns the solution
;;;
;;; returns one of 3 things:
;;; 1.  if the variables are bound, returns true or false
;;; 2.  if the solution is bound it returns one less
;;; 3.  if the original is bound, it returns the solution as a binding list.
;;;


(defun one-greater (bignumber number)
  (cond ((and (is-variable number )
	      (is-variable bignumber)) 'no-match-attempted)
	((is-variable bignumber) 
       	 (binding-list bignumber (list (1+ number))))
	((is-variable number) 
       	 (binding-list number (list (1- bignumber))))
	(T (= bignumber (1+ number)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; LESS-THAN / GREATER-THAN
;;;
;;; strictly relational.  This is not for use with unbound variables,
;;; and it wouldn't make sense since we can't return all possible
;;; values less than, since we don't know the domain of the variable

(defun less-than (num1 num2)
      (< num1 num2))

(defun greater-than (num1 num2)
      (> num1 num2))
      
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Greater-than-or-equal
;;;
;;; strictly a relational comparitor.  Has no definition for unbound
;;; variables

(defun greater-than-or-equal (num1 num2)
	(>= num1 num2))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; BETWEEN-NUMBERS
;;;
;;; Tells whether number is in the range lower-bound to upper-bound.
;;;
;;; has no definition for unbound variables.
;;;
(defun between-numbers (number lower-bound upper-bound)
  (and
   (>= number lower-bound)
   (<= number upper-bound)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;; ADJACENT-X / ADJACENT-Y
;;;
;;; this either checks whether adj-{x,y} is in the range {x,y} +/- 1
;;; or returns a list of all of the numbers in that range.
;;;

(defun adjacent-x (x adj-x)  ;; returns all the x squares +1 or -1
  (cond ((is-variable x) 'no-match-attempted)
	((is-variable adj-x)
	 (binding-list adj-x
		(remove-if-not #'(lambda(xpos) (in-xrange xpos))
		   (list (1+ x) x (1- x)))))
	(T (member adj-x 
		(remove-if-not #'(lambda(xpos) (in-xrange xpos))
		   (list (1+ x) x (1- x)))))))

(defun adjacent-y (y adj-y)  ;; returns all the x squares +1 or -1 of y
  (cond ((is-variable y) 'no-match-attempted)
	((is-variable adj-y)
	 (binding-list adj-y
		(remove-if-not #'(lambda(ypos) (in-yrange ypos))
		   (list (1+ y) y (1- y)))))
	(T (member adj-y 
		(remove-if-not #'(lambda(ypos) (in-xrange ypos))
		   (list (1+ y) y (1- y)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ADDTO/SUB
;;;
;;; This is for mathematical addition/subtraction.  In both
;;; of these functions, if 2 variables are unbound it returns
;;; without matching, if 1 variable is unbound it fills it in and
;;; if none are it returns a boolean value.
;;;



(defun addto (value1 value2 result)
  (cond ((or
	 (and (is-variable value1) (is-variable result))
	 (and (is-variable value1) (is-variable value2))
	 (and (is-variable value2) (is-variable result)))
	   'no-match-attempted)
	((is-variable value1) 
	      (binding-list value1 (list (- result value2))))
	((is-variable value2) 
	      (binding-list value2 (list (- result value1))))
	((is-variable result)
	      (binding-list result (list (+ value1 value2))))
	(T (= (+ value1 value2) result))))

(defun sub (value1 value2 result)
  (cond ((or
	 (and (is-variable value1) (is-variable result))
	 (and (is-variable value1) (is-variable value2))
	 (and (is-variable value2) (is-variable result)))
	   'no-match-attempted)
	((is-variable value1) 
	      (binding-list value1 (list (+ result value2))))
	((is-variable value2) 
	      (binding-list value2 (list (- value1 result))))
	((is-variable result)
	      (binding-list result (list (- value1 value2))))
	(T (= (+ value1 value2) result))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; This function answers the pression quetion:
;;;
;;; how much will a given plant grow given non-optimal water?
;;; (or even non-optimal)
;;;
;;; this function will return either 0 or 1, depending on whether it
;;; is good or bad (growth or not) based on the following criteria:
;;;
;;; if the tolerance of the plant is 0 (intolerant), the actual
;;; water must be within the range NEEDED +/- 1 for the
;;; plant to grow 1.  If it is 1, (tolerant), the actual water must
;;; be within the range NEEDED +/- 3 for the plant to grow
;;; 1.  In all other cases, the value of growth is 0.
;;;
;;; this will fill in unbound "has" values for "needed" values, which
;;; is done as follows:  if the growth is 1 it's easy, it's in
;;; the range needs +/- 1 or 3, depending.  This is done
;;; with the function "range-list", which returns a list of numbers
;;; in a certain range.
;;; When the value is 0, it gets weird , however, since we
;;; want the set complement of this value.  For this I use a function
;;; called not-range-list, which returns all numbers within
;;; a range that are NOT within the specified secondary range.
;;;


(defun calc-water-growth (has needs tolerance will-grow) 

(cond ((is-variable needs)     'no-match-attempted)
      ((is-variable tolerance) 'no-match-attempted)
      ((and (is-variable will-grow) 
	    (is-variable has)) 'no-match-attempted)
      ((is-variable will-grow)
       (binding-list will-grow
		     (list 
		      (if (= 0 tolerance)
			  (in-range has (- needs 1) (+ needs 3) 1 0)
			  (in-range has (- needs 3) (+ needs 3) 1 0)))))
      ((is-variable has)
       (binding-list has
          (cond
	       ((and (= 0 tolerance) (= will-grow 1))
		(range-list (- needs 1) (+ needs 1)))
	       ((and (= 0 tolerance) (= will-grow 0))
		 (not-range-list 0 *max-water* (- needs 1) (+ needs 1)))
	       ((and (= 1 tolerance) (=  will-grow 1))
		(range-list (- needs 3) (+ needs 3)))
	       ((and (= 1 tolerance) (= will-grow 0))
		(not-range-list 0 *max-water* (- needs 3) (+ needs 3))))))
      
(T
 (if (= tolerance 0) 
     (= will-grow (in-range has (- needs 1) (+ needs 1) 1 0))
     (= will-grow (in-range has (- needs 3) (+ needs 3) 1 0))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; CALC-FERT-GROWTH
;;;
;;; this is almost exactly the same as
;;; calc-water-growth.  for a description see above.
;;;

(defun calc-fert-growth (has needs tolerance will-grow) 
;; how much will a plant grow due to fert amount?
;; either 1 (good) or 0 (bad)
;; it is "good " if the water tolerance is high (1)

(cond ((is-variable needs)     'no-match-attempted)
      ((is-variable tolerance) 'no-match-attempted)
      ((and (is-variable will-grow) 
	    (is-variable has)) 'no-match-attempted)
      ((is-variable will-grow)
       (binding-list will-grow
		     (list 
		      (if (= 0 tolerance)
			  (in-range has (- needs 1) (+ needs 1) 1 0)
			  (in-range has (- needs 3) (+ needs 3) 1 0)))))
      ((is-variable has)
       (binding-list has
          (cond
	       ((and (= 0 tolerance) (= will-grow 1))
		(range-list (- needs 1) (+ needs 1)))
	       ((and (= 0 tolerance) (= will-grow 0))
		 (not-range-list 0 *max-fert* (- needs 1) (+ needs 1)))
	       ((and (= 1 tolerance) (=  will-grow 1))
		(range-list (- needs 3) (+ needs 3)))
	       ((and (= 1 tolerance) (= will-grow 0))
		(not-range-list 0 *max-fert* (- needs 3) (+ needs 3))))))
      
(T
 (if (= tolerance 0) 
     (= will-grow (in-range has (- needs 1) (+ needs 1) 1 0))
     (= will-grow (in-range has (- needs 3) (+ needs 3) 1 0))))))
     
     
			  
;;;
;;; Helper functions

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 1-TO-VALUE
;;;
;;; 1-to-value returns a list of values between 1 and value.
;;; useful with binding-list for ranges.

(defun 1-to-value (value)
  (cond
    ((zerop value) NIL)
    (T (cons value (1-to-value  (1- value))))))


(defun not-range-list (l-bound u-bound not-lbound not-ubound)
;; makes a range list of l-bound - u-bound excluding those in the
;; range not-lbound to not-ubound.  for a description of this
;;; usage look at calc-water-usage


  (remove-if #'(lambda(x) (and (>= x not-lbound )(<= x not-ubound)))
	     (range-list l-bound u-bound)))

(defun range-list (l-bound u-bound)
  (remove-if #'(lambda(x) (or (< x 0) (>  *max-water* *max-fert*)))
	     (range-list-aux l-bound u-bound)))


(defun range-list-aux (l-bound u-bound)
  (cond ((= u-bound l-bound) (list u-bound))
	(T (cons l-bound (range-list-aux (1+ l-bound) u-bound)))))
	  
 

(defun in-range (value l-bound u-bound ret-true ret-false)
;; if value is in the range l-bound...u-bound return "true"
;; if it's not, return "false"
  (cond
    ((and 
      (>= value l-bound)
      (<= value u-bound)) ret-true)
    (T ret-false)))
      


;  BINDING-LIST returns a binding list for a single variable, only
;
(defun binding-list (var val-list)
  (cond ((null val-list) nil)
        ((null (car val-list)) (binding-list var (cdr val-list)))
        (t (append (list (list (list var (car val-list))))
                   (binding-list var (cdr val-list))))))


;;;
;;; For debugging
(defun trace-value (something)
  (print (list 'value 'is something))
   T)

(defun print-value (something value )
  (print (list something 'eq value))
T)

(defun fail (something)
NIL)
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;   meta-fn stuff:
;;;;

(defun parent-op (node op)
  (and (not-top-level-node node)
       (current-op (node-parent node) op)))

(defun parent-goal (node goal)
  (and (not-top-level-node node)
       (current-goal (node-parent node) goal)))

(defun grandparent-goal (node goal)
  (and (not-top-level-node node)
       (current-goal (node-parent (node-parent node)) goal)))

(add-meta-fn 'parent-op)
(add-meta-fn 'parent-goal)
(add-meta-fn 'grandparent-goal)

       