;;; Sean T. Lamont
;;; CSE 473
;;; Farmworld specification:
;;; Operators


(setq *OPERATORS* '(
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Fertilize:
;;;
;;; Preconditions: we have fertilizer stored in the state
;;; Postconditions: x/y/t has some amount of fertilizer
;;;
;;; this attempts to make an x/y location at some time to have
;;; an amount of fertilizer.  it is dependent on the amount of
;;; fertilizer we have, which is represented by the predicate 
;;; stored-fertilizer.  It changes this value in the state by a 
;;; value that is equal to the amount planted at x/y/t.
;;; 
;;; Since we want to find out how much stored fertilizer is in
;;; the state, and this is a possibly unbounded number, the
;;; generator for <total-fert-amount> is unstatic.
;;; addto is the static generator for <new-fert-amount>
;;; the conditional effects on this operator insure that 
;;; a value less than or equal to *MAX-FERT* is asserted
;;; into the state.
;;;;;;;;;;;;;;;;;;;;;;;;

(FERTILIZE
 (params  (<crop1> <x> <y> <time> <amount-to-fert>))
 (preconds 
  (and
   (stored-fertilizer <total-fert-amount>)           ;; static generation
   (valid-fert <amount-to-fert>)                     ;; impossible here
   (valid-fert <old-fert-amount>)
   (has-fert <x> <y> <time> <old-fert-amount>)
   (greater-than-or-equal <total-fert-amount> <amount-to-fert>)
   (sub <total-fert-amount> <amount-to-fert> <new-total-fert>)
   (addto <old-fert-amount> <amount-to-fert> <new-fert-amount>)))

 (effects (
	   (del (stored-fertilizer <total-fert-amount>))
	   (del (has-fert <x> <y> <time> <old-fert-amount>))
	   (add (stored-fertilizer <new-total-fert>))
	   (if  (greater-than-or-equal <new-fert-amount> 5)
	    (add (has-fert <x> <y> <time> 5)))
	   (if  (less-than <new-fert-amount> 5)
	    (add (has-fert <x> <y> <time> <new-fert-amount>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Water:
;;; 
;;; Preconditions:  we have water stored in the state
;;; Postconditions: x/y/t has some amount of water
;;;
;;; This puts water at an x/y position at some time.
;;; 
;;; This attempts to make an x/y location at some time to
;;; have an amount of water.  It is dependent on the amount
;;; of water stored in the state, represented by stored-water.
;;; It changes the value of the state by a value equal to the
;;; amount placed at x/y/t.
;;;
;;; Since the value of water in the state is possibly unbounded,
;;; static generation of total-water-amount is impossible.
;;;
;;; if the value of the amount of water at an x/y/t value exceeds
;;; *MAX-WATER*, only the value *MAX-WATER* is asserted to be true.
;;;


(WATER
 (params  (<crop1> <x> <y> <time> <amount-to-water>))
 (preconds 
  (and
   (stored-water <total-water-amount>)              ; static generation is
   (valid-water <amount-to-water>)                  ; impossible
   (valid-water <old-water-amount>)
   (has-water <x> <y> <time> <old-water-amount>)
   (greater-than-or-equal <total-water-amount> <amount-to-water>)
   (sub <total-water-amount> <amount-to-water> <new-total-water>)
   (valid-water <new-total-water>)
   (addto <old-water-amount> <amount-to-water> <new-water-amount>)))

 (effects (
	   (del (stored-water <total-water-amount>))
	   (del (has-water <x> <y> <time> <old-water-amount>))
	   (add (stored-water <new-total-water>))
	   (if  (greater-than-or-equal <new-water-amount> 10)
	    (add (has-water <x> <y> <time> 10)))
	   (if  (less-than <new-water-amount> 10)
	    (add (has-water <x> <y> <time> <new-water-amount>))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Plant:
;;;
;;; Preconditions:  Plantable crop
;;; Postconditions: make the crop at location x/y/t be growth stage 1.
;;;
;;; most of the work done here is in is-plantable.  this is what
;;; starts the growing cycle.

(PLANT
  (params (<crop1> <x> <y> <time>))
  (preconds (plantable <crop1> <x> <y> <time>))

  
 (effects (
	   (del (current-growth <crop1> <x> <y> <time> 0))
	   (add (current-growth <crop1> <x> <y> <time> 1)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Grow:
;;;
;;; preconditions:
;;;     The crop is completely growable, which is a boolean value
;;;      calculated by light, fertilizer, and water factors.
;;; Postconditions:  the square x/y at time t is increased in its
;;;     growth cycle, until it reaches its maximum growth cycle.
;;;
;;; 
;;;
;;; This really shouldn't be a part of the planner, since the fact
;;; that growth is a kind of an implicit thing that just "happens."
;;; However, it is difficult to represent ongoing events in prodigy without
;;; making it an operator, so here it is.
;;;
;;; this attempts to increase the so-called growth cycle, the number
;;; of passes through grow that it takes a plant to fully mature.
;;;
;;; this  simply adds one to the growth cycle of the selected
;;; x/y/time cycle, trying to grow it into the next subsequent time
;;; spot.  It also checks whether the next time slot is free
;;; of plants (hence the forall)  , and if the growth cycle
;;; of the square we're growing FROM is 0, it deletes it. (in all
;;; other cases, it doesn't delete the previous time value, because
;;; zero just means "we can plant something here", which is only
;;; thing which needs to be invalidated when we grow from 0 to 1. if
;;; we didn't do this, it would take on both 0 and the value it
;;; is growing to IE 1,2,... which is OK for all values except 0)
;;;
;;; most of the determining of plantability here is done in the
;;; inference rules that determine whether something is plantable
;;; by checking the amount of fertilizer, the proximity of its nearest
;;; neighbors, and its current water
;;;
;;; since the current-growth <x> <y> <new-time> <new-crop-growth>
;;; will always be the goal, we can consider new-time and new-crop-growth
;;; to be statically generated (IE bound before the preconditions)
;;; thus, <time> and <crop-growth> are bound by virtue of being
;;; one greater by the one-greater function


(GROW     
 (params (<crop1> <X> <Y> <TIME>  ))
 (preconds (and (one-greater <new-time> <time>)  ;; generate the time
	        (one-greater <new-crop-growth> <crop-growth>)
	        (valid-growth <crop-growth>)     ;; bounds check
	        (valid-time <time>)              ;; bounds check
	        (current-growth <crop1> <x> <y> <time> <crop-growth>)
;;	        (forall (<some-crop>) (crop <some-crop>)
;;			 (current-growth <crop1> <x> <y> <new-time> 0))
	        (growth <crop1> <max-growth>)   ;; generate the max growth
	        (crop-will-grow <crop1> <x> <y> <time>)
	        (less-than <crop-growth> <max-growth>)))
(effects (
	  (del (current-growth <crop1> <x> <y> <new-time> 0))
	  
  (add (current-growth <crop1> <x> <y> <new-time> <new-crop-growth>)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Harvest:
;;;
;;; Preconditions:
;;; X/Y at time T has a crop that is at its max point in the growth cycle
;;; Postconditions:  X/Y at time T+1 is zero, Harvested (x,y,t) is true,
;;; we have one more of whatever product we harvested (has-product)
;;;
;;; This is the means of achieving has-product, which is most
;;; of the time going to be the goal. all it basically does it
;;; check to see if we're in a good range, generate the successor
;;; time,  and add "harvested" to the state, indicating that we can't
;;; harvest this square again, and then generate one more than the
;;; current amount of whatever we're harvesting, so we can say
;;; (has-product(corn)= has-product(corn) + 1 )

(HARVEST
 (params (<crop1> <X> <Y> <time> ))
 (preconds      (and
		(in-xrange <x>)
		(in-yrange <y>)
		(one-greater <new-amount> <amount>)    ;; generator for new-amt
		(has-product <crop1> <amount>)
		(greater-than-or-equal <amount> 0)
		(valid-time <new-time>)               ;; generator for new-time
		(one-greater <new-time> <time>)       ;; generator for time
		(valid-time <time>)
	     	(valid-growth <max-growth>)
	        (growth <crop1> <max-growth>)
		(current-growth <crop1> <x> <y> <time> <max-growth>)
   	        (~ (harvested <crop1> <x> <y> <time>))))
 (effects (
	    (del (has-product <crop1> <amount>))
	    (add (harvested <crop1> <x> <y> <time>))
	    (add (has-product <crop1> <new-amount>))
	    (if (valid-time <newtime>)
		(add (current-growth <crop1> <x> <y> <new-time> 0))))))
))


(setq *INFERENCE-RULES* '(
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; Inference rules:
;;;
;;; with the notable exception of infer-plantable,
;;; all of these are used to check conditions under which we want
;;; to plant.  Each of the 3 cases that contribute to the ability of 
;;; a plant to grow at a given x/y/time (light/water/fertilizer)
;;; is represented as a "good" (asserted as true in the state) value
;;; or a "bad" (asserted as false in the state) value.
;;;
;;; the infer-total-growth-{good,bad} rules give a formula for the
;;; effective growth of the plant based on these factors.  Although
;;; this is kind of an arbitrary figure, the formula for calculation of
;;; the "growability" of a square is as follows:
;;;
;;; (x,y,t) will grow IF it has adequate water
;;;         AND it has either valid fertilizer or valid light satisfied.
;;;
;;; "adequate water" and "adequate fertilizer" just means that there
;;; is some in the state at that point.  
;;; "valid light" is calculated in an interesting way.  In order
;;; for  the plant to have valid light, ALL of the adjacent
;;; squares (ne/n/nw/se/s/sw) have to have heights that are less
;;; than or equal to the height of the square (x,y,t).  This is
;;; to simulate the blocking out of the sun from other, higher plants.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-NO-HIGHER-NEIGHBORS
;;;
;;; this asserts no-higher neighbor into the state
;;; if all of the heights of adjacent squares are less than or equal
;;; to the square we're examining.
;;;
;;; the "height" is not the same as the growth factor, but is calculated
;;; as a factor based on the growth factor, since the growth cycle
;;; and the height have the same rate of growth.
;;; this is represented with "height-based-on-growth"
;;; 
;;; the "adjacent-x" and "adjacent-y" return lists of 
;;; possible x's within the valid range of x's (or y's) which are
;;; x +/- 1 square.  In this way all 9 squares are generated.
;;; it doesn't matter that the middle one is checked, it is greater than
;;; or equal to itself by definition.

  (INFER-NO-HIGHER-NEIGHBORS
   (params (<crop1> <x> <y> <time>))
   (preconds 
    (and 
;; NONSTATIC, DEBUG.  reverse args
     (current-growth <crop1> <x> <y> <time> <crop-growth>)  ;; my growth
     (height-based-on-growth <crop1> <crop-growth> <crop-height>) ;; my height
     (valid-growth <crop-growth>)                           ;; bind me and
     (forall (<adj-x>)     (adjacent-x <x> <adj-x>)
      (forall (<adj-y>)     (adjacent-y <y> <adj-y>)
       (and
	(crop <neighbor>)                  ;; bind each neighbor
;; DEBUG nonstatic generators
	(current-growth <crop1> <adj-x> <adj-y> <time> <n-growth>)
	(height-based-on-growth <crop1> <n-growth> <n-height>) ;; bind n-ht
	(greater-than-or-equal <crop-height> <n-height>))))))

   (effects (
	     (add (no-higher-neighbor <crop1> <x> <y> <time> )))))
			 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-TOTAL-GROWTH-GOOD
;;;
;;; this tries to prove that the "total" growth, the factor based
;;; on the 3 different elements, is true.  It tries to figure
;;; out that there is good water, and that there is either good
;;; light or good fertilizer amounts.
;;;

(INFER-TOTAL-GROWTH-GOOD
 (params (<crop1> <x> <y> <time>))
 (preconds
  (and
    (good-water-growth <crop1> <x> <y> <time> )
   (or
    (good-fert-growth <crop1> <x> <y> <time>)
    (good-light-growth <crop1> <x> <y> <time>))))

 (effects (  
	   (add (crop-will-grow <crop1> <x> <y> <time>)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-TOTAL-GROWTH-BAD
;;;
;;; This tries to assert that the total growth is bad.  I'm not quite
;;; sure why you would want to do this, but It's in for symmetry
;;; I guess, or expandability.  It checks to see whether
;;; either the water is bad, or both the light growth and
;;; the fertilizer amount are bad.

(INFER-TOTAL-GROWTH-BAD
 (params (<crop1> <x> <y> <time>))
 (preconds
  (or
   (~ (good-water-growth <crop1> <x> <y> <time>))
   (and
    (~ (good-light-growth <crop1> <x> <y> <time> ))
    (~ (good-fert-growth <crop1> <x> <y> <time>)))))
 (effects (  
	   (add (~ (crop-will-grow <crop1> <x> <y> <time>))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-GROWTH-DUE-TO-GOOD-LIGHT
;;;
;;; this uses the no-highest-neighbor predicate to make sure
;;; that there are nothing adjacent to block out the light. If this
;;; is true, it asserts that the light growth is "good"
;;;

(INFER-GROWTH-DUE-TO-GOOD-LIGHT
 (params (<crop1> <x> <y> <time>))
 (preconds
  (no-higher-neighbor <crop1> <x> <y> <time>))

 (effects (
	   (add (good-light-growth <crop1> <x> <y> <time> )))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-GROWTH-DUE-TO-BAD-LIGHT
;;;
;;; the complement of infer-growth-due-to-good-light, this tries
;;; to prove that one of the neighbors is higher, and thus blocks
;;; out the light.  Again, this is mostly for symmetry.
;;;
;;;


(INFER-GROWTH-DUE-TO-BAD-LIGHT                 ;; this operator handles the
 (params (<crop1> <x> <y> <time> <will-grow>)) ;; case of a plant being blocked
 (preconds                                     ;; by other neighboring plants
    (~ (no-higher-neighbor <crop1> <x> <y> <time>)))
 
(effects (
	  (add (~ (good-light-growth <crop1> <x> <y> <time>))))))
    

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-GROWTH-DUE-TO-GOOD-WATER
;;;
;;; this, like infer-growth-due-to-good-fertilizer, tries to 
;;; find in the state that the fertilizer is of a valid amount.  
;;; 
;;; the formula for finding whether this is true is as follows:
;;;
;;; if the tolerance = 0 (intolerant) and the actual current fertilizer
;;; is within +/- 1 of the needed , it will grow.  if the tolerance = 1
;;; (tolerant) and the actual is within +/- 3 of the current, it will
;;; grow.  this same formula is used in the formula for calculating
;;; fertilizer amounts
;;;
;;; this MUST be satisfied for something to grow.

(INFER-GROWTH-DUE-TO-GOOD-WATER

 (params (<crop1> <x> <y> <time>))
 (preconds
 (and
  (is-equal <will-grow> 1)
  (water-tolerance <crop1> <c-water-tol>)
  (needs-water     <crop1> <c-water-need>)
  (calc-water-growth <c-water-has> <c-water-need> <c-water-tol> <will-grow>)
  (has-water <x> <y> <time> <c-water-has>)))

(effects (
	      (add (good-water-growth <crop1> <x> <y> <time>)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-GROWTH-DUE-TO-BAD-WATER
;;;
;;; The complement of infer-growth-due-to-bad-water, this
;;; tries to determine from the state whether the water amount
;;; is currently BAD , meaning no growth.
;;;
;;;

(INFER-GROWTH-DUE-TO-BAD-WATER

 (params (<crop1> <x> <y> <time>))
 (preconds
 (and
  (is-equal <will-grow> 0)
  (water-tolerance <crop1> <c-water-tol>)
  (needs-water     <crop1> <c-water-need>)
  (calc-water-growth <c-water-has> <c-water-need> <c-water-tol> <will-grow>)
  (has-water <x> <y> <time> <c-water-has>)))


(effects (
	      (add (~ (good-water-growth <crop1> <x> <y> <time>))))))
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; INFER-GROWTH-DUE-TO-GOOD-FERT
;;; 
;;; this tries to find out whether the x/y/t location
;;; can grow, based on good fertilizer.  This is used in the calculation
;;; of total growth.   The formula is as is described in the 
;;; infer-growth-due-to-good-water function.
;;;


(INFER-GROWTH-DUE-TO-GOOD-FERT

 (params (<crop1> <x> <y> <time>))
 (preconds
 (and
  (is-equal <will-grow> 1)
  (fert-tolerance <crop1> <c-fert-tol>)
  (needs-fert     <crop1> <c-fert-need>)
  (calc-fert-growth <c-fert-has> <c-fert-need> <c-fert-tol> <will-grow>)
  (has-fert <x> <y> <time> <c-fert-has>)))


(effects (
	      (add (good-fert-growth <crop1> <x> <y> <time>)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-GROWTH-DUE-TO-BAD-FERT
;;;
;;; the complement of INFER-GROWTH-DUE-TO-GOOD-FERT, this 
;;; determines from the state whether the current fertilizer
;;; of an x/y/t point is unsuitable for growing.
;;;


(INFER-GROWTH-DUE-TO-BAD-FERT

 (params (<crop1> <x> <y> <time>))
 (preconds
 (and
  (is-equal <will-grow> 0)
  (fert-tolerance <crop1> <c-fert-tol>)
  (needs-fert     <crop1> <c-fert-need>)
  (calc-fert-growth <c-fert-has> <c-fert-need> <c-fert-tol> <will-grow>)
  (has-fert <x> <y> <time> <c-fert-has>)))

(effects (
	      (add (~ (good-fert-growth <crop1> <x> <y> <time>))))))
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; INFER-PLANTABLE
;;;
;;; a crop is PLANTABLE at (X,Y,T) if:
;;; 
;;; the planting time is within the range (min-time) to (max-time),
;;; where these values are defined as constants for the crop we want
;;; to grow and are defined in the lisp database.
;;;
;;; there is nothing at the square (current-crowth is 0)
;;;
;;; if all of this is true it states that "plantable' of that square
;;; is true at that time.
;;;

(INFER-PLANTABLE
(params (<crop1> <x> <y> <time>))
  
(preconds 
     (and
      (current-growth <crop1> <x> <y> <time> 0)
      (plant-time-max <crop1> <max-planting-time>)
      (plant-time-min <crop1> <min-planting-time>)
      (between-numbers <time> <min-planting-time> <max-planting-time>)))
(effects (
	  (add (plantable <crop1> <x> <y> <time>)))))))
			  
