#|
The SIGN and SIGNED RANK TESTS.  (Etzioni and Etzioni censored-data extension)

Author:  Oren Etzioni  (etzioni@cs.washington.edu)
Date:    June 10, 1992.

Copyright 1992 by Oren Etzioni.
This is experimental software for research purposes only.
This software is made available under the following conditions:
1) It will only be used for internal, noncommercial research purposes.
2) Any bugs, bug fixes, or extensions will be forwarded to the author.

To understand the code in this file see ``Statistical Methods for
Analyzing Speedup Learning Experiments'' By Oren Etzioni and Ruth
Etzioni.  

To obtain a copy of the paper, do an anonymous ftp to
june.cs.washington.edu and cd into pub/etzioni/papers.  The paper is
in compressed postscript form: statistical-methods.ps.Z

The *-data files in this directory contain the data used in that paper.

DIRECTIONS:

To apply the SIGN TEST:
1. Set F=data from the system presumed to be faster according to H_a.
2. Set S=data from the system presumed to be slower according to H_a.

NOTE: I expect the data to be in the following format: 
(value1 value2..valueN).

3. Set *bound* to the appropriate bound.
4. Call (sign-test F S)
5. The function returns the test statistic, which is a normally
distributed random variable with mean 0 and variance 1.  The statistic
is referred to as z.

NOTE: although z is normally distributed, the tests are *not* assuming
that the data is normally distributed!

NOTE: if z is negative, the result is not significant.  If z > 3.08
then the p-value is bounded above by 0.000, and the result is ``very''
significant.

6. The p-value of the test is the area to the right of the z value
under the normal curve.  Tables that enable you to compute the
p-value, based on z, appear in the back of virtually every statistics
textbook.

NOTE: the total area under the normal curve is one.  This fact enables
you to determine the area to the *right* of z, based on the
information in the textbook.



The procedure for applying the SIGNED RANK TEST is the same, but
the call is (signed-rank-test F S).

|#


;;; The resource bound.
(defvar *bound* 150)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  SIGN TEST.

;; TOP-LEVEL FUNCTION.
;;; The two arguments to the function are not symmetric:
;;; f-data is the data for the faster system (according to H_a). 
;;; A positive difference is a data point on which the value of the
;;; f-data point is smaller than the value of the corresponding s-data point.
;;; Splits non-censored ties evenly between pos-diff and neg-diff.
;;; Counts doubly censored points as neg-diff.

(defun sign-test (f-data s-data)
	(check-data f-data s-data)

  (let* (
		 (sample-size (length f-data))
		 (pos-diff (number-of f-data s-data #'pos-diff?))
		 (nc-ties (number-of f-data s-data #'nc-tie?))
		 )

	(sign-statistic (+ pos-diff (floor (/ nc-ties 2))) sample-size)
	))


;;; One sided test: only considers the positive differences and the
;;; sample size.  
(defun sign-statistic (pos-diff sample-size)
  (when (< sample-size 25)
	(error "This test assumes the sample size is at least 25 ~%"))
  (/ (- pos-diff (/ sample-size 2.0)) (sqrt (* sample-size 0.25))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; SIGNED RANK TEST

;; TOP-LEVEL FUNCTION.
;;; The two arguments to the function are not symmetric:
;;; f-data is the data for the faster system (according to H_a). 
;;; A positive difference is a data point on which the value of the
;;; f-data point is smaller than the value of the corresponding s-data point.
;;; The function splits non-censored ties evenly between pos-diff and neg-diff,
;;; and counts doubly censored points as maximal rank negative differences.

(defun signed-rank-test (f-data s-data)
	(check-data f-data s-data)

  (let* (
		 (sample-size (length f-data))
		 (sorted-diff (sort (compute-differences f-data s-data) #'<
							:key #'car))
		 (ranks (compute-ranks sorted-diff))
		 (pos-ranks (find-pos-ranks ranks))
		 )

	(signed-rank-statistic (sum pos-ranks) sample-size)
	))


(defun signed-rank-statistic (pos-ranks-sum n)
  (when (< n 25)
	(error "This test assumes the sample size is at least 25 ~%"))
  (/ (- pos-ranks-sum (/ (* n (1+ n))
						 4))
	 (sqrt (/ (* n (1+ n) (+ (* 2 n) 1))
			  24))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Utilities.


(defun check-data (f-data s-data)
  (let ((fsum (sum f-data))
		(ssum (sum s-data)))
  (unless (= (length s-data) (length f-data))
		  (error "Data sets have differing number of data points ~%"))

	(format t "~% ~% Number of censored points for faster system (including doubly censored points): ~s ~%"
			(count-if #'(lambda (v) (>= v *bound*)) f-data))

	(when (>= fsum ssum)
		  (format t "WARNING: The alternate hypothesis is that the first
argument to the test routine represents the faster system ~%")
		  (format t "But the total times for the two systems, in order, are ~s and ~s ~%" fsum ssum) 
		  (format t "The arguments to the test routine may be inverted. ~%")
		  )))



(defun number-of (f-data s-data compare-func)
  (loop for f in f-data
		for s in s-data
		count (funcall compare-func f s)
		))


(defun nc-tie? (v1 v2)
  (and (= v1 v2)
	   (< v1 *bound*)))

(defun pos-diff? (v1 v2)
  (and (< v1 v2)
	   (< v1 *bound*)))

(defun sum (l)
  (loop for n in l
  summing n))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; low-level functions for signed rank test.


(defun find-pos-ranks (list)
  (loop for x in list
		when (= (cadr x) 1) collect (car x)))

(defun compute-ranks (diff-list)
  (reverse (compute-ranks-aux 1 nil diff-list)))

(defun compute-ranks-aux (cur-rank ranked-list diff-list)
  (if diff-list
	  (let* ((ranked-sublist (rank-sublist cur-rank diff-list))
			 (advance (length ranked-sublist))
			 )
		(compute-ranks-aux
		 (+ cur-rank advance)
		 (append ranked-sublist ranked-list)
		 (nthcdr advance diff-list))
		)
	ranked-list))


(defun rank-sublist (rank diff-list)
  (let ((diffs (find-equal-diffs diff-list)))
	(create-ranked-list (+ rank (/ (1- (length diffs)) 2.0)) diffs)))


(defun find-equal-diffs (list)
  (if (or (= (length list) 1)
		  (not (= (caar list) (caadr list))))
	  (list (car list))
	(append (list (car list))
			(find-equal-diffs (cdr list))))
  )


(defun create-ranked-list (rank diffs)
  (cond ((null diffs) nil)
		(t (cons (list rank (cadr (car diffs)))
				 (create-ranked-list rank (cdr diffs))))))
				 
  

;; Returns ((diff-magnitude sign)...)
;; Sign=1 (pos, i.e. f is faster) or -1.
;; Splits ties evenly between pos and neg.
;; Assigns maximal negative magnitude to any f-data point that exceeds
;; the bound.  This covers all censoring.
(defun compute-differences (f-data s-data)
  (let (diff
		(tie-sign -1))
	(loop for f in f-data
		  for s in s-data
		  if (>= f *bound*) collect (list most-positive-fixnum -1)
		  else do (setq diff (min (- s f) *bound*))
		  and if (= diff 0)
		  collect (list diff tie-sign)
		  and do (setq tie-sign (flip-sign tie-sign))
		  else collect (list (abs diff) (signum diff)))
	))
				   

(defun flip-sign (n)
  (if (= n 1) -1 1))




