;;;---------------------------------------------------------------
;;;  Strips.lisp
;;;  Nick Short  Leonard Dickens
;;;  7/92
;;;
;;;  DESCRIPTION:  This file contains the Strips planner, which
;;;  allows deadlines and backtracking.
;;;
;;;
;;;---------------------------------------------------------------


;;; ----------------------------------------------------
;;;
;;;  Strips data structures
;;;
;;; ----------------------------------------------------

(defvar *backtracking-number*
  "The number of times the planner backtracks")

(defvar *debug-strips* nil
  "Variable which indicates whether the debug trace is on")

;;; Note:  in the documentation, we refer to an instantiated
;;;        strips operator as a structure whose bindings slot
;;;        is non-nil.  

(defstruct (strips-operator (:conc-name sonode-) 
			    (:print-function print-strips-operator))
  name                ;;; name of strips operator
  args                ;;; variable arguments for strips operator
  preconditions       ;;; list of precondition formulae which must be made true
  filter              ;;; list of formulae used to select an operator
  add-list            ;;; list of formulae which are added to the state
  delete-list         ;;; list of formulae which are deleted from the state
  (command-string " ") ;; string that pprints the output
  (duration 0)	      ;;; for use in meeting deadlines
  ;; Instantiation-specific information:
  (bindings nil)      ;;; variable bindings for this frame from unification
  (heuristic-exprs nil) ;;; heuristic expression, will be evalled.
)

(defun print-strips-operator (node stream depth)
    (let* ((*print-gensym* nil)
	   (bindings (sonode-bindings node))
	   (preconditions (strips-subst (sonode-preconditions node)
					    bindings))
	   (filter (strips-subst (sonode-filter node)
				 bindings))
	   (add-list (strips-subst (sonode-add-list node)
				   bindings))
	   (delete-list (strips-subst (sonode-delete-list node)
				      bindings))
	   (command-string (strips-subst (read-from-string
					  (format nil "(~A)"
					      (sonode-command-string node)))
						 bindings)))
			

	  (format stream "{strips:~S ~S~%" (sonode-name node)
		                           (sonode-args node))
	  (format stream "    Command: ~{~S ~}~%" command-string)
	  (if bindings
	   (format stream 
		         "   Bindings: ~{~S ~}~%" bindings))
	  (format stream "   Preconds: ~{~S ~}~%" preconditions)
	  (format stream "     Filter: ~{~S ~}~%" filter)
	  (format stream "   Add-list: ~{~S ~}~%" add-list)
	  (format stream "   Del-list: ~{~S ~}~%" delete-list)
	  (if (sonode-duration node) 
	      (format stream
		         "   Duration: ~S~%}~%" (sonode-duration node)))
	  (if (sonode-heuristic-exprs node)
	      (format stream
		      "    H-Value: ~S~%}~%" (sonode-heuristic-exprs node))
	    (format stream "}~%"))))

;;; This constructor function converts  strips syntax into a frame structure;
;;; strips frame can either be a list frame or another structure
;;;  An action is a list defined as
;;;
;;;        (strips-op <op-name> 
;;;                   :filter (<state-formula>*)
;;;                   :preconditions (<state-formula>*)
;;;                   :add-list (<state-formula>*)
;;;                   :delete-list (<state-formula>*)
;;;                   :duration <n>
;;;                   :command-string <string>
;;;                   :variables (<variable>*)
;;;

(defun create-strips-structure (strips-frame &optional bindings)
   (if (strips-operator-p strips-frame)		 
       (make-strips-operator  :name (get-args strips-frame 'name)
			      :args (get-args strips-frame 'args)
			      :filter (get-args strips-frame 'filter)
			      :preconditions 
			      (get-args strips-frame 'preconditions)
			      :add-list (get-args strips-frame 'add-list)
			      :delete-list (get-args strips-frame 'delete-list)
			      :command-string (get-args strips-frame 
							'command-string)
			      :duration (get-args strips-frame 'duration)
			      :bindings bindings)))


;;;------------------------------------
;;;  strips syntax conversion routines 					   

(defmacro strips-op (name &rest body)
  `(apply 'readstrips ',name ',body))

(defun readstrips (name &rest body &key add-list delete-list preconditions
			                filter duration command-string 
					variables)
  (insert-op-frame (make-strips-operator :name name
					  :args variables
					  :preconditions preconditions
					  :add-list add-list
					  :delete-list delete-list
					  :command-string command-string
					  :duration (if duration duration 0)
					  :filter filter)))

;;; print the entire strips plan library

(defun dump-stripsops (&optional (stream t))
  (maphash #'(lambda (key value)
	       (format stream "~%--------------------~S--------------------~%"
		       key)
	       (format stream "~S~%" value))
	   *operators*))


;;; accessor function for strips-operator structure;
;;; (done in the philosophy of CLOS)

(defun get-args (op-frame arg-type) 
  ;;; for args, preconds, delete-lists, add-lists, filters, bindings, etc.
  (if (strips-operator-p op-frame)
      (case arg-type
	    (name (sonode-name op-frame))
	    (args (sonode-args op-frame))
	    (preconditions (sonode-preconditions op-frame))
	    (filter (sonode-filter op-frame))
	    (add-list (sonode-add-list op-frame))
	    (delete-list (sonode-delete-list op-frame))
	    (duration (sonode-duration op-frame))
	    (variables (sonode-variables op-frame))
	    (command-string (sonode-command-string op-frame))
	    (bindings (sonode-bindings op-frame))
	    (evalled-exprs (sonode-heuristic-exprs op-frame))
	    (otherwise (warn "func:get-args : illegal accessor name")))))

;;;-----------------------------------------------------
;;;  Data Structures for plan selection from the 
;;;  plan library, which is stored as a hash table
;;;

;;; hash table from operator-name -> operator-frame-structure
(defvar *operators* (make-hash-table)
  "hash table for operators")

;;; returns a operator structure
(defun get-op-frame (opname)
  (declare  (special *operators*))
  (gethash opname *operators*))

;;; inserts a strips operator structure into  the plan library  *operators*
(defun insert-op-frame  (opframe)
  (declare  (special *operators*))
  (if (strips-operator-p opframe)
      (setf (gethash (get-args opframe 'name) *operators*) opframe)))
	    
(defun reset-stripsops ()
  (declare  (special *operators*))
  (setf *operators* (make-hash-table)))

;;; ----------------------------------------------------
;;;
;;;   Func: STRIPS
;;;
;;;   This is the main routine that is called for finding
;;;   a plan.
;;;
;;; ----------------------------------------------------

(defun strips (init-state goal &key deadline
			  (subplan-filter #'(lambda (x) t))
			  (dist-from-goal #'get-difference-set)
			  (actions-filter #'get-ordered-actions))
  (declare (special *backtracking-number*
		    *debug-strips*))
  (catch 'failed-plan
    (setf *backtracking-number* 0)
    (mapcar #'(lambda (x) (cons (get-args x 'name)
				  (strips-subst (get-args x 'args)
						(get-args x 'bindings))))
		(strips-1 init-state goal :deadline deadline
			  :subplan-filter subplan-filter
			  :dist-from-goal dist-from-goal
			  :actions-filter actions-filter))))

(defun strips-1 (init-state goal-state &key deadline 
		 (subplan-filter #'(lambda (x) t))
		 (dist-from-goal #'get-difference-set)
		 (actions-filter #'get-ordered-actions))
  (declare (special *backtracking-number*
		    *debug-strips*))
   (do* ((difference (funcall dist-from-goal goal-state init-state))
	  (actions (let ((x (funcall actions-filter 
				     difference init-state goal-state)))
		     (progn 
		       (if *debug-strips*
			   (format t "ACTS: ~A~%" x))
		       x))
		     (cdr actions))
	  (action (car actions) (car actions))
	  (ground-preconditions (strips-subst (get-args action 'preconditions)
					      (get-args action 'bindings))
				(strips-subst (get-args action 'preconditions)
					      (get-args action 'bindings)))
	  (count 0 (1+ count))
	  pre-duration pre-actions middle-state post-actions plan)
	((or plan (null difference)) plan)
	(if (null actions) (throw 'failed-plan 
				  (progn (incf *backtracking-number*)
					 :failure)))
	(if *debug-strips*
	    (format t "trying action ~A: ~A~%" count action))
	(catch 'failed-plan
	  (setf pre-actions (strips-1 init-state ground-preconditions
				      :deadline deadline
				      :subplan-filter subplan-filter
				      :dist-from-goal dist-from-goal
				      :actions-filter actions-filter))
	  (multiple-value-setq (middle-state pre-duration)
			       (has-effects (append pre-actions
						    (list action))
					    init-state))
	  (if *debug-strips*
	    (format t 
	    "===========~%(preconds: ~A, preact: ~A , act: ~A, midd-state ~A~%"
		  ground-preconditions pre-actions action middle-state))

	  ;;; if not a good plan, then failure
	  (if (not (funcall subplan-filter (append pre-actions (list action))))
	      (throw 'failed-plan 
				  (progn (incf *backtracking-number*)
					 :failure)))
	  ;;; if plan takes longer than deadine , then failure
	  (if (and deadline (> pre-duration deadline))
	      (throw 'failed-plan 
		     (progn (incf *backtracking-number*) :failure)))

	  ;;; calculate the actions after this action is chosen
	  (setf post-actions (strips-1 middle-state goal-state 
				       :deadline 
				         (if deadline
					   (- deadline pre-duration))
				      :subplan-filter subplan-filter
				      :dist-from-goal dist-from-goal
				      :actions-filter actions-filter))	 
	  (if *debug-strips*
	    (format t 
	    "Postactions: ~A~%" post-actions))

	  (if (not (equal post-actions :failed-plan))
	    (setf plan (append pre-actions (list action) post-actions))))))

;;;---------------------------------------------------------------
;;;
;;; 
;;;
;;;---------------------------------------------------------------


;;; --------------------------------------------------------------
;;; has-effect returns the effects and the duration for executing
;;; the actions.
;;;

(defun has-effects (actions state)
  (do* ((curr-state state)
       (action-list actions (cdr action-list))
       (action (car action-list)
	       (car action-list))
       (delete-list (strips-subst
		     (get-args action 'delete-list) 
		     (get-args action 'bindings))
		    (strips-subst
		     (get-args action 'delete-list) 
		     (get-args action 'bindings)))
       (add-list (strips-subst
		     (get-args action 'add-list) 
		     (get-args action 'bindings))
		 (strips-subst
		     (get-args action 'add-list) 
		     (get-args action 'bindings)))
       (total-duration 0 (+ total-duration 
			    (if action 
				(get-args action 'duration) 0))))
   ((null action-list) (values curr-state total-duration))
     ;;; delete states
	 (setq curr-state
	       (remove-if #'(lambda (x) (member x delete-list :test 'equal))
		    curr-state))
     ;;; add new states
	 (setq curr-state
	       (remove-duplicates 
		(append add-list curr-state) :test 'equal))))
	 

(defun evalled-list-> (l1 l2)
  (cond ((null l2) (not (null l1)))
	((null l1) nil)
	(t 
	 (let ((v1 (eval (car l2)))
	       (v2 (eval (car l1))))
	   (cond ((< v1 v2))
		 ((> v1 v2) nil)
		 (t (evalled-list-> (cdr l1) (cdr l2)))))
	 )))

	 
;;; --------------------------------------------------------------
;;;  returns a list of actions ordered by how many formulae they
;;;  satisfy in the diff.

(defun get-ordered-actions (diff currstate goalstate)
  (let* ((possible-acts (get-possible-actions diff currstate)))
    (calculate-action-value possible-acts diff currstate goalstate)
    (sort possible-acts #'evalled-list-> :key #'sonode-heuristic-exprs)))

;;; --------------------------------------------------------------
;;; assigns score based on the number of goals the operators match
;;; Note: the actions formal is a list of instantiated strips operators;
;;;       diff is a list of formulae
;;;  
;;; calculate-action-order returns a list ( (<n> <strips-operator>)* )
;;; where <n> indicates the number of goals satisfied in diff and
;;;       <strips-operator> is unique in the return list
;;;  

(defun calculate-action-value (actions diff currstate goalstate)
  (mapc
   #'(lambda (x)
       (setf (sonode-heuristic-exprs x)
	     (list 
	      ;; The most important thing about the operator is how many
	      ;;  goals (things in diff) its addlist fulfills
	      (length (intersection
			  diff
			  (strips-subst (get-args x 'add-list) 
					(get-args x 'bindings))
			  :test #'strips-unify))
	      (- (length (intersection
			  goalstate
			  (strips-subst (get-args x 'delete-list) 
					(get-args x 'bindings))
			  :test #'strips-unify)))
	      ;; Second most important is how closely the operator's
	      ;;  preconditions match the current state
	      (- (length (intersection
			  currstate
			  (strips-subst (get-args x 'preconditions)
					(get-args x 'bindings))
			  :test #'strips-unify))
		 (length (get-args x 'preconditions)))
	      ;; Finally, we throw in a random value to insure that each
	      ;;  time operators get generated, we get random tie-breaking
	      ;;  of valuewise- identical operators
	      (random 10))))
   actions))


						      
;;; --------------------------------------------------------------

(defun get-possible-actions (goals currstate)
  (do ((curr-goals goals (cdr curr-goals))
       result)
      ((null curr-goals) (reverse result))
    (do* ((inst-ops (get-ops (car curr-goals))
		    (cdr inst-ops))
	  (trial-op (car inst-ops) (car inst-ops))
	  preconditions filters op-bindings)
	((null inst-ops))
      (setf preconditions (get-args trial-op 'preconditions))
      (setf filters       (get-args trial-op 'filter))
      (setf op-bindings 
	(filter-check currstate filters (sonode-bindings trial-op)))
      (dolist (binding op-bindings)
	(if (and binding
		 (not (vars-in-preconds-p (strips-subst preconditions
							binding)))
		 (filter-check-not-equal filters binding))
	    (push (create-strips-structure (car inst-ops) binding)
		  result))))))
		       
;;; --------------------------------------------------------------           
;;; checks to see if state-2 is possibly true given state-1
;;; state-1 is a list of ground formulae
;;; state-2 is a list of formulae possibly containing variables
;;; Returns a list of possible bindings, each of which is an
;;; augmentation (more specific binding) of the binding argument;
;;; or it returns nil (no bindings found) signifying failure.

(defun filter-check (state-1 state-2 binding)
  ;; Each item in instantiated-state-2 *must* be unifyiable
  ;;   with some item in state-1, or else failure
  (if binding
      (let ((instantiated-state-2 (strips-subst state-2 binding)))
	(bindings-intersect-binding
	 (filter-check-1 state-1 instantiated-state-2)
	 binding))))


(defun filter-check-1 (state-1 state-2)
  ;; Each item in state-2 must be unifiable with some item in state-1, 
  ;;  or else failure
  (cond 
   ((null state-2) 
    ;; If state-2 has no items, it will match any state-1 trivially.
    '((nil)))
   ((null state-1)
    ;; There are items to match in state-2, but no items to match with.  Fail.
    nil)
   (t
    ;; Attempt to match the first item in state-2 with some formula from
    ;;  state-1, by looking at every formula in state-1 for a match.
    (let ((item (car state-2))
	  (new-binding nil)
	  (return-bindings nil))
      (cond 
       ((eq (car item) :equal)
	;; Unify one item of the equal with the other
	(setf new-binding (strips-unify (cadr item) (caddr item)))
	(filter-check state-1 (cdr state-2) new-binding))
       ((eq (car item) :not-equal)
	;; Put off processing not-equal until after all binding
	(filter-check-1 state-1 (cdr state-2)))
       (t
	(do ((state-1-x    state-1  (cdr state-1-x))
	     (formula (car state-1) (car state-1-x)))
	    ((null state-1-x) return-bindings)
	  (setf new-binding (strips-unify item formula))
	  (cond 
	   ((null new-binding)
	    ;; No possible unification found on this pair of formulae.
	    ;; Continue looking through state-1 for unifiers
	    )
	   ((null (car new-binding))
	    ;; exactly matching formulae found; no action needed; 
	    ;; return result of the rest of state-2
	    (return (filter-check-1 state-1 (cdr state-2))))
	   (t
	    ;; A formula that unified with item was found.
	    ;; Here we have choice possibilities; we check both:
	    ;; (1) Recursively call filter-check on the situtation
	    ;;     with the new binding, minus the matched formula.
	    ;; (2) Keep checking through state-1 for other formulae
	    ;;     which will match item
	    (setf return-bindings 
	      (bindings-union
	       return-bindings
	       (filter-check state-1 (cdr state-2) new-binding)))
	    ;; Continue looking through state-1 for unifiers
	    )
	   ))))))))
	 
	     


(defun vars-in-preconds-p (preconditions)
   (cond ((null preconditions) nil)
	 ((var-member (car preconditions)))
	 ((vars-in-preconds-p (cdr preconditions)))))

(defun var-member (precond)
  (cond ((null precond) nil)
	((and (listp (car precond))
	      (var-member (car precond))
	      (var-member (cdr precond))))
	((pcvar-p (car precond)))
	((var-member (cdr precond)))))


;;; --------------------------------------------------------------           
;;; Checks any :not-equal conditions in the list of filters to see that
;;;  they are indeed not equal.
;;;
(defun filter-check-not-equal (filters binding)
  ;; Each filter of the form (:not-equal <a> <b>) most have <a> and <b>
  ;;  not unifiable with each other.
  (let (new-binding)
    (dolist (filter (strips-subst filters binding) t)
      (cond ((eq (car filter) :not-equal)
	     (setf new-binding (strips-unify (cadr filter) (caddr filter)))
	     (cond 
	      ((null new-binding)
	       ;; No possible unification found on this pair of formulae.
	       ;; This is good.  Keep searching for other filters to fail.
	       )
	      ((null (car new-binding))
	       ;; exactly matching formulae found; definite failure
	       (return nil))
	      (t
	       ;; A formula that unified with filter was found.
	       ;; Here what we ought to do in the ideal world of lots of
	       ;;  time to dork around with strips, is to keep the binding
	       ;;  around somehow and contrain the state not to have it true.
	       ;;  Instead, we simply filter out this operator.
	       (return nil))
	      ))))))


;;; --------------------------------------------------------------
;;;
;;;
;;; Gets a list of action structures which have the formula
;;; in their add lists; it returns a list of instantiated-strips-operators
;;;

(defun get-ops (formula)
  (let (result)
    (maphash #'(lambda (key strips-operator)
		 (declare (ignore key))
		 (let ((bindings (check-single-operator strips-operator
							formula)))
		   (if bindings 
		       (push 
			(create-strips-structure strips-operator bindings)
			result))))
		 *operators*)
    result))

;;; --------------------------------------------------------------
;;; checks to see if a formula is in the op's add-list
;;; if so, then the bindings are returned

(defun check-single-operator (op formula)
  (do* ((add-list (get-args op 'add-list) (cdr add-list))
	(bindings (strips-unify (car add-list) formula)
		  (strips-unify (car add-list) formula))) 
       ((or (null add-list)
	    bindings) bindings)))
       

;;; the unify function comes from unify.lisp of the UMD
;;; Nonlin implementation;  the code also comes from
;;; Charniak & McDermott's AI programming book

(defun strips-unify (pat1 pat2)
  (unify pat1 pat2))

(defun strips-subst (pattern bindings) 
   (safe-replace-variables pattern (car bindings)))

;;; merge two binding lists from strips-unify -- note: no two 
;;; duplicate bindings will be included due to the 
;;; union

(defun merge-bindings (b1 b2)
  (let ((new-bindings (union (car b1) (car b2) :test #'equal)))
    (if new-bindings (list new-bindings))))


;;; if there are no formula in common from state-1 and state-2
;;;  then state-1 is the difference

(defun get-difference-set (state-1 state-2) 
  (set-difference state-1 state-2
			 :test #'strips-unify))
