;;; Lucid CL interface to C code implementing UNIX socket client and server
;;; Copyright (c) 1992  Leonard Dickens
;;; Intelligent Data Management, NASA Goddard Space Flight Center
;;;
;;; This software is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
;;; to anyone for the consequences of using it or for whether it serves any 
;;; particular purpose or works at all.
;;;
;;; You may copy, modify and generally use this software freely, as long as
;;; this copyright notice is left intact in every copy or derivative work.
;;;

(in-package 'socket)

(export '(open-client
          server-initialize
          server-startup
          server-get-connection
          server-poll
          server-end-of-file
          server-shutdown))

;;;;;;;;;;;;;;;;;;;;;;; Client (output) interface ;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; Open socket function takes two strings (port and host),
;;;  and returns a file descriptor for the socket.
;;;
(def-foreign-function 
  (c-open-socket
   (:return-type :signed-32bit)
   (:name "_open_socket"))
  (port :simple-string)
  (host :simple-string))

(load-foreign-files "open_socket.o")

(defun open-client (&key port (host "localhost"))
  (declare (type simple-string port)
	   (type simple-string host))
  ;; See the Advanced User's Guide, p 8-20 for more on this:
  (let ((socket-descriptor (c-open-socket host port)))
    (if (< socket-descriptor 0)
	(warn "failed to open output socket")
      (make-lisp-stream
       :input-handle socket-descriptor
       :output-handle socket-descriptor
       :auto-force t))))
		


;;;;;;;;;;;;;;;;;;;;;;; Server (input) interface ;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(def-foreign-struct connection 
  (connect_fd     :type :signed-32bit)
  (max_clients    :type :signed-32bit)
  (n_clients      :type :signed-32bit)
  (port           :type (:pointer :character))
  (input_procedure           :type :unsigned-32bit)  ; actually a proc pointer
  (new_connection_procedure  :type :unsigned-32bit)
  (end_of_file_procedure     :type :unsigned-32bit))


(def-foreign-function 
  (c-socket-server-initialize
   (:name "_lisp_socket_server_initialize")))

(def-foreign-function 
  (c-socket-server-startup
   (:return-type :signed-32bit)
   (:name "_socket_server_startup"))
  (port :simple-string)
  (input-handler :unsigned-32bit)
  (max-clients :signed-32bit)
  (new-connection-handler :unsigned-32bit)
  (end-of-file-handler :unsigned-32bit)
  )




(def-foreign-callable
  (new-connection-handler (:return-type :signed-32bit))
  ((file-descriptor :signed-32bit))

  (setf (aref *input-stream-lookup* file-descriptor)
	(make-lisp-stream :input-handle file-descriptor
			  :output-handle file-descriptor
			  :auto-force t))
  0)

(def-foreign-callable
  (end-of-file-handler  (:return-type :signed-32bit))
  ((file-descriptor :signed-32bit))

  (close (aref *input-stream-lookup* file-descriptor))
  (setf  (aref *input-stream-lookup* file-descriptor) *standard-output*)
  0)


(def-foreign-callable
  (all-input-handler  (:return-type :signed-32bit))
  ((file-descriptor :signed-32bit))

  (setq *input-file-descriptor* file-descriptor)
  (let* ((s (aref *input-stream-lookup* file-descriptor))
	(port (foreign-string-value 
		   (connection-port (socket:server-get-connection))))
	(ch nil)
	(handler (gethash port *function-hash*)))
    (funcall handler s)
    (setq ch (read-char-no-hang s nil :eof))
    (cond ((eq ch :eof)
	   (c-socket-server-end-of-file 
	    file-descriptor))
	  ((eq ch nil))
	  (t (unread-char ch s))))
  0)



;;;
;;; C object code that implements a generic server interface.
;;;
(load-foreign-files "socket_server.o")
(load-foreign-files "lisp-interface.o")


;;;
;;; Global variables to allow us to see values in handle-input that
;;;  get initialized in socket-server-loop.
;;;
(defvar *input-file-descriptor* nil)
(defvar *input-handlers* nil)
(defvar *input-stream-lookup* nil)

(defvar *function-hash* nil)


;;; 
;;; Definitions for the size external functions defined here:
;;;

(defun server-initialize ()
  (setq *function-hash* (make-hash-table :test #'equal :size 7))
  (setq *input-handlers* nil)
  (setq *input-stream-lookup*
	;; Note the assumed limit of 64 fds following:
	(make-array 64 :initial-element *standard-output*))
  (c-socket-server-initialize))

(defun server-startup (&key port read-stream-input-proc)
  (declare (type simple-string port)
	   (type function read-stream-input-proc))
  (let* ((n-c-h (foreign-variable-address "_new_connection_handler"))
	 (eof-h (foreign-variable-address "_end_of_file_handler"))
	 (inp-h (foreign-variable-address "_all_input_handler"))
	 )
    (setf (gethash port *function-hash*) read-stream-input-proc)
    (c-socket-server-startup port inp-h 0 n-c-h eof-h)))

(def-foreign-function 
  (server-get-connection
   (:name "_socket_server_get_connection")
   (:return-type (:pointer connection))))

(def-foreign-function 
  (server-poll
   (:name "_lisp_socket_server_poll"))
  (milliseconds :signed-32bit))

(def-foreign-function 
  (server-end-of-file
   (:name "_socket_server_end_of_file"))
  (fd :signed-32bit))

(def-foreign-function 
  (server-shutdown
   (:name "_socket_server_shutdown")))


#|
;;; This didn't work: the closure was probably being garbage collected
;;; somehow or something like that.  If I had time to really debug this
;;; for a while, this is a much more elegant way to get callback functions.

(defun server-startup (&key port read-stream-input-proc)
  (declare (type simple-string port)
	   (type function read-stream-input-proc))
  (let* ((n-c-h (foreign-variable-address "_new_connection_handler"))
	 (eof-h (foreign-variable-address "_end_of_file_handler"))
	 (inp-h-symbol (gensym "input_handler"))
	 inp-h inp-h-string inp-h-function
	 )
    (if read-stream-input-proc
	(progn
	  (setq inp-h-function
		(eval `(def-foreign-callable
			 (,inp-h-symbol  (:return-type :signed-32bit))
			 ((file-descriptor :signed-32bit))
			 ;;
			 (setq *input-file-descriptor* file-descriptor)
			 (let ((s (aref *input-stream-lookup* file-descriptor))
			       (ch nil))
			   (funcall ,read-stream-input-proc s)
			   (setq ch (read-char-no-hang s nil :eof))
			   (cond ((eq ch :eof)
				  (c-socket-server-end-of-file 
				    file-descriptor))
				 ((eq ch nil))
				 (t (unread-char ch s))))
			 0)))
	  (setq inp-h-string (concatenate 'string "_" 
					  (symbol-name inp-h-symbol)))
	  (setq inp-h (foreign-variable-address inp-h-string))
	  (push inp-h-function *input-handlers*)
	  )
      (setq inp-h 0))
    (setf (gethash *function-hash* port) read-stream-input-proc)
    (c-socket-server-startup port inp-h 0 n-c-h eof-h)))
|#
