;;; random utility functions

(defun memberp (item list)
  (member item list :test #'equalp))

(defun keep (item list &key (test #'eql) (key #'identity))
  (remove item list :test-not test :key key))

;;;;
;;;; functions to deal with negation and expressions
;;;;
;;;; ("expressions" are simply conditions or negations thereof.)

(defun negated-expression? (expression)
  ;; is this a negated expression?
  ;; e.g. (NOT (A)) -> t, (NOT (NOT (A))) -> t, (A) -> nil
  ;; compare to expression-sign !!
  (eq (first expression) 'NOT))

(defun de-negate-expression (expression)
  ;; remove ONE layer or negation
  ;; e.g. (NOT (NOT (A))) -> (NOT (A)), (A) -> meaningless
  ;; compare to negatify-expression !!
  (second expression))

(defun positive-expression (expression)
  ;; remove ALL layers of negation
  ;; e.g. (A) -> (A), (NOT (A)) -> (A), (NOT (NOT (A))) -> (A)
  (if (negated-expression? expression)
      (positive-expression (de-negate-expression expression))
      expression))

(defun expression-sign (expression)
  ;; is this expression "really" negated?  t == positive, nil == negative
  ;; e.g. (NOT (A)) -> nil, (NOT (NOT (A))) -> t
  (if (negated-expression? expression)
      (not (expression-sign (de-negate-expression expression)))
      t))

(defun simplify-expression (expression)
  ;; remove excess layers of negation
  ;; e.g. (A) -> (A), (NOT (NOT (A))) -> (A),
  ;; (NOT (NOT (NOT (A)))) -> (NOT (A))
  (let ((positive-expression (positive-expression expression)))
    (if (expression-sign expression)
	positive-expression
        `(NOT ,positive-expression))))

(defun negatify-expression (expression)
  ;; returns the negation of this expression
  ;; e.g. (A) -> (NOT (A)), (NOT (A)) -> (A)
  (simplify-expression `(NOT ,expression)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A simple best first search strategy.  Returns 3 values: the found state,
;;;  the average branching factor, and the number of generated but unexplored 
;;;  states.  The search will only generate up to LIMIT states.  
;;;  Calls search-trace for debugging purposes.
(defun BESTF-SEARCH (initial-state daughters-fn goal-p rank-fn limit)
  (let ((branches nil))                         ; compute average branch factor
    (do* ((current-entry nil (car search-queue))
          (current-state initial-state (cdr current-entry))
          (search-queue nil (cdr search-queue)))
         ((or (null current-state)
	      (funcall goal-p current-state)
              (> 0 limit))
          (values current-state
                  (if (null branches) 0
                      (div* (apply #'+ branches) (length branches)))
                  (length search-queue)))
      (incf *nodes-visited*)
      (when (and (> *trace* -1) (zerop (mod *nodes-visited* 100)))
	(format t "~&~S nodes visited; ~S nodes created~%"
		*nodes-visited* *plan-id*))
      (search-trace current-state search-queue rank-fn)
      (let ((children (funcall daughters-fn current-state)))
        (setf limit (- limit (length children)))
        (setf search-queue
              (merge
                  'list search-queue
                  (sort (mapcar #'(lambda (x) (cons (funcall rank-fn x) x))
                                children)
                        #'< :key #'car)
                  #'< :key #'car))
        (push (length children) branches)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Search trace function 
(defun SEARCH-TRACE (current-state search-queue &optional (rank-fn nil))
  (when (> *trace* 0)
    (format t "~%CURRENT PLAN: ~S (rank ~a)"
	    current-state (funcall rank-fn current-state))
    (format t "~%QUEUE   -  Length ~a" (length search-queue)))
  (when (and (> *trace* 4) search-queue)
    (dolist (q-plan search-queue)
      (let ((plan (cdr q-plan)))
	(format t "~%~3T~S (rank ~a)"
		plan (funcall rank-fn plan))))))

;;;;;;;;;;;;;;;

(defun partition (list &key (test #'eql) (key #'identity))
  (let ((results nil))
    (dolist (element list)
      (let* ((key (funcall key element))
	     (assoc (assoc key results :test test)))
	(if assoc
	    (push element (cdr assoc))
	    (push `(,key . (,element)) results))))
    (mapcar #'cdr results)))

(defun set-equal (set1 set2 &key (test #'equal))
  (and (every #'(lambda (element) (member element set1 :test test)) set2)
       (every #'(lambda (element) (member element set2 :test test)) set1)))

#|
(defmacro walk-list-slow (((before element after) list) &body body)
  ;; a fancy dolist that keeps track of what comes before and after
  `(do ((,before nil (nreverse (cons ,element (nreverse ,before))))
	(,after (cdr ,list) (cdr ,after))
	(,element (car ,list) (car ,after)))
       ((null ,element))
     ,@body))
|#

(defmacro walk-list (((before element after) list) &body body)
  ;; a fancy dolist that keeps track of what comes before and after
  ;; this version gets the before out of order but does much less
  ;; consing!!
  `(do ((,before nil (cons ,element ,before))
	(,after (cdr ,list) (cdr ,after))
	(,element (car ,list) (car ,after)))
       ((null ,element))
     ,@body))

#|

    for example:

USER(40): (walk-list ((before element after) '(a b c d e f))
	    (format t "b=~S, e=~S, a=~S~%" before element after))
b=NIL, e=A, a=(B C D E F)
b=(A), e=B, a=(C D E F)
b=(B A), e=C, a=(D E F)
b=(C B A), e=D, a=(E F)
b=(D C B A), e=E, a=(F)
b=(E D C B A), e=F, a=NIL
NIL
USER(41): (walk-list-slow ((before element after) '(a b c d e f))
	    (format t "b=~S, e=~S, a=~S~%" before element after))
b=NIL, e=A, a=(B C D E F)
b=(A), e=B, a=(C D E F)
b=(A B), e=C, a=(D E F)
b=(A B C), e=D, a=(E F)
b=(A B C D), e=E, a=(F)
b=(A B C D E), e=F, a=NIL
NIL

|#


;; for bayes assessor and examples that might want to use it to 
;; automatically construct states

(defun bit-set? (bit-number int)
  (not (zerop (logand (expt 2 bit-number) int))))
