;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Code used to handle a detected unsafety condition.  Returns a list
;;;  of plan refinements that resolve the threat.

(defun HANDLE-UNSAFE (plan)
  (let ((unsafe (get-an-unsafe plan))
	(*verbose-otpaths?* nil)) ;; verbosity is especially obnoxious here
    (when (> *trace* 1)
      (format t "~%Resolve ~S?" unsafe))
    (cond ((still-threatens? unsafe plan)
	   (nconc (demote unsafe plan)
		  (promote unsafe plan)
		  (confront unsafe plan)))
	  (t
	   (when (> *trace* 1)
	     (format t "~%~3T* Unsafe evaporated"))
	   (list
	    (tweak-plan
	     plan
	     :reason `(:evaporated ,unsafe)
	     :unsafe (remaining-unsafes plan)))))))

(defun still-threatens? (unsafe plan)
  (let ((link (unsafe-link unsafe)))
    (member (unsafe-threatener unsafe)
	    (possibly-between (link-producer link)
			      (link-consumer link)
			      plan))))

;; unsafes are removed as if from a stack.  so these two functions, called
;; together, are used to extract an unsafe and then return the list of
;; unsafges without that one.  note that handling unsafes is much easier than
;; handling opens -- they don't need to get stuck on a "linked" list.

(defun get-an-unsafe (plan)
  (car (plan-unsafe plan)))

(defun remaining-unsafes (plan)
  (cdr (plan-unsafe plan)))

;;
;; the refinement operations
;;

(defun demote (unsafe plan)
  (when (> *trace* 1)
    (format t "~%~3TResolve ~S by demotion?" unsafe))
  (let* ((threatener
	  (unsafe-threatener unsafe))
	 (producer
	  (link-producer (unsafe-link unsafe)))
	 (demotable?
	  (member threatener (possibly-prior producer plan))))
    (if demotable?
	(let ((new-plan
	       (tweak-plan
		plan
		:reason `(:demote ,unsafe)
		:ordering (add-ordering plan threatener producer)
		:unsafe (remaining-unsafes plan))))
	  (when (> *trace* 1)
	    (format t "~%~6T* Yes! ~S" new-plan))
	  (list new-plan)))))

(defun promote (unsafe plan)
  (when (> *trace* 1)
    (format t "~%~3TResolve ~S by promotion?" unsafe))
  (let* ((threatener
	  (unsafe-threatener unsafe))
	 (consumer
	  (link-consumer (unsafe-link unsafe)))
	 (promotable?
	  (member consumer (possibly-prior threatener plan))))
    (if promotable?
	(let ((new-plan
	       (tweak-plan
		plan
		:reason `(:promote ,unsafe)
		:ordering (add-ordering plan consumer threatener)
		:unsafe (remaining-unsafes plan))))
	  (when (> *trace* 1) 
	    (format t "~%~6T* Yes! ~S" new-plan))
	  (list new-plan)))))

(defun confront (unsafe plan)
  ;; adds links from all good outcomes of the threatener to the producer,
  ;; and records that this step is an undermineer of the links if it has any
  ;; bad outcomes.
  (when (> *trace* 1)
    (format t "~%~3TResolve ~S by confrontation?" unsafe))
  (let* ((steps
	  (plan-steps plan))
	 (links
	  (plan-links plan))
	 (old-link
	  (unsafe-link unsafe))
	 (link
	  ;; the link may have changed since it was made into an unsafe.
	  ;; in particular, underminers may have been added.  so get
	  ;; a fresh copy of the old link from the current list.  so we need
	  ;; to do an expensive test.
	  (find-if #'(lambda (ln) (same-link? old-link ln))
		   links))
	 (condition
	  (link-condition link))
	 (not-condition
	  (negatify-expression condition))
	 (producer
	  (link-producer link))
	 (consumer
	  (link-consumer link))
	 (threatener
	  (unsafe-threatener unsafe))
	 (threatener-step
	  (find threatener steps :key #'plan-step-id))
	 (old-threatener-template
	  (plan-step-template threatener-step))
	 (old-threatener-otpaths
	  (step-template-otpaths old-threatener-template))
	 (old-bad-otpaths
	  (remove-if-not
	   #'(lambda (otpath)
	       (memberp not-condition (otpath-outcomes otpath)))
	   old-threatener-otpaths))
	 (old-good/neutral-otpaths
	  (get-good/neutral-otpaths condition old-threatener-otpaths))
	 ;; things get tricky now.  we need to rebuild the threatening step's
	 ;; good/neutral outcomes to include the special underminer condition.
	 ;; this means we also need to change all the links coming from
	 ;; the threatener to use these new outcomes instead of what they
	 ;; have now.  but all this this only need to be done if the threatener
	 ;; actually has bad outcomes
	 (underminer-condition
	  ;; the condition used to asses p(threats)
	  (if old-bad-otpaths
	      (list
	       (gentemp
		(format nil "S-P~AT~AC~A-" producer threatener consumer)))))
	 (underminer-open
	  ;; an open condition corresponding to this condition
	  (if old-bad-otpaths
	      (make-openc :condition underminer-condition
			  :consumer consumer)))
	 (new-threatener-otpaths	
	  ;; the new outcomes with the underminer condition added where needed
	  (let ((old-otpaths
		 (step-template-otpaths old-threatener-template)))
	    (if old-bad-otpaths
		(mapcar
		 #'(lambda (otpath)
		     (if (member otpath old-good/neutral-otpaths)
			 (make-otpath
			  :trigger (otpath-trigger otpath)
			  :outcomes `(,underminer-condition
				      ,@(otpath-outcomes otpath))
			  :label (otpath-label otpath))
		         otpath))
		 old-otpaths)
	        old-otpaths)))
	 (new-threatener-step-template
	  ;; build a new template for the threatening step; later we'll add
	  ;; the underminer condition to the good outcomes
	  (if old-bad-otpaths
	      (make-step-template
	       :action (step-template-action old-threatener-template)
	       :otpaths new-threatener-otpaths)))
	 (new-threatener-step
	  ;; the new threatener step with this new template
	  (if old-bad-otpaths
	      (make-plan-step
	       :template new-threatener-step-template
	       :id threatener)))
	 (new-good-otpaths
	  ;; the otpaths that actually support the threatned condition
	  (get-good-otpaths condition new-threatener-otpaths))
	 (new-links-to-consumer
	  ;; the links from the good otpaths to the consumer of the link, 
	  ;; but don't add links that already exist.  siunce the otpaths
	  ;; were just changed -- the underminer condition was added --
	  ;; we can't just compare the otpaths directly.  instead we
	  ;; will look at the labels.  this is slightly dangerous!!
	  ;; note that if we find one of these links already there, it
	  ;; will get fixed up to use the new otpath just a little later.
	  ;;
	  ;; 12/19/92: it looks like this code makes all links to the
	  ;; consumer from the threatener.  it seems to me it shouldn't add
	  ;; any -- that should be the job of the NEW-LINK code.  the lines
	  ;; marked with (*+*) below are fixes to make this happen.  (i guess
	  ;; another option would be to produce N new children here, where
	  ;; there are N good otpaths but this seems less pure, too.)
	  ;; note that this fix makes everything slow down since these
	  ;; links have to be added one by one.
	  ;;
	  (and nil ;; (*+*)
	       (apply
		#'nconc
		(mapcar
		 #'(lambda (otpath)
		     (unless
			 (find-if #'(lambda (l)
				      (and (eql (link-consumer l) consumer)
					   (eql (link-producer l) threatener)
					   (eql (otpath-label (link-otpath l))
						(otpath-label otpath))
					   (equal (link-condition l) condition)))
				  links)
		       (list
			(make-link
			 :condition
			 condition
			 :denouement
			 (make-denouement :step-id threatener :otpath otpath)
			 :consumer
			 consumer))))
		 new-good-otpaths))))
	 (replacement-link
	  ;; replacement for the threatened link, because we add to its
	  ;; underminers (but only when the threatener actually has bad
	  ;; outcomes)
	  (if old-bad-otpaths
	      (let ((rl (copy-link link)))
		(push underminer-open (link-underminers rl))
		rl)))
	 (links-from-threatener
	  ;; we need to fix up the links from the threatening step so
	  ;; so that they see the new otpaths not the old ones
	  (if old-bad-otpaths
	      (keep threatener links :key #'link-producer)))
	 (new-links-from-threatener
	  ;; change the denouement's to use the new otpaths.  again we
	  ;; do this by examining the labels.  so this, too, may be dangerous!
	  (if old-bad-otpaths
	      (mapcar
	       #'(lambda (l)
		   (make-link
		    :condition
		    (link-condition l)
		    :consumer
		    (link-consumer l)
		    :denouement
		    (make-denouement
		     :step-id threatener
		     :otpath (find (otpath-label (link-otpath l))
				   new-threatener-otpaths
				   :key #'otpath-label))))
	       links-from-threatener)))
	 (new-plan
	  (tweak-plan
	   plan
	   :reason
	   `(:confront ,unsafe)
	   :links
	   (if old-bad-otpaths
	       (cons replacement-link
		     (append ;; new-links-to-consumer (*+*)
			     new-links-from-threatener
			     (set-difference (remove link links)
					     links-from-threatener)))
	     links);; was (append new-links-to-consumer links) (*+*)
	   :steps
	   (if old-bad-otpaths
	       (cons new-threatener-step (remove threatener-step steps))
	       steps)
	   :open
	   (let ((new-open (new-opens-from-links nil plan)))
	     ;; above, 'nil' was 'new-opens-from-plans' (*+*)
	     (if old-bad-otpaths
		 (cons underminer-open new-open)
	         new-open))
	   :ordering
	   (add-ordering plan producer threatener threatener consumer)
	   :unsafe
	   (nconc ;; hope nconc doesn't destory last arg
	    (apply #'nconc
		   (mapcar #'(lambda (link) (test-link plan link))
			   nil))	; was 'new-links-to-consumer' (*+*)
	    (remaining-unsafes plan)))))
    (unless (or new-good-otpaths old-bad-otpaths)
      (error "internal error -- ~S seems to have mysteriously evaporated"
	     unsafe))
    (when (> *trace* 0.5)
      (format t "~%~9T* Yes! ~S" new-plan))
;; from a nasty debugging session:
;;    (when (member (plan-id new-plan) '(24146 39154 22 218))
;;      (format t "CONFRONT:~%~3T~S~%~3T~S~%~3T~S~%~3T~S~%~3T~S~%"
;;	      plan new-plan unsafe link replacement-link)
;;      (when (= (plan-id new-plan) 39154)
;;	(setf **p** new-plan)
;;	(cerror "got the plan!!" "do nothing")))
    (list new-plan)))

(defun same-link? (link1 link2)
  ;; just like (equalp link1 link2) but more efficient
  ;; it compares the consumer, producer and outcome which is enough to
  ;; prove that the two links are in fact the same. outcomes are compared
  ;; using just the label which is a bit dangerous I suppose.  underminers
  ;; don't need to be checked, and they aren't which allows a little slop:
  ;; if an old version of the link is floating around (from before
  ;; confrontation) then this will find it anyway.  this is for the benefit
  ;; of 'confront' only -- nothing else should rely on this!!
  (and (eql (link-consumer link1) (link-consumer link2))
       (eql (link-producer link1) (link-producer link2))
       (eql (otpath-label (link-otpath link1))
	    (otpath-label (link-otpath link2)))
       (equal (link-condition link1) (link-condition link2))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6.2. Detecting unsafety conditions

;;;  Tests whether the link is possibly threatened by any steps possibly in the
;;;  "range" of the link.  Returns a list (possible empty) of the unsafes.

(defun TEST-LINK (plan link)
  (let* ((new-unsafe
	 nil)
	 (condition
	  (link-condition link))
	 (betweens
	  (possibly-between (link-producer link) (link-consumer link) plan))
	 (*verbose-otpaths?*
	  nil))
    (when betweens ;; just for efficiency
      (when (> *trace* 1) 
	(format t "~%~9TTest ~S with betweens ~S" link betweens))
      (dolist (step (plan-steps plan))
	(let ((step-id (plan-step-id step)))
	  (when (member step-id betweens)
	    (when (> *trace* 2) 
	      (format t "~%~12TTest ~S" step))
	    (when (possibly-threatens? step condition)
	      (let ((unsafe (make-unsafe :threatener step-id :link link)))
		(push unsafe new-unsafe)
		(when (> *trace* 2) 
		  (format t "~%~15T* New unsafe ~S" unsafe))))))))
    new-unsafe))


;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a step possibly threatens any link.  Returns nil if all
;;;  links are safe, otherwise returns a list of unsafes.
;;;  Assumes that step is only constrained wrt start and a single other point.
;;;  This lets us only have to check if a link's source is prior to the step.

(defun TEST-STEP (plan step)
  (let* ((new-unsafe
	  nil)
	 (step-id
	  (plan-step-id step))
	 (priors
	  (possibly-prior step-id plan))
	 (*verbose-otpaths?*
	  nil))
    (when priors ;; just for efficiency (but 0 is always a prior! oh well)
      (when (> *trace* 1) 
	(format t "~%~9TTest ~S with priors ~S" step priors))
      (dolist (link (plan-links plan))
	(let ((producer (link-producer link)))
	  (when (member producer priors)
	    (when (> *trace* 2) 
	      (format t "~%~12TTest ~S" link))
	    (when (possibly-threatens? step (link-condition link))
	      (let ((unsafe (make-unsafe :threatener step-id :link link)))
		(push unsafe new-unsafe)
		(when (> *trace* 2) 
		  (format t "~%~15T* New unsafe ~S" unsafe))))))))
    new-unsafe))

;;;

(defun possibly-threatens? (step condition)
  ;; does this step have an outcome that touches condition or (not condition)?
  (let ((not-condition (simplify-expression `(not ,condition))))
    (some #'(lambda (otpath)
	      (let ((outcomes (otpath-outcomes otpath)))
		(memberp not-condition outcomes)
		;; doesn't seem to be a good reason to detect C - C non-threats
		;; just check for real threats: C - (NOT C)
		;; was:
		;; (or (memberp condition outcomes)
		;;     (memberp not-condition outcomes))
		))
	  (step-template-otpaths (plan-step-template step)))))
