(defvar *ss-iterations* 100)

;;;  Possible values are :MIN :MAX :RANDOM
(defvar *ss-partial-order-policy* :MIN)

(defvar *ss-print-assess-stats*)
(setf *ss-print-assess-stats* NIL)

;;;(setf *plan-assessor-fn* 'stochastic-simulation-assessor)

;;;********************************************************

(defun stochastic-simulation-assessor (plan tau)
  (let ((output (ss-assess plan tau)))
    (when *ss-print-assess-stats*
      (format t "~&Value with ~a iterations is ~a, exact is ~a~%"
              *ss-iterations* 
              (float output)
              (dumb-assessor plan tau)))
    output))
    
(defun ss-assess (plan tau)
  (let ((initial-step     (find 0 (plan-steps plan) :key #'plan-step-id))
        (goal-expression  (ss-extract-goal-expression plan))
        (orderings        (plan-ordering plan))
        (steps            (ss-remove-dummy-steps plan)))
    (case *ss-partial-order-policy*
      ((:MIN :MAX) 
       (let* ((sequences (ss-extract-total-orders plan))
              (probs (mapcar #'(lambda (seq) 
                                 (ss-assess-sequence initial-step
                                                     goal-expression
                                                     seq
                                                     NIL))
                             sequences)))
         (apply (if (eq *ss-partial-order-policy* :MIN)
                    #'MIN
                    #'MAX)
                probs)))
      ((:RANDOM) 
       (ss-assess-sequence initial-step
                           goal-expression
                           steps
                           orderings))
      (OTHERWISE (error "Bad arg to ss-assessor ~a"
                        *ss-partial-order-policy*)))))

      
(defun ss-extract-total-orders (plan)
  (let ((steps (plan-steps plan)))
    (mapcar #'(lambda (index-sequence)
                (mapcar #'(lambda (index) (find index 
                                                steps
                                                :key #'plan-step-id
                                                :test #'eql))
                        index-sequence))
            (compute-total-orders plan))))

;;;***********************************

(defun ss-assess-sequence (initial-step goal-expression steps orderings)
  (let ((count 0))
    (dotimes (n *ss-iterations*)
      (when (ss-assess-once initial-step 
                            goal-expression 
                            steps
                            orderings)
        (incf count)))
    (/ count *ss-iterations*)))

;;;********************************************************

(defun ss-extract-goal-expression (plan)
  (trigger-conditions
   (otpath-trigger
    (car (step-template-otpaths
          (plan-step-template
           (find :goal (plan-steps plan) :key #'plan-step-id)))))))

(defun ss-remove-dummy-steps (plan)
  (remove-if #'(lambda (s) (member (plan-step-id s) '(0 :goal) :test #'eql))
             (plan-steps plan)))

;;;*********************************************************************

(defun ss-assess-once (initial-step goal-expression remaining-steps orderings)
  (do ((state (ss-choose-state initial-step))
       (steps (copy-list remaining-steps)))
      ((null steps) (ss-true-in-state goal-expression state))
    (let ((next-step (ss-choose-step-randomly 
                      (ss-possible-first-steps steps orderings))))
      (setf state (ss-progress-step next-step state))
      (setf steps (delete next-step steps))
      (when orderings
        (setf orderings 
              (delete-if #'(lambda (o) 
                             (or (eql (ordering-1st o) 
                                      (plan-step-id next-step))
                                 (eql (ordering-2nd o) 
                                      (plan-step-id next-step))))
                         orderings))))))

;;;****************

(defun ss-choose-state (initial-step)
  (let ((random-prob (random 1.0))
        (prob-so-far 0.0))
    (dolist (otp (step-template-otpaths (plan-step-template initial-step)))
      (setf prob-so-far (+ prob-so-far (trigger-prob (otpath-trigger otp))))
      (when (>= prob-so-far random-prob)
        (return (copy-tree (otpath-outcomes otp)))))))

;;;************

(defun ss-choose-step-randomly (step-list)
  (cond
   ((= 1 (length step-list)) (car step-list))
   (t (nth (random (length step-list)) step-list))))

(defun ss-possible-first-steps (step-list orderings)
  (cond 
   ((null orderings) (list (car step-list)))
   (t (remove-if #'(lambda (step) 
                     (let ((step-id (plan-step-id step)))
                       (some #'(lambda (ord) (eql step-id (ordering-2nd ord)))
                             orderings)))
                 step-list))))

;;;***********

(defun ss-progress-step (step state)
  (let* ((otpaths (step-template-otpaths (plan-step-template step)))
         (probs (mapcar #'(lambda (otp) (ss-eval-otpath otp state)) otpaths))
         (pairs (mapcar #'cons otpaths probs))
         (random-prob (random 1.0))
         (prob-so-far 0.0))
    (dolist (pair pairs)
      (let ((otp (car pair))
            (prob (cdr pair)))
        (setf prob-so-far (+ prob-so-far prob))
        (when (>= prob-so-far random-prob)
          (return (ss-progress-outcome (otpath-outcomes otp) state)))))))

;;;********************

(defun ss-eval-otpath (otp state)
  (let ((trigger (otpath-trigger otp)))
    (cond
     ((ss-true-in-state (trigger-conditions trigger) state)
      (trigger-prob trigger))
     (t 0.0))))

(defun ss-true-in-state (cond-list state)
  (every #'(lambda (c) (member c state :test #'equal))
         cond-list))

;;;********************

(defun ss-progress-outcome (change-list state)
  (dolist (change change-list)
    (setf state (ss-remove-basic-change change state))
    (setf state (cons change state)))
  state)

(defun ss-remove-basic-change (change state)
  (let ((basic-change (if (eq (first change) 'not) (second change) change)))
    (setf state (delete-if #'(lambda (state-prop) 
                               (or (equal state-prop basic-change)
                                   (and (eq (first state-prop) 'not)
                                        (equal (second state-prop) basic-change))))
                           state))))


