;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. handle partial orderings.

(defun POSSIBLY-BETWEEN (s1 s2 plan)
  (intersection (possibly-after s1 plan) (possibly-prior s2 plan)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly prior to a given
;;; step.  Possibly prior always includes the initial conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly prior all
;;; steps that aren't in the not-prior list.
(defun POSSIBLY-PRIOR (step-id plan)
  (case step-id
    ;; a couple of special cases
    (:bhudda
     (mapcar #'plan-step-id (plan-steps plan)))
    (0
     nil)
    (otherwise
     (let ((not-prior (list step-ID :goal))
	   (poss-prior (list '0)))
       (do* ((queue (list step-id) (cdr queue))
	     (np-step step-id (car queue)))
	   ((null queue) not-prior)
	 (setf not-prior
	   (dolist (order (plan-ordering plan) not-prior)
	     (when (eql np-step (ordering-1st order))
	       (let ((2nd (ordering-2nd order)))
		 (unless (member 2nd not-prior)
		   (setf queue (nconc queue (list 2nd))
			 not-prior (nconc not-prior (list 2nd)))))))))
       (dotimes (n (plan-high-step plan))
	 (if (not (member (+ n 1) not-prior))
	     (setf poss-prior (cons (+ n 1) poss-prior))))
       poss-prior))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Returns list of step-id's of steps possibly after a given
;;; step.  Possibly after always includes the goal conditions.
;;; First build a list of steps constrained to be not prior by
;;; the ordering constraints.  Then add to possibly after all
;;; steps that aren't in the not-after list.
(defun POSSIBLY-AFTER (step-id plan)
  (case step-id
    ;; a couple of special cases
    (:goal
     nil)
    (:bhudda
     (warn "internal error: should never need to check what's after :bhudda")
     nil)
    (otherwise
     (let ((not-after (list step-ID '0))
	   (poss-after (list :goal)))
       (do* ((queue (list step-id) (cdr queue))
	     (np-step step-id (car queue)))
	   ((null queue) nil)
	 (dolist (order (plan-ordering plan))
	   (let ((1st (ordering-1st order)))
	     (when (and (eql np-step (ordering-2nd order))
			(not (member 1st not-after)))
	       (setf queue (nconc queue (list 1st))
		     not-after (nconc not-after (list 1st)))))))
       (dotimes (n (plan-high-step plan))
	 (if (not (member (+ n 1) not-after))
	     (setf poss-after (cons (+ n 1) poss-after))))
       poss-after))))

;;; Returns correct order: first step at head
;;; Input: max is an integer
;;;    Ordering is a list of pairs (f l) where step number f must be before l
;;;    f, l <= max
;;; See Aho, Hopcoft, Ullman p70 for faster way
(defun TOP-SORT (ordering max)
  (let ((a
	 (top-sort1
	  (mapcar #'(lambda (o)
		      (list (ordering-1st o) (ordering-2nd o)))
		  ordering)
	  max))
	(b nil))
    (dotimes (i max (nconc a b))
      (when (not (member (1+ i) a :test #'eql))
	(push (1+ i) b)))))

;;; Topological Sort util  -   This code is DESTRUCTIVE!  Pass it a copy!
(defun TOP-SORT1 (ordering max)
  (when ordering
    (let ((as (mapcar #'cadr ordering)))
      (do ((p ordering (cdr p)))
	  ((not (member (caar p) as))
	   (cons (caar p)
		 (top-sort1 (delete-if #'(lambda (x) 
					   (eql (car x) (caar p))) ordering)
			    (- max 1))))))))

;;;;;;;

(defun add-ordering (plan &rest steps)
  (let ((ordering (plan-ordering plan)))
    (loop
      (if steps
	  (let ((1st (pop steps))
		(2nd (pop steps)))
	    (cond ((null 2nd)
		   (error "not an even number of steps: ~S" steps))
		  (;; don't insert orderings that always hold
		   (not (or (eql 1st 0)
			    (eql 2nd :goal)
			    ;; these are used only for the fake final step
			    (eql 1st :goal) 
			    (eql 2nd :bhudda)))
		   (pushnew (make-ordering :1st 1st :2nd 2nd) ordering
			    :test #'equalp))))
	  (return)))
    ordering))


;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;

;; ;; ;; all total orders of a partial order  (thanks to jordan!!)

(defstruct (ocache
	    (:print-function ocache-printer))
  ;; a cache for ordering information -- node must come after parents
  node
  parents)

(defun ocache-printer (ocache stream depth)
  (declare (ignore depth))
  (format stream "{~S>~S}" (ocache-node ocache) (ocache-parents ocache)))

(defun compute-total-orders (plan)
  ;; returns a list of all total orders consistent with the given plan
  (let ((ocaches nil))
    (dolist (ordering (plan-ordering plan))
      (let ((1st (ordering-1st ordering))
	    (2nd (ordering-2nd ordering)))
	;; first, record the fact that 1st<2nd
	(let ((ocache (find 2nd ocaches :key #'ocache-node)))
	  (unless ocache
	    (let ((new-ocache (make-ocache :node 2nd :parents '(0))))
	      (push new-ocache ocaches)
	      (setf ocache new-ocache)))
	  (pushnew 1st (ocache-parents ocache)))
	;; second, record the fact that 2nd is a node (even though can't say
	;; what comes before it from just this ordering constraint)
	(let ((ocache (find 1st ocaches :key #'ocache-node)))
	  (unless ocache
	    (let ((new-ocache (make-ocache :node 1st :parents '(0))))
	      (push new-ocache ocaches)
	      (setf ocache new-ocache))))))
    ;; now fix up the ocaches for 0, G and any completely unconstrained steps
    (let ((step-ids nil))
      (dotimes (i (plan-high-step plan))
	(push (1+ i) step-ids))
      ;; (i) step G comes after everything
      (push (make-ocache :node :goal :parents (cons 0 step-ids))
	    ocaches)
      ;; (ii) make sure every step has en ocache entry
      (dolist (step-id step-ids)
	(unless (find step-id ocaches :key #'ocache-node)
	  (push (make-ocache :node step-id :parents '(0))
		ocaches))))
    ;; (iii) step 0 comes before anything
    (push (make-ocache :node 0 :parents nil)
	  ocaches)
    ;; now actually compute the total orders, and throw out the 0
    ;; and :GOAL steps (which always occur first and last) because
    ;; that what consumer of these things wants (see 'dumb-assess-ordered')
    ;; they are there becase really-c-t-o needs them.
    (mapcar #'(lambda (total-order)
		(nbutlast (cdr total-order)))
	    (really-compute-total-orders ocaches))))

(defun really-compute-total-orders (ocaches)
  (let ((nexts
	 (mapcar #'ocache-node
		 (remove nil ocaches :test-not #'eq :key #'ocache-parents)))
	(new-total-orders
	 nil))
    (dolist (next nexts)
      (let ((new-ocaches
	     (mapcar
	      #'(lambda (ocache)
		  (make-ocache :node (ocache-node ocache)
			       :parents (remove next (ocache-parents ocache))))
	      (remove next ocaches :key #'ocache-node))))
	(cond (new-ocaches
	       (setf new-total-orders
		 (nconc
		  (mapcar #'(lambda (new-total-order)
			      (cons next new-total-order))
			  (really-compute-total-orders new-ocaches))
		  new-total-orders)))
	      (t
	       (setf new-total-orders
		 (cons (list next) new-total-orders))))))
    new-total-orders))
