;; stuff related to outcome trees, otpaths, etc

;;; an outcome tree is converted into a set of OTPATHs, a
;;; representation of a path through the tree.  a otpath
;;; stores a list of the conditions that must be true, the "accumulated"
;;; probability along the path, and the list of outcomes.
;;; for example:

#|
(if (A)
    (with-probability
	(0.2
	 (if (B)
	     (with-probability
		 (0.4
		  (if (NOT (C))
		      (outcomes (P) (Q) (NOT (R)))  ;; 1
		      (outcomes (W) (X))))          ;; 2
	         (otherwise		 
		  (outcomes (NOT (Z)))))            ;; 3
	     (outcomes)))                           ;; 4
        (otherwise
	 (outcomes (NOT (G)) (NOT (H)))))           ;; 5
    (outcomes))                                     ;; 6
|#

;;; would get translated in to the following six otpaths:

;;;      conditions          prob        outcomes
;;;      ---------------------------------------------------
;;;  1   ((A) (B) (NOT (C))) 0.08        ((P) (Q) (NOT (R)))
;;;  2   ((A) (B) (C))       0.08        ((W) (X))
;;;  3   ((A) (B))           0.12        (NOT (Z))
;;;  4   ((A) (NOT (B)))     0.2         ()
;;;  5   ((A))               0.8         ((NOT (G)) (NOT (H)))
;;;  6   (NOT (A))            1          ()

;;; three abbreviations:
;;; 1.
;;; '(with-probability (p1 ..) (p2 ..) .. (pn ..) (otherwise ..))'
;;;      gets treated as if it were
;;; '(with-probability (p1 ..) (p2 ..) .. (pn ..) ((- 1 p1 p2 .. pn) ..))'
;;; 2.
;;; '(if test then)' gets treated as if it were '(if test then (outcomes))'
;;; (ie, if you leave out the else, then it means the same as "this step has
;;; has no effect under these conditions")
;;; 3.
;;; '(with-probability (p1 ..) (p2 ..) .. (pn ..))' but the sum
;;; p1+p2+..+pn < 1, then the tree gets treated as if it were
;;; '(with-probability (p1 ..) (p2 ..) .. (pn ..) (otherwise (outcomes)))'

;; short labels are used to print otpaths instead of a full
;; symbolic description, iff the variable *verbose-otpaths* is nil

(defvar *verbose-otpaths?* t)

;; and here are the labels: each otpath is given a unique
;; one-character label.  call 'otpath-reset-label' to start the
;; labelling over again (ie, between def-steps).

(defvar *otpath-current-label*)

(defun otpath-inc-label ()
  (let ((label *otpath-current-label*))
    (setf *otpath-current-label*
      (code-char (1+ (char-code *otpath-current-label*))))
    label))

(defun otpath-reset-label ()
  (setf *otpath-current-label* #\A))

;;;;;

;; are these two denouments equal. cheat by looking just at labels -- this
;; is a little dangerous, I suppose.

(defun denouement-equalp (d1 d2)
  (and (eq (denouement-step-id d1)
	   (denouement-step-id d2))
       (eq (otpath-label (denouement-otpath d1))
	   (otpath-label (denouement-otpath d2)))))

;;;;;

(defun outcome-tree->otpaths (outcome-tree)
  (otpath-reset-label)
  (ot->otpaths outcome-tree nil 1))

(defun ot->otpaths (outcome-tree conditions prob)
  (if outcome-tree
      (ecase (first outcome-tree)
	(IF
	 (let* ((condition (second outcome-tree))
		(then (third outcome-tree))
		(else (or (fourth outcome-tree) '(outcomes))))
	   (append
	    (ot->otpaths
	     then
	     (cons (simplify-expression condition) conditions)
	     prob)
	    (ot->otpaths
	     else
	     (cons (simplify-expression `(NOT ,condition)) conditions)
	     prob))))
	(WITH-PROBABILITY
	 (let* ((branches (cdr outcome-tree))
		(last-branch (car (last branches))))
	   (if (eq (first last-branch) 'OTHERWISE)
	       (setf (first last-branch)
		 (- 1 (apply #'+ (mapcar #'first (butlast branches))))))
	   (let ((sum (apply #'+ (mapcar #'first branches))))
	     (cond ((> sum 1)
		    (error "WITH-PROBABILITY branches sum can't exceed 1: ~S"
			   outcome-tree))
		   ((< sum 1)
		    (unless (= 1 sum)
		      (push `(,(- 1 sum) (outcomes)) branches)))))
	   (apply #'append
		  (mapcar #'(lambda (branch)
			      (ot->otpaths
			       (second branch)
			       conditions
			       (* prob (first branch))))
			  branches))))
	(OUTCOMES
	 (list (make-otpath
		:trigger (make-trigger :conditions conditions :prob prob)
		:outcomes (cdr outcome-tree)))))))

(defun retrieve-otpaths (template condition)
  (remove-if-not #'(lambda (otpath)
		     (memberp condition (otpath-outcomes otpath)))
		 (step-template-otpaths template)))

#|
;; if |prob(t1)-prob(t2)| <= epsilon, then t1 and t2 are put in the
;; same equivalance class and they are minimized together.
;; zero is the "right" number but for heuristic purposes it might
;; be better to use a larger number.  however, to take care of
;; arithmetic errors, it's probably best to use a small positive number

(defun build-triggers (otpaths)
  (let* ((triggers (mapcar #'otpath-trigger otpaths))
	 (min-triggers (minimize-triggers triggers)))
    (when (and nil ;; miminimization removed -- no point in checking any more
	       (> *trace* 3)
	       (not (set-equal triggers min-triggers :test #'equalp)))
      (format t "~&~3T[min: ~S => ~S]" triggers min-triggers))
    min-triggers))

(defvar *minimization-epsilon* 0.01) 

(defun minimize-triggers (triggers)
  (let* ((new-triggers
	  nil)
	 (merged-triggers
	  ;; first, merge the triggers together by adding probabilities
	  ;; of those triggers with identical conditions.
	  (mapcar
	   #'(lambda (penultimate-group)
	       (make-trigger
		:conditions (trigger-conditions (first penultimate-group))
		:prob (apply #'+ (mapcar #'trigger-prob penultimate-group))))
	   (partition triggers :test #'set-equal :key #'trigger-conditions)))
	 (equivalence-classes
	  ;; second, parition the merged triggers by probability
	  (partition merged-triggers
		     :test #'(lambda (p1 p2)
			       (<= (abs (- p1 p2)) *minimization-epsilon*))
		     :key #'trigger-prob)))
    ;; third, minimize the conditions associated with each trigger in the
    ;; partitioned, merged triggers, and get a lower-bound by taking the
    ;; minimum probability from the class.  note that this will not by very
    ;; far (a few multiples of episilon?) from the max/average/etc.
    ;; this could use some real analysis, I suppose.
    (dolist (class equivalence-classes)
      (let ((prob (apply #'min (mapcar #'trigger-prob class)))
	    (disjunction (mapcar #'trigger-conditions class)))
	(dolist (conjunct (espresso disjunction))
	  (push (make-trigger :prob prob :conditions conjunct)
		new-triggers))))
    new-triggers))
|#

(defun get-good-otpaths (condition otpaths)
  (remove-if-not
   #'(lambda (otpath)
       (memberp condition (otpath-outcomes otpath)))
   otpaths))

(defun get-good/neutral-otpaths (condition otpaths)
  (let ((not-condition (simplify-expression `(not ,condition))))
    (remove-if
     #'(lambda (otpath)
	 (memberp not-condition (otpath-outcomes otpath)))
     otpaths)))
