;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Handling open conditions

(defun HANDLE-OPEN (plan)
  (apply #'nconc
	 (mapcar #'(lambda (open)
		     (nconc (add-step open plan)
			    (new-link open plan)))
		 (get-some-opens plan))))

(defun GET-SOME-OPENS (plan)
  ;; If any conditions are truly OPEN, return just one of them (the 1st);
  ;; if there are no more open conditions, return ALL of them.
  (let ((opens (plan-open plan)))
    (if opens
	(list (first opens))
        (plan-linked plan))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 5.1. Adding new steps

(defun ADD-STEP (open plan)
  (let* ((new-plans nil)
	 (condition (openc-condition open))
	 (consumer (openc-consumer open))
	 (producer (1+ (plan-high-step plan)))
	 (links (plan-links plan))
	 (steps (plan-steps plan)))
    (when (> *trace* 1)
      (format t "~%New step for ~S?" open))
    (dolist (template *templates*) 
      (when (> *trace* 1)
	(format t "~%~3Tusing ~S?" template))
      (let ((otpaths (retrieve-otpaths template condition)))
	(dolist (otpath otpaths)
	  (when (> *trace* 1)
	    (format t "~%~6Tusing otpath ~S?" otpath))
	  (cond ((zerop (trigger-prob (otpath-trigger otpath)))
		 ;; seems like a reasonable thing to do, e.g.
		 ;; if the initial state has lots of p=0
		 ;; outcomes explicitly listed
		 (when (> *trace* 1)
		   (format t "~%~9T* No! zero probability")))
		(t
		 (let* ((new-step
			 (make-plan-step
			  :template template
			  :id producer))
			(new-denouement
			 (make-denouement
			  :step-id producer
			  :otpath otpath))
			(new-link
			 (make-link
			  :denouement new-denouement
			  :condition condition
			  :consumer consumer))
			(new-plan
			 (tweak-plan
			  plan
			  :reason `(:new-step ,open ,producer ,otpath)
			  :steps `(,new-step ,@steps)
			  :links `(,new-link ,@links)
			  :ordering (add-ordering plan producer consumer)
			  :open (new-opens producer otpath open plan)
			  :linked (adjoin open (plan-linked plan))
			  :high-step producer)))
		   ;; need to do this now because we unsafes detection
		   ;; depend on this new step
		   (setf (plan-unsafe new-plan)
		     (nconc ;; hope nconc doesn't destroy last arg!
		      (test-step new-plan new-step) 
		      (test-link new-plan new-link)
		      (plan-unsafe new-plan)))
		   (when (> *trace* 0.5)
		     (format t "~%~9T* Yes! ~S" new-plan))
		   (push new-plan new-plans)))))))
    new-plans))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5.2. Adding links from old steps

(defun NEW-LINK (open plan)
  (let* ((new-plans nil)
	 (condition (openc-condition open))
	 (consumer (openc-consumer open))
	 (priors (possibly-prior consumer plan))
	 (links (plan-links plan))
	 (steps (plan-steps plan)))
    (when (> *trace* 1)
      (format t "~%New link for ~S?" open))
    (dolist (step steps)
      (let ((producer (plan-step-id step)))
	(when (member producer priors)
	  (when (> *trace* 1)
	    (format t "~%~3Tfrom ~S?" step))
	  (let* ((template (plan-step-template step))
		 (otpaths (retrieve-otpaths template condition)))
	    (dolist (otpath otpaths)
	      (when (> *trace* 1)
		(format t "~%~6Tusing otpath ~S?" otpath))
	      (cond ((find-if
		      #'(lambda (link)
			  (and (eq (link-producer link) producer)
			       (eq (link-otpath link) otpath)
			       (eq (link-consumer link) consumer)
			       (equal (link-condition link) condition)))
		      links)
		     (when (> *trace* 2)
		       (format t "~%~9T* No! link already exists")))
		    ((zerop (trigger-prob (otpath-trigger otpath)))
		     ;; see comment in NEW-STEP
		     (when (> *trace* 2)
		       (format t "~%~9T* No! zero probability")))
		    (t
		     (let* ((new-step
			     (make-plan-step
			      :template template
			      :id producer))
			    (new-denouement
			     (make-denouement
			      :step-id producer
			      :otpath otpath))
			    (new-link
			     (make-link
			      :denouement new-denouement
			      :condition condition
			      :consumer consumer))
			    (new-plan
			     (tweak-plan
			      plan
			      :reason `(:new-link ,open ,producer ,otpath)
			      :steps `(,new-step ,@(remove step steps))
			      :links `(,new-link ,@links)
			      :open (new-opens producer otpath open plan)
			      :ordering (add-ordering plan producer consumer)
			      :linked (adjoin open (plan-linked plan))
			      :unsafe (nconc (test-link plan new-link) 
					     (plan-unsafe plan)))))
		       (when (> *trace* 0.5)
			 (format t "~%~9T* Yes! ~S" new-plan))
		       (push new-plan new-plans)))))))))
    new-plans))

;;; a few functions for constructing a new list of open conditions

(defun conditions->opens (conditions step-id)
  (mapcar #'(lambda (condition)
	      (make-openc :condition condition :consumer step-id))
	  conditions))

;; new open conditions for a step just being added.
;; OPEN' = OPEN' U NEW
;; note that this is a special case of new-link-opens
;; since NEW intersect LINKED == {} for a new step, and UNION
;; isn't needed -- APPEND will do -- because OPEN intersect NEW
;; will also necessarily be {}.  so this could be rewritten for efficiency

(defun new-opens (step-id otpath just-closed plan)
  ;; for NEW-STEP and NEW-LINK: new open conditions for a link produced by
  ;; this otpath and step, given that we just closed the open condition
  ;; just-closed
  (remove
   just-closed
   (merge-opens (conditions->opens
		 (trigger-conditions (otpath-trigger otpath))
		 step-id)
		plan)))

(defun new-opens-from-links (links plan)
  ;; for CONFRONT: new open conditions for this set of links.
  (merge-opens (apply #'nconc
		      (mapcar #'(lambda (link)
				  (conditions->opens
				   (trigger-conditions
				    (otpath-trigger
				     (link-otpath link)))
				   (link-producer link)))
			      links))
	       plan))

(defun merge-opens (new-opens plan)
  ;; merge new open conditions with current:
  ;;      OPEN' =  OPEN U (NEW - LINKED)
  (union
   (set-difference new-opens (plan-linked plan) :test #'equalp)
   (plan-open plan)
   :test #'equalp))
