;; (require 'buridan)

;; (setq *trace* -1/2)

;; to make testing a whole bunch of examples easier

(defvar *registered-examples*)

(setf *registered-examples* nil)

(defun register-example (fn)
  (pushnew fn *registered-examples*))

(defun test-examples (&optional (verbose t))
  (time
   (dolist (fn *registered-examples*)
     (when verbose (format t "~%++++Doing example ~S++++:~%" fn))
     (funcall fn))))

;;
;; very simple test: shows that buridan is a superset of snlp
;;

(defun vanilla-blocks-world ()
  (reset-domain)
  (defstep (dry-gripper)
      (outcomes (gripper-dry)))
  (defstep (pickup)
      (if (gripper-dry)
	   (outcomes (holding)))))

(defun test0 ()
  (vanilla-blocks-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(outcomes (not (holding)) (not (gripper-dry)))))
    (plan initial-conditions '((holding)) 1)))

(register-example #'test0)


;; the slippery blocks-world example
;; This is the example used in the intro of the AIJ paper!!

(defun slippery-blocks-world ()
  (reset-domain)
  ;; drying off the gripper works 80% of the time
  (defstep (dry-gripper)
      (with-probability
	(0.8 (outcomes (gripper-dry)))))
  ;; picking up an object has a 50% chance of failure with a wet
  ;; gripper, and a 5% chance of failure with a dry gripper
  (defstep (pickup)
      (if (gripper-dry)
	(with-probability
	    (0.95 (outcomes (holding))))
	(with-probability
	    (0.5 (outcomes (holding)))))))

(defun test1 (&optional (tau 0.9))
  (slippery-blocks-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (0.3 (outcomes (not (holding)) (not (gripper-dry))))
	   (0.7 (outcomes (not (holding)) (gripper-dry))))))
    (plan initial-conditions '((holding)) tau)))

(register-example #'test1)


;; no solution possible to this problem, which is the right thing to happen
;; BUT: I don't know yet how to get the planner to figure this out (planning
;; is undecidable, after all....) , so it just keeps going forever.

(defun schitzophrenia-world ()
  (reset-domain)
  (defstep (step-1)
      (if (B)
	  (outcomes (C)))))

(defun test2 ()
  (schitzophrenia-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (0.5 (outcomes (A) (not (B))))
	   (0.5 (outcomes (not (A)) (B))))))
    (plan initial-conditions '((A)(C)) 0.1)))


;;
;; A test of disjunctive goals: each disjunct can get a fairly high prob,
;; but both are needed to get the desired threshold
;;

(defun waste-time-world ()
  (reset-domain)
  (defstep (grab-a-latte)
      (if (hub-open)
	  (outcomes (latte-consumed))))
  (defstep (read-news)
      (if (network-up)
	  (outcomes (news-read)))))

(defun test3 ()
  (waste-time-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (0.65 (outcomes (hub-open) (network-up)
		  (not (latte-consumed)) (not (news-read))))
	   (0.2  (outcomes (hub-open) (not (network-up))
		  (not (latte-consumed)) (not (news-read))))
	   (0.1  (outcomes (not (hub-open)) (network-up)
		  (not (latte-consumed)) (not (news-read))))
	   (0.05 (outcomes (not (hub-open)) (not (network-up))
		  (not (latte-consumed)) (not (news-read)))))))
    (plan initial-conditions
	  '(:or ((latte-consumed)) ((news-read)))
	  0.90)))

;;;; NOTE THAT THESE TWO GOALS ARE NOT MUTUALLY EXCLUSIVE!!
;;;; recalll that the complicated assessor need to do this as a special
;;;; case: not treat 2 different outcomes of the goal as mutex automatically
;;;; although they might be turn out that way in which case it will eventually
;;;; be caught -- mutex detection at the outcome level is just short-circuiting
;;;; what would happen at the chance proposition level!! 
;;;; the dumb assessor also needs to do a similar trick.
;;;; if we require goal disjuncts to be mutex then this would go away.  
;;;; advantage: then goal really would be a step *just like any other*.

(register-example #'test3)

;;;
;;; a simple case of clobbering fixed by reording: grind, then polish
;;;

(defun simple-lens-world ()
  (reset-domain)
  (defstep (polish)
      (outcomes (smooth)))
  (defstep (grind)
      (outcomes (ground) (not (smooth)))))

(defun test4 ()
  (simple-lens-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(outcomes (not (ground)) (not (smooth)))))
    (plan initial-conditions '((ground)(smooth)) 1)))

(register-example #'test4)

;;
;; a (slightly) more complicated clobbereing example: paint, then
;; wait, then pickup
;;

(defun chocolate-blocks-world ()
  (reset-domain)
  (defstep (paint)
      (if (holding)
	   (outcomes (painted) (wet-paint) (not (clean)))
	   (outcomes (painted) (wet-paint) (clean))))
  (defstep (pickup)
      (if (wet-paint)
	   (outcomes (not (clean)) (holding))
	   (outcomes (clean) (holding))))
  (defstep (wait)
      (outcomes (not (wet-paint)))))

(defun test5 ()
  (chocolate-blocks-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 `(outcomes (not (holding)) (not (painted)) (not (wet-paint))
		    (clean))))
    (plan initial-conditions '((painted) (clean) (holding)) 1)))

(register-example #'test5)

;;;
;;; the lens example: lots of clobbering, relatively big N
;;;

(defun lens-world ()
  (reset-domain)
  (defstep (have-reflective-lens)
      (if (glass)
	   (if (smooth)
	       (if (clean)
		   (if (concave)
		       (if (reflective)
			   (outcomes (reflective-lens))))))))
  (defstep (make-glass)
      (if (sand)
	   (if (hot)
	       (outcomes (glass)))))
  (defstep (heat)
      (if (have-fuel)
	   (outcomes (hot))))
  (defstep (fill-fuel-tank)
      (outcomes (have-fuel)))
  (defstep (wait)
      (outcomes (not (hot))))
  (defstep (polish)
      (if (glass)
	   (if (not (hot))
	       (if (clean)
		   (outcomes (smooth) (not (reflective)))))))
  (defstep (vacuum)
      (if (glass)
	   (if (not (hot))
	       (outcomes (clean)))))
  (defstep (grind)
      (if (glass)
	   (if (not (hot))
	       (outcomes (concave) (not (clean)) (not (smooth))
		         (not (reflective))))))
  (defstep (aluminize)
      (if (glass)
	   (if (not (hot))
	       (outcomes (reflective) (not (clean)))))))

(defun test6 ()
  (lens-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 `(outcomes (sand) (not (glass)) (not (hot)) (not (smooth))
		    (not (clean)) (not (concave)) (not (reflective))
		    (not (reflective-lens)) (not (have-fuel))))
	(*search-limit* 2500))
    (plan initial-conditions '((reflective-lens)) 1)))

(register-example #'test6)

;;
;; an example that requires biting the bullet: A and B must occur
;; in that order, and C must not occur before A nor after B.  so
;; the only possible order is A,C,B.  but that causes a threat
;; on support from A to B
;;

(defun bite-bullet-world ()
  (reset-domain)
  (defstep (step-A)
      (if (not (a-blocked))
	  (outcomes (needed-by-b))))
  (defstep (step-B)
      (if (needed-by-B)
	  (outcomes (goal-1) (c-blocked))
	  (outcomes (c-blocked))))
  (defstep (step-C)
      (if (not (c-blocked))
	  (with-probability
	      (0.9 (outcomes (goal-2) (a-blocked)))
	      (otherwise (outcomes (not (needed-by-b)) (goal-2) (a-blocked))))
	  (outcomes (a-blocked)))))

(defun test7 ()
  (bite-bullet-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
         `(outcomes (not (needed-by-b)) (not (a-blocked)) (not (c-blocked))
		    (not (goal-1)) (not (goal-2)))))
    (plan initial-conditions '((goal-1) (goal-2)) 0.8)))

(register-example #'test7)

;; show that "abstraction" works
;; (really only interesting in the old/broken buridan.  it works now,
;; but the planner must make links for (P) which was the point: it shouldn't
;; need to.  but if the "exhastivity" heuristic is added to the assessor then
;; this should allow the new buridan to get it right too.)

(defun simple-abstraction-world ()
  (reset-domain)
  (defstep (the-step)
      (if (P)
	  (outcomes (desired) (other))
	  (outcomes (desired) (another)))))

(defun test8 ()
  (simple-abstraction-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (1/2
	    (outcomes (P) (not (desired)) (not (other)) (not (another))))
	   (1/2
	    (outcomes (not (P)) (not (desired)) (not (other)) (not (another)))))))
    (plan initial-conditions '((desired)) 1)))

(register-example #'test8)

;;
;; the infamous bombs & toilet example
;;

(defun bombs-toilet-world ()
  (reset-domain)
  (defstep (dunk-suitcase1)
      (if (is-bomb-suitcase1)
	  (outcomes (defused))))
  (defstep (dunk-suitcase2)
      (if (is-bomb-suitcase2)
	  (outcomes (defused)))))

(defun test9 ()
  (bombs-toilet-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 ;; exactly 1 of the two things is a bomb
	 '(with-probability
	   (1/2
	    (outcomes (not (defused)) (is-bomb-suitcase1) (not (is-bomb-suitcase2))))
	   (1/2
	    (outcomes (not (defused)) (not (is-bomb-suitcase1)) (is-bomb-suitcase2))))))
    (plan initial-conditions '((defused)) 1)))

(register-example #'test9)

;;
;; dan's favorites  -- they don't do much that is too exciting

(defun dan-favorite-world-1 ()
  (reset-domain)
  (defstep (the-step)
      (if (A)
	  (outcomes (G)))))

(defun test10 ()
  (dan-favorite-world-1)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (0.5 (outcomes (A) (not (G))))
	   (0.5 (outcomes (not (A)) (G))))))
    (plan initial-conditions '((G)) 1)))

(register-example #'test10)

;; frankly i'm not sure what this is supposed to test
;; it certainly is impossible to find a p=1 plan!

(defun dan-favorite-world-2 ()
  (reset-domain)
  (defstep (the-step)
      (if (B)
	  (with-probability
	      (0.5 (outcomes (C) (not (A))))
	      (0.5 (outcomes (C) (A)))))))

(defun test11 ()
  (dan-favorite-world-2)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 '(with-probability
	   (0.9 (outcomes (A) (B) (not (C))))
	   (0.1 (outcomes (not (A)) (not (B)) (not (C)))))))
    (plan initial-conditions '((A)(C)) 0.4)))

(register-example #'test11)

;;
;; the example from the paper -- show off all the bells and whistles with
;; a small domain.
;;
;; slippery blocks world + cholocate blocks world
;;    except: no dry-off period
;;            p(dry@init)=0.7 instead of 0.3 so that just 1 dry-gripper
;;            is needed, to make this doable.
;;
;; note that the plan step labels are the same as in the paper EXCEPT paint-block:
;;   the first and third labels are reversed: paper: c b a, here: a b c.

(defun mocha-blocks-world ()
  (reset-domain)
  (defstep (paint-block)
      (if (holding-block)
	  (outcomes (block-painted) (not (clean-hands)))
	  (with-probability
	      (0.9 (outcomes (block-painted)))
	      (0.1 (outcomes (block-painted) (not (clean-hands)))))))
  (defstep (dry-gripper)
      (with-probability
	  (0.8 (outcomes (gripper-dry)))
	  (0.2 (outcomes))))
  (defstep (pickup-block)
      (if (gripper-dry)
	  (with-probability
	      (0.95 (outcomes (holding-block)))
	      (0.05 (outcomes)))
	  (with-probability
	      (0.5 (outcomes (holding-block)))
	      (0.5 (outcomes))))))

(defun test12 (&optional (threshold 0.8))
  (mocha-blocks-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 30000)
	(initial-conditions
	 '(with-probability
	   (0.7
	    (outcomes (gripper-dry) (not (holding-block))
	              (clean-hands) (not (block-painted))))
	   (0.3
	    (outcomes (not (gripper-dry)) (not (holding-block))
	              (clean-hands) (not (block-painted)))))))
    (plan initial-conditions
	  '((holding-block) (block-painted) (clean-hands))
	  threshold)))

;; (register-example #'test12)


;;
;; the not so famous bombs & toilet example:
;;    original bomb & toilet, plus clogging
;;

(defun bombs-clogging-toilet-world ()
  (reset-domain)
  (defstep (dunk-p1)
      (if (is-bomb-p1)
	  (with-probability
	      (0.95 (outcomes (defused)))
	      (0.05 (outcomes (defused) (toilet-clogged))))
	  (with-probability
	      (0.95 (outcomes))
	      (0.05 (outcomes (toilet-clogged))))))
  (defstep (dunk-p2)
      (if (is-bomb-p2)
	  (with-probability
	      (0.95 (outcomes (defused)))
	      (0.05 (outcomes (defused) (toilet-clogged))))
	  (with-probability
	      (0.95 (outcomes))
	      (0.05 (outcomes (toilet-clogged)))))))

(defun test13 ()
  (bombs-clogging-toilet-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 200000)
	(initial-conditions
	 '(with-probability
	   (1/2
	    (outcomes
	     (is-bomb-p1) (not (is-bomb-p2))
	     (not (toilet-clogged)) (not (defused))))
	   (1/2
	    (outcomes
	     (not (is-bomb-p1)) (is-bomb-p2)
	     (not (toilet-clogged)) (not (defused)))))))
    (plan initial-conditions '((defused) (not (toilet-clogged))) 0.9)))

;; (register-example #'test13)   -- very long!


;;;;;;;; test14: hard for forward because many different states
;;;;;;;;         but most are irrelevant,  and many useless outcomes
;;;;;;;;         per action, and many partial orders are compatible
;;;;;;;;         with the solution

(defun test14-world ()
  (reset-domain)
  (defstep (step-a)
      (if (a)
	  (outcomes (b))
	  (if (w)
	      (outcomes (not (w)))
	      (outcomes (w)))))
  (defstep (step-b)
      (if (b)
	  (outcomes (c))
	  (if (x)
	      (outcomes (not (x)))
	      (outcomes (x)))))
  (defstep (step-c)
      (if (c)
	  (outcomes (d))
	  (if (y)
	      (outcomes (not (y)))
	      (outcomes (y)))))
  (defstep (step-anywhere-e)
      (outcomes (e)))
  (defstep (step-anywhere-f)
      (outcomes (f))))

(defun test14 ()
  (test14-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 10000)
	(initial-conditions
	 `(with-probability
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (w) (x) (y)))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (w) (x) (not(y))))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (w) (not(x)) (y)))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (w) (not(x)) (not(y))))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (not(w)) (x) (y)))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (not(w)) (x) (not(y))))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (not(w)) (not(x)) (y)))
	    (1/8 (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
			   (not(w)) (not(x)) (not(y)))))))
    (plan initial-conditions '((d)(e)(f)) 1
	  :rank-fun #'rank-links-are-good)))

;;;;;;;; test15: hard for complicated but easy for dumb:
;;;;;;;;         no partial order, no unnecessary states or outcomes,
;;;;;;;;         but a pretty hairy assessment expression

(defun test15-world ()
  (reset-domain)
  (defstep (step-a)
      (if (a)
	  (outcomes (b))))
  (defstep (step-b)
      (if (b)
	  (outcomes (c))
	  (outcomes (xy))))
  (defstep (step-x)
      (if (xy)
	  (if (x)
	      (outcomes (y)))))
  (defstep (step-y)
      (if (y)
	  (outcomes (c)))))

(defun test15 ()
  (test15-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 20000)
	(initial-conditions
	 `(with-probability
	    (1/2 (outcomes (a) (not(b)) (not(c))
			   (not(x)) (not(y)) (not (xy))))
	    (1/2 (outcomes (not(a)) (not(b)) (not(c))
			   (x) (not(y)) (not (xy)))))))
    (plan initial-conditions '((c)) 1
	  :rank-fun #'rank-steps-are-bad)))

;;;;;;;; test16  like 14 (hard for dumb) but without cheat of
;;;;;;;;         lots of total orders. (since that is all that
;;;;;;;;         was makes test14 work.)


(defun build-uniform-distribution (fixed varying)
  (let ((outcomes nil))
    (dotimes (state (expt 2 (length varying)))
      (let ((varying-outcome nil))
	(dolist (prop varying)
	  (if (bit-set? (position prop varying) state)
	      (push prop varying-outcome)
	      (push (negate-expression prop) varying-outcome)))
	(push (cons 'OUTCOMES
		    (append fixed varying-outcome))
	      outcomes)))
    (let ((p (/ 1 (length outcomes))))
      (cons 'WITH-PROBABILITY
	    (mapcar #'(lambda (outcome)
			(list p outcome))
		    outcomes)))))

(defun test16-world ()
  (reset-domain)
  (macrolet
      ((build-hairy-step (name prop-in prop-out)
	 `(defstep (,name)
	      (if ,prop-in
		  (outcomes ,prop-out)
		  (if (x1)
		      (if (x2)
			  (if (x3)
			      (outcomes (not(x1)) (not(x2)) (not(x3)))
			      (outcomes (not(x1)) (not(x2)) (x3)))
			  (if (x3)
			      (outcomes (not(x1)) (x2) (not(x3)))
			      (outcomes (not(x1)) (x2) (x3))))
		      (if (x2)
			  (if (x3)
			      (outcomes (x1) (not(x2)) (not(x3)))
			      (outcomes (x1) (not(x2)) (x3)))
			  (if (x3)
			      (outcomes (x1) (x2) (not(x3)))
			      (outcomes (x1) (x2) (x3)))))))))
    (build-hairy-step step-a (a) (b))
    (build-hairy-step step-b (b) (c))
    (build-hairy-step step-c (c) (d))
    (build-hairy-step step-d (d) (e))
    (build-hairy-step step-e (e) (f))
    (build-hairy-step step-f (f) (g))
    (build-hairy-step step-g (g) (h))
    (build-hairy-step step-h (h) (i))
    (build-hairy-step step-i (i) (j))
    (build-hairy-step step-j (j) (k))
    (build-hairy-step step-k (k) (l))
    (build-hairy-step step-l (l) (m))
    (build-hairy-step step-m (m) (n))
    (build-hairy-step step-n (n) (o))
    (build-hairy-step step-o (o) (p))
    (build-hairy-step step-p (p) (q))
    (build-hairy-step step-q (q) (r))
    (build-hairy-step step-r (r) (s))
    (build-hairy-step step-s (s) (t))
    (build-hairy-step step-t (t) (u))
    (build-hairy-step step-u (u) (v))
    (build-hairy-step step-v (v) (w))
    (build-hairy-step step-w (w) (x))
    (build-hairy-step step-x (x) (y))
    (build-hairy-step step-y (y) (z))))

(defun test16 ()
  (test16-world)
  (setq *plan-id* 0)
  (let ((initial-conditions
	 (build-uniform-distribution
	  '((a)       (not (b)) (not (c)) (not (d)) (not (e)) (not (f))
	    (not (g)) (not (h)) (not (i)) (not (j)) (not (k)) (not (l))
	    (not (m)) (not (n)) (not (o)) (not (p)) (not (q)) (not (r))
	    (not (s)) (not (t)) (not (u)) (not (v)) (not (w)) (not (x))
	    (not (y)) (not (z)))
	  '((x1) (x2) (x3)))))
    (plan initial-conditions '((z)) 1
	  :rank-fun #'rank-links-are-good)))

;;;;;;;;;;;;;;;;;;;;;;;;


(defun test17-world ()
  (reset-domain)
  (macrolet ((build-messy-step (name test-prop main-prop distractor)
	       `(defstep (,name)
		    (if ,test-prop
			(outcomes ,main-prop)
			(with-probability
			  (0.5 (outcomes ,distractor))
			  (0.5 (outcomes (not ,distractor))))))))
    (build-messy-step step-a (a) (b) (x1))
    (build-messy-step step-b (b) (c) (x2))
    (build-messy-step step-c (c) (d) (x3))
    (build-messy-step step-d (d) (e) (x4))
    (build-messy-step step-e (e) (f) (x5))
    (build-messy-step step-f (f) (g) (x6))
    (build-messy-step step-g (g) (h) (x7))
    (build-messy-step step-h (h) (i) (x8))))
;; extras:
;;  (build-messy-step step-i (i) (j) (x9))
;;  (build-messy-step step-j (j) (k) (x10))
;; add (x9) and (x10) to initial-conditions below!

(defun test17 (goal)
  (test17-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 10000)
	(initial-conditions
	 `(with-probability
	    (0.8       
	     (outcomes (a) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))
		       (not(g)) (not(h)) (not(i)) (not(j)) (not (k))
		       (x1) (x2) (x3) (x4) (x5) (x6) (x7) (x8)))
	    (0.2
	     (outcomes (not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not (f))
		       (not(g)) (not(h)) (not(i)) (not(j)) (not (k))
		       (x1) (x2) (x3) (x4) (x5) (x6) (x7) (x8))))))
    (plan initial-conditions (list goal) 0.7 :rank-fun #'rank-steps-are-good)))

;; use goal = (a), goal=(b), ..., goal=(i) to make a plan of length
;; 0,1,2,...,8

;;;;;;;;;;;;;;;;;;;;;;;;


(defun test18-world ()
  (reset-domain)
  (macrolet ((build-messy-step (name main-prop kill-props)
	       `(defstep (,name)
		    (if ,main-prop
			(with-probability
			    (0.5 (outcomes (Q) ,@kill-props))
			    (0.5 (outcomes ,@kill-props)))
			(outcomes ,@kill-props)))))
    (build-messy-step step-a (a) ((not(a))))
    (build-messy-step step-b (b) ((not(a)) (not(b))))
    (build-messy-step step-c (c) ((not(a)) (not(b)) (not(c))))
    (build-messy-step step-d (d) ((not(a)) (not(b)) (not(c)) (not(d))))
    (build-messy-step step-e (e) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e))))
    (build-messy-step step-f (f) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not(f))))
    (build-messy-step step-g (g) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not(f)) (not(g))))
    (build-messy-step step-h (h) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not(f)) (not(g)) (not(h))))))

;; extras:
;; (build-messy-step step-i (i) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not(f)) (not(g)) (not(h)) (not(i))))
;; (build-messy-step step-j (j) ((not(a)) (not(b)) (not(c)) (not(d)) (not(e)) (not(f)) (not(g)) (not(h)) (not(i)) (not(j))))))
;; add (i) and (j) to initial-conditions below!!

(defun test18 (tau)
  (test18-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 1000000)
	(initial-conditions
	 '(outcomes (a) (b) (c) (d) (e) (f) (g) (h) (not (Q)))))
    (plan initial-conditions '((Q)) tau :rank-fun #'RANK-orderings-are-good-duplicate-steps-are-evil-links-are-neutral)))
    
;; use tau=0.0, 0.5, 0.75, 0.875, ..., 0.99609375  [see 'exp/exp.lisp'
;; for the equation] to make a plan of length 0,1,2,3,...,8

;;;;;;;;;;;;;;;;;;;;;;;;


(defun test19-world ()
  (reset-domain)
  (macrolet ((build-messy-step (name goal distractor)
	       `(defstep (,name)
		    (with-probability
			(0.5 (outcomes ,goal ,distractor))
		        (0.5 (outcomes ,goal (not (,distractor))))))))
    (build-messy-step step-a (b) (x1))
    (build-messy-step step-b (c) (x2))
    (build-messy-step step-c (d) (x3))
    (build-messy-step step-d (e) (x4))
    (build-messy-step step-e (f) (x5))
    (build-messy-step step-f (g) (x6))
    (build-messy-step step-g (h) (x7))
    (build-messy-step step-h (i) (x8))))

(defun test19 (goal)
  (test19-world)
  (setq *plan-id* 0)
  (let ((*search-limit* 1000000)
	(initial-conditions
	 '(outcomes (a) (not (b)) (not (c)) (not (d))
	            (not (e)) (not (f)) (not (g)) (not (h)) (not (i))
	            (x1) (x2) (x3) (x4) (x5) (x6) (x7) (x8))))
    (plan initial-conditions
	  goal
	  1.0
	  :rank-fun #'RANK-links-are-great-duplicates-and-orderings-are-evil)))

;; use goal = ((a)), ((a)(b)), ((a)(b)(c)), ..., ((a)(b)...(i)) for n=0,1,...,8