;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; the dumb plan assessment algorithm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun dumb-assessor (plan tau)
  (let ((state-distrib
	 (get-state-distrib plan))
	(goal-conjunctions
	 (mapcar
	  #'trigger-conditions
	  (mapcar
	   #'otpath-trigger
	   (step-template-otpaths
	    (plan-step-template
	     (find :goal (plan-steps plan) :key #'plan-step-id))))))
	(min-prob 1))
    ;; this loop is basically an (apply #'min ...) but we want to
    ;; be able to short-circuit the loop if the min is definitely
    ;; less than tau.  note that this means this function does NOT
    ;; return a true lower bound.  rather, if the true lower bound is
    ;; less than tau, then this funciton returns something that is
    ;; less than tau, but if the true lower bound is greater than tau
    ;; it returns the true lower bound.
    (dolist (order (compute-total-orders plan))
      (let ((prob (dumb-assess-ordered
		   plan state-distrib goal-conjunctions order)))
	(cond ((< prob tau)
	       (return-from dumb-assessor prob))
	      ((< prob min-prob)
	       (setf min-prob prob)))))
    min-prob))

(defun dumb-assessor-max (plan tau)
  ;; an alternative assessment algorithm, for free;
  ;; takes the MAX over all total orders instead of the MIN.
  ;; if this were used "for real" then the generator should
  ;; use the total order and constrain the plan
  ;; to respect it.  BURIDAN does not currently do this but
  ;; it easily could.
  (let ((state-distrib
	 (get-state-distrib plan))
	(goal-conjunctions
	 (mapcar
	  #'trigger-conditions
	  (mapcar
	   #'otpath-trigger
	   (step-template-otpaths
	    (plan-step-template
	     (find :goal (plan-steps plan) :key #'plan-step-id))))))
	(max-prob 0))
    ;; this loop is basically an (apply #'min ...) but we want to
    ;; be able to short-circuit the loop if the min is definitely
    ;; less than tau.  note that this means this function does NOT
    ;; return a true lower bound.  rather, if the true lower bound is
    ;; less than tau, then this funciton returns something that is
    ;; less than tau, but if the true lower bound is greater than tau
    ;; it returns the true lower bound.
    (dolist (order (compute-total-orders plan))
      (let ((prob (dumb-assess-ordered
		   plan state-distrib goal-conjunctions order)))
	(cond ((>= prob tau)
	       (return-from dumb-assessor-max prob))
	      ((> prob max-prob)
	       (setf max-prob prob)))))
    max-prob))

(defun dumb-assessor2 (plan tau)
  ;; an alternative plan assessment algorithm, for free:
  ;; this is just like dumb-assessor except that we only check one
  ;; consistent total order.  THIS ASSESSMENT ALGORITHM IS NOT SOUND!!!
  (declare (ignore tau))
  (let ((state-distrib
	 (get-state-distrib plan))
	(goal-conjunctions
	 (mapcar
	  #'trigger-conditions
	  (mapcar
	   #'otpath-trigger
	   (step-template-otpaths
	    (plan-step-template
	     (find :goal (plan-steps plan) :key #'plan-step-id)))))))
    (dumb-assess-ordered
     plan
     state-distrib
     goal-conjunctions
     (top-sort (plan-ordering plan) (plan-high-step plan)))))

;; ;; ;; outcomes, triggers, states 

;; outcomes and states are implemented just as lists of conditions

(defun result (outcome state)
  ;; the new state that results from realizing the given outcome
  ;; in the given state
  ;; NOTE that if the initial states are not complete then this
  ;; straightforward implementation won't work: domains must not
  ;; sprout new propositions!!  note that this does the right thing
  ;; for safety conditions: these get sprouted by steps, so they get
  ;; ignored, but that's the right thing to do.
  (let ((new-state nil))
    (dolist (proposition state)
      (if (member proposition outcome :test #'equal)
	  ;; case 1: outcome makes the proposition true
	  (push proposition new-state)
	  (let ((not-proposition (negatify-expression proposition)))
	    (if (member not-proposition outcome :test #'equal)
		;; case 2: outcome makes the proposition false
		(push not-proposition new-state)
		;; case 3: outcome leaves the proposition along
		(push proposition new-state)))))
    new-state))

(defun outcome-prob-state (trigger state)
  ;; the probability that the trigger is realized in the given state
  (* (expression-prob-state (trigger-conditions trigger) state)
     (trigger-prob trigger)))

(defun expression-prob-state (expression state)
  (if (subsetp expression state :test #'equal)
      1
      0))

;; ;; ;; state distributions

(defstruct (stateprob
	    (:print-function stateprob-printer))
  state
  rho)

(defun stateprob-printer (stateprob stream depth)
  (declare (ignore depth))
  (format stream "{~S,~S}"
	  (stateprob-state stateprob) (stateprob-rho stateprob)))

;; state distributions are lists of stateprob's

(defun get-state-distrib (plan)
  ;; get the initial state distribution from step 0
  (mapcar #'(lambda (otpath)
	      (make-stateprob :state (otpath-outcomes otpath)
			      :rho (trigger-prob (otpath-trigger otpath))))
	  (step-template-otpaths
	   (plan-step-template
	    (find 0 (plan-steps plan) :key #'plan-step-id)))))

(defun expression-prob-state-distrib (expression state-distrib)
  (apply #'+
	 (mapcar #'(lambda (stateprob)
		     (* (stateprob-rho stateprob)
			(expression-prob-state expression
					       (stateprob-state stateprob))))
		 state-distrib)))

;; ;; ;; execution of an action in a state distribution

(defun exec (step state-distrib)
  ;; the state distribution that results from executing the given action
  ;; in the given state distribution
  (let ((new-state-distrib nil)
	(otpaths (step-template-otpaths (plan-step-template step))))
    (dolist (stateprob state-distrib)
      (let ((state (stateprob-state stateprob))
	    (rho (stateprob-rho stateprob)))
	(dolist (otpath otpaths)
	  (let ((new-state
		 (result (otpath-outcomes otpath) state))
		(new-rho
		 (* rho (outcome-prob-state (otpath-trigger otpath) state))))
	    (unless (zerop new-rho)
	      (push (make-stateprob :state new-state :rho new-rho)
		    new-state-distrib))))))
    ;; now merge duplicate states by summing probabilities
    ;; (yes we could do it incrementally as the new state distrib is built
    ;; but that would require many linear scans to check for duplication
    ;; instead of just this one pass.)
    (mapcar
     #'(lambda (state)
	 (make-stateprob
	  :state state
	  :rho (apply #'+
		      (mapcar #'stateprob-rho
			      (remove state new-state-distrib
				      :test-not #'equal
				      :key #'stateprob-state)))))
     (remove-duplicates
      (mapcar #'stateprob-state new-state-distrib)
      :test #'equal))))

;; ;; ;; the main function: compute the prob of a totally ordered plan

(defun dumb-assess-ordered (plan state-distrib goal-expressions order)
  ;; returns the prob of plan assuming its steps are ordered as given
  ;; (this assumes 0 and :GOAL are NOT included in the order!!)
  (let ((steps (plan-steps plan)))
    ;; first, get the final state distribution by executing every step
    (dolist (step-id order)
      (when (> *trace* 9)
	(format t "      SD(~S-) = ~S~%" step-id state-distrib))
      (setf state-distrib
	(exec (find step-id steps :key #'plan-step-id)
	      state-distrib)))
    (when (> *trace* 9)
      (format t "      SD(G-) = ~S~%" state-distrib))
    ;; second, compute probability of the goal outcomes using
    ;; P(AvB) = P(A)+P(B)-P(A&B).  this is needed because P(A&B)
    ;; might not be 0 because different goal conjuncts don't need
    ;; to be mutually exclsuive although arguably they ought to be
    ;; for purity.
    (flet ((multiple-expressions-prob (expression-sets)
	     (mapcar #'(lambda (expressions)
			 (expression-prob-state-distrib
			  (remove-duplicates (apply #'append expressions)
					     :test #'equal)
			  state-distrib))
		     expression-sets)))
      (multiple-value-bind (adds subtracts)
	  (disjunction->conjunction goal-expressions)
	(- (apply #'+ (multiple-expressions-prob adds))
	   (apply #'+ (multiple-expressions-prob subtracts)))))))
