;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; the complicated plan assessment algorithm
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *eager-complicated-assessor?* t)

(defun complicated-assessor (plan tau)
  (declare (ignore tau))
  (let ((a (get-assessment-structure plan)))
    (cond ((not (or *eager-complicated-assessor?*
		    (plan-complete plan)))
	   (when (> *trace* 8)
	     (format t "  + incomplete plan! p=0~%"))
	   0)
	  (t
	   (when (> *trace* 8)
	     (let ((*verbose-otpaths?* nil))
	       (format t "  + Got assessment structure:~%      ~S~%" a)))
	   (let ((p (assess-assessment-structure a)))
	     (when (> *trace* 8)
	       (format t "  + Done assessing ~S: p=~S~%" plan p))
	     p)))))

#|

special version to monitor the number of times "eager" assessment (ie,
asessing incomplete plans by giving unsupported propositions and
unresolved threats to links probbaility zero instead of zero for the
whole plan) actually helped, ie the number of times that such plans
had > 0 probability.

(defvar *n-calls*)
(defvar *n-incompletes*)
(defvar *n-incomplete-nonzeroes*)

(defun reset-compl ()
  (setf *n-calls* 0
	*n-incompletes* 0
	*n-incomplete-nonzeroes* 0))
    
(defun complicated-assessor (plan)
  (incf *n-calls*)
  (let ((complete? (plan-complete plan)))
    (unless complete?
      (incf *n-incompletes*))
    (let ((a (get-assessment-structure plan)))
      (when (> *trace* 8)
	(let ((*verbose-otpaths?* nil))
	  (format t "  + Got assessment structure:~%      ~S~%" a)))
      (let ((p (assess-assessment-structure a)))
	(when (> *trace* 8)
	  (format t "  + Done assessing ~S: p=~S~%" plan p))
	(unless (or complete? (zerop p))
	  (incf *n-incomplete-nonzeroes*))
	p))))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; data structures

(defstruct (pleaf-set
	    (:print-function pleaf-set-printer))
  frontier ;; denouements yet to be expanded
  visited) ;; denouements that have been expanded (alway includes frontier)

;; an 'assessment structure' is a list of pleaf-sets

(defun pleaf-set-printer (pleaf-set stream depth)
  (declare (ignore depth))
  (format stream "{f:~S v:~S}"
	  (pleaf-set-frontier pleaf-set)
	  (pleaf-set-visited pleaf-set)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun make-init-assessment-structure (plan)
  (mapcar #'(lambda (link)
	      (let ((denouement-list (list (link-denouement link))))
		(make-pleaf-set
		 :frontier denouement-list
		 :visited denouement-list)))
	  (get-links '(nirvana) ':bhudda plan)))

;;;

(defun assess-assessment-structure (assessment-structure)
  ;; this fn assumes we've traversed the links all the way back to "initial-
  ;; state-like" steps -- ie, it ignores anything in the pleaf-sets' frontiers,
  ;; and it assumes that anything in visited has all it preconditions checked
  ;; out.  in short, don't call this on an arbitrary assessment structure;
  ;; only trust what get-assessment-structure returns.
  (multiple-value-bind (adds subtracts)
      (disjunction->conjunction assessment-structure)
    (- (apply #'+ (assess-subsets adds))
       (apply #'+ (assess-subsets subtracts)))))

;; do simplification BEFORE d->c not after!!!  (but after as well!)

(defun assess-subsets (list-of-pleaf-sets)
  ;; note that we this does a final consistency check and convergence
  ;; adjustment, so the transformations that do these things don't need
  ;; to be done-- they are just for efficiency
  ;; note that while within a pleaf-set the transformations will simplify stuff
  ;; as much as possible we still need to do this simplification because
  ;; here pleaf sets are being ANDed together, so additional simplifications
  ;; may be neccesary.
  (mapcar #'(lambda (pleaf-sets)
	      (let ((all-visited
		     (apply #'append (mapcar #'pleaf-set-visited pleaf-sets))))
		(if (consistent-denouements? all-visited)
		    (apply
		     #'*
		     (mapcar #'(lambda (denouement)
				 (trigger-prob
				  (otpath-trigger
				   (denouement-otpath
				    denouement))))
			     (remove-redundants all-visited)))
		    0)))
	  list-of-pleaf-sets))
;;;


(defun get-assessment-structure (plan)
  ;; refines the initial assessment structure (based on 'plan') as far as
  ;; possible, returning eirther the maximally refined assessment structure, or
  ;; 'nil', meaning that no consistent assessment structure exists.
  (let ((*verbose-otpaths?* nil) ;; verbosity is especially obnoxious here
	(assessment-structure (make-init-assessment-structure plan)))
    (when (> *trace* 7)
      (format t "~2TInitial assessment structure:~%~6T~S~%"
	      assessment-structure))
    ;; keep looping until ... (**)
    (loop
      ;; first, apply all the transformations that are applicable
      (loop
	(when (null assessment-structure)
	  (when (> *trace* 7)
	    (format t "~2TDone -- nothing left!~%"))
	  ;; empty assessment expression (**)
	  (return-from get-assessment-structure nil))
	(multiple-value-bind (xformed xform-desc xformable?)
	    (select-transformation assessment-structure)
	  (cond (xformable?
		 (when (> *trace* 7)
		   (format t "~4TApplying xform ~S:~%~6T~S~%"
			   xform-desc xformed))
		 (setf assessment-structure xformed))
		(t
		 ;; no more xformations applicable; exit this inner loop (**)
		 (return)))))
      ;; second, expand one pleaf (exit if nothing can be further expanded)
      (multiple-value-bind (expanded exp-desc expandable?)
	  (do-expansion assessment-structure plan)
	(cond (expandable?
	       (when (> *trace* 7)
		 (format t "~4TExpanding ~S:~%~6T~S~%"
			 exp-desc expanded))
	       (setf assessment-structure expanded))
	      (t
	       ;; nothing left to expand, return from main loop (**)
	       (when (> *trace* 7)
		 (format t "~2TDone~%"))
	       (return-from get-assessment-structure
		 assessment-structure)))))))

;;;; transformations

(defvar *transformations* nil)

(defun select-transformation (assessment-structure)
  (dolist (xform *transformations*)
    (let ((xformed-children (funcall xform assessment-structure)))
      (when xformed-children
	(return-from select-transformation
	  (values (car xformed-children) xform t)))))
  (values nil nil nil))

;; a transformation function must take a single arg, the input assessment
;; structure, and return a list of transformed children.

;;;; specific transformations

;; 1. inconsistency

(defun remove-inconsistencies (assessment-structure)
  ;; remove pleaf sets having at least one pair of "duplicate" pleaves that
  ;; were visited
  (let* ((changes? nil)
	 (new-assessment-structure
	  (remove-if-not
	   #'(lambda (pleaf-set)
	       (cond ((consistent-denouements? (pleaf-set-visited pleaf-set))
		      t)
		     (t
		      (setf changes? t)
		      nil)))
	   assessment-structure)))
    (if changes?
	(list new-assessment-structure)
        nil)))

(defun consistent-denouements? (denouements)
  ;; is this set of denouements consistent?
  ;; yes, if there is no step associated with two different otpaths
  (every #'(lambda (same-step-denouements)
	     (let ((first (first same-step-denouements)))
	       (if (eq (denouement-step-id first) :goal)
		   ;; special case: the goal "otpaths" are not
		   ;; mutually exclusive! this is needed for ':or' goals
		   t
		   (every #'(lambda (denouement)
			      (denouement-equalp denouement first))
			  (rest same-step-denouements)))))
	 (partition denouements :key #'denouement-step-id)))

;;; 2: and-redundancy:
;;     {... or (... & P & P & ...) or ...} --> {... or (... & P & ...) or ...}

(defun and-redundancy (assessment-structure)
  (let* ((changes? nil)
	 (new-assessment-structure
	  (mapcar #'(lambda (pleaf-set)
		      (let ((visited (pleaf-set-visited pleaf-set))
			    (frontier (pleaf-set-frontier pleaf-set)))
			;; try to bit a little wise about consing needlessly
			(cond ((has-redundants? visited)
			       ;; since frontier is always a subset of visited,
			       ;; we know frontier may have redundants, too
			       (setf changes? t)
			       (make-pleaf-set
				:frontier (remove-redundants frontier)
				:visited (remove-redundants visited)))
			      (t
			       pleaf-set))))
		  assessment-structure)))
    (if changes?
	(list new-assessment-structure)
        nil)))

(defun has-redundants? (denouements)
  (do ((walk-denouements (cdr denouements) (cdr walk-denouements))
       (denouement (first denouements) (first walk-denouements)))
      ((null walk-denouements)
       nil)
    (if (member denouement walk-denouements :test #'denouement-equalp)
	(return t))))

(defun remove-redundants (denouements)
  (remove-duplicates denouements :test #'denouement-equalp))

;; or-redundancy: remove two identical pleaf sets from the assessment
;;     {... or P ... or ..  P or ... } --> {... or P or ...}

(defun or-redundancy (assessment-structure)
  (let* ((changes? nil)
	 (new-assessment-structure
	  (remove-duplicates
	   assessment-structure
	   :test #'(lambda (pleaf-set-1 pleaf-set-2)
		     (cond ((set-equal (pleaf-set-visited pleaf-set-1)
				       (pleaf-set-visited pleaf-set-2)
				       :test #'denouement-equalp)
			    (setf changes? t)
			    t)
			   (t
			    nil))))))
    (if changes?
	(list new-assessment-structure)
        nil)))

;;;;

(setf *transformations*
  (list #'remove-inconsistencies #'or-redundancy #'and-redundancy))

;;;; expansion

(defun do-expansion (assessment-structure plan)
  ;; find the first expandable pleaf and expand it.  no search control hooks
  ;; here! returns three values: the new assessment structure, the thing that
  ;; was expanded, and a flag: was anything actually expanded or not?
  (walk-list ((before-sets pleaf-set after-sets) assessment-structure)
    (let ((frontier (pleaf-set-frontier pleaf-set))
	  (visited (pleaf-set-visited pleaf-set)))
      (walk-list ((before-frontier denouement after-frontier) frontier)
	(let ((new-sets
	       (generate-new-sets
		denouement
		(nconc before-frontier after-frontier)
					; previous line is slightly dangerous!!
					; it works because if we get this far
					; then we'll won't need these variables
					; anymore.  if that changes, switch it
					; to 'append'.
		visited
		plan)))
	  (return-from do-expansion
	    (values (nconc before-sets (nconc new-sets after-sets))
					; same thing!!!
		    denouement
		    t))))))
  ;; if we get this far then nothing could be expanded: there was nothing in
  ;; any of the pleaf-sets' frontiers
  (values nil nil nil))

(defun generate-new-sets (denouement rest-frontier visited plan)
  (let* ((otpath
	  (denouement-otpath denouement))
	 (conditions
	  (trigger-conditions (otpath-trigger otpath)))
	 (step-id
	  (denouement-step-id denouement))
	 (sop-denouements
	  (get-sop-denouements conditions step-id plan)))
    (cond (sop-denouements
	   ;; we now have, in sop form, the denouements that this denouement
	   ;; expands into (including underminers).
	   ;; note that if one of the conditions has no incoming links, then
	   ;; that sum term in pos-links will be empty, so sop-links will be
	   ;; (), which is the right thing to have happen: if no support
	   ;; exists for some link to a denouement, then the denouement is
	   ;; assigned a probability zero.  this happens because this
	   ;; entire pleaf-set vanishes: because we return nil for new-sets
	   ;; also, we treat all "unincorporated threats" as if they had
	   ;; been confronted and added to the list of underminers
	   (mapcar #'(lambda (anded-denouements)
		       (make-pleaf-set
			:frontier (append anded-denouements rest-frontier)
			:visited (append anded-denouements visited)))
		   sop-denouements))
	  (conditions
	   ;; there were some trigger conditions for this denouements,
	   ;; but they must be unsupported, so get rid of this pleaf-set
	   nil)
	  (t
	   ;; no conditions (ie, a pleaf) and no underminer denouements
	   ;; either, so we just move this denouement to visited.
	   (list (make-pleaf-set :frontier rest-frontier :visited visited))))))
	  
(defun get-sop-denouements (conditions step-id plan)
  (let ((condition-denouements
	 (mapcar #'(lambda (condition)
		     (get-pos-condition-denouements condition step-id plan))
		 conditions)))
    (cond ((member '() condition-denouements)
	   ;; a condition was unsupported (we need to do this because pos->sop
	   ;; treats () as "true")
	   nil)
	  (t
	   (pos->sop (apply #'nconc condition-denouements))))))

(defun get-pos-condition-denouements (condition step-id plan)
  ;; the denouments into which condition expands, in POS form
  (sop->pos (apply #'nconc
		   (mapcar #'(lambda (link)
			       (get-sop-link-denouements link plan))
			   (get-links condition step-id plan)))))

(defun get-sop-link-denouements (link plan)
  ;; the denouements into which the link expands, in SOP form
  ;; if the link has unresolved threats or some confronted threat's safety
  ;; condition isn't supported, then we return (), ie "false", which the
  ;; 'nconc' in the previous function removes, as it should since
  ;; we want to just throw away this link under those conditions.
  ;; `yikes!!' indeed.
  (if (member link (plan-unsafe plan) :key #'unsafe-link :test #'same-link?)
      ;; this link has unresolved threats.  so it doesn't expand into
      ;; anything.  ie it exands into logical FALSE and since the links are
      ;; being ORed together it will just drop out.
      nil
      ;; normal case: all threats resolved
      (let ((link-denouement (link-denouement link))
	    (underminers (link-underminers link)))
	(if underminers
	    ;; underminers, so create the SOP expression
	    (simplify-sop
	     (mapcar #'(lambda (underminer-denouements)
			 (cons link-denouement
			       underminer-denouements))
		     (pos->sop
		      ;; the following is the POS form of the outcomes
		      ;; supporting all the safety conditions.  Note that
		      ;; if one of the get-links below returns NIL
		      ;; then the above pos->sop nukes the whole expression
		      ;; (ie, it treats () as "false") which is the right
		      ;; thing to do.
		      (mapcar #'(lambda (underminer)
				  (mapcar #'link-denouement
					  (get-links
					   (openc-condition underminer)
					   (openc-consumer underminer)
					   plan)))
			      underminers))))
	    ;; no underminers, so this is an easy case
	    `((,link-denouement))))))

(defun get-links (condition step-id plan)
  (remove-if-not #'(lambda (link)
		     (and (equal condition (link-condition link))
			  (eq step-id (link-consumer link))))
		 (plan-links plan)))

;; the following functions know nothing about probabilities, pleafs,
;; plans, denouements or any of that.  they just operate on lists.

(defun disjunction->conjunction (things)
  ;; (a b c) --> {((a)(b)(c)(a b c)); ((a b)(b c)(a c))}
  (case (length things)
    ((0 1)
     (values (mapcar #'list things) nil))
    (2
     (values (mapcar #'list things) (list things)))
    (otherwise
     (let* ((thing (first things))
	    (other-things (rest things))
	    (distributed (mapcar #'(lambda (other) (list thing other))
				 other-things)))
       (multiple-value-bind (distrib-adds distrib-subtracts)
	   (disjunction->conjunction distributed)
	 (multiple-value-bind (other-adds other-subtracts)

	     (disjunction->conjunction other-things)
	   (values (mapcar #'canonicalize
			   `((,thing) ,@other-adds ,@distrib-subtracts))
		   (mapcar #'canonicalize
			   `(,@other-subtracts ,@distrib-adds)))))))))

(defun canonicalize (list)
  (remove-duplicates (flatten list)))

(defun flatten (list)
  (cond ((null list)
	 nil)
	((atom (car list))
	 (cons (car list) (flatten (cdr list))))
	(t
	 (nconc (flatten (car list)) (flatten (cdr list))))))

(defun pos->sop (expression)
  ;; treat pos as a boolean expression and gets the corresponding
  ;; sop form
  ;; (a+b)&(c+d+e):    ((a b)(c d e)) --> ((a c)(a d)(a e)(b c)(b d)(b e))
  ;; (a+b)&()&(c+d+e): ((a b)()(c d e)) --> nil
  ;; (ie, () treated as "false")
  (labels
      ((internal-pos->sop (exp)
	 (cond ((= (length exp) 1)
		;; normal terminating condition
		(mapcar #'list (first exp)))
	       (t
		(let ((disjunction (first exp)))
		  (if disjunction
		      (let ((sop-rest
			     (simplify-sop (internal-pos->sop (rest exp)))))
			(apply #'nconc
			       (mapcar #'(lambda (disjunct)
					   (mapcar #'(lambda (product)
						       (cons disjunct product))
						   sop-rest))
				       disjunction)))))))))
    (simplify-sop (internal-pos->sop expression))))

(defun sop->pos (expression)
  ;; treat pos as a boolean expression and gets the corresponding
  ;; sop form
  ;; (a&b)+(c&d&e):    ((a b)(c d e)) --> ((a c)(a d)(a e)(b c)(b d)(b e))
  ;; (a&b)+()+(c&d&e): ((a b)()(c d e)) --> nil
  ;; (ie, () is treated as "true")
  (labels
      ((internal-sop->pos (exp)
	 (cond ((null exp)
		;; unusual special case
		nil)
	       ((= (length exp) 1)
		;; normal terminating condition
		(mapcar #'list (first exp)))
	       (t
		(let ((conjunction
		       (first exp))
		      (pos-rest
		       (simplify-pos (internal-sop->pos (rest exp)))))
		  (if conjunction
		      (apply #'nconc
			     (mapcar #'(lambda (conjunct)
					 (mapcar #'(lambda (product)
						     (cons conjunct product))
						 pos-rest))
				     conjunction))
		    pos-rest))))))
    (simplify-pos (internal-sop->pos expression))))

;; these functions do simplification.  they aren't just "generic" SOP/POS
;; tools because they examine the denouments to detect inconsistency.
;; this sort of thing properly belongs with the xform stuff since there's
;; so much overlap but that would be far too slow.  this stuff could really
;; use a rewrite.

(defun simplify-sop (sop)
  ;; first: remove inconsistencies
  (setf sop
    (remove-if-not #'consistent-denouements? sop))
  ;; second: remove AND redundancy
  (setf sop
    (mapcar
     #'(lambda (product)
	 (if (has-redundants? product)
	     (remove-redundants product)
	     product))
     sop))
  ;; third: remove OR redundancy
  (setf sop
    (remove-duplicates
     sop
     :test #'(lambda (product1 product2)
	       (set-equal product1 product2 :test #'denouement-equalp))))
  sop)

(defun simplify-pos (pos)
  ;; first: remove inconsistencies
  ;;   (too hard -- skip it!)
  ;; second: remove OR redundancy
  (setf pos
    (mapcar
     #'(lambda (sum)
	 (if (has-redundants? sum)
	     (remove-redundants sum)
	   sum))
     pos))
  ;; third: remove AND redundancy
  (setf pos
    (remove-duplicates
     pos
     :test #'(lambda (sum1 sum2)
	       (set-equal sum1 sum2 :test #'denouement-equalp))))
  pos)
