(defvar *display-plans-when-done?* t) ;; just for experimentation

(defun plan (initial
	     goals
	     threshold
	     &key
	     (rank-fun #'rank-default)
	     (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (plan-internal initial (->disjunction goals) threshold
		     rank-fun search-fun)
    (declare (ignore done? time q-len av-branch))
    (if (plan-p plan)
	(if *display-plans-when-done?*
	    (display-plan plan)
	    plan))))

(defmacro defstep (action outcome-tree)
  `(push (build-step-template ',action ',outcome-tree)
	 *templates*))

(defun reset-domain ()
  (setf *templates* nil))

(defvar *vcr-loaded* nil) ;; stupid hack

(defun plan-internal (i-outcome-tree goal threshold rank-fun search-fun)
  (reset-stat-vars)  ;; clear globals for metering
  (let* ((init-time
	  (get-internal-run-time))
	 (i-step
	  (make-plan-step
	   :id 0
	   :template (build-step-template '(init) i-outcome-tree)))
	 (g-otpaths
	  (progn
	    (otpath-reset-label) ;; for the goal's otpaths' labels
	    (mapcar
	     #'(lambda (conj)
		 (make-otpath :trigger (make-trigger :conditions conj :prob 1)
			      :outcomes '((nirvana))))
	     goal)))
	 (g-step
	  (make-plan-step
	   :id :goal
	   :template (make-step-template :action '(goal) :otpaths g-otpaths)))
	 (n-open
	  (make-openc :condition '(nirvana) :consumer :bhudda))
	 (initial-plan
	  (make-plan
	   :steps (list i-step g-step)
	   :open (list n-open)
	   :high-step 0)))
    (when *vcr-loaded*
      (vcr-frame nil `(:init ,threshold) initial-plan))
    (multiple-value-bind (plan bfactor qlength)
	(funcall search-fun
		 initial-plan
		 #'plan-refinements
		 #'(lambda (plan) (plan-test plan threshold t))
		 rank-fun
		 *search-limit*)
      (values plan			; the plan itself
	      (and plan (plan-test plan threshold nil))
	      (- (get-internal-run-time) init-time) ; time
	      qlength			; final length of the queue
	      bfactor))))		; average branching factor

;;;;;;;;;;;;;;;;;;;;;;;;

(defun plan-complete (plan)
  (not (or (plan-open plan)
	   (plan-unsafe plan))))

(defvar *plan-assessor-fn*)

;; *plan-assessor-fn* must be bound to a funciton taking two arguments.
;; the first is a plan, the second is the probability threshold.  it
;; must return the assessed probability. (tau is passed in so that the
;; assessor can be clever if it knows how to assess a plan faster 
;; given tau (ie, steve's algorithm))

;; by default, use the recursive impementation of the reverse (complicated)
;; algorithm

(setf *plan-assessor-fn* #'recur-assessor)

(defun plan-test (plan threshold verbose?)
  ;; (and (plan-complete plan) ...) not in here because
  ;; if the assessor doesn't know how to deal with an incomplete plan
  ;; then it should return 0 or whatver it can.  but an incomplete plan
  ;; CAN be OK, as the dumb assessment algorithm demonstrates.
  (when (> *trace* 7)
    (format t "~%Assessing ~S:~%" plan))
  (let* ((p (funcall *plan-assessor-fn* plan threshold))
	 (good-enough? (>= p threshold)))
    (when (and verbose? (> *trace* 0.5))
      (format t "~&PROBABILITY(~S) >= ~,3F ~:[<~;>=~] ~S"
	      plan p good-enough? threshold))
    good-enough?))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a list of one step refinements to the current plan.

(defun plan-refinements (plan)
  (if (plan-unsafe plan)
      (handle-unsafe plan)
      (handle-open plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. ranking partial plans

;; lower rank ==> better plan

(defun RANK-default (plan)
  ;; the standard rankiong function, carried over directly from SNLP
  (let ((steps (length (plan-steps plan)))
        (unsafe (length (plan-unsafe plan)))
        (open (length (plan-open plan)))
	(links (length (plan-links plan))))
    (+ unsafe open steps links)))

(defun RANK-links-are-good (plan)
  ;; count more causal link as better
  (let ((steps (length (plan-steps plan)))
        (unsafe (length (plan-unsafe plan)))
        (open (length (plan-open plan)))
    	(links (length (plan-links plan))))
    (- (+ unsafe open steps)
       links)))

(defun RANK-steps-are-bad (plan)
  ;; penalize extra steps more
  (let ((steps (length (plan-steps plan)))
        (unsafe (length (plan-unsafe plan)))
        (open (length (plan-open plan)))
    	(links (length (plan-links plan))))
    (+ unsafe open steps steps links)))

(defun RANK-steps-are-good (plan)
  ;; reward extra steps
  (let ((steps (length (plan-steps plan)))
        (unsafe (length (plan-unsafe plan)))
        (open (length (plan-open plan)))
    	(links (length (plan-links plan))))
    (- (+ unsafe open links)
       steps)))

(defun RANK-orderings-are-good-duplicate-steps-are-evil-links-are-neutral (plan)
  ;; reward orderings and the number of different steps
  (let ((duplicates (n-duplicates (plan-steps plan) :key #'plan-step-template))
	(unsafe (length (plan-unsafe plan)))
	(open (length (plan-open plan)))
	(orderings (length (plan-ordering plan))))
    (- (+ unsafe open (* 100 duplicates))
       orderings)))

(defun n-distinct (list &key (key #'identity))
  (let ((keys nil)
	(n-elements 0))
    (dolist (element list)
      (let ((the-key (funcall key element)))
	(unless (member the-key keys)
	  (push the-key keys)
	  (incf n-elements))))
    n-elements))

(defun n-duplicates (list &key (key #'identity))
  (- (length list) (n-distinct list :key key)))

(defun RANK-links-are-great-duplicates-and-orderings-are-evil (plan)
  (let ((duplicates (n-duplicates (plan-steps plan) :key #'plan-step-template))
	(unsafe (length (plan-unsafe plan)))
	(open (length (plan-open plan)))
	(links (length (plan-links plan)))
	(orderings (length (plan-ordering plan))))
    (- (+ unsafe open (* 100 (+ duplicates orderings)))
       (* 10 links))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 8. Handy utility functions

(defun tweak-plan (plan &key           ; initially written by jsp
                        (steps :same)
                        (links :same)
                        (unsafe :same)
                        (open :same)
                        (linked :same)
                        (ordering :same)
                        (high-step :same)
			(reason '(:unspecified)))
  "Return a new plan that is a copy of PLAN with the supplied
   instance variable tweaked."
  (flet ((tweak-it (keyword plan-accessor)
           (if (eq keyword :same)
               (funcall plan-accessor plan)
	     keyword)))
    (let ((new-plan
	   (make-plan
	    :steps (tweak-it steps #'plan-steps)
	    :links (tweak-it links #'plan-links)
	    :unsafe (tweak-it unsafe #'plan-unsafe)
	    :open (tweak-it open #'plan-open)
	    :linked (tweak-it linked  #'plan-linked)
	    :ordering (tweak-it ordering  #'plan-ordering)
	    :high-step (tweak-it high-step #'plan-high-step))))
      (when *vcr-loaded*
	(vcr-frame plan reason new-plan))
      new-plan)))

(defvar *intern-plans?* nil) ;; save a copy of plan #j in the variable 'planj'?

(defun MAKE-PLAN (&rest args)
  (incf *plans-created*)
  (let ((plan (apply #'make-plan* args)))
    (when *intern-plans?*
      (set (intern (format nil "PLAN~S" (plan-id plan))) plan))
    plan))

(defun unintern-plans ()
  (dotimes (i *plan-id*)
    (makunbound (intern (format nil "PLAN~S" (1+ i))))))

(defun ->disjunction (goals)
  ;; converts a goal expression into disjunctive normal form.  for example:
  ;;    goals                       meaning          (->disjunction goals)
  ;;    -----                       -------          ---------------------
  ;;   ((A)(B))                     (A)&(B)               (((A)(B)))
  ;;   (:or ((A)(B)) ((C)(D))) (A)&(B) v (C)&(D)     (((A)(B)) ((C))(D))
  ;; note that the input expression can NOT be in an arbitrary form.
  (if (eq (first goals) :OR)
      (cdr goals)
      (list goals)))

(defun build-step-template (action outcome-tree)
  (make-step-template :action action
		      :otpaths (outcome-tree->otpaths outcome-tree)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  9. Print functions 

;;;;;;;;;;;;;;;;
;;; This version does a toplogical sort and then prints out the steps
;;; in the order in which they should be executed

(defun DISPLAY-PLAN (plan &optional (stream t))
  (format stream "~&plan ~S:" (plan-id plan))
  (let ((steps (plan-steps plan)))
    (format stream "~&~2Tinitial:~:[~15T<irrelevant>~%~;~:*~{~15T~A~%~}~]"
	    (mapcar
	     #'(lambda (links)
		 (let ((otpath
			(denouement-otpath (link-denouement (first links)))))
		   (format nil "~S, p=~,3F, for~{ ~S~}"
			   (otpath-outcomes otpath)
			   (trigger-prob (otpath-trigger otpath))
			   (remove-duplicates (mapcar #'link-consumer links)))))
	     (plan-step-links (find 0 steps :key #'plan-step-id) plan)))
    (let ((index 0)
	  (order (top-sort (plan-ordering plan) (plan-high-step plan))))
      (dolist (step-id (remove '0 order)) ; ignore initial step
	(let ((step (find step-id steps :key #'plan-step-id)))
	  (format stream "~4Tstep ~S:~15T~S~60T[~S]~%~{~18T~A~%~}"
		  (incf index)
		  (step-template-action (plan-step-template step))
		  step-id
		  (mapcar #'(lambda (link)
			      (let ((otpath (link-otpath link)))
				(format nil "~S for ~A via ~A: ~S"
					(link-condition link)
					(let ((consumer (link-consumer link)))
					  (if (eq consumer :goal)
					      "goal"
					    consumer))
					(otpath-label otpath)
					(otpath-trigger otpath))))
			  (apply #'append (plan-step-links step plan)))))))
    (let ((disjunction
	   (mapcar #'trigger-conditions
		   (mapcar #'otpath-trigger
			   (mapcar #'(lambda (links)
				       (denouement-otpath
					(link-denouement (first links))))
				   (plan-step-links
				    (find :goal steps :key #'plan-step-id)
				    plan))))))
      (format stream "~&~2Tgoal:~15T~S~{ :or~%~15T~S~}~&"
	      (car disjunction) (cdr disjunction)))
    (let ((*verbose-otpaths?* nil))
      (format stream "~:[~;~:*~2Tunsafe:~{~15T~S~%~}~]~&" (plan-unsafe plan)))
    (format stream "~:[~;~:*~2Topen:~{~15T~S~%~}~]~&" (plan-open plan))
    (let ((*trace* 0))
      (format stream "~2Tprob:~15T~,3F~%"
	      ;; below, we don't know tau because it isn't attached to
	      ;; the plan.  but use 1.0 since that is certain to give
	      ;; us the lower bound.  (maybe not for steve's assessor!!)
	      (funcall *plan-assessor-fn* plan 1))))
  plan)

(defun plan-step-links (step plan)
  ;; returns all the links coming from this step, partitioned by which otpath
  ;; they come from.  used just by display-plan
  (partition
   (keep (plan-step-id step) (plan-links plan) :key #'link-producer)
   :key #'(lambda (link) (denouement-otpath (link-denouement link)))))

(defun display-plans ()
  ;; display all current plans -- just a debugging hack
  (dotimes (i *plan-id*)
    (display-plan (eval (intern (format nil "PLAN~S" (1+ i)))))))

