(require 'ideal)

(defun bayes-assessor (plan tau)
  (let ((min-prob 1))
    ;; this loop is basically an (apply #'min ...) but we want to
    ;; be able to short-circuit the loop if the min is definitely
    ;; less than tau.  note that this means this function does NOT
    ;; return a true lower bound.  rather, if the true lower bound is
    ;; less than tau, then this funciton returns something that is
    ;; less than tau, but if the true lower bound is greater than tau
    ;; it returns the true lower bound.
    (dolist (order (compute-total-orders plan))
      (build-net-from-plan plan order)
      (ideal::jensen-infer (ideal::create-jensen-join-tree ideal::*diagram*)
			   ideal::*diagram*)
      (let ((prob
	     (ideal::belief-of
	      (ideal::make-conditioning-case
	       (list (cons (ideal::find-node :SUCCESS)
			   (ideal::find-label :TRUE :SUCCESS)))))))
	(cond ((< prob tau)
	       (return-from bayes-assessor prob))
	      ((< prob min-prob)
	       (setf min-prob prob)))))
    min-prob))

(defun bayes-assessor-max (plan tau)
  (let ((max-prob 0))
    ;; an alternative assessment algorithm, for free;
    ;; takes the MAX over all total orders instead of the MIN.
    ;; if this were used "for real" then the generator should
    ;; use the total order and constrain the plan
    ;; to respect it.  BURIDAN does not currently do this but
    ;; it easily could.
    (dolist (order (compute-total-orders plan))
      (build-net-from-plan plan order)
      (ideal::jensen-infer (ideal::create-jensen-join-tree ideal::*diagram*)
			   ideal::*diagram*)
      (let ((prob
	     (ideal::belief-of
	      (ideal::make-conditioning-case
	       (list (cons (ideal::find-node :SUCCESS)
			   (ideal::find-label :TRUE :SUCCESS)))))))
	(cond ((>= prob tau)
	       (return-from bayes-assessor-max prob))
	      ((> prob max-prob)
	       (setf max-prob prob)))))
    max-prob))

;;  ******
;;  DANGER -- ugly code ahead.  don't bother.  turn back.
;;  ******

(defun build-net-from-plan (plan order)
  ;; ... first, add the success node, which will be true if STEP-G has a
  ;; successful outcome. (this will this handle disjunction right i think!!)
  (when (> *trace* 8)
    (format t "  + assessing ~S with order ~S~%" plan order))
  (when (> *trace* 8)
    (format t "    + adding SUCCESS node~%"))
  (setf ideal::*diagram*
    (ideal::add-node nil ;; this is the initial "empty" diagram
		     :name :SUCCESS
		     :type :chance
		     :relation-type :prob
		     :state-labels '(:TRUE :FALSE))) 
  ;; ... second: add the goal step, its trigger conditions, and the links
  ;; between them (since the goal step has the special SUCCESS and
  ;; FAILURE pseduo-outcome is needs to be treated specially -- it gets added
  ;; and linked here instead of in 'insert-steps-into-net'.)
  (when (> *trace* 8)
    (format t "    + adding STEP-G node~%"))
  (let* ((step-G
	  (find :goal (plan-steps plan) :key #'plan-step-id))
	 (step-G-otpaths
	  (step-template-otpaths (plan-step-template step-G)))
	 (step-G-node-name
	  (build-description step-G)))
    ;; 2a -- add the goal step itself
    (setf ideal::*diagram*
      (ideal::add-node ideal::*diagram*
		       :name step-G-node-name
		       :type :chance
		       :relation-type :prob
		       :state-labels '(:STEP-G-SUCCESS :STEP-G-FAILURE)))
    (let* ((step-G-node
	    (ideal::find-node step-G-node-name))
	   (conditions
	    (remove-duplicates
	     (mapcar #'positive-expression
		     (apply #'append
			    (mapcar #'trigger-conditions
				    (mapcar #'otpath-trigger
					    step-G-otpaths))))
	     :test #'equal))
	   (conditions-node-names
	    (mapcar #'(lambda (prop)
			(build-description (list prop 'G)))
		    conditions)))
      (when (> *trace* 8)
	(format t "    + adding STEP-G's trigger conditions~%"))
      (dolist (condition-node-name conditions-node-names)
	;; 2b -- add the goal's trigger conditions
	(setf ideal::*diagram*
	  (ideal::add-node ideal::*diagram*
			   :name condition-node-name
			   :type :chance
			   :relation-type :prob
			   :state-labels '(:TRUE :FALSE)))
	;; 2c -- add the arc from this condition
	(ideal::add-arcs step-G-node
			 (list (ideal::find-node condition-node-name))))
      ;; 2c -- now fill in all 2^(|conditions|) elements of STEP-G's
      ;; probability matrix such that
      ;;    p(STEP-G=SUCCESS|state) = 1  if state makes at least
      ;; one of triggers true and p(STEP-G=SUCCESS|state) = 0 otherwise.
      (let ((step-G-success-conditioning-case
	     (ideal::make-conditioning-case
	      (list (cons step-G-node
			  (ideal:find-label :STEP-G-SUCCESS
					    step-G-node-name)))))
	    (step-G-failure-conditioning-case
	     (ideal::make-conditioning-case
	      (list (cons step-G-node
			  (ideal:find-label :STEP-G-FAILURE
					    step-G-node-name))))))
	(dotimes (int-state (expt 2 (length conditions)))
	  (let (true-conditions false-conditions)
	    (dolist (condition conditions)
	      (if (bit-set? (position condition conditions) int-state)
		  (push condition true-conditions)
		  (push condition false-conditions)))
	    ;; ... third: add the arc from step-G to the success node
	    (let ((success-prob
		   (if (some
			;; is there at least one outcome which is true?
			;; (remember these are goal outcomes so they aren't mutex
			;; so more than 1 might be true.)
			#'(lambda (otpath)
			    (let ((tc
				   (trigger-conditions (otpath-trigger otpath))))
			      (and
			       ;; all the positive triggers must hold
			       (subsetp
				(remove-if-not #'expression-sign tc)
				true-conditions
				:test #'equal)
			       ;; all the negative triggers must not hold
			       (subsetp
				(mapcar #'negatify-expression
					(remove-if #'expression-sign tc))
				false-conditions
				:test #'equal))))
			step-G-otpaths)
		       1
		       0))
		  (state-cond-case
		   (ideal::make-conditioning-case
		    (append
		     (mapcar #'(lambda (condition)
				 (let ((node-name
					(build-description
					 (list condition 'G))))
				   (cons (ideal::find-node node-name)
					 (ideal::find-label :TRUE node-name))))
			     true-conditions)
		     (mapcar #'(lambda (condition)
				 (let ((node-name
					(build-description
					 (list condition 'G))))
				   (cons (ideal::find-node node-name)
					 (ideal::find-label :FALSE node-name))))
			     false-conditions)))))
	      (setf (ideal::prob-of
		     step-G-success-conditioning-case
		     state-cond-case)
		success-prob)
	      (setf (ideal::prob-of
		     step-G-failure-conditioning-case
		     state-cond-case)
		(- 1 success-prob))))))
      ;; ... third: add the arc from step-G to the success node
      (when (> *trace* 8)
	(format t "    + adding arc from STEP-G to SUCCESS~%"))
      (let ((success-node (ideal::find-node :SUCCESS)))
	;; 4a: add the arc itself
	(ideal::add-arcs success-node (list step-G-node))
	;; 4b: assign the probability distribution:
	;; p{SUCCESS=true|STEP-G=a} = 1  and  p{S=F|G=a} = 0,
	;; for each good outcome a of G and 
	;; p{SUCCESS=true|STEP-G=OTHER} = 0  and  p{S=F|G=OTHER} = 1
	;; actually I never set the S=F cases -- I assume IDEAL will do
	;; this automatically and correctly since p=1 in the S=T cases!!
	(let ((success-true-cond-case
	       (ideal::make-conditioning-case
		(list (cons success-node (ideal::find-label :TRUE :SUCCESS)))))
	      (success-false-cond-case
	       (ideal::make-conditioning-case
		(list (cons success-node (ideal::find-label :FALSE :SUCCESS)))))
	      (step-G-success-cond-case
	       (ideal::make-conditioning-case
		(list (cons step-G-node (ideal::find-label :STEP-G-SUCCESS step-G-node-name)))))
	      (step-G-failure-cond-case
	       (ideal::make-conditioning-case
		(list (cons step-G-node (ideal::find-label :STEP-G-FAILURE step-G-node-name))))))
	  (setf (ideal::prob-of success-true-cond-case step-G-success-cond-case)
	    1)
	  (setf (ideal::prob-of success-true-cond-case step-G-failure-cond-case)
	    0)
	  (setf (ideal::prob-of success-false-cond-case step-G-success-cond-case)
	    0)
	  (setf (ideal::prob-of success-false-cond-case step-G-failure-cond-case)
	    1)))
      ;; ... fourth: add the rest of the steps (this is where most of
      ;; the work is done)
      (when (> *trace* 8)
	(format t "    + adding the rest of the steps....~%"))
      (setf ideal::*diagram*
	(insert-steps-into-net
	 (mapcar #'(lambda (condition)
		     (cons condition
			   (ideal::find-node
			    (build-description (list condition 'G)))))
		     conditions)
	 plan order))))
  ;; done building diagram
  ideal::*diagram*)

(defun insert-steps-into-net (conditions plan order)
  ;; add all the steps to the plan except step-G which was added above,
  ;; keeping track of conditions (conditions is actually a list
  ;; of (condition . node) pairs where node is the node representing the
  ;; condition at the previous time step.)
  (dolist (step-id (reverse order))
    ;; to add step step-id, we....
    (let* ((step (find step-id (plan-steps plan) :key #'plan-step-id))
	   (step-node-name (build-description step))
	   (otpaths (step-template-otpaths (plan-step-template step)))
	   (otpath-label-names (mapcar #'build-description otpaths)))
      (when (> *trace* 8)
	(format t "      # adding ~S~%" step))
      ;; ... 1: make a node for the step
      (setf ideal::*diagram*
	(ideal::add-node ideal::*diagram*
			 :name step-node-name
			 :type :chance
			 :relation-type :prob
			 :state-labels otpath-label-names))
      ;; ... 2: add a layer of nodes for the conditions inherited from
      ;; the previous iteration
      (let ((new-conditions nil))
	(when (> *trace* 8)
	  (format t "        - adding new nodes for conditions~%"))
	(dolist (condition.node conditions)
	  (let* ((condition (car condition.node))
		 (new-node-name (build-description (list condition step-id))))
	    (when (> *trace* 9)
	      (format t "          > adding node for old condition ~S: ~S~%"
		      condition new-node-name))
	    (setf ideal::*diagram*
	      (ideal::add-node ideal::*diagram*
			       :name new-node-name
			       :type :chance
			       :relation-type :prob
			       :state-labels '(:TRUE :FALSE)))
	    (push (cons condition (ideal::find-node new-node-name))
		  new-conditions)))
	;; ... 3: set up any triggers conditions that aren't already
	;; being handled
	(let* ((all-trigger-conditions
		(remove-duplicates
		 (mapcar #'positive-expression
			 (apply #'append
				(mapcar #'trigger-conditions
					(mapcar #'otpath-trigger
						otpaths))))
		 :test #'equal))
	       (new-trigger-conditions
		(remove-if #'(lambda (new-trigger)
			       (member new-trigger conditions
				       :key #'car :test #'equal))
			   all-trigger-conditions)))
	  (dolist (new-trigger new-trigger-conditions)
	    (let ((node-name (build-description (list new-trigger step-id))))
	      (when (> *trace* 9)
		(format t "          > adding node for new condition ~S: ~S~%"
			new-trigger node-name))
	      (setf ideal::*diagram*
		(ideal::add-node ideal::*diagram*
				 :name node-name
				 :type :chance
				 :relation-type :prob
				 :state-labels '(:TRUE :FALSE)))
	      (push (cons new-trigger (ideal::find-node node-name))
		    new-conditions)))
	  ;; ... 4: add arcs from this step to the inherited condition's nodes
	  ;; (note that arcs get drawn from the new the new node for the
	  ;; thee inherited conditions as well.)
	  (let ((step-node (ideal::find-node step-node-name)))
	    ;;   4a: add the arcs themselves
	    (let ((affected-conditions
		   (remove-if-not
		    #'(lambda (condition.node)
			(let* ((condition (car condition.node))
			       (not-condition (negatify-expression condition)))
			  (some #'(lambda (otpath)
				    (let ((outcomes (otpath-outcomes otpath)))
				      (or (member condition outcomes
						  :test #'equal)
					  (member not-condition outcomes
						  :test #'equal))))
				otpaths)))
		    conditions)))
	      (when (> *trace* 8)
		(format t "        - adding arc from ~S to affecteds: ~S~%"
			step-node (mapcar #'cdr affected-conditions)))
	      (dolist (affected (mapcar #'cdr affected-conditions))
		(ideal::add-arcs affected (list step-node)))
	      ;; BY THE WAY.....
	      ;; DANGER: the notation is a bt strange: I refer to "next" and
	      ;; "prev" to mean the nodes from the-next/this iteration, and
	      ;; "prev" to mean the nodes from the last iteration, although
	      ;; we are constructing the net backwards in plan-time so "prev"
	      ;; REALLY means future and "next" means temporally earlier.
	      (when (> *trace* 8)
		(format t "        - adding arc to previous conditions~%"))
	      (dolist (condition.node conditions)
		(let* ((condition
			(car condition.node))
		       (prev-node
			(cdr condition.node))
		       (next-node
			(cdr (find condition new-conditions
				   :key #'car :test #'equal))))
		  (when (> *trace* 8)
		    (format t "          > adding arc from ~S to ~S~%"
			    next-node prev-node))
		  (ideal::add-arcs prev-node (list next-node))))
	      ;;   4b: set probability matrices for each unaffected condition
	      ;;     to just p(D=T|prev=T) = p(D=F|prev=F) = 1
	      (when (> *trace* 8)
		(format t "        - setting dist of prev conditions~%"))
	      (dolist (condition.node conditions)
		(unless (member condition.node affected-conditions
				:test #'equal) ;; eq ought to do but to be safe!
		(when (> *trace* 9)
		  (format t "          > setting dist for unaffected ~S~%"
			  condition.node))
		  (let* ((condition
			  (car condition.node))
			 (node
			  (cdr condition.node))
			 (name
			  (ideal::node-name node))
			 (node-true-cond-case
			  (ideal::make-conditioning-case
			   (list
			    (cons node
				  (ideal::find-label :TRUE name)))))
			 (node-false-cond-case
			  (ideal::make-conditioning-case
			   (list
			    (cons node
				  (ideal::find-label :FALSE name)))))
			 (next-condition.node
			  (find condition new-conditions
				:test #'equal :key #'car))
			 (next-node
			  (cdr next-condition.node))
			 (next-name
			  (ideal::node-name next-node))
			 (next-node-true-cond-case
			  (ideal::make-conditioning-case
			   (list
			    (cons next-node
				  (ideal::find-label :TRUE next-name)))))
			 (next-node-false-cond-case
			  (ideal::make-conditioning-case
			   (list
			    (cons next-node
				  (ideal::find-label :FALSE next-name))))))
		    (setf (ideal::prob-of
			   node-true-cond-case
			   next-node-true-cond-case)
		      1)
		    (setf (ideal::prob-of
			   node-true-cond-case
			   next-node-false-cond-case)
		      0)
		    (setf (ideal::prob-of
			   node-false-cond-case
			   next-node-true-cond-case)
		      0)
		    (setf (ideal::prob-of
			   node-false-cond-case
			   next-node-false-cond-case)
		      1))))
	      ;;   4c: set probility matrices such that:
	      ;;   i.   if step/a makes D true then p(D=T|step=a,next=?)=1
	      ;;   ii.  if step/a makes D false then p(D=T|step=a,next=?)=0
	      ;;   iii. if step/a doesn't touch D the np(D=T|step=a,next=T)=1
	      ;;        and p(D=F|step=a,next=F)=1
	      ;;   where "?" means "don't care" and D ranges over each affected
	      ;;   condition (note that case (iii) is needed because "affected"
	      ;;   just means does ANY outcome touch the condition but any
	      ;;   particular outcome might or might not).
	      (dolist (condition.node affected-conditions)
		(when (> *trace* 9)
		  (format t "          > setting dist for affected ~S~%"
			  condition.node))
		(let* ((condition
			(car condition.node))
		       (node
			(cdr condition.node))
		       (name
			(ideal::node-name node))
		       (node-true-cond-case
			(ideal::make-conditioning-case
			 (list
			  (cons node (ideal::find-label :TRUE name)))))
		       (node-false-cond-case
			(ideal::make-conditioning-case
			 (list
			  (cons node (ideal::find-label :FALSE name)))))
		       (next-condition.node
			(find condition new-conditions :test #'equal :key #'car))
		       (next-node
			(cdr next-condition.node))
		       (next-name
			(ideal::node-name next-node))
		       (next-node-true-raw-cond-case
			(cons next-node (ideal::find-label :TRUE next-name)))
		       (next-node-false-raw-cond-case
		      (cons next-node (ideal::find-label :FALSE next-name))))
		  (dolist (otpath otpaths)
		    (let ((step-raw-cond-case
			   (cons step-node
				 (ideal::find-label
				  (build-description otpath)
				  step-node-name))))
		      ;; the differences between the 3 cases are (*)'ed
		      (cond ((member condition
				     (otpath-outcomes otpath) :test #'equal)
			     ;; case i: step/outcome makes condition TRUE
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       1)
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       1)
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       0)
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       0))
			    ((member (negatify-expression condition)
				     (otpath-outcomes otpath) :test #'equal)
			     ;; case ii: step/outcome makes condition FALSE
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       1)
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       1)
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       0)
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       0))
			    (t
			     ;; case iii: step/outcome doesn't touch condition
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       1)
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       1)
			     (setf (ideal::prob-of
				    node-false-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-true-raw-cond-case))) ;; (*)
			       0)
			     (setf (ideal::prob-of
				    node-true-cond-case ;; (*)
				    (ideal::make-conditioning-case
				     (list step-raw-cond-case
					   next-node-false-raw-cond-case)));; (*)
			       0)))))))
	      ;; ... 5: add the arcs from the trigger conditions to the step
	      (when (> *trace* 8)
		(format t "        - adding arcs from triggers to the step~%"))
	      (dolist (condition all-trigger-conditions)
		(let* ((condition.node
			(find condition new-conditions :test #'equal :key #'car))
		       (node
			(cdr condition.node)))
		  (when (> *trace* 9)
		    (format t "          > adding arc to ~S from ~S~%"
			    step-node node))
		  (ideal::add-arcs step-node (list node))))
	      ;; ... 6: set the probability matrix of the step.  this is a bit
	      ;;  tricky in that we encode chance propositions by a probability
	      ;; distribution over the outcomes -- all  2^(|conditions|) elements
	      ;; of it.
	      ;;    p(step=a|state) = p   if state makes a's trigger true,
	      ;;        (where p is a's trigger's probability), and
	      ;;    p(step=a|state) = 0   otherwise
	      (when (> *trace* 8)
		(format t "        - setting dist of step ~S~%" step-node))
	      (dotimes (int-state (expt 2 (length all-trigger-conditions)))
		(let (true-conditions false-conditions)
		  (dolist (condition all-trigger-conditions)
		    (if (bit-set? (position condition all-trigger-conditions)
				  int-state)
			(push condition true-conditions)
			(push condition false-conditions)))
		  (when (> *trace* 9)
		    (format t "          > state ~S: true=~S; false=~S~%"
			    int-state true-conditions false-conditions))
		  (dolist (otpath otpaths)
		    (let ((step-cond-case
			   (ideal::make-conditioning-case
			    (list 
			     (cons step-node
				   (ideal::find-label
				    (build-description otpath)
				    step-node-name))))))
		      (when (> *trace* 9)
			(format t "            * setting dist for outcome ~S~%"
				(build-description otpath)))
		      (setf (ideal::prob-of
			     step-cond-case
			     (ideal::make-conditioning-case
			      (append
			       (mapcar #'(lambda (condition)
					   (let ((node-name
						  (build-description
						   (list condition step-id))))
					     (cons 
					      (ideal::find-node node-name)
					      (ideal::find-label :TRUE node-name))))
				       true-conditions)
			       (mapcar #'(lambda (condition)
					   (let ((node-name
						  (build-description
						   (list condition step-id))))
					     (cons (ideal::find-node node-name)
						   (ideal::find-label :FALSE node-name))))
				       false-conditions))))
			(let* ((trigger
				(otpath-trigger otpath))
			       (trigger-conditions
				(trigger-conditions trigger)))
			  (if (and
			       (subsetp
				(remove-if-not #'expression-sign trigger-conditions)
				true-conditions :test #'equal)
			       (subsetp
				(mapcar
				 #'negatify-expression
				 (remove-if #'expression-sign trigger-conditions))
				false-conditions
				:test #'equal))
			      (trigger-prob trigger)
			      0)))))))))
	  ;; ... 7: done -- do the next iteration, inhereting the new set
	  ;; of conditions
	  (setf conditions new-conditions)))))
  ;; FINALLY, add STEP-0
  ;;  (a) add the node itself
  (when (> *trace* 8)
    (format t "        - adding step 0~%"))
  (let* ((step-0
	  (find 0 (plan-steps plan) :key #'plan-step-id))
	 (step-0-name
	  (build-description step-0))
	 (step-0-otpaths
	  (step-template-otpaths (plan-step-template step-0))))
    (setf ideal::*diagram*
      (ideal::add-node ideal::*diagram*
		       :name step-0-name
		       :type :chance
		       :relation-type :prob
		       :state-labels (mapcar #'build-description step-0-otpaths)))
    ;; (b) set the distribution of step 0 based on the probability
    ;; in each trigger
    (let ((step-0-node (ideal::find-node step-0-name)))
      (when (> *trace* 9)
	(format t "          > setting dist of ~S~%" step-0-node))
      (dolist (otpath step-0-otpaths)
	(setf (ideal::prob-of
	       (ideal::make-conditioning-case
		(list
		 (cons
		  step-0-node
		  (ideal::find-label (build-description otpath) step-0-name))))
	       nil) ;; step-0 isn't conditional on anything!
	  (trigger-prob (otpath-trigger otpath))))
      ;;  (c) add arcs to all the conditions and set probability matrices
      ;; (c-1): add the arcs themselves
      (when (> *trace* 9)
	(format t "          > adding arcs from ~S to ~S~%"
		step-0-node (mapcar #'cdr conditions)))
      (dolist (condition.node conditions)
	(ideal::add-arcs (cdr condition.node) (list step-0-node)))
      ;; (c-2): now set the probabilities
      (dolist (condition.node conditions)
	(let* ((condition
		(car condition.node))
	       (node
		(cdr condition.node))
	       (condition-true-cond-case
		(ideal::make-conditioning-case
		 (list
		  (cons
		   node
		   (ideal::find-label	
		    :TRUE
		    (ideal::node-name node))))))
	       (condition-false-cond-case
		(ideal::make-conditioning-case
		 (list
		  (cons
		   node
		   (ideal::find-label	
		    :FALSE
		    (ideal::node-name node)))))))
	  (when (> *trace* 9)
	    (format t "          > setting dist for ~S~%" node))
	  (dolist (otpath step-0-otpaths)
	    (let ((step-cond-case
		   (ideal::make-conditioning-case
		    (list
		     (cons
		      step-0-node
		      (ideal::find-label
		       (build-description otpath)
		       step-0-name)))))
		  (indicator
		   (if (member condition (otpath-outcomes otpath) :test #'equal)
		       1
		       0))) ;; assume either C or ~C is in every step 0 outcome	
	      (setf (ideal::prob-of condition-true-cond-case step-cond-case)
		indicator)
	      (setf (ideal::prob-of condition-false-cond-case step-cond-case)
		(- 1 indicator))))))))
  ideal::*diagram*)

(defun build-description (thing)
  ;; returns a label for each kind of "thing" in the net
  ;; nodes might be steps or proposition (or the special SUCCESS node
  ;; but that isn't handled here)
  ;; state-labels are outcomes (or TRUE/FALSE for propositions but
  ;; that isn't handled here)
  (intern
   (cond ((plan-step-p thing)
	  (format nil "S.~A" (plan-step-id thing)))
	 ((otpath-p thing)
	  (format nil "~A" (otpath-label thing)))
	 ((listp thing)
	  ;; assume this is a proposition, of the form ((TC) 1)
	  ;; where 1 is the step index and (TC) is the proposition
	  (format nil "~S@~S-" (car (first thing)) (second thing))))))
