;
;	BUILD II  --  a blocks world
;
;						John Nagle
;						Version 1.7 of 5/15/87
;
;
;	IBM RS-1 robot communications interface
;
;	Uses serial port connected to special interface conversion unit
;	which communicates with the RS-1 via a parallel port.
;
;	The link accepts commands in AML (A Manufacturing Language)
;	and returns the result of evaluating the command, plus anything
;	else the command wants to print to the line.  The link is line-oriented.
;
(defparameter robotport "/dev/ttya" "Port to which robot is attached")

(defvar robotstream-in nil)		; stream if open
(defvar robotstream-out nil)
(defvar echo-robot t "Echo communications with robot if T")
(defconstant readerval '(FAULT COMMUNICATIONS READ-ERROR)) ; returned on err
(defconstant esc (int-char 27))		; ASCII ESC
(defparameter robot-stty-command
"stty new 9600 -echo -tabs -nl litout stop ")
;
;	open-robot  --  open communications with robot
;
(defun open-robot (&optional (port robotport))
	(when robotstream-in (return-from open-robot t)) ; ignore if open
	(setq robotstream-in (open port :direction :input)) ; open for input
	(unless robotstream-in
		(error "Unable to open robot I/O port ~a" port)) ; bad
	(set-robot-comm-modes port)		; set TTY modes
	(clear-input robotstream-in)		; clear any pending input
	(setq robotstream-out (open port :direction :output
		:if-exists :append		; normally append to dev
		:if-does-not-exist :error))	; error if not appendable
	)
;
;	set-robot-comm-modes
;
(defun set-robot-comm-modes (dev)
  (let ((s (concatenate 'string robot-stty-command " > " dev)))
       (unless (= (system s) 0)	; do command
	       (error "UNIX error executing: ~a" s))))
;
;	write-to-robot  --  write to robot stream
;
(defun write-to-robot (s)
	(open-robot)				; open if not already open
	(write-line s robotstream-out)		; write to robotstream-out
	(force-output robotstream-out)		; force everything out
	(when echo-robot (format t "-> ~a~%" s))) ; echo if requested.
;
;	read-from-robot  --  read from robot stream
;
;	Reads one line from the robot stream.
;
(defun read-from-robot nil
  (open-robot)				; open if not already open
  (let* ((s (read-line robotstream-in nil readerval))) ; read a line
	;	Parse into an S-expression if possible.
	(when echo-robot (format t "<- ~a~%" s)) ; echo if requested
	s))
;
;	probe-robot  --  is data available from robot?
;
(defun probe-robot nil
  (open-robot)				; open if needed
  (listen robotstream-in))		; T if valid
;
;	drain-input  --  drain any available input
;
;	Used in resynchronizing communications.
;
(defun drain-input (str)
  (loop 
    (unless (listen str) (return))	; if nothing to read, escape
    (read-char str)))			; drain any chars available.
;
;	halt-robot  --  force ATTN at robot, stopping motion or computation
;
;	This sends an ESC, which is recognized by the interface unit as
;	an ATTN request. The interface unit sends a signal on a special
;	interface line which is monitored by the AML program that handles
;	remote operation.  The AML program stops the robot with HALTMOVE,
;	and forces an error to cause normal error processing; eventually
;	control comes back to the read-eval-print loop at the robot.
;	The interface unit discards all pending input and output when
;	this occurs, and returns (REQUEST ATTN) to the serial line.
;
;	Halt-robot stops robot motion IMMEDIATELY; the halt can be quite
;	abrupt.  It is not desirable to use this as part of normal operation;
;	the RS-1 manual warns that it is bad for the robot to use this
;	type of halt during fast robot motion.
;
;	Halt-robot does not normally turn off hydraulics, although a
;	request issued during a fast move may result in enough positional
;	error to force a tolerance error shutdown.
;
;	The position of the robot after a halt will not change if the
;	robot was not moving and will be somewhere along the path of the
;	current move if one was in progress.
;
;	***NEED UNIX-DEPENDENT CODE TO CLEAR XOFF HOLD IF CLEAR-OUTPUT DOESNT***
;
(defun halt-robot (&optional (usermsg "==> ROBOT HALT REQUESTED <=="))
  (let ((starttime (get-universal-time))) ; timestamp start of halt process
       (open-robot)				; open if not already open
       (format t "~a~%" usermsg)		; notify user
       (clear-output robotstream-out)		; discard any pending output
       (write-char esc robotstream-out)		; send an ESC
       (clear-input robotstream-in)		; clear any pending input
       (drain-input robotstream-in)		; clear any pending input
       (write-char esc robotstream-out)		; send again just in case.
       (force-output robotstream-out)		; force out
       ;	Resynchronize serial link.
       ;	Note that this busy-waits 
       (loop
	(when (> (- (get-universal-time) starttime) 15) ; if taking too long
	      (error "==> NO RESPONSE TO ATTN SIGNAL <==")) ; note problem
	(if (probe-robot)
	    (let ((s (read-from-robot)))	; get any output available
		 (unless echo-robot
			 (format t "<- ~a~%" s)) ; echo if indicated.
		 (when (equal s "(ATTN)")
		       (format t "Halted.")
		       (return))
		 (format t "===>ATTN INCOMPLETE: ~a <===~%" s))
	    (sleep 1)))))			; no input, wait 1 sec.
;
;	synchronize-robot-io  -- make sure data path is synchronized.
;
(defun synchronize-robot-io nil
  (when (probe-robot) 
	(format t "Resynchronizing robot I/O channel.~%")
	(halt-robot)))
;
;	command-robot  --  send command to robot
;
;	Expects exactly one line in reply.
;
(defun command-robot (s)
	(synchronize-robot-io)		; make sure we are synchronized.
	(write-to-robot s)
	(read-from-robot))
