;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.23 of 4/18/87
;
;	Goal functions of the planning system.
;	
(require 'database "database")
(use-package 'database)
(require 'builddefs "builddefs")			; prerequisite
(use-package 'builddefs)
(require 'vector "vector")
(use-package 'vector)
(require 'buildutil "buildutil")
(use-package 'buildutil)
(require 'collision "build5")
(use-package 'collision)
(require 'stability "build6")
(use-package 'stability)
;
(require 'plandefs "plandefs")
;
;	move  --  move a block, with checking
;
;	(primitive planning function)
;
;	If a gripe handler returns, the problem should have been dealt
;	with and we should be in a valid state; we make some checks to
;	insure that this is the case.
;
(defun move (b locp locg
	       &aux x item (startcontext context))
  ;	Preliminary checks
  (verify-at b locp)			; B must be at LOCP.
  (verify-stable)			; moderately expensive
  ;	If the move is null, done.
  (when (equalp locp locg)
	(return-from move plan))	; happy
  ;	If B is immovable, no good.
  (when (isimmovable b)			; if immovable
	(gripe (list 'immovable b)))	; yes, cannot move.
  ;	If B is already in place, we would prefer not to move it,
  ;	and will gripe, but MOVE will move it if the caller insists.
  (when (setq item (present (list 'in-place b '*)))
	(setq x (gripe (list 'in-place b (third item)) t))
	(assert (eq x plan))		; recovery must not change plan!
	(context-delete item))		; caller insisted.
  ;	Make sure that block is where it is supposed to be.
  (verify-at b locp)			; internal error trap
  ;	Establish new working context.
  (setq context (push-context))
  ;	Remove block from old position.
  (remblock b)				; remove the block
  ;	Show state with old block gone.
  (when showinter (showcontext context (format nil "Moving ~a" b)))
  ;	Check stability of state with block removed.
  (unless (eq t (checkstate))
	  (gripe (list 'unstab-rem b (cdr x))))
  ;	Add block at new location.
  (addat b locg)			; add B at goal location.
  ;	Show state with block at new location.
  (when showstep (showcontext context (format nil "After move of ~a" b)))
  ;	Collision at new location?
  ;	This is recoverable.  
  (when (eq 'hit (car (setq x (gettouch b))))
	(setq context startcontext)	; back out failed move.
	(setq plan (gripe (list 'hit b (cdr x)) t)) ; get help
	(setq context (second (first plan)))	; accept new context
	;	Try again, recursively.
	(return-from move (move b (getat b) locg)))
  ;	Unstable after add at new location?
  ;	This is recoverable.
  (unless (eq t (setq x (checkstate)))	; unstable now?
	  (setq context startcontext)	; back out failed move.
	  (setq plan (gripe (list 'unstab-add b (cdr x)) t))
	  (setq context (second (first plan)))	; accept new context
	  ;	Try again, recursively.
	  (return-from move (move b (getat b) locg)))
  (verify-at b locg)			; block must be where wanted.
  ;	Success - update and return plan.
  (push (list startcontext 		; state before action
	      context 			; state after action
	      (list 'move b locp locg)	; action
	      reason) 			; reason for action
	plan)
  plan)					; return plan
;
;	moveg  --  move group of blocks as a unit.
;
;	If TEST is t, check whether legal movable sub-assembly.
;	***UNTESTED AND NEEDS WORK***
;
(defun moveg (baseg riders locp locg &optional (test t)
		    &aux x y (startcontext context))
  ;	If the move is null, no good.
  (when (equalp locp locg)
	(gripe (list 'already-at baseg locg)))
  ;	Gripe if anything immovable.
  (when (setq x (anyimmovable (cons baseg riders)))
	(gripe (list 'immovable-g x)))
  ;	Gripe if anything already in place, but accept it if caller insists.
  (when (setq x (anyinplace (cons baseg riders)))
	(gripe (list 'in-place-g x) t)		; recoverable
	(reminplace x))				; caller insists.
  ;	Will anything fall off if we pick up this group?
  (when (setq x (fallout baseg riders))
	(gripe (list 'fallout baseg riders x)))
  ;	Preliminary checks passed, we will try to do the move.
  (setq context (push-context))
  (verify-at baseg locp)			; make sure BASEG at LOCP.
  (remblock baseg)				; remove block BASEG.
  (setq y (saverel locp riders))		; save rel configuration.
  (remblocks riders)				; remove riding blocks.
  (when showinter 
	(showcontext context 
		     (format nil "While moving group ~a ~a" baseg riders)))
  (unless (eq t (setq x (checkstate)))
	  (gripe (list 'unstab-rem-g baseg riders (cdr x)))
	  (verify-stable))			; must still be stable
  ;	Add blocks at new location.
  (addat baseg locg)				; add base block.
  (reladd locg y)				; add riders.
  (when showstep
	(showcontext context  (format nil "After MOVEG of ~a ~a" baseg riders)))
  ;	Check for collision in final state.  Recoverable.
  (when (eq 'hit (car (setq x (gettouch))))
	(gripe (list 'hit-g baseg riders (cdr x)) t)
	(verify-unhit))				; must clear collision
  ;	Check for instability in final state.  Recoverable.
  (unless (eq t (setq x (checkstate)))
	  (gripe (list 'unstab-add-g baseg riders (cdr x)) t)
	  (verify-unhit)			; must clear collision
	  (verify-stable))			; must be stable
  ;	Perform shake test on group unless caller requested otherwise.
  ;	Not recoverable; the selection of RIDERS was no good.
  ;	Really ought to perform this test earlier.
  (when test
	(setq x (shakeout baseg riders))
	(gripe (list 'not-msa baseg riders x)))
  ;	Finally return success.
  (push (list 
	 startcontext 				; context before move
	 context 				; context after move
	 (list 'moveg baseg riders locp locg)	; action
	 reason)				; reason for action
	plan) ; update plan
  plan)						; return new plan
;
;	anyimmovable  --  list any immovable objects in list of blocks L.
;
(defun anyimmovable (l &aux immovables)
  (mapc
   (function
    (lambda (b) 
	    (when (isimmovable b) 
		  (push b immovables))))
   l)
  immovables)
;
;	anyinplace  --  list any objects in L that are already in
;			place.
;
;	Returns list of the form
;		((blk at) (blk at) ...)
;
(defun anyinplace (l &aux inplace item)
  (mapc
   (function
    (lambda (b)
	    ;	If IN-PLACE item exists, add to list.
	    (when (setq item (present (list 'in-place b '*)))
		  (push (list b (third item)) inplace))))
   l)
  inplace)
;
;	reminplace  --  remove the IN-PLACE items for blocks in L.
;
;	Format of L is that returned by IN-PLACe.
;
(defun reminplace (l)
  (mapc
   (function
    ;	Remove all relevant IN-PLACE items.
    (lambda (x) (context-delete (list 'in-place (first x) (second x)))))
   l)
  t)
;
;	saverel  --  save relative locations of blocks to B in LOC.
;
;	This is part of the mechanics of moving a group of blocks.
;
(defun saverel (loc l)
  (mapcar 
   (function
    (lambda (b) 
	    (list b (relate-location (getat b) loc))))
   l))  
;
;	reladd  --  replace blocks in same relation to new LOC.
;
;	This is how we move a group of blocks.
;
(defun reladd (loc l)
  (mapc
   (function
    (lambda (x)
	(addat (first x) (unrelate-location (second x) loc))))
   l) t)
;
;	remblocks  --  remove listed blocks
;
(defun remblocks (l)
  (mapc
   (function (lambda (b) (remblock b)))
   l)
  t)
