;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.30 of 5/19/87
;
(require 'builddefs "builddefs")
(require 'database "database")
(require 'plandefs "plandefs")
(use-package 'builddefs)
(use-package 'database)
;
;	Planning portion of build system
;	New control framework
;
(require 'builddefs "builddefs")		; prerequisite
;
;	goal  --  work on a new goal.
;
;	Goal rebinds a number of special variables and invokes FN
;	on ARGS.  FN should be a choice function.
;
(defun goal (fn args			; what to do and who to do it to
		&optional (gripehandler nil)
		&aux
		(reason reason)		; maintain chain of reasons
		(context context)	; rebind and save context
		(plan plan)		; rebind and save plan
		(*choices *choices)	; list of gripe catchers
		(goaldepth (1+ goaldepth))	; depth into subgoaling
		(pfn (pname-of-fn fn))	; printable name of fn
		newplan)
  (tracks "Beginning work on goal: ~a ~a." pfn args)
  (when dumpstates (dumpstate context))	; debug
  (push (cons pfn args) reason)		; add new reason
  (when gripehandler
	(push gripehandler *choices))	; add new gripe handler if present
  (null-plan-step-check)		; generate null step if needed
  (setq newplan (apply fn args))	; do the goal
  (tracks "Achieved goal: ~a ~a." pfn args)
  (unless (member (first (third (first plan))) '(move moveg nop))
	  (error "INVALID PLAN STEP ~a." 
		 (first (third (first plan)))))	; ***TEMP*** debug trap
  (when dumpstates (dumpstate context))	; debug
  newplan)				; return result of evaluating goal.
;
;	prepgoal  --  generate internal preparatory goal
;
;	Prepgoal is like goal, except that reason for this goal is to
;	prepare for the action given in reason.
;
(defun prepgoal (why fn args &optional (gripehandler nil))
          (tracks "Preparing for ~a." why)
          (push (cons 'prepare-for why) reason) ; update reason
          (goal fn args gripehandler))
;
;	null-plan-step-check
;
;	We maintain the invariant that the after-context in the last step
;	of the current plan is the current context, and sometimes we have
;	to insert a null step in the plan to achieve this.  For a null
;	step in the plan, only derived database items (such as SUP-BY)
;	may differ.  AT items must not change.
;
(defun null-plan-step-check nil
  (unless (and plan (eq (second (first plan)) context))
	  ;	Make dummy plan step.
	  (push (list 
		 (if plan (second (first plan)) context)		
		 context		; with current after-context
		 (list 'nop)		; NOP action
		 reason) 		; why
		plan)))
;
;	gripe  --  report failure with a reason
;
;	Griping is probably the best idea in the whole system.
;
;	Gripe passes its message and a return tag back to last choice point.
;	MESSAGE is a form which the caller should understand and act on.
;	It is not a text string.
;
;	Gripe does not usually return to its caller, but it is possible
;	for it to do so, in which case the condition causing the gripe
;	has presumably been removed.  
;	
;	If RETURNABLE is non-nil, the caller is prepared to retry if a
;	gripe handler can deal with the gripe.
;
(defun gripe (message &optional (returnable nil)
		      &aux gripehandler pfn (*choices *choices))
  (tracks "Gripe: ~a." message)
  (setq gripehandler (pop *choices))	; get top gripe handler
  (unless gripehandler
	  (error "No gripe handlers remain."))
  ;;;(setq pfn (pname-of-fn gripehandler))	; get print name for messages
  (setq pfn "#<function>")		; ***TEMP***
  (tracks "Applying gripe handler ~a." pfn)
  (null-plan-step-check)		; add dummy step if needed
  (setq plan (funcall gripehandler message returnable)); call gripe handler
  (tracks "Gripe ~a dealt with by ~a." message pfn) ; recovered
  (unless (member (first (third (first plan))) '(move moveg nop))
	  (error "INVALID PLAN STEP ~a." 
		 (first (third (first plan)))))	; ***TEMP*** debug trap
  (unless returnable 
	  (error "Gripe handler ~a returned but should not have." pfn))
  (null-plan-step-check)		; add dummy step if needed
  plan)				; return reply of gripe handler.
;
;	pass  --  pass a gripe along
;
;	Pass is used in gripe handlers to pass gripe upward to next
;	higher level gripe handler.
;
;	Behavior much like that of GRIPE.
;
(defun pass (message &optional (returnable nil))
  (tracks "Passing on gripe: ~a." message)
  (gripe message returnable))
;
;	final-gripe-handler  --  nothing worked.
;
;	We are not supposed to get here.
;
(defun final-gripe-handler (message returnable)
  (declare (ignore returnable))
  (tracks "Reached final gripe handler - ~a" message)
  (error "BUILD gives up"))		; ***TEMP***
(setq *choices (list #'final-gripe-handler))	; initialize gripe handlers
;
;	dobuild  --  user's entry to BUILD
;
;	Tries to get to state FINALGOAL from state START.
;	Returns plan in forward order.
;
(defun dobuild (start finalgoal 
		      &optional (why (list "Because you told me to."))
		      &aux (context start) plan reason)
  (gripehandlers
   ((build-by-dobuild (message returnable)
		      (identity returnable)	; make compiler happy
		      (format t "DOBUILD unsuccessful: ~a~%" message)
		      (return-from dobuild nil)))
   (setq context (push-context))		; protect original
   (push why reason)			; initialize reason
   (setq plan (goal 'build
		    (list finalgoal)
		    build-by-dobuild)))
  (reverse plan))
