;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.36 of 4/21/87
;
;      Touch and collision testing stuff.
;
(require 'builddefs "builddefs")
(require 'buildutil "buildutil")
(require 'vector "vector")
(require 'database "database")
(provide 'collision)
(in-package 'collision)
(export '(
	  gettouch
	  separate
	  delete-collision-relationships
	  ))
(use-package '(builddefs buildutil vector database))
;
;      gettouch2  --  get touch relation of two given objects.
;
(defun gettouch2 (b1 b2 &aux relation hits)
  ;	Did we make this check already?
  (cond ((setq relation (present (list 'touches b1 b2 '*)))
	 (fourth relation))	; yes, return result
	((setq relation (present (list 'touches b2 b1 '*)))
	 (flip (fourth relation))) ; did it from b1
	(t (setq hits (hitlisp b1 b2))	; get collisions
	   (context-add (list 'touches b1 b2 hits)) ; remember for next
	   hits)))
;
;      flip  --  given (hitlisp a b) generate (hitlisp b a).
;
(defun flip (l)
  (cond ((null l) nil)				; if none, nil
	((eq l 'hit) l)				; if hit, hit
	((cons (reverse-plane (first l)) (rest l))))) ; otherwise reverse plane
;
;      gettouch1  --  get all touch or hit relations of given block
;
(defun gettouch1 (b)
  (prog (touchers lb x lost)
	(setq touchers nil lb (getobs))
	loop (cond ((null lb)
		    (and lost (return (cons 'hit touchers)))
		    (return touchers))
		   ((eq	(first lb) b))
		   ((null (setq	x (gettouch2 b (first lb)))))
		   (lost (and (eq x 'hit)
			      (setq touchers (cons (first lb) touchers))))
		   ((eq x 'hit)
		    (setq lost t touchers (list (first lb))))
		   ((setq touchers (cons (cons (first lb)	x) touchers))))
	(setq lb (rest lb))
	(go loop)))
;	       
;      gettouch0  --  get all touch or hit relations in context
;
(defun gettouch0 nil
  (prog (touchers lb1 lb2 x lost)
	(setq touchers nil 
	      lb1 (getobs)
	      lost nil)
	loop1  (or lb1
		   (and lost (return (cons 'hit touchers)))
		   (return touchers))
	(setq lb2 (rest lb1))
	loop2  (cond ((null lb2) (setq lb1 (rest lb1)) (go loop1))
		     ((null (setq x (gettouch2 (first lb1) (car lb2)))))
		     (lost (and (eq x 'hit)
				(setq touchers
				      (cons (list (first lb1) (car lb2))
					    touchers))))
		     ((eq x 'hit)
		      (setq lost t touchers (list (list (first lb1)
							(first lb2)))))
		     ((setq touchers 
			    (cons (cons (first lb1) (cons (car lb2) x))
				  touchers))))
	(setq lb2 (rest lb2))
	(go loop2)))
;	      
;     	gettouch  --  main entry to touch detection
;
;	Usage:	(gettouch <block> <block)	-- return touching points
;		(gettouch <block>)
;		(gettouch)
;
(defun gettouch	(&rest arglist &aux n)
  (setq n (length arglist))			; number of args
  (cond ((= n 2) (gettouch2 (first arglist) (second arglist)))
	((= n 1) (gettouch1 (first arglist)))
	((= n 0) (gettouch0))
	(t (error "Wrong number of args to gettouch"))))
;
;      hitlisp	--  test two objects
;
;      If clear, return	nil.
;      If colliding, return 'hit.
;      If touching, return (list normal-plane point1 point2 ...)
;
;	The approach is to find a plane which lies between the blocks.
;	Given this, most of the rest is straightforward.
;
(defun hitlisp (b1 b2)
  (prog (t1 t2 pts1 pts2 plane pl1 pl2 l1 l2 x y z)
	(setq t1 (getat b1))		; location of B1
	(setq t2 (getat b2))		; location of B2
	;       Perform cheap checks to eliminate most possiblities.
	(and (cond ((eq b1 table)	; table special cases
		    (> (vector3-z t2) (+ zmin (block-maxdimension b2))))
		   ((eq b2 table)
		    (> (vector3-z t1) (+ zmin (block-maxdimension b1))))
 		   ;	Too far apart to collide?
		   ((> (point-point-distance-squared t1 t2)
		       (sq (+ tol1
			      (block-maxdimension b1)
			      (block-maxdimension b2))))))
	     (return nil))			; cannot collide
	;	End of cheap checks, will have to do some real work.
	(setq pts1 (getverts b1))
	(setq pts2 (getverts b2))
	;       SEPARATE does all the work.
	(setq x (separate pts1 pts2 (getcgrav b1) (getcgrav b2)))
	(unless x (return 'hit)) 	; no separation plane, hit.
	(when (eq x t) (return nil))	; nothing touches separation plane.
	;       PL1 and PL2 are lists of the points on the separation plane.
	;	T1 and T2 are counts of the number of points from each block
	;	found in the separation plane. 
	;	Zero points indicates no contact.  One point indicates point
	;	contact.  Two points indicates line contact.  More than two
	;	points indicates planar contact.
	(setq plane (first x))   
	(setq pl1 (second x))
	(setq pl2 (third x))
	(setq t1 (length pl1))
	(setq t2 (length pl2))
	(cond ((= t1 0) (return nil))	; no contact 
	      ((= t2 0) (return nil))	; no contact
	      ((and (= t1 1) (= t2 1))
	       ;       One point from each, see if same.
	       (and (v3equalp (aref pts1 (first pl1)) (aref pts2 (car pl2)))
		    (return (list plane (aref pts1 (first pl1)))))
	       (return nil))
	      
	      ((and (= t1 1) (= t2 2))
	       ;       One and two, see if point-on-line contact.
	       ;       Check for collinearity.
	       (and (online   (aref pts1 (first pl1))
			      (aref pts2 (first pl2))
			      (aref pts2 (second pl2)))
		    (return (list plane (aref pts1 (first pl1)))))
	       (return nil))
	      
	      ((and (= t1 2) (= t2 1))
	       ;       Two and one, ditto
	       (and (online   (aref pts2 (first pl2))
			      (aref pts1 (first pl1))
			      (aref pts1 (second pl1)))
		    (return (list plane (aref pts1 (first pl1)))))
	       (return nil)))
	;	Line-on-line contact is handled in the general case below.
	(setq x nil l1 nil l2 nil)
	;
	;       If two points, get line.  If more get face boundaries.
	;       Report all verts from other block inside face bounds.
	;       Also all boundary line intersections.
	;
	(cond ((= t1 2) (setq l1 (cons pl1 l1)))
	      ((> t1 2)
	       (setq l1 (findlf b1 pl1 plane))
	       (unless l1 (return nil))		; no match, no touch
	       (setq y (getfacep b1))
	       (setq z (second l1) l1 (first l1))
	       (mapc
		(function (lambda (w)
				  (and (inall pts2 w y z)
				       (setq x
					     (addpoint (aref pts2 w) x)))))
		pl2)))
	(cond ((= t2 2) (setq l2 (cons pl2 l2)))
	      ((> t2 2)
	       (setq l2 (findlf b2 pl2 plane))
	       (unless l2 (return nil))		; no match, no touch
	       (setq y (getfacep b2))
	       (setq z (second l2) l2 (first l2))
	       (mapc
		(function (lambda (w)
				  (and	(inall pts1 w y	z)
					(setq x
					      (addpoint	(aref pts1 w) x)))))
		pl1)))
	;
	;       General case, n points vs m points.  Plane-to-plane and
	;	line-to-line contact are handled here.
	;	
	;	This code is executed very frequently and should be rewritten
	;	for better performance.
	;
	(mapc
	 (function
	  (lambda (a)
		  (mapc
		   (function
		    (lambda (b)
			    (prog (c aa bb cc dd)
				  (setq aa (aref pts1 (first a)))
				  (setq bb (aref pts1 (second a)))
				  (setq cc (aref pts2 (first b)))
				  (setq dd (aref pts2 (second b)))
				  (setq c nil)
				  ;       Check for collinear points
				  (cond ((and (= t1 2) (= t2 2))
					 (and (online aa cc dd)
					      (setq c t)
					      (setq x (addpoint aa x)))
					 (and (online bb cc dd)
					      (setq c t)
					      (setq x (addpoint bb x)))
					 (and (online cc aa bb)
					      (setq c t)
					      (setq x (addpoint cc x)))
					 (and (online dd aa bb)
					      (setq c t)
					      (setq x (addpoint dd x)))))
				  ;       Not collinear, compute intersection.
				  (and (null c)
				       (setq c (intersect aa bb cc dd))
				       (setq x (addpoint c x))))))
		   l2)))
	 l1)
	(and x (return (cons plane x)))
	(return nil)))
;
;      separate  --  given two arrays full of points, finds the
;		     separating plane (if any).  Also returns a
;		     list of points of each object found to be on
;		     this plane.  Scans points of each block for
;		     offside of	current	plane, fixes plane if it
;		     finds a bad point.
;
;	A return of NIL indicates a hit.
;	A return of T indicates a clean miss.
;	Otherwise returns
;		(PLANE PL1 PL2)
;
(defun separate	(pts1 pts2 cg1 cg2)
  (prog (plane p1 p2 p3 np1 np2 x1 x2 y z w pl1 pl2 n)
	;	If CGs are coincident, this is a hit.
	;	Otherwise compute vector between CGs.
	(when (v3zerop (setq x1 (v3diff cg2 cg1))) (return nil))
	(setq np1 (array-dimension pts1 0) ; vertex count of B1
	      np2 (array-dimension pts2 0) ; vertex count of B2	
	      p1 (maxp pts1 x1 nil)		; point of B1 furthest along X1
	      x2 (v3negate x1)	 		; reversed X1 vector.
	      p3 (maxp pts2 x2 nil)		; point of B2 furthest toward B1
	      y (list p1))
	;	Check if the plane normal to the vector between the
	;	CG's is a good separating plane.
	(when (separated pts1 pts2 p1 p3 x1)
	      (return-from separate t))	; no contact possible
	;	Cheap check failed, have to really compute separating plane.
	;	Look for the point other than P1 furthest along vector X1
	;	toward B2 but not on the line CG-P1.
	;	This normally takes only one iteration.
	(setq p2 (maxp pts1 x1 (list p1)))	; second point of B1 toward B2.
	(loop
	 (unless (v3zerop (v3cross (v3diff (aref pts1 p1)
					   (aref pts1 p2))
				   x1))
		 (return))			; good point, quit search
	 (setq y (cons p2 y))			; add bad point to reject list.
	 (unless (setq p2 (maxp pts1 x1 y)) ; try for a new P2.
		 (return-from separate nil)))
	;	We now have two points from B1.
	;	Find the point from B2 furthest along X toward B1.
	;	The three points define a plane.
	loop2   (cond ((null (setq plane	
				   (make-plane-from-points
				    (aref pts1 p1)
				    (aref pts1 p2)
				    (aref pts2 p3)))))
		      ((and (setq w (point-plane-test cg1 plane))
			    (setq y (point-plane-test cg2 plane))
			    nil))
		      ((and (eq w 'inside) (eq y 'outside))
		       (go newplane))
		      ((and (eq w 'outside) (eq y 'inside))
		       (setq w p1 )
		       (setq p1 p2) 
		       (setq p2 w)
		       (setq  plane (reverse-plane plane))
		       (go newplane)))
	(setq z (cons p3 z))
	(unless (setq p3 (maxp pts2 x2 z)) (return nil))
	(go loop2)
	newplane
	(setq pl1 nil pl2 nil n 0)
	loop3   (cond   ((= n np2) (setq n 0) (go loop4))
			((eq (setq x2 (point-plane-test (aref pts2 n) plane))
			     'on)
			 (setq pl2 (cons n pl2)))
			((eq x2 'inside)
			 (setq x2
			       (make-plane-from-points (aref pts1 p1)
						       (aref pts1 p2)
						       (aref pts2 n)))
			 (or (checkplane x2 cg1 cg2) (return nil))
			 (setq plane x2 p3 n)
			 (go newplane)))
	(setq n (1+ n))
	(go loop3)
	loop4   (cond ((= n np1) (return (list plane pl1 pl2)))
		      ((eq (setq x2 (point-plane-test (aref pts1 n) plane))
			   'on)
		       (setq pl1 (cons n pl1)))
		      ((eq x2 'inside)) ; if x is inside, skip rest
		      ((checkplane (setq x2
					 (make-plane-from-points
					  (aref pts1 n)
					  (aref pts1 p2)
					  (aref pts2 p3)))
				   cg1
				   cg2)
		       (setq plane x2 p1 n)
		       (go newplane))
		      ((checkplane (setq x2
					 (make-plane-from-points
					  (aref pts1 p1)
					  (aref pts1 n)
					  (aref pts2 p3)))
				   cg1
				   cg2)
		       (setq plane x2 p2 n)
		       (go newplane))
		      ((return nil)))
	(setq n (1+ n))
	(go loop4)))
;
;     maxp  --	 get point from	P farthest in direction	of vector V.
;
;     Members of L are not eligible.
;
(defun maxp (p v l)
  (prog (best x np)
	(setq np (array-dimension p 0))	; number of points in P
	(setq best nil)
	(do ((i 0 (1+ i)))
	    ((= i np) nil)
	    (cond ((member i l))	; ignore if member of L
		  ((null best)
		   (setq best
			 (cons i (v3dot (aref p i) v))))	; dist in dir of V
		  ((> (setq x (v3dot (aref p i) v))
		      (rest best))
		   (setq best (cons i x)))))
	(unless best (return nil))
	(return (first best))))			; return index of best point
;
;	separated  --  test if blocks cannot touch because furthest
;		       points are on opposite sides of plane normal
;		       to CG1-CG2 vector.
;	
;	We compute the vector between the two points and see if it is
;	in the proper direction.  If not, the blocks cannot touch.
;	T indicates touch not possible.
;
(defun separated (pts1 pts2 pnt1 pnt2 vec)
  ;	Compute vector between furthest points, dot with desired dir.
  (<
   (v3dot
    (v3diff (aref pts1 pnt1)
	    (aref pts2 pnt2))
    vec)
   mtol1))
;
;      checkplane  --  check if	plane cuts between two centers of gravity.
;
;      Normal must point from cg1 size to cg2 side.
;
(defun checkplane (plane cg1 cg2)
  (and (eq 'inside (point-plane-test cg1 plane))
       (eq 'outside (point-plane-test cg2 plane))))
;
;      online  --  is point A on line B-C?
;
(defun online (a b c)
  (prog (d md me)
	(and (v3zerop (setq d (v3diff a b))) (return t))
	(and (> (setq md (v3mag d))
		(+ tol1 (setq me (v3mag (v3diff c b)))))
	     (return nil))
	(return (v3zerop (v3diff c
			       (v3plus b
				      (v3scale d
					      (/ me md))))))))
;
;      findlf  --  find	lines, faces of	B that lie on plane P
;
;	The vertices of B in list PL, and no others from B, are known
;	to lie in plane P. 
;
(defun findlf (b pl p)
  (prog (f facel lv x first)
	(setq facel (getfacel b))	; get face list of B.
	(setq f (matchface b pl p))	; which face?
	(unless f (return nil))		; fails if no match
	(setq lv (aref facel f 0))	
	(setq x nil first (first lv))
	loop  (cond ((rest lv)
		     (setq x (cons (list (first lv) (second lv)) x))
		     (setq lv (rest lv))
		     (go loop))
		    ((return (list (cons (list (first lv) first) x)
				   (aref facel f 1)))))))
;
;	matchface  --  find which face of B matches point list PL.
;
;	We have found at least three points from the same block in
;	plane P.  We also know, supposedly, that the plane does not
;	penetrate the block and that the block is a convex polyhedron.
;	From this we would expect that all the points would belong to
;	the same face and that this face would be in plane P.
;
;	Because of roundoff and tolerance errors, the possibility exists
;	that two blocks very close to touching may generate unexpected
;	situations here.  The most likely cause of trouble is a block 
;	with either a very narrow face or a very oblique angle.  In
;	such a case we may get two faces which appear to be in contact
;	with the plane.
;
(defun matchface (b pl p
		    &aux facel cnt)
  (declare (ignore p))			; don't seem to need plane info
  (assert (> (length pl) 2))		; must have 3 to match face
  (setq facel (getfacel b))		; get face point list of B.
  ;	Examine faces.  Is there some face that accounts for all the
  ;	points?
  (setq cnt (array-dimension facel 0))	; number of faces
  (do ((n 0 (1+ n)))			; iterate over faces
      ((= n cnt) nil)
      (when (subsetp pl (aref facel n 0)); when this face matches
	    (return-from matchface n))) ; success
  ;	That didn't work.  Abnormal.  Recoverable error for now.
  (cerror "Continue" "Matchface could not match.")
  nil)
;
;      inall  --  is point P on	inside of all faces in list FL?
;
(defun inall (pts p fcs	fl)
  (map nil
       (function
	(lambda (facenum)		; test this face
		(when (< tol1 (point-plane-distance 
			       (aref pts p) 
			       (aref fcs facenum)))
		      (return-from inall nil))))	; found bad point
       fl)			; map over all faces
  t)				; true if no bad points.
;
;      addpoint	 --  add point P to list L unless already there.
;
;	Inefficient.
;
(defun addpoint	(p l)
  (prog (x) (setq x l)
	loop (cond ((null x) (return (cons p l)))
		   ((v3zerop (v3diff p (first x))) (return l)))
	(setq x (rest x))
	(go loop)))
;
;      intersect  --  does A-B intersect C-D?		   
;
;      If so, where?
;
(defun intersect (a b c	d)
  (prog (ab mab pdc pdd pti ps)
	(setq ab (v3diff b a) mab (v3mag ab))
	(setq pdc (/  (v3mag (v3cross (v3diff c a ) ab)) mab))
	(and (< (abs pdc) tol1) (return nil))
	(setq pdd (/  (v3mag (v3cross (v3diff d a) ab)) mab))
	(and (< (abs pdd) tol1) (return nil))
	(setq ps (+ pdc pdd))
	(setq pti (v3plus (v3scale c (/  pdd ps))
			 (v3scale d (/  pdc ps))))
	(return (and (online pti a b)
		     (online pti c d)
		     pti))))
;
;	delete-collision-relationships  --  delete relationships between blocks
;
;	Called when something is moved or deleted.
;	All relevant touching and supported-by relationships must go.
;
(defun delete-collision-relationships (b)
  (mapc #'context-delete (context-lookup (list 'touches b '* '*)))
  (mapc #'context-delete (context-lookup (list 'touches '* b '*))))
