;
;	BUILD II  --  a blocks world
;
;						John Nagle
;						Version 1.5 of 5/12/87
;
;	AML (A Manufacturing Language) expression parsing.
;	Used to digest results from AML robot command execution.
;
;
(defconstant lparen #\()		; avoid editor trouble
(defconstant rparen #\))		; 
;
;	parse-aml-expression  --  convert AML expression to S-expression
;
;	Input is a string, containing one of the following:
;
;	<aml-expression> ::= <number> | <aggregate> | <S-msg> | <msg>
;	<number> ::= <nonnegative-number> | "-" <nonnegative-number>
;	<nonnegative-number> ::= <integer> | <real>
;	
;	<aggregate> ::= "<" <value-list> ">"
;	<value-list> ::= list of 0 or more of <value> separated by ",".
;	<value> ::= <number>
;	<S-msg> ::= "(" <S-msg-item>* ")"
;	<S-msg-item> ::= <number> | <atom> | <S-msg>
;	
;	
(defun parse-aml-expression (s
			     &aux (ch nil) (pos 0) (slen (length s)))
  (labels
   (
    ;
    ;	chget  -- get next character
    ;
    (chget nil
	   (setq ch
		 (if (< pos slen)
		     (char s pos) nil))		; return NIL after end
	   (incf pos))
    ;
    ;	ignore-whitespace  -- skips over blanks
    ;
    (ignore-whitespace nil
		       (loop
			(unless (eq #\Space ch) (return))
			(chget)))
    ;
    ;	syntax-error  --  report syntax error
    ;
    (syntax-error nil
		  (error "Syntax error in AML value ~a at char ~a" s ch))
    ;
    ;	parse-symbol  --  parse a symbol
    ;
    (parse-symbol 
     (&aux (ss (make-array 20
			   :element-type 'string-char
			   :adjustable t
			   :fill-pointer 0)))
     (loop
      (unless (or (alpha-char-p ch)		; non-alpha, end
		  (eq ch #\-)
		  (eq ch #\_))
	      (return (intern ss)))		; return interned symbol
      (vector-push-extend (char-upcase ch) ss)	; alpha, add this char
      (chget)))
    ;
    ;	parse-s-msg  --  parse string containing S-expression
    ;
    ;	An S-msg is a string with a form similar to an S-expression.
    ;	We generate these in our AML programs.
    ;
    (parse-s-msg
     (&aux items)
     (unless (eq ch lparen) (syntax-error)); must begin with
     (chget)					; use up 
     (loop
      (ignore-whitespace)			; if any
      (cond
       ((null ch) (syntax-error))		; premature end
       ((eq ch rparen) (return))		; normal end of s-msg
       ((alpha-char-p ch)			; symbol
	(push (parse-symbol) items))	; parse it
       ((or (digit-char-p ch)		; number
	    (eq ch #\-)
	    (eq ch #\.))
	(push (parse-number) items))	; parse one number
       ((eq ch #\<)			; AML aggregate
	(push (parse-aggregate) items))
       ((eq ch lparen )			; sublist 
	(push (parse-s-msg) items))		; do sublist
       (t (syntax-error))))			; invalid
     (chget)				; use up rparen
     (return-from parse-s-msg (reverse items)))
    ;
    ;	parse-aggregate  --  parse string containing numeric AML aggr.
    ;
    (parse-aggregate 
     (&aux items)
     (unless (eq ch #\<) (syntax-error)); must begin with <
     (chget)				; use up <
     (loop
      (ignore-whitespace)		; if any
      (cond
       ((null ch) (syntax-error))	; premature end
       ((or (digit-char-p ch)		; number
	    (eq ch #\-)
	    (eq ch #\.))
	(push (parse-number) items)	; parse one number
	;	Number must be followed by ',' or '>'.
	(ignore-whitespace)		; skip any blanks
	(cond
	 ((eq ch #\,) (chget))		; separator, OK
	 ((eq ch #\>) (return))		; end of aggregate	
	 (t (syntax-error))))		; invalid
       ((eq ch #\>) (return))))		; indicates null aggregate
     (chget)				; use up >
     (return-from parse-aggregate (reverse items)))
    ;
    ;	parse-number  --  parse a single number, integer or real.
    ;
    (parse-number
     (&aux (val 0) (sign 1) (scale 0) (decimal nil))
     (ignore-whitespace)
     (loop
      (cond ((eq ch #\-)			; if sign
	     (chget)	 			; use up
	     (setq sign (* sign -1)))
	    ((eq ch #\.)			; if decimal
	     (when decimal (syntax-error))
	     (chget) 				; use up
	     (setq decimal t))
	    ((and ch (digit-char-p ch))		; if digit
	     (when decimal (incf scale))	; tally digits after .
	     (setq val (+ (* val 10) (digit-char-p ch)))
	     (chget)) 				; use up
	    (t 		
	     (return (* sign (if (= scale 0)		; if decimal
				 val
				 (/ val (expt 10.0 scale))))))))))
   ;
   ;	Main
   (chget)				; get first character
   (ignore-whitespace)			; ignore any lead whitespace
   (cond ((null ch)			; end of string already?
	  nil)				; return NIL.
	 ((eq ch lparen)		; if begins with left paren
	  (parse-s-msg))		; S-msg
	 ((eq ch #\<)			; if begins with <
	  (parse-aggregate))	; parse aggregate of numbers.
	 ((or (digit-char-p ch)		; if begins with digit
	      (eq ch #\-)		; or -
	      (eq ch #\.))		; or .
	  (parse-number))		; parse as number
	 (t s))))			; otherwise return string.
