/* 
 *  proof - An interactive proofreader (terminal version)
 *  
 *  Copyright (c) 19{89-91} by Craig Latta. All Rights Reserved.
 *  
 *  Permission to use, copy, and/or distribute for any purpose and
 *  without fee is hereby granted, provided that both the above copyright
 *  notice and this permission notice appear in all copies and derived works.
 *  Fees for distribution or use of this software or derived works may only
 *  be charged with express written permission of the copyright holder.
 *  This software is provided ``as is'' without express or implied warranty.
 *  
 *  FILE:		rules.c
 *  
 *  DESCRIPTION:	The rules and rule packages of the left-associative
 *                      parser used to check grammar in proof.
 *  
 */

#include  <errno.h>
#include  <stdio.h>
#include  <string.h>
#include  <ctype.h>
#include  <sys/types.h>
#include  <sys/stat.h>
#include  <sys/file.h>
#include  <malloc.h>
#include  <stuff.h>

#include  "gdbm.h"
#include  "proof.h"


/*      Global variables        */


/*	Internal prototypes	*/


/* 
 *  
 *  FUNCTION      void combine(STATE *sentence, lexentry *next)
 *  
 *  PURPOSE       takes a sentence start and a next lexeme, and forms the
 *                cartesian product of all readings of the sentence start 
 * 	and all readings of the next lexeme. for each pair, combine()
 * 	applies the first element of the first member (i.e. the
 * 	rule package) to the second element of the first and second
 * 	members (i.e., the expressions, each consisting of a surface 
 * 	and a category). 
 *  
 *  INPUT         STATE    *sentence   -    the current sentence
 *                lexentry *next       -    the (evaluated) next lexeme
 *  
 *  OUTPUT        for each successful combination, combine() builds a history 
 *                section. If it fails on any member of a set of pairs,
 *                combine()
 * 	returns a suitable error message and the last pair.
 *  
 *  SIDE EFFECTS  none.
 * 
 */

void
  combine(start, next)
LEXENTRY  *start, *next;
{
  int   index;
  RULE  rule;
  
  /* pass control to the rule package indicated by the scan history */
  index = history[curr_deriv - 1].rules[0];
  rule = history[curr_deriv - 1].rules[index];
  
  /* diagnostic follows */
  printf("\n[%d", curr_deriv);
  for (index = 0; index < backtracked; index++)
    printf("'");
  printf("]: ");

  switch(rule)
    {
    case ADD_MINUS_NOM:
      printf("add minus nominative\n");
      print_status(start, next);
      p_add_minus_nom(start, next);
      break;

    case ADD_MINUS_VERB:
      printf("add minus verb\n");
      print_status(start, next);
      p_add_minus_verb(start, next);
      break;

    case DET_PLUS_ADJ:
      printf("determiner plus adjective\n");
      print_status(start, next);
      p_det_plus_adj(start, next);
      break;

    case DET_PLUS_NOUN:
      printf("determiner plus noun\n");
      print_status(start, next);
      p_det_plus_noun(start, next);
      break;
      
    case FVERB_PLUS_MAIN:
      printf("finite verb plus main clause\n");
      print_status(start, next);
      p_fverb_plus_main(start, next);
      break;
     
    case FVERB_PLUS_NOM:
      printf("finite verb plus nominative\n");
      print_status(start, next);
      p_fverb_plus_nom(start, next);
      break;

    case NOM_PLUS_FVERB:
      printf("nominative plus finite verb\n");
      print_status(start, next);
      p_nom_plus_fverb(start, next);
      break;
 
    case NOUN_PLUS_PNM:
      printf("noun plus post-nominal phrase\n");
      print_status(start, next);
      p_noun_plus_pnm(start, next);
      break; 
      
    case NOM_PLUS_RELPRO:
      printf("nominative plus relative pronoun\n");
      print_status(start, next);
      p_nom_plus_relpro(start, next);
      break;
 
    case PREP_PLUS_NP:
      printf("preposition plus noun phrase\n");
      print_status(start, next);
      p_prep_plus_np(start, next);
      break; 
      
    case START:
      printf("start\n");
      print_status(start, next);
      p_start(start, next);
      break;
      
    case START_MINUS_RELCL:
      printf("start minus relative clause\n");
      print_status(start, next);
      p_start_minus_relcl(start, next);
      break;

    case START_MINUS_SUBCL:
      printf("start minus subclause\n");
      print_status(start, next);
      p_start_minus_subcl(start, next);
      break;

    case TOP_MINUS_MAIN:
      printf("top minus main clause\n");
      print_status(start, next);
      p_top_minus_main(start, next);
      break;
      
    case WH_PLUS_VERB:
      printf("wh-phrase plus verb\n");
      print_status(start, next);
      p_wh_plus_verb(start, next);
      break;

      default:
      printf("Scan history contains an invalid rule number: %d.\n", rule);
      return;
    }
  
} /* end: combine() : finished */


/* *** RULE PACKAGES *** */

/* 
 *  FUNCTION      extern void  p_add_minus_nom(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of some other rules.
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_nom_plus_fverb()
 */

extern void
  p_add_minus_nom(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  NOM_PLUS_FVERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else {
	fprintf(stderr, "   Rule package 'add minus nominative' throws out rule 'nominative plus finite verb'.\n");
	if (!backtracked)
	  {
	    strcpy(error, "   Rule package 'add minus nominative' has failed. You tried to follow the add phrase '");
	    strcat(error, start->surface);
	    strcat(error, "' with '");
	    strcat(error, next->surface);
	    strcat(error, "' which is ungrammatical, since it cannot start a nominative phrase.\n");
	  }
	curr_condition = INVALID;
      }
    }
  
} /* end: extern void  p_add_minus_nom(LEXENTRY *start, *next): f template */

/* 
 *  
 *  FUNCTION      extern void  p_add_minus_verb(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_add_minus_verb().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_fverb_plus_main()
 *                r_add_minus_verb()
 *                r_complete()
 */

extern void
  p_add_minus_verb(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  FVERB_PLUS_MAIN) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else  {
	fprintf(stderr, "   Rule package 'add minus verb' throws out rule 'finite verb phrase plus nominative'.\n");
      }	
    }

  failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == ADD_MINUS_VERB)
	failed = 1;
    }
  if (!failed)  
    /*
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_add_minus_verb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == ADD_MINUS_VERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'add minus verb' throws out rule 'add minus verb'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == COMPLETE)
	failed = 1;
    }
  if (!failed)  
    /*
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_complete(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == COMPLETE) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'add minus verb' throws out rule 'complete'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'add minus verb' has failed. You tried to follow the add phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot properly start a verb phrase.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: extern void  p_add_minus_verb(LEXENTRY *start, *next): f template */


/* 
 *  
 *  FUNCTION      void  p_det_plus_adj(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of the rule r_det_plus_adj().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        A (LEXENTRY *) indicating...
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start
 * 
 *  CHILDREN      r_det_plus_noun()
 *                r_det_plus_adj()
 */

void
  p_det_plus_adj(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0; 
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in 
     * derivation-package space 
     */
    {
      r_det_plus_noun(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'determiner plus adjective' throws out rule 'determiner plus noun'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == DET_PLUS_ADJ)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_det_plus_adj(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_ADJ)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'determiner plus adjective' throws out rule 'determiner plus adjective'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'add minus verb' has failed. You tried to follow the add phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot properly start a verb phrase.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: LEXENTRY *p_det_plus_adj(LEXENTRY *start, *next) : finished */


/* 
 *  
 *  FUNCTION      void  p_det_plus_noun(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_det_plus_noun().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_fverb_plus_main()
 *                r_top_minus_main()
 *                r_nom_plus_fverb()
 *                r_start_minus_relcl()
 * 	          r_fverb_plus_nom()
 *                r_complete()
 */

void
  p_det_plus_noun(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  

  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'finite verb plus main'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_top_minus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'top minus main'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_nom(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'finite verb plus nominative'.\n");
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == START_MINUS_RELCL)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_start_minus_relcl(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == START_MINUS_RELCL)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'start minus relative clause'.\n");
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'nominative plus fverb'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == COMPLETE)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_complete(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == COMPLETE)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'determiner plus noun' throws out rule 'complete'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'determiner plus noun' has failed. You tried to follow the determiner phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "', which is not grammatically correct.\n");
	    }
	  
	  curr_condition = INVALID;
	}
    }
  
} /* end: LEXENTRY *p_det_plus_noun(LEXENTRY *start, *next) : finished */


/*
 *  
 *  FUNCTION      void  p_fverb_plus_main(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_fverb_plus_main().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_det_plus_noun()
 *                r_prep_plus_np()
 *                r_nom_plus_fverb()
 *                r_fverb_plus_main()
 */

void
  p_fverb_plus_main(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;

 for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_det_plus_noun(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'finite verb plus main' throws out rule 'determiner plus noun'.\n");
    }
  
  failed = 0;

  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == PREP_PLUS_NP)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_prep_plus_np(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == PREP_PLUS_NP)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'finite verb plus main' throws out rule 'preposition plus noun phrase'.\n");
	}
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'finite verb plus main' throws out rule 'nominative plus finite verb'.\n");
	}
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'finite verb plus main' throws out rule 'finite verb phrase plus main clause'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == COMPLETE)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_complete(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == COMPLETE)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'finite verb plus main' throws out rule 'complete'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'finite verb plus main clause' has failed. You tried to follow the phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "', which cannot continue a finite verb phrase.");
	      curr_condition = INVALID;
	    }
	}
    }

} /* end: LEXENTRY *p_fverb_plus_main(LEXENTRY *start, *next): f template */


/* 
 *  
 *  FUNCTION      extern void  p_fverb_plus_nom(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_fverb_plus_nom().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_wh_plus_verb()
 *                r_add_minus_verb()
 * 
 */

extern void
  p_fverb_plus_nom(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == WH_PLUS_VERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_wh_plus_verb(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  WH_PLUS_VERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'finite verb plus nominative' throws out rule 'wh-phrase plus verb'.\n");
    }      
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    if (history[curr_deriv].rules[index] == ADD_MINUS_VERB)
      failed = 1;
  
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_add_minus_verb(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  ADD_MINUS_VERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'finite verb plus nominative' throws out rule 'add minus verb'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'add minus verb' has failed. You tried to follow the add phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot properly start a verb phrase.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: extern void  p_fverb_plus_nom(LEXENTRY *start, *next): f template */


/* 
 *  
 *  FUNCTION      void  p_nom_plus_fverb(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_nom_plus_fverb().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_add_minus_verb()
 *                r_fverb_plus_main()
 *                r_complete()
 */

void
  p_nom_plus_fverb(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == ADD_MINUS_VERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_add_minus_verb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == ADD_MINUS_VERB)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'add minus verb'.\n");
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else 
	fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'finite verb plus main'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == COMPLETE)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_complete(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == COMPLETE)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'complete'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule 'nominative plus finite verb phrase' has failed. You tried to follow the phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "', which cannot continue a finite verb phrase.");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: LEXENTRY *p_nom_plus_fverb(LEXENTRY *start, *next) : finished */


/* 
 *  
 *  FUNCTION      extern void  p_nom_plus_relpro(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_nom_plus_relpro().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      (identical to p_det_plus_noun)
 *                r_fverb_plus_main()
 *                r_top_minus_main()
 *                r_nom_plus_fverb()
 * 	          r_fverb_plus_nom()
 * 
 */

extern void
  p_nom_plus_relpro(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;


  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'finite verb plus main'.\n");
    }
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_top_minus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'top minus main'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_nom(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
        fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'finite verb plus nominative'.\n");
    }
    
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'nominative plus finite verb' throws out rule 'nominative plus fverb'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'nominative plus finite verb' has failed. You tried to follow the determiner phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "', which is not grammatically correct.\n");
	    }
	  
	  curr_condition = INVALID;
	}
    }
  
} /* end: extern void  p_nom_plus_relpro(LEXENTRY *start, *next): f template */


/* 
 *  FUNCTION      extern void  p_prep_plus_np(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of rule r_prep_plus_np().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_det_plus_noun()
 *                r_fverb_plus_main()
 *                r_complete()
 */

extern void
  p_prep_plus_np(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;

  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	    failed = 1;
	}
  if (!failed)  
    /*
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_det_plus_noun(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'preposition plus noun phrase' throws out rule 'determiner plus noun'.\n");
    }

  failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == FVERB_PLUS_MAIN)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_main(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  FVERB_PLUS_MAIN) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'preposition plus noun phrase' throws out rule 'finite verb phrase plus main clause'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == COMPLETE)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_complete(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  COMPLETE) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else {
	fprintf(stderr, "   Rule package 'preposition plus noun phrase' throws out rule 'complete'.\n");
	if (!backtracked)
	  {
	    strcpy(error, "   Rule package 'preposition plus noun phrase' has failed. You tried to follow the prepositional phrase '");
	    strcat(error, start->surface);
	    strcat(error, "' with '");
	    strcat(error, next->surface);
	    strcat(error, "' which is ungrammatical, since it cannot begin a main clause.\n");
	  }
	curr_condition = INVALID;
      }
    }
  
} /* end: extern void  p_prep_plus_np(LEXENTRY *start, *next): f template */


/* 
 *  FUNCTION      extern void  p_noun_plus_pnm(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of r_noun_plus_pnm().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_prep_plus_np()
 */

extern void
  p_noun_plus_pnm(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  

  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == PREP_PLUS_NP)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_prep_plus_np(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  PREP_PLUS_NP) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'noun plus post-nominal phrase' throws out rule 'preposition plus noun phrase'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'noun plus post-nominal phrase' has failed. You tried to follow the noun phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since <reason>.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: extern void  p_noun_plus_pnm(LEXENTRY *start, *next): f template */


/* 
 *  
 *  FUNCTION      extern void  p_top_minus_main(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of r_top_minus_main
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_fverb_plus_nom()
 * 
 */

extern void
  p_top_minus_main(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_nom(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  FVERB_PLUS_NOM) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'topicalized main constituent' throws out rule 'finite verb plus nominative'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'add minus verb' has failed. You tried to follow the add phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot properly start a verb phrase.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} /* end: extern void  p_top_minus_main(LEXENTRY *start, *next): f template */


/* 
 *  FUNCTION      void  *p_start(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the first two expressions of a sentence.
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start
 * 
 *  CHILDREN      r_det_plus_adj()
 *                r_det_plus_noun()
 *                r_top_minus_main()
 *                r_fverb_plus_nom()
 * 	          r_nom_plus_fverb()
 */

void
  p_start(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == DET_PLUS_ADJ)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_det_plus_adj(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_ADJ)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else       
	fprintf(stderr, "   Rule package 'start' throws out rule 'determiner plus adjective'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_det_plus_noun(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == DET_PLUS_NOUN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else       
	fprintf(stderr, "   Rule package 'start' throws out rule 'determiner plus noun'.\n");
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == NOM_PLUS_RELPRO)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_relpro(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_RELPRO)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else       
	fprintf(stderr, "   Rule package 'start' throws out rule 'nominative plus relative pronoun'.\n");
    }

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_top_minus_main(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == TOP_MINUS_MAIN)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else       
	fprintf(stderr, "   Rule package 'start' throws out rule 'top minus main'.\n");
    }    

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	failed = 1;
    }
  if (!failed)
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_fverb_plus_nom(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == FVERB_PLUS_NOM)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else       
	fprintf(stderr, "   Rule package 'start' throws out rule 'finite verb phrase plus nominative'.\n");
    }   

  failed = 0;
  
  for (index = 1; index <= history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'start' throws out rule 'nominative plus fverb'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'start' has failed. You tried to follow the phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "', which is ungrammatical.");
	    }
	  
	  curr_condition = INVALID;
	}
    }
  
  /* ... etc. */
  
} /* end: LEXENTRY *p_start(LEXENTRY *start, *next) : finished*/


/* 
 *  FUNCTION      extern void  p_start_minus_relcl(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of r_start_minus_relcl().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_nom_plus_fverb()
 *                r_add_minus_nom()
 */

extern void
  p_start_minus_relcl(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
    
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  NOM_PLUS_FVERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'start minus relative clause' throws out rule 'nominative plus finite verb'.\n");
    }
  
  failed = 0;

  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == ADD_MINUS_NOM)
	    failed = 1;
	}
  if (!failed)  
    /*
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_add_minus_nom(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == ADD_MINUS_NOM) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'start minus relative clause' throws out rule 'add minus nominative'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'start minus relative clause' has failed. You tried to follow the start phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot start a relative clause.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} 
/* 
 * end: extern void  p_start_minus_relcl(LEXENTRY *start, *next): f template 
 */


/* 
 *  FUNCTION      extern void  p_start_minus_subcl(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of r_start_minus_subcl().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN    
 *                r_nom_plus_fverb()
 *                r_add_minus_nom()
 */

extern void
  p_start_minus_subcl(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  

  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB)
	    failed = 1;
	}
  if (!failed)  
    /*
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_nom_plus_fverb(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == NOM_PLUS_FVERB) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	fprintf(stderr, "   Rule package 'start minus subclause' throws out rule 'nominative phrase plus finite verb'.\n");
    }
  
  failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
    {
      if (history[curr_deriv].rules[index] == ADD_MINUS_NOM)
	failed = 1;
    }
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_add_minus_nom(start, next);
      index = history[curr_deriv].rules[0];
      if (history[curr_deriv].rules[index] == ADD_MINUS_NOM) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else
	{
	  fprintf(stderr, "   Rule package 'start minus subclause' throws out rule 'add minus nominative'.\n");
	  if (!backtracked)
	    {
	      strcpy(error, "   Rule package 'start minus subclause' has failed. You tried to follow the start phrase '");
	      strcat(error, start->surface);
	      strcat(error, "' with '");
	      strcat(error, next->surface);
	      strcat(error, "' which is ungrammatical, since it cannot begin a subphrase.\n");
	    }
	  curr_condition = INVALID;
	}
    }
  
} 
/* 
 * end: extern void  p_start_minus_subcl(LEXENTRY *start, *next): f template 
 */


/* 
 *  FUNCTION      extern void  p_wh_plus_verb(start, next)
 *  
 *  PURPOSE       To apply all the possibly successful rule packages 
 *                to the output of r_wh_plus_verb().
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 *  
 *  SIDE EFFECTS  The control flow is altered (heh), as is the given start.
 * 
 *  CHILDREN      r_start_minus_subcl()
 */

extern void
  p_wh_plus_verb(start, next)
LEXENTRY    *start, *next;
{
  int  index, failed = 0;
  
  for (index = 1; index < history[curr_deriv].rules[0]; index++)
	{
	  if (history[curr_deriv].rules[index] == START_MINUS_SUBCL)
	    failed = 1;
	}
  if (!failed)  
    /* 
     * this rule hasn't yet failed from this point in
     * derivation-package space 
     */
    {
      r_start_minus_subcl(start, next);
      if (history[curr_deriv].rules[history[curr_deriv].rules[0]] == 
	  START_MINUS_SUBCL) 
	/* the rule has suceeded, and control will be passed to its package */
	return;
      else {
	fprintf(stderr, "   Rule package 'wh-phrase plus verb' throws out rule 'start minus subclause'.\n");
	if (!backtracked)
	  {
	    strcpy(error, "   Rule package 'wh-phrase plus verb' has failed. You tried to follow the wh-phrase '");
	    strcat(error, start->surface);
	    strcat(error, "' with '");
	    strcat(error, next->surface);
	    strcat(error, "' which is ungrammatical, since the package has no branches.\n");
	  }
	curr_condition = INVALID;
      }
    }
  
} /* end: extern void  p_wh_plus_verb(LEXENTRY *start, *next): f template */



/* *** RULES *** */

/*
 * FUNCTION      void  r_add_minus_nom(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine an add phrase with a nominative phrase.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  n_seg is the the first next segment
 *
 *      	 The input condition of this rule says:
 *                  the start must contain an SC or SD segment.
 *
 *        	 The output specification of this rule says:
 *                  the result is the start with n_seg inserted as
 *                  the third segment, followed by SD if it isn't already.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_add_minus_nom(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i = 0;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start */
      for (index = 1; index <= start->segments[s_cat_index][0]; index++)
	if ((start->segments[s_cat_index][index] == SC) ||
	    (start->segments[s_cat_index][index] == SD))
	  i = 1;
      if (i)
	{
	  /* 
	   * allow for each interpretation of "next", paying attention to
	   * the size of result.segments.
	   */
	  for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	       (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	    {
	      n_seg = next->segments[n_cat_index][1];
	      
	      /* denote success */
	      curr_condition = VALID;
	      history[curr_deriv].rules[0] +=1;
	      index = history[curr_deriv].rules[0];
	      history[curr_deriv].rules[index] =
		ADD_MINUS_NOM;
	      
	      /* output spec */
	      copy_category(result.segments[r_cat_index],
			    start->segments[s_cat_index]);
	      
	      splice[0] = 1;
	      splice[1] = n_seg;
	      
	      splice_category(result.segments[r_cat_index], 3, splice);
	      if (result.segments[r_cat_index][4] != SD)
		{
		  splice[1] = SD;
		  splice_category(result.segments[r_cat_index], 4, splice);
		}		
	      
	      /* 
	       * increment r_cat_index at end of successful rule 
	       * processing. Failures are not recorded.
	       */
	      r_cat_index += 1;
	      
	      /* diagnostics follow */
	      printf("   Rule 'add minus nominative' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
	      if (r_cat_index == MAX_CATEGORIES)
		{
		  printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			 start->surface, next->surface);
		}
	    }
	}
      else
	{
	  fprintf(stderr, "   Rule 'add minus nominative' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
		  n_cat_index, next->surface, s_cat_index);
	} 
    }
  
  if (--r_cat_index == 0)  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'add minus nominative' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a nominative phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_add_minus_nom(LEXENTRY *start, *next) : finished */


/*
 * FUNCTION      void  r_add_minus_verb(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine an add phrase and a verb.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is set to the second segment of the start.
 * 
 *      	 The input condition of this rule says:
 *                  The first segment of start must be V, VI, or SH. The first 
 *                  segment of next must be V, HV, CA, M, or __CA.
 *                 
 *        	 The output specification of this rule says:
 *                  The result category is comprised of the first and last
 *                  segments of the start, and a start M, DO, HV, or B
 *                  valency is cancelled and replaced by the cdr of next,
 *                  unless next has a NOM valency, in which case it is replaced
 *                  by the cddr of next, or, if start has a
 *                  B valency in second position and next has an HV
 *                  segment in initial position and its second segment
 *                  is not B, in which case the B valency is cancelled
 *                  with the cddr of next and AG.
 *                  An A next segment cancels a start WH valency. 
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *                packages, the control flow is altered.
 */

void
  r_add_minus_verb(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i = 0, j = 0, k = 0;
  SEG       s_seg, n_seg;
  
  
  result.segments[0][0] = 0;

  /*  start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][2];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  n_seg = next->segments[n_cat_index][1];
	  
	  /* condition on next */
	  if ((n_seg == V) || (n_seg == M) || (n_seg == HV) || (n_seg == B) ||
	      (n_seg == CA) || (n_seg == _CA))
	    {
	      /* central rule logic may proceed */
	      
	      /* input spec */
	      if ((start->segments[s_cat_index][1] == V) ||
		  (start->segments[s_cat_index][1] == VI) ||
		  (start->segments[s_cat_index][1] == SH))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    ADD_MINUS_VERB;
		  
		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);

		  for (i = 1; i <= next->segments[n_cat_index][0]; i++)
		    {
		      if (next->segments[n_cat_index][i] == A)
			j = i;
		    }
		  for (i = 1; i <= result.segments[r_cat_index][0]; i++)
		    {
		      if (start->segments[s_cat_index][i] == WH)
			k = i;
		    }
		  if (j && k)
		    {
		      compact_category(result.segments[r_cat_index], k);
		      compact_category(next->segments[n_cat_index], j);
		    }
		  
		  if ((s_seg == M) || (s_seg == DO) || (s_seg == HV) ||
		      (s_seg == B))
		    {
		      for (index = 1; ((index <= 
					next->segments[n_cat_index][0]) &&
				       (next->segments[n_cat_index][index]
					!= NOM)); index++)
			;
		      
		      if (index != next->segments[n_cat_index][0] + 1)
			{
			  /* there is a NOM segment in next */
			  compact_category(result.segments[r_cat_index], 2);
			  compact_category(next->segments[n_cat_index], 1);
			  compact_category(next->segments[n_cat_index], 1);
			  splice_category(result.segments[r_cat_index], 2,
					  next->segments[n_cat_index]);
			}
		      
		      else if ((next->segments[n_cat_index][1] == HV) &&
			       (next->segments[n_cat_index][2] != B))
			{
			  compact_category(result.segments[r_cat_index], 2);
			  compact_category(next->segments[n_cat_index], 1);
			  compact_category(next->segments[n_cat_index], 1);
			  copy_category(splice, next->segments[n_cat_index]);
			  splice[++splice[0]] = AG;
			  splice_category(result.segments[r_cat_index], 2,
					  splice);
			}
		      
		      else
			{
			  compact_category(result.segments[r_cat_index], 2);
			  compact_category(next->segments[n_cat_index], 1);
			  splice_category(result.segments[r_cat_index], 2,
					  next->segments[n_cat_index]);
			}
		    }
		  
		  result.segments[0][0] += 1;
		  
		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1; 

		  /* diagnostics follow */
		  printf("   Rule 'add minus verb' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been \
reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'add minus verb' throws \
out meaning %d of next lexeme '%s', uses start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked))  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'add minus verb' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot continue an add phrase.");
    }
  
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;

      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_add_minus_verb(LEXENTRY *start, *next) : finished template */


/*
 * FUNCTION      void  r_complete(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a declarative or interrogative sentence and
 *               a period, semicolon, question mark, or exlamation point,
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  If start consists of a single segment, s_seg is set to it.
 * 
 *      	 The input condition of this rule says:
 *                  The start is valid if it doesn't not consist of a
 *                  single segment SH, S_H, P1, PH, or P_H, or if it
 *                  contains multiple segments, none of which is S, C, SC,
 *                  WS, WU, AG, A, NP, or P.
 *
 *        	 The output specification of this rule says:
 *                  the output is DECL if declarative, and INTERROG if
 *                  interrogative.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_complete(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index = 0, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i = 0;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      if (start->segments[s_cat_index][0] == 1)
	s_seg = start->segments[s_cat_index][1];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* condition on next */
	  if ((strcmp(next->surface, "?") == 0) ||
	      (strcmp(next->surface, "!") == 0) ||  
	      (strcmp(next->surface, ";") == 0) ||  
	      (strcmp(next->surface, ".") == 0))
	    {
	      /* central rule logic may proceed; set n_seg */
	      n_seg = next->segments[n_cat_index][1];
	      
	      /* input spec */
	      for (i = 1; i <= start->segments[s_cat_index][0]; i++)
		if ((start->segments[s_cat_index][i] == S) ||
		    (start->segments[s_cat_index][i] == C) ||
		    (start->segments[s_cat_index][i] == SC) ||
		    (start->segments[s_cat_index][i] == WS) ||
		    (start->segments[s_cat_index][i] == WU) ||  
		    (start->segments[s_cat_index][i] == AG) || 
		    (start->segments[s_cat_index][i] == A) ||   
		    (start->segments[s_cat_index][i] == NP) ||   
		    (start->segments[s_cat_index][i] == P))
		  index = 1;
	      
	      if (((--i == 1) &&
		  ((s_seg != SH) && (s_seg != S_H) && (s_seg != P1) &&
		   (s_seg != PH) && (s_seg != P_H))) ||
		  ((i != 1) && (index == 0)))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    COMPLETE;

		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);

		  result.segments[r_cat_index][0] = 1;
		  result.segments[r_cat_index][1] = n_seg;

		  result.segments[0][0] += 1;

		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'complete' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'complete' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }

  if (--r_cat_index == 0)  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'complete' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot end this sentence.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
     
} /* end: void r_complete(LEXENTRY *start, *next) : finished template */


/*
 *  
 *   FUNCTION      void  *r_det_plus_adj(LEXENTRY *start, *next)
 *  
 *   PURPOSE       To combine a determiner and the next lexeme, iff it is an 
 *                 adjective or a suitable beginning to an adjective phrase.
 *   
 *   DECL          For all interpretations of both the start and the next lexeme:
 *   
 *                 The selection specification of this rule says:
 * 
 * 		   If the category of the start consists of a single 
 * 		   segment x, then the local variable s_seg is set to x. 
 * 		   Otherwise the value of s_seg is set to the second segment 
 * 		   of the category of the start. If any categorization of 
 * 		   the next lexeme has more than one segment, then that it 
 * 		   denotes a not an adjective, and the rule fails for that 
 * 		   categorization. Otherwise, the local variable n_seg is 
 * 		   set to the first segment of that categorization (heh)
 *   
 * 		   The input condition of this rule says:
 * 		   1. the value of s_seg must be U, S, P, CA, or _CA, and 
 * 		      n_seg must be CA or _CA, or
 * 		   2. the value of s_seg must be __C, and n_seg must be _CA.
 * 		   3. the value of s_seg must be C, and n_seg must be CA.
 *      
 * 		   The output specification of this rule says:
 * 		   1. If s_seg is U, S, or P, then the output category is
 * 		      the category of the start.
 * 		   2. If s_seg is CA or _CA, then the output category 
 * 		      is the category of the start, except that s_seg is
 * 		      replaced by P.
 * 		   3. If s_seg is C or __C, then the 
 * 		      output category is the category of the start, except 
 * 		      that s_seg is replaced by S.
 *   
 *   INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                 and (evaluated) next lexeme.
 *   
 *   OUTPUT        none
 *   
 *   PARENTS
 *  
 *   SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *                 packages, the control flow is altered.
 */

void
  r_det_plus_adj(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  int       index, s_seg_index, s_cat_index, r_cat_index = 1, n_cat_index, i;
  SEG       s_seg, n_seg, s_len;
  
  
  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* 
	If the category of the start consists of a single segment x, 
	then the local variable s_seg is set to x...
	*/
      if (start->segments[s_cat_index][0] == 1)
	{
	  s_seg_index = 1;
	  s_seg = start->segments[s_cat_index][1];
	}
      else
	/* 
	  ...Otherwise the value of seg is set to the second segment 
	  of the current start category.
	  */
	{
	  s_seg_index = 2;
	  s_seg = start->segments[s_cat_index][2];
	}
      
      /* 
	allow for each interpretation of "next", paying attention to the 
	size of result.segments, and keeping in mind that if "next" is 
	has no categorizations with one segment then it is not an
	adj, and the rule fails.
	*/
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  if (next->segments[n_cat_index][0] != 1)
	    {
	      /* diagnostic follows */
	      printf("   Start meaning %d of rule 'determiner plus adj' throws out meaning %d of next lexeme '%s', as it is not an adjective.\n", s_cat_index, n_cat_index, next->surface);
	    }
	  
	  else
	    {
	      /* central rule logic may proceed; set n_seg */
	      
	      n_seg = next->segments[n_cat_index][1];
	      
	      /* input spec */
	      if ((((s_seg == U) || (s_seg == S) || 
		    (s_seg == P) || (s_seg == CA) || (s_seg == _CA)) &&
		   ((n_seg == CA) || (n_seg == _CA))) ||
		  
		  (((s_seg == __C)) &&
		   ((n_seg == _CA))) ||
		  
		  (((s_seg == C)) &&
		   ((n_seg == CA))))
		{
		  /* denote success */
		  curr_condition |= VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] = DET_PLUS_ADJ;
		  result.segments[0][0] += 1;
		  
		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);
		  if (!((s_seg == U) || (s_seg == S) || (s_seg == P)))
		    {
		      /* 
		       * first output case handled by the failure of the "if"
		       * condition, number of result segments is the same as
		       * start's in any case
		       */
		      if ((s_seg == CA) || (s_seg == _CA))
			result.segments[r_cat_index][s_seg_index] = P;
		      if ((s_seg == C) || (s_seg == __C))
			result.segments[r_cat_index][s_seg_index] = S;
		    }
		  
		  /* 
		    increment r_cat_index at end of successful rule 
		    processing. Failures are not recorded.
		    */
		  r_cat_index += 1;
		  
		  /* diagnostics follow */
		  printf("   Rule 'determiner plus adjective' combines meaning %d of determiner phrase '%s' with meaning %d of adjective '%s'.\n", s_cat_index, start->surface, n_cat_index, next->surface);
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been \
reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked))  /* there were no successes */
    /* this is, of course, very wimpy */
    {
      strcpy(error, "   Rule 'determiner plus adjective' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin an adjective phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  	
} /* end: void  r_det_plus_adj(LEXENTRY *start, *next) : finished */


/* 
 *  FUNCTION      void  r_det_plus_noun(LEXENTRY *start, *next)
 *  
 *  PURPOSE       To combine a determiner and the next lexeme, iff it is a 
 *                noun or a suitable beginning to a noun phrase.
 *  
 *  DECL          For all interpretations of both the start and the next lexeme:
 *   
 *                The selection specification of this rule says:
 * 	          If the category of the start consists of a single 
 * 	          segment x, then the local variable s_seg is set to x. 
 * 	          Otherwise the value of s_seg is set to the second segment 
 * 	          of the category of the start. If any categorization of 
 * 	          the next lexeme has more than one segment, then that it 
 * 	          denotes a not a noun, and the rule fails for that 
 * 	          categorization. Otherwise, the local variable n_seg is 
 * 	          set to the first segment of that categorization.
 * 
 * 	          The input condition of this rule says:
 * 	          1. the value of s_seg must be U, S, C, or WU, and 
 * 	             n_seg must be CH or C_H, or
 * 	          2. the value of s_seg must be U, S, __C, or WU, and 
 * 	             n_seg must be _CH or _C_H, or
 * 	          3. the value of s_seg must be U, P, CA, _CA, or WU, 
 * 	             and the category of next must be PH or P_H.
 *           
 * 	          The output specification of this rule says:
 * 	          1. If s_seg is WU, then if n_seg is CH, _CH, 
 * 	             C_H, or _C_H, then the output category is WS; 
 * 	             otherwise it is WP.
 * 	          2. If n_seg is CH or _CH, then the output category 
 * 	             is the category of the start minus
 * 	             s_seg, plus SH.
 * 	          3. If the category of next is C_H or _C_H, then the 
 * 	             output category is the category of the first 
 * 	             expression minus s_seg, plus S_H.
 *                4. If none of the above apply, the result has the
 *                   same segments as next
 * 
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                 and (evaluated) next lexeme.
 *  
 *  OUTPUT        none
 * 
 *  PARENTS
 *  
 *  SIDE EFFECTS  As with all rules and rule packages, the control flow
 *                 is altered.
 */

void
  r_det_plus_noun(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1, n_cat_index, i;
  SEG       s_seg, n_seg, s_len;  


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)  
    {
      /* 
	If the category of the start consists of a single segment x, 
	then the local variable s_seg is set to x...
	*/
      if (start->segments[s_cat_index][0] == 1)
	{
	  s_seg = start->segments[s_cat_index][1];
	  s_seg_index = 1;
	}
      else
	/* 
	  ...Otherwise the value of seg is set to the second segment 
	  of the current start category.
	  */
	{
	  s_seg = start->segments[s_cat_index][2];
	  s_seg_index = 2;
	}
      
      /* 
	allow for each interpretation of "next", paying attention to the size 
	of result.segments, and keeping in mind that if "next" is has no 
	categorizations with more than one segment, then it is not a noun, and 
	the rule fails
	*/
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]); 
	   n_cat_index++)  
	{
	  if (next->segments[n_cat_index][0] != 1)
	    {
	      /* diagnostic follows */
	      printf("   Start meaning %d of rule 'determiner plus noun' throws out meaning %d of next lexeme '%s'.\n", s_cat_index, n_cat_index, next->surface);
	    }
	  
	  else
	    {
	      /* central rule logic may proceed; set n_seg */
	      
	      n_seg = next->segments[n_cat_index][1];
	      
	      /* input spec */
	      if ((((s_seg == U) || (s_seg == S) || 
		    (s_seg == C) || (s_seg == WU)) &&
		   ((n_seg == CH) || (n_seg == C_H))) ||
		  (((s_seg == U) || (s_seg == S) || 
		    (s_seg == __C) || (s_seg == WU)) &&
		   ((n_seg == _CH) || (n_seg == _C_H))) ||
		  (((s_seg == U) || (s_seg == P) || 
		    (s_seg == CA) || (s_seg == _CA) ||
		    (s_seg == WU)) &&
		   ((n_seg == PH) || (n_seg == P_H))))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] = DET_PLUS_NOUN;

		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);
		  if (s_seg == WU)
		    {
		      if ((n_seg == CH) || 
			  (n_seg == _CH) ||
			  (n_seg == C_H) || 
			  (n_seg == _C_H))
			result.segments[r_cat_index][1] = WS;
		      else
			result.segments[r_cat_index][1] = WP;
		    }
		  else if
		    ((n_seg == CH) || (n_seg == _CH))
		      {
			for (index = s_seg_index; 
			     (index + 1) <= result.segments[r_cat_index][0]; 
			     index++)
			  result.segments[r_cat_index][index] =
			    result.segments[r_cat_index][index + 1];
			result.segments[r_cat_index][index] = SH;
			result.segments[r_cat_index][0] = index;
		      }
		  else if
		    ((n_seg == C_H) || (n_seg == _C_H))
		      {
			for (index = s_seg_index; 
			     (index + 1) <= result.segments[r_cat_index][0]; 
			     index++)
			  result.segments[r_cat_index][index] =
			    result.segments[r_cat_index][index + 1];
			result.segments[r_cat_index][index] = S_H;
			result.segments[r_cat_index][0] = index;
		      } 
		  else
		    copy_category(result.segments[r_cat_index],
				  next->segments[n_cat_index]);
		  
		  result.segments[0][0] += 1;
		  
		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'determiner plus noun' combines meaning %d of determiner '%s' with meaning %d of noun '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface);
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been reached, in rule 'determiner plus noun'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	      else
		{
		  /* diagnostic follows */
		  printf("   Start meaning %d, of rule 'determiner plus noun' throws out meaning %d of next lexeme '%s', as it is not a noun.\n", s_cat_index, n_cat_index, next->surface);
		} 
	    }
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked))  /* there were no successes */
    /* this is, of course, very wimpy */
    {
      strcpy(error, "   Rule 'determiner plus noun' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a noun phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }

} /* end: void  r_det_plus_noun(LEXENTRY *start, *next) : finished */


/* 
 *  FUNCTION     void  r_fverb_plus_main(LEXENTRY *start, *next)
 *  
 *  PURPOSE      To combine a finite verb phrase and the beginning of a
 *               main clause.
 *  
 *  DECL         For all interpretations of both the start and the next lexeme:
 *  
 *               The selection specification of this rule says:
 * 	         The first segment of the start must be V or VI, or the
 *               second segment must be A. If the start
 * 	         consists of a single segment x, then the local
 * 	         variable s_seg is set to x. Otherwise the value of
 * 	         s_seg is set to the second segment of the category of
 * 	         the start. The local variable n_seg is 
 * 	         set to the first segment of the next categorization.
 * 
 * 	         The input condition of this rule says:
 * 	           1. n_seg next must be either AG (and s_seg is AG), AV,
 *                    C, _C, CA, _CA, CH, C_H, _CH, _C_H, P, P1, P3, OB_P3,
 *                    PH, P_H, SC, SH, TO (and s_seg is TO), or U.
 * 
 * 	         The output specification of this rule says:
 * 	           1. The output category is the start category minus s_seg, 
 * 	           compacted, with n_seg added at the end, unless s_seg is
 *                 AG, D, U, or TO, in which case it is start with s_seg
 *                 replaced by the last segment of next, unless s_seg is
 *                 D, in which case it is start with D cancelled and
 *                 n_seg appended.
 * 
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 *
 * PARENTS
 * 
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_fverb_plus_main(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  int       index, r_cat_index = 1, n_cat_index, s_cat_index, i;
  SEG       s_seg, n_seg, s_len, s_seg_index;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      if ((start->segments[s_cat_index][1] != V) && 
	  (start->segments[s_cat_index][1] != VI) &&
	  ((start->segments[s_cat_index][0] > 1) &&
	  (start->segments[s_cat_index][2] != A)))
	{
	  fprintf(stderr, "   Rule 'fverb plus main' throws out meaning %d of start '%s'.\n", s_cat_index, start->surface);
	  continue;
	}
      else
	{
	  if (start->segments[s_cat_index][0] == 1)
	    {
	      s_seg = start->segments[s_cat_index][1];
	      s_seg_index = 1;
	    }
	  else
	    {
	      s_seg = start->segments[s_cat_index][2];
	      s_seg_index = 2;
	    }
	}

      /* 
	allow for each interpretation of "next", paying attention to the size 
	of result.segments.
	*/
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  {
	    /* central rule logic may proceed; set n_seg */
	    n_seg = next->segments[n_cat_index][1];
	    
	    /* input spec */
	    if (((next->segments[n_cat_index][1] == AG) || (s_seg == AG)) ||
		(next->segments[n_cat_index][1] == AV) ||
		(next->segments[n_cat_index][1] == C) ||
		(next->segments[n_cat_index][1] == _C) ||
		(next->segments[n_cat_index][1] == CA) ||
		(next->segments[n_cat_index][1] == _CA) ||
		(next->segments[n_cat_index][1] == CH) ||
		(next->segments[n_cat_index][1] == C_H) ||
		(next->segments[n_cat_index][1] == _C_H) ||
		(next->segments[n_cat_index][1] == P) ||
		(next->segments[n_cat_index][1] == P1) ||
		(next->segments[n_cat_index][1] == P3) ||
		(next->segments[n_cat_index][1] == OB_P3) ||
		(next->segments[n_cat_index][1] == PH) ||
		(next->segments[n_cat_index][1] == P_H) ||
		(next->segments[n_cat_index][1] == SC) ||
		(next->segments[n_cat_index][1] == SH) ||
		((next->segments[n_cat_index][1] == TO) && (s_seg == TO)) ||
		(next->segments[n_cat_index][1] == U))
	      {
		/* denote success */
		curr_condition |= VALID;
		history[curr_deriv].rules[0] += 1;
 		index = history[curr_deriv].rules[0];
		history[curr_deriv].rules[index] = FVERB_PLUS_MAIN;
		
		/* output spec */
		/* the number of result segments is the same as next's */
		copy_category(result.segments[r_cat_index],
			      start->segments[s_cat_index]);

		if ((s_seg == U) || ((s_seg == AG) &&
		    (start->segments[s_cat_index][3] != TO)) ||
		    (s_seg == TO))
		  result.segments[r_cat_index][2] =
		    next->segments[n_cat_index]
		      [next->segments[n_cat_index][0]];
		else
		  {
		    if (start->segments[s_cat_index][0] > 1)
		      compact_category(result.segments[r_cat_index],
				       s_seg_index);
		    result.segments[r_cat_index][++result.segments[r_cat_index]
						 [0]] = n_seg;
		  }
		
		/* 
		  increment r_cat_index at end of successful rule 
		  processing. Failures are not recorded.
		  */
		r_cat_index += 1;

		/* diagnostics follow */
		printf("   Rule 'finite verb plus main clause' combines meaning %d of finite verb phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		if (r_cat_index == MAX_CATEGORIES)
		  {
		    printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			   start->surface, next->surface);
		  }
	      }
	    else
	      {
		fprintf(stderr, "   Start meaning %d of rule 'fverb plus main' throws out meaning %d of next lexeme '%s'.\n", s_cat_index, n_cat_index, next->surface);
	      }  
	  }
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    /* this is, of course, very wimpy */
    {
      strcpy(error, "   Rule 'finite verb plus main clause' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot continue a finite verb phrase.");
    }
  
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void  r_fverb_plus_main(LEXENTRY *start, *next) : finished */


/*
 *  
 * FUNCTION      void  r_fverb_plus_nom(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a finite verb phrase and a nominative.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is set to the second segment of start
 *                  n_seg is set to the first segment of next
 *
 *      	 The input condition of this rule says:
 *                  The first category of start must be V or VI. Furthermore,
 *                  if s_seg is S3, n_seg must be SH or S_H
 *
 *        	 The output specification of this rule says:
 *                  The output category is the start category minus s_seg,
 *                  plus n_seg.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 *  
 * PARENTS
 * 
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *                packages, the control flow is altered.
 *  
 */

void
  r_fverb_plus_nom(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, i;
  SEG       s_seg, n_seg, s_len;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      if (start->segments[s_cat_index][0] > 1)
	s_seg = start->segments[s_cat_index][2];
      else
	continue;
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* condition on start */
	  if ((start->segments[s_cat_index][1] == VI) ||
	      (start->segments[s_cat_index][1] == V))
	    {
	      /* central rule logic may proceed; set n_seg */
	      n_seg = next->segments[n_cat_index][1];
	      
	      /* input spec */
	      if ((s_seg == S3) && ((n_seg == SH) || (n_seg == S_H)) ||
		  (s_seg != S3))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    FVERB_PLUS_NOM;
		  
		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);
		  for (i = 2; i < result.segments[s_cat_index][0]; i++)
		    {
		      result.segments[s_cat_index][i] =
			result.segments[s_cat_index][i + 1];
		    }
		  result.segments[s_cat_index][i] = n_seg;
		  result.segments[0][0] += 1;

		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'finite verb plus nominative' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'finite verb plus nominative' throws \
out meaning %d of next lexeme '%s', uses start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }

  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "Rule 'finite verb phrase plus nominative' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot continue a finite verb phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_fverb_plus_nom(LEXENTRY *start, *next) : finished template */


/*
 *  FUNCTION      void  r_nom_plus_fverb(LEXENTRY *start, *next)
 *  
 *  PURPOSE       To combine a nominative and the next lexeme, iff it is a
 *                finite verb or a suitable beginning to an adjective phrase.
 *  
 *  DECL          For all interpretations of both the start and the next lexeme:
 *  
 *                The selection specification of this rule says:
 * 
 * 	If the category of the start contains WH, W_H, WS, or WP,
 * 	then the local variable s_seg is set to the last segment
 * 	of the start. Otherwise, s_seg is set to the first category
 * 	segment of the start. n_seg is set to the second segment
 * 	of the the next category.
 *  
 * 	The input condition of this rule says:
 *         	    The category of the start may not contain a DO, HV, or B,
 * 	    and the next segment must be a finite verb, i.e.,
 * 	    it must have an initial segment of V or VI, and at
 * 	    least one other segment. Furthermore:
 * 	    1. if s_seg is S1, then the second segment of next
 * 	    must be S1, NOM, or N;
 * 	    2. if s_seg is PH, P_H, WP, P1, or P3, then the second
 * 	    segment of next must be NM, NOM, or N;
 * 	    3. if s_seg is SH, S_H, S3, WH, W_H, or WS, then
 * 	    the second segment of next must be S3 or N.
 * 
 * 	The output specification of this rule says:
 * 	    if s_seg is SH, S_H, PH, or P_H, the output category is the
 *          next category with NOM, N, or S3 cancelled, and s_seg
 *          appended, unless the second start segment is s_seg. Otherwise,
 *          it is the last segment of next appended to the first segment
 *          of start, unless n_seg is DO, in which case start WS valencies
 *          are retained, or if start contains WH or W_H, in which case a
 *          produced inital V segment becomes VI.
 *  
 *  INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *                 and (evaluated) next lexeme.
 *   
 *  OUTPUT        none
 *  
 *  PARENTS
 * 
 *  SIDE EFFECTS  As with all rules and rule packages, the control flow
 *                is altered.
 */

void
  r_nom_plus_fverb(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, s_len, s_cat_len, failed = 0, i;
  SEG       s_seg = NOSEG, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++) 
    {
      /* 
       * If the category of the start contains WH, W_H, WS, or WP,
       * then the local variable s_seg is set to the last segment
       * of the start.
       */
      s_cat_len = start->segments[s_cat_index][0];

      for (s_seg_index = 1; s_seg_index <= s_cat_len; s_seg_index++)
	if ((start->segments[s_cat_index][s_seg_index] == WH) ||
	    (start->segments[s_cat_index][s_seg_index] == W_H) ||
	    (start->segments[s_cat_index][s_seg_index] == WS) ||
	    (start->segments[s_cat_index][s_seg_index] == WP))
	  s_seg = TRUE;
      if (s_seg != TRUE)
	  s_seg = start->segments[s_cat_index][1];
      else
	s_seg = start->segments[s_cat_index][s_cat_len];
      
      /*  
       * allow for all interpretations of "next", paying attention to 
       * the size of result.segments, and keeping in mind that if "next"
       * has no categorizations with V in the initial position, then
       * it is not a finite verb, and the rule fails
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  if (((next->segments[n_cat_index][1] == V) ||
	       (next->segments[n_cat_index][1] == VI)) &&
	      (next->segments[n_cat_index][0] > 1))
	    {
	      /*  
	       * central rule logic may proceed; set n_seg to the
	       * second segment of next
	       */

	      n_seg = next->segments[n_cat_index][2];
	      
	      /* input spec */
	      for (s_seg_index = 1; s_seg_index <= s_cat_len; s_seg_index++)
		{
		  if (((start->segments[s_cat_index][s_seg_index] != DO) &&
		       (start->segments[s_cat_index][s_seg_index] != HV) &&
		       (start->segments[s_cat_index][s_seg_index] != B)))
		    {
		      if (((s_seg == S1) && ((n_seg == S1) || (n_seg == NOM) ||
					     (n_seg == N))) ||
			  
			  (((s_seg == PH) || (s_seg == P_H) || (s_seg == WP) ||
			    (s_seg == P1) || (s_seg == P3)) && 
			   ((n_seg == NM) || (n_seg == NOM) ||(n_seg == N))) ||
			  
			  (((s_seg == SH) || (s_seg == S_H) || (s_seg == S3) ||
			    (s_seg == WH) || (s_seg == W_H) || 
			    (s_seg == WS)) && 
			   ((n_seg == S3) || (n_seg == N))) ||
			  
			  (s_seg == SD))
			{
			  /* denote success */
			  curr_condition |= VALID;
			  history[curr_deriv].rules[0] +=1;
			  index = history[curr_deriv].rules[0];
			  history[curr_deriv].rules[index] = NOM_PLUS_FVERB;
			  
			  /* output spec */
			  copy_category(result.segments[r_cat_index],
					next->segments[n_cat_index]);

			  if ((s_seg == SH) || (s_seg == S_H) ||
			      (s_seg == PH) || (s_seg == P_H))
			    {
			      for (index = 1; (index <= 
				   result.segments[r_cat_index][0]) &&
				   (result.segments[r_cat_index][index] !=
				    NOM) &&
				   (result.segments[r_cat_index][index] !=
				    N) &&
				   (result.segments[r_cat_index][index] !=
				    S3); index++)
				;
			      compact_category(result.segments[r_cat_index],
					       index);
			      if ((s_cat_len > 1) &&
				  (start->segments[s_cat_index][2] != s_seg))
				result.segments[r_cat_index]
				  [++result.segments[r_cat_index][0]] = s_seg;
			    }
			  else
			    {
			      if (n_seg == DO)
				{
				  splice[0] = 1;
				  splice[1] = n_seg;
				  splice_category(result.segments[r_cat_index],
						  2, splice);
				}
			      else
				{
				  result.segments[r_cat_index][0] = 2;
				  result.segments[r_cat_index][2] =
				    next->segments[n_cat_index]
				      [next->segments[n_cat_index][0]];
				}
			    }

			  for (i = 1; i < start->segments[s_cat_index][0]; 
			       i++)
			    if ((start->segments[s_cat_index][i] == WH) ||
				(start->segments[s_cat_index][i] == W_H))
			      {
				result.segments[r_cat_index][1] = VI;
				for (index = 1;
				     index < result.segments[r_cat_index][0];
				     index++)
				  if (result.segments[r_cat_index][index] == 
				      A)
				    compact_category(result.segments
						     [r_cat_index], index);
			      }				

			  result.segments[0][0] += 1;
			  
			  /* 
			   * increment r_cat_index at end of successful rule 
			   * processing. Failures are not recorded ('cause we
			   * have backtracking).
			   */
			  r_cat_index += 1;

			  /* diagnostics follow */
			  printf("   Rule 'nominative phrase plus finite verb' combines meaning %d of nominative phrase '%s' with meaning %d of finite verb '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface);
			  if (r_cat_index == MAX_CATEGORIES)
			    {
				printf("   The maximum number of interpretations has been reached, in rule 'nominative plus finite verb'. Start = '%s', next = '%s'.\n", start->surface, next->surface);
			    }
		      }
		  }
	      }
	  }
	  else
	    {
	      fprintf(stderr, "   Start meaning %d of rule 'nominative plus finite verb' throws out meaning %d of next lexeme '%s'.\n", s_cat_index,n_cat_index, next->surface);
	    }
      }
    }
  
  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */  
    /* this is, of course, very wimpy */
    {
      strcpy(error, "   Rule 'nominative plus finite verb' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a finite verb phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: r_nom_plus_fverb(LEXENTRY *start, *next) : unfinished */


/*
 * FUNCTION      void  r_noun_plus_pnm(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a noun phrase with a post-nominal modifier.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is the last segment of start. n_seg is the 
 *                  last segment of next.
 *
 *      	 The input condition of this rule says:
 *                  the first segment of next must PNM. s_seg must be SH or S_H
 *
 *        	 The output specification of this rule says:
 *                  result is the start with s_seg cancelled and n_seg
 *                  added as a valency in second position.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_noun_plus_pnm(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][start->segments[s_cat_index][0]];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* condition on next */
	  if (next->segments[n_cat_index][1] == PNM)
	    {
	      /* central rule logic may proceed; set n_seg */
	      n_seg = next->segments[n_cat_index][next->segments
						  [n_cat_index][0]];
	      /* input spec */
	      if ((s_seg == SH) || (s_seg == S_H))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    NOUN_PLUS_PNM;

		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);

		  compact_category(result.segments[r_cat_index],
				   start->segments[s_cat_index][0]);
		  
		  splice[0] = 1;
		  splice[1] = n_seg;
		  splice_category(result.segments[r_cat_index], 2, splice);

		  result.segments[0][0] += 1;

		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'noun plus post-nominal modifier' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'noun plus post-nominal modifier' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }

  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'noun plus post-nominal modifier' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a post-nominal modifier phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_noun_plus_pnm(LEXENTRY *start, *next) : finished template */


/*
 * FUNCTION      void  r_nom_plus_relpro(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a nominative phrase with a relative pronoun.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  The start must end with SH, PH, S1, OB_S1, S3, OB_S3,
 *                  P1, OB_P1, P3, OB_P3, S_H, or P_H, to which s_seg is set.
 *                  The next lexeme must be a relative pronoun ("who" or 
 *                  "which").
 *
 *      	 The input condition of this rule says:
 *                  The next lexeme must be "who" if s_seg is SH, PH, S1,
 *                  OB_S1, S3, OB_S3, P1, OB_P1, P3, or OB_P3, and "which"
 *                  if s_seg is S_H, P_H, P3, or OB_P3.
 *        	 The output specification of this rule says:
 *                  The start is unchanged.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_nom_plus_relpro(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* 
   * start->segments[0][0] gets corrupted by the output spec, so we
   * set it outside its loop 
   */

  s_len = start->segments[0][0];
  
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][start->segments[s_cat_index][0]];
      
      if ((s_seg == SH) || (s_seg ==PH) || (s_seg ==S1) || 
	  (s_seg == OB_S1) || (s_seg == S3) || (s_seg == OB_S3) || 
	  (s_seg == P1) || (s_seg == OB_P1) || (s_seg == P3) || 
	  (s_seg == OB_P3) || (s_seg == S_H) || (s_seg == P_H))
	{
	  /* 
	   * allow for each interpretation of "next", paying attention to 
	   * the size of result.segments.
	   */
	  for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	       (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	    {
	      /* condition on next */
	      if ((strcmp(next->surface, "who") == 0) ||
		  (strcmp(next->surface, "which") == 0))
		{
		  /* central rule logic may proceed; set n_seg */
		  
		  /* input spec */
		  if ((strcmp(next->surface, "who") == 0) &&
		      ((s_seg == SH) || (s_seg ==PH) || (s_seg ==S1) || 
		       (s_seg == OB_S1) || (s_seg == S3) || (s_seg == OB_S3) || 
		       (s_seg == P1) || (s_seg == OB_P1) || (s_seg == P3) || 
		       (s_seg == OB_P3)) || 
		      ((strcmp(next->surface, "which") == 0) && 
		       ((s_seg == S_H) || (s_seg == P_H))))
		    {
		      /* denote success */
		      curr_condition = VALID;
		      history[curr_deriv].rules[0] +=1;
		      index = history[curr_deriv].rules[0];
		      history[curr_deriv].rules[index] =
			NOM_PLUS_RELPRO;
		      
		      /* output spec */
		      
		      /* 
		       * increment r_cat_index at end of successful rule 
		       * processing. Failures are not recorded.
		       */
		      r_cat_index += 1;
		      
		      /* diagnostics follow */
		      printf("   Rule 'nominative plus relative pronoun' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		      if (r_cat_index == MAX_CATEGORIES)
			{
			  printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
				 start->surface, next->surface);
			}
		    }
		}
	      else
		{
		  fprintf(stderr, "   Rule 'nominative plus relative pronoun' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
			  n_cat_index, next->surface, s_cat_index);
		}
	    }
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'nominative plus relative pronoun' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which doesn't agree.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* (there are no redundancies to remove) */
    }
  
} /* end: void r_nom_plus_relpro(LEXENTRY *start, *next) : finished template */


/*
 * FUNCTION      void  r_prep_plus_np(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a prepositional phrase and a noun phrase
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is the second segment of start, n_seg is
 *                  the first category of the start
 * 
 *      	 The input condition of this rule says:
 *                  The start must have a noun-phrase valency (s_seg
 *                  must be NP)
 *
 *        	 The output specification of this rule says:
 *                  The result is the start with the NP valency called,
 *                  and n_seg added at the end, unless n_seg is U, in
 *                  which case n_seg cancels it directly.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 *  
 * PARENTS
 * 
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_prep_plus_np(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /*  start->segments[0][0] may be corrupted by the output spec */
  s_len = start->segments[0][0];
  
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][2];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* central rule logic may proceed; set n_seg */
	  n_seg = next->segments[n_cat_index][1];
	  
	  /* input spec */
	  if (((n_seg == SH) || (n_seg == U) || (n_seg == S_H) ||
	       (n_seg == PH) || (n_seg == P_H)) && (s_seg == NP))
	    {
	      /* denote success */
	      curr_condition = VALID;
	      history[curr_deriv].rules[0] += 1;
	      index = history[curr_deriv].rules[0];
	      history[curr_deriv].rules[index] =
		PREP_PLUS_NP;
	      
	      /* output spec */
	      /* segment count doesn't change */
	      copy_category(result.segments[r_cat_index],
			    start->segments[s_cat_index]);

	      if (n_seg == U)
		result.segments[r_cat_index][2] = n_seg;
	      else
		{
		  compact_category(result.segments[r_cat_index], 2);
		  result.segments[r_cat_index][++result.segments
					       [r_cat_index][0]] = n_seg;
		}
	      
	      result.segments[0][0] += 1;
	      
	      /* 
	       * increment r_cat_index at end of successful rule 
	       * processing. Failures are not recorded.
	       */
	      r_cat_index += 1;
	      
	      /* diagnostics follow */
	      printf("   Rule 'preposition plus noun phrase' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
	      if (r_cat_index == MAX_CATEGORIES)
		{
		  printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			 start->surface, next->surface);
		}
	    }
	}
    }
  
  if (--r_cat_index == 0)  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'preposition plus noun phrase' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot continue a prepositional phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_prep_plus_np(LEXENTRY *start, *next) : finished template */


/*
 * FUNCTION      void  r_start_minus_relcl(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a <start description> with a <next description>.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is set to the first start segment, and n_seg is
 *                  set to the first next segment.
 *
 *      	 The input condition of this rule says:
 *                  s_seg must be SH, S_H, PH or P_H, and n_seg must be
 *                  WH or W_H.
 *
 *        	 The output specification of this rule says:
 *                  the output category is s_seg plus WS if s_seg is
 *                  SH or S_H, or WP if s_seg is PH or P_H, and SD.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_start_minus_relcl(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;
  
  
  result.segments[0][0] = 0;
  
  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
  
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][1];
      if ((s_seg == SH) || (s_seg == S_H) || (s_seg == PH) || (s_seg == P_H))
	{
	  /* 
	   * allow for each interpretation of "next", paying attention to
	   * the size of result.segments.
	   */
	  for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	       (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	    {
	      /* condition on next */
	      n_seg = next->segments[n_cat_index][1];
	      if ((n_seg == WH) || (n_seg == W_H))
		{
		  /* central rule logic may proceed */
		  
		  /* input spec */
		  if (VALID)
		    {
		      /* denote success */
		      curr_condition = VALID;
		      history[curr_deriv].rules[0] +=1;
		      index = history[curr_deriv].rules[0];
		      history[curr_deriv].rules[index] =
			START_MINUS_RELCL;
		      
		      /* output spec */
		      copy_category(result.segments[r_cat_index],
				    start->segments[s_cat_index]);
		      
		      splice[0] = 2;
		      splice[2] = SD;
		      if ((s_seg == SH) || (s_seg == S_H))
			splice[1] = WS;
		      else
			splice[1] = WP;
		      splice_category(result.segments[r_cat_index],
				      result.segments[r_cat_index][0] + 1,
				      splice);
		      
		      result.segments[0][0] += 1;
		      
		      /* 
		       * increment r_cat_index at end of successful rule 
		       * processing. Failures are not recorded.
		       */
		      r_cat_index += 1;
		      
		      /* diagnostics follow */
		      printf("   Rule 'start minus relative clause' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		      if (r_cat_index == MAX_CATEGORIES)
			{
			  printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
				 start->surface, next->surface);
			}
		    }
		}
	      else
		{
		  fprintf(stderr, "   Rule 'start minus relative clause' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
			  n_cat_index, next->surface, s_cat_index);
		} 
	    }
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'start minus relative clause' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a relative clause.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);  
    }  
  
} /*
   * end: void r_start_minus_relcl(LEXENTRY *start, *next) : finished template
   */


/*
 * FUNCTION      void  r_start_minus_subcl(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a start with a subclause.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is set to the second start segment. n_seg is
 *                  set to the first next segment.
 *
 *      	 The input condition of this rule says:
 *                  
 *
 *        	 The output specification of this rule says:
 *                  (n_seg == SC) cancels (s_seg == A),
 *                  (n_seg == SH) or (n_seg == S_H) cancels (s_seg == A)
 *                  with SC SH SD.
 * 
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_start_minus_subcl(start, next)
LEXENTRY    *start, *next;
{
  CATEGORY  splice;
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      if (start->segments[s_cat_index][0] > 1)
	s_seg = start->segments[s_cat_index][2];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  if ((next->segments[n_cat_index][1] == SC) ||
	      (next->segments[n_cat_index][1] == SH) ||
	      (next->segments[n_cat_index][1] == S_H))
	    {
	      /* central rule logic may proceed; set n_seg */
	      n_seg = next->segments[n_cat_index][1];
	      /* input spec */
	      if (s_seg == A)
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    START_MINUS_SUBCL;
		  
		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				start->segments[s_cat_index]);

		  if ((n_seg == SH) || (n_seg == S_H))
		    {
		      splice[0] = 3;
		      splice[1] = SC;
		      splice[2] = SH;
		      splice[3] = SD;
		    }
		  else
		    {
		      splice[0] = 1;
		      splice[1] = n_seg;
		    }
		  
		  compact_category(result.segments[r_cat_index], 2);
		  splice_category(result.segments[r_cat_index], 2, splice);

		  result.segments[0][0] += 1;

		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'start minus subclause' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'start minus subclause' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }

  if ((--r_cat_index == 0) && (!backtracked))  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'start minus subclause' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot begin a continuing subclause.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
   
} /* end: void r_start_minus_subcl(LEXENTRY *start, *next) : finished template */


/*
 *
 * FUNCTION      void  r_top_minus_main(LEXENTRY *start, *next)
 * 
 * PURPOSE       To combine a subject and a verb phrase (?).
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 * 
 *               The selection specification of this rule says:
 *                  s_seg is set to the first segment of the start 
 *                  category. The first segment of the next category must be V.
 * 
 *	         The input condition of this rule says:
 *                  s_seg must be WP, WH or W_H.
 *
 *	         The output specification of this rule says:
 *                  the output category is the next category with the 
 *                  first category changed to VI, plus s_seg.
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 *
 * OUTPUT        None.
 *
 * PARENTS
 * 
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 * 
 */

void
  r_top_minus_main(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, i;
  SEG       s_seg, n_seg, s_len;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];
 
  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][1];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* condition on next */
	  if (next->segments[n_cat_index][1] == V)
	    {
	      /* central rule logic may proceed; set n_seg */

	      /* input spec */
	      if ((s_seg == WP) || ((s_seg == WH) || s_seg == W_H))
		{
		  /* denote success */
		  curr_condition = VALID;
		  history[curr_deriv].rules[0] +=1;
		  index = history[curr_deriv].rules[0];
		  history[curr_deriv].rules[index] =
		    TOP_MINUS_MAIN;

		  /* output spec */
		  copy_category(result.segments[r_cat_index],
				next->segments[n_cat_index]);
		  result.segments[r_cat_index][1] = VI;
		  result.segments[r_cat_index][++result.segments
					       [r_cat_index][0]] = s_seg;
		  result.segments[0][0] += 1;
			  
		  /* 
		   * increment r_cat_index at end of successful rule 
		   * processing. Failures are not recorded.
		   */
		  r_cat_index += 1;

		  /* diagnostics follow */
		  printf("   Rule 'topicalized main constituent' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
		  if (r_cat_index == MAX_CATEGORIES)
		    {
		      printf("   The maximum number of interpretations has been \
reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			     start->surface, next->surface);
		    }
		}
	    }
	 else
	    {
	      fprintf(stderr, "   Rule 'topicalized main constituent' throws \
out meaning %d of next lexeme '%s', uses start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }

  if ((--r_cat_index == 0) && (!backtracked))  /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "Rule 'topicalized main constituent' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which cannot continue the sentence.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }

} /* end: void r_top_minus_main(LEXENTRY *start, *next) : finished template */


/*
 * FUNCTION      void  r_wh_plus_verb(LEXENTRY *start, *next)
 *
 * PURPOSE       To combine a wh-phrase with a verb.
 * 
 * DECL          For all interpretations of both the start and the next lexeme:
 *
 *               The selection specification of this rule says:
 *                  s_seg is the second start segment, n_seg is the
 *                  second next segment.
 *
 *      	 The input condition of this rule says:
 *
 *        	 The output specification of this rule says:
 *                  (n_seg == NOM) cancels (s_seg == DO) with cddr(next).
 *
 * INPUT         LEXENTRY  *start, *next    -    the current sentence start
 *               and (evaluated) next lexeme.
 * 
 * OUTPUT        None.
 * 
 * PARENTS
 *  
 * SIDE EFFECTS  The passed start is changed. As with all rules and rule 
 *               packages, the control flow is altered.
 */

void
  r_wh_plus_verb(start, next)
LEXENTRY    *start, *next;
{
  LEXENTRY  result;  /* local result, will be loaded into "start" */
  SURFACE   last_surface;
  int       index, s_seg_index, s_cat_index, r_cat_index = 1;
  int       n_cat_index, n_len, s_len, i;
  SEG       s_seg, n_seg;


  result.segments[0][0] = 0;

  /* start->segments[0][0] gets corrupted by the output spec */
  s_len = start->segments[0][0];

  /* Allow for each interpretation (categorization) of "start" */
  for (s_cat_index = 1; s_cat_index <= s_len; s_cat_index++)
    {
      /* condition(s) on start, setting s_seg */
      s_seg = start->segments[s_cat_index][2];
      
      /* 
       * allow for each interpretation of "next", paying attention to the size 
       * of result.segments.
       */
      for (n_cat_index = 1; (n_cat_index <= next->segments[0][0]) && 
	   (r_cat_index < MAX_CATEGORIES); n_cat_index++)
	{
	  /* central rule logic may proceed; set n_seg */
	  if (next->segments[n_cat_index][0] > 1)
	    n_seg = next->segments[n_cat_index][2];
	  else
	    continue;
	  
	  if ((n_seg == NOM) && (s_seg == DO))
	    {
	      /* denote success */
	      curr_condition = VALID;
	      history[curr_deriv].rules[0] +=1;
	      index = history[curr_deriv].rules[0];
	      history[curr_deriv].rules[index] =
		WH_PLUS_VERB;
	      
	      /* output spec */
	      copy_category(result.segments[r_cat_index],
			    start->segments[s_cat_index]);
	      
	      compact_category(result.segments[r_cat_index], 2);
	      compact_category(next->segments[n_cat_index], 1);
	      compact_category(next->segments[n_cat_index], 1);
	      splice_category(result.segments[r_cat_index], 2,
			      next->segments[n_cat_index]);

	      result.segments[0][0] += 1;
	      
	      /* 
	       * increment r_cat_index at end of successful rule 
	       * processing. Failures are not recorded.
	       */
	      r_cat_index += 1;
	      
	      /* diagnostics follow */
	      printf("   Rule 'wh-phrase plus verb' combines meaning %d of phrase '%s' with meaning %d of lexeme '%s'\n", s_cat_index, start->surface, n_cat_index, next->surface); 
	      if (r_cat_index == MAX_CATEGORIES)
		{
		  printf("   The maximum number of interpretations has been reached, in rule 'determiner plus adj'. Start = '%s', next = '%s'.\n", 
			 start->surface, next->surface);
		}
	    }
	  else
	    {
	      fprintf(stderr, "   Rule 'wh-phrase plus verb' throws out meaning %d of next lexeme '%s', moves to start meaning %d.\n",
		      n_cat_index, next->surface, s_cat_index);
	    } 
	}
    }
  
  if ((--r_cat_index == 0) && (!backtracked)) /* there were no successes */
    {
      /* this is, of course, very wimpy */
      strcpy(error, "   Rule 'wh-phrase plus verb' has failed. You tried to follow the phrase '");
      strcat(error, start->surface);
      strcat(error, "' with '");
      strcat(error, next->surface);
      strcat(error, "', which is not a suitable verb for continuing a we-phrase.");
    }
  else
    {
      /* load the result into start */
      for (index = 1; index <= r_cat_index; index++)
	copy_category(start->segments[index],
		      result.segments[index]);
      start->segments[0][0] = r_cat_index;
      
      /* remove redundancies */
      comb_definition(start->segments);
    }
  
} /* end: void r_wh_plus_verb(LEXENTRY *start, *next) : finished template */
