/* File: lhip.pl
 * Created: 4-Dec-93
 * Last Update:  4-Dec-93
 * Author: Afzal Ballim (C) 1993
 * Purpose: left head-corner island parser compiler.
 *			Rules can have multiple "heads" (known as "anchors"
 *			here), parsing is based on finding the anchors, then
 *			filling in the gaps, left-right. Anchors are searched
 *			for left to right, and depth-first.
 *			Oh, can also find unattached constituents afterwards.
 *
 * This version has user-configuarable thresholding.
 *
 * Needs: member/2,append/3
 */

% operators needed for lhip rules
:- op(1200,xfx,[ ~~> ]).
:- op(1150,xf,[ ? ]).
:- op(1151,fx,[ ? ]).
:- op(500,fy,[ ~ ]).
:- op(500,fy,[ * ]).
:- op(100,fx,[ @ ]).
:- op(500,yfx,[ # ]).
:- op(1000,xfy,[ : ]).

% dynamic clauses used for the chart and identifying lhip clauses
:- dynamic lhip_known/5,lhip_known_fail/2,lhip_ccount/1,
   lhip_threshold_value/1,lhip_success_counter/1.


% Default threshold value. For a rule to succeed 
%  Cover/Span > Threshold
lhip_threshold_value(0).

% counter for referencing successful rule firing
lhip_success_counter(0).

% need to distinguish clauses, otherwise lhip_known and lhip_fail
% cannot be used properly. Add an extra argument (the lhip clause 
% counter) to each clause to distinguish it.
lhip_ccount(0).


/************************************************************
 *
 * Term expansion code
 *
 ************************************************************/

/*===========================================================
 * Predicate: term_expansion/2
 * Modes: 
 * 
 * Purpose: File reading causes invokation of term_expansion
 *			under sicstus, which gives us a way of hooking
 *			in to compile the grammar. Term expansion "compiles"
 *			the lhip rules into prolog clauses, complete with
 *			memoization (which gives a "chart" like effect).
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update: 
 *=========================================================*/

% a delete rule
term_expansion(
	(-X # T ~~>P),
	(Xin
	 :-	
	   copy_term(Xin2,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		lhip_known(Cntr,Ref,Rf,Xfrz,Xin);
		% if not known to have succeeded or failed, try it
		\+lhip_known_fail(Cntr,Xfrz),
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		P1,
		C/(De-Ds) >= T,
		lhip_asserts(lhip_known(Cntr,Ref,Rf,Xfrz,Xin2),Ref);
		% not successful
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		assertz(lhip_known_fail(Cntr,Xfrz)),
		fail)))
		
:-
	!,Xin = lhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL]/DL,Rf),
	Xin2 = lhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL2]/DL2,Rf),
	lhip_body(P,P1,FS,S,E,Ds,De,C,Rf/[]),
	lhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(lhip_ccount(Cntr)),
	assert(lhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a normal rule
term_expansion(
	(X # T ~~>P),
	(Xin
	 :-
	   copy_term(Xin2,Xfrz),numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		lhip_known(Cntr,Ref,Rf,Xfrz,Xin);
		% if not known to have succeded or failed, try it
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		P1,
		C/(De-Ds) >= T,
		lhip_asserts(lhip_known(Cntr,Ref,Rf,Xfrz,Xin2),Ref);
		% not successful
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		assertz(lhip_known_fail(Cntr,Xfrz)),
		fail)))
:-
	!,lhip_clause(X,Xin,[FS,S,E,Ds,De,C,[Ref|DL]/DL,Rf]),
	lhip_clause(X,Xin2,[FS,S,E,Ds,De,C,[Ref|DL2]/DL2,Rf]),
	lhip_body(P,P1,FS,S,E,Ds,De,C,Rf/[]),
	lhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(lhip_ccount(Cntr)),
	assert(lhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a delete rule
term_expansion(
	(-X ~~> P),
	(Xin
	 :-	
	   lhip_threshold_value(T),
	   copy_term(Xin2,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		lhip_known(Cntr,Ref,Rf,Xfrz,Xin);
		% if not known to have succeeded or failed, try it
		\+lhip_known_fail(Cntr,Xfrz),
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		P1,
		C/(De-Ds) >= T,
		lhip_asserts(lhip_known(Cntr,Ref,Rf,Xfrz,Xin2),Ref);
		% not successful
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		assertz(lhip_known_fail(Cntr,Xfrz)),
		fail)))
:-
	!,Xin = lhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL]/DL,Rf),
	Xin2 = lhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL2]/DL2,Rf),
	lhip_body(P,P1,FS,S,E,Ds,De,C,Rf/[]),
	lhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(lhip_ccount(Cntr)),
	assert(lhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a normal rule
term_expansion(
	(X ~~> P),
	(Xin
	 :-
	   lhip_threshold_value(T),
	   copy_term(Xin2,Xfrz),numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		lhip_known(Cntr,Ref,Rf,Xfrz,Xin);
		% if not known to have succeded or failed, try it
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		P1,
		C/(De-Ds) >= T,
		lhip_asserts(lhip_known(Cntr,Ref,Rf,Xfrz,Xin2),Ref);
		% not successful
		\+lhip_known(Cntr,Ref,Rf,Xfrz,Xin),
		\+lhip_known_fail(Cntr,Xfrz),
		assertz(lhip_known_fail(Cntr,Xfrz)),
		fail)))
:-
	!,lhip_clause(X,Xin,[FS,S,E,Ds,De,C,[Ref|DL]/DL,Rf]),
	lhip_clause(X,Xin2,[FS,S,E,Ds,De,C,[Ref|DL2]/DL2,Rf]),
	lhip_body(P,P1,FS,S,E,Ds,De,C,Rf/[]),
	lhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(lhip_ccount(Cntr)),
	assert(lhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

/*===========================================================
 * Predicate: lhip_body/9
 * Modes: lhip_body(+P,-Px,+FS,+S,+E,-Ds,-De,-C,+Rf/?Rl)
 * 
 * Purpose: performs the translation of the body of a lhip
 *			clause P, into Px, where FS is the sentence
 *			represented as a function, S is the start of the
 *			span over which the search occurs, E is the end
 *			boundary of the span (i.e., search occurs in S..E-1
 *			inclusive), Ds is the start of the found constituent
 *			De is the end boundary of it, and C is the number of
 *			tokens covered by the constituent in Ds..De-1.
 *			In addition, a diff-list is maintained as a store
 *			for tracking successful rule firing.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_body(P,Px,FS,S,E,Ds,De,C,Rf/Rl)
:-
	lhip_constrain(P,Knt,NonKnt,FS,Ds,De,E,S,S,_SNK,_SNI,[[],[]],
					Covers,Rf/Rl),
	!,lhip_unravel(Knt,NonKnt,Px,C,Covers).

lhip_body(P,_,_,_,_,_,_,_,_)
:-
	format("ERROR: could not compile lhip body ~q~n",[P]),
	!,fail.

/*===========================================================
 * Predicate: lhip_constrain/14
 * Modes: lhip_constrain(
 *			+Body,
 *			-Anchors,
 *			-NonAnchors,
 *			+Sent,
 *			+CvS,
 *			+CvE,
 *			+End,
 *			+EndPrevAnchor,
 *			+EndPrevItem,
 *			-StartNextAnchor,
 *			-StartNextItem,
 *			[+PrevCls,+PrevDCls],
 *			-Covers,
 *			+Rf/+Rl)
 * 
 * Purpose: 
 * 
 * Licensed Use: 
 * 
 * Created: 20-Nov-93
 * Last Update:  6-Dec-93
 *=========================================================*/

%                    +ERROR conditions+

%					ILLEGAL HEADS

lhip_constrain(((*!),Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a head ~q~n",[((*!),Q)]),
	!,fail.

lhip_constrain((*[],_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a head~n",[]),
	!,fail.

lhip_constrain((*P,_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

lhip_constrain(((*!):Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a head ~q~n",[((*!),Q)]),
	!,fail.

lhip_constrain((*[]:_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a head~n",[]),
	!,fail.

lhip_constrain((*P:_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

lhip_constrain((*!),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a head ~q~n",[(*!)]),
	!,fail.

lhip_constrain(*[],_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a head~n",[]),
	!,fail.

lhip_constrain(*P,_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

%					ILLEGAL SPECIFIC DELETE RULES

lhip_constrain(((-!),Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a delete rule ~q~n",[((-!),Q)]),
	!,fail.

lhip_constrain((-[],_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a delete rule~n",[]),
	!,fail.

lhip_constrain((-P,_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

lhip_constrain(((-!):Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a delete rule ~q~n",[((-!),Q)]),
	!,fail.

lhip_constrain((-[]:_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a delete rule~n",[]),
	!,fail.

lhip_constrain((-P:_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

lhip_constrain((-!),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("ERROR: cut cannot be a delete rule ~q~n",[(-!)]),
	!,fail.

lhip_constrain(-[],_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("ERROR: general delete cannot be a specific delete rule~n",[]),
	!,fail.

lhip_constrain(-P,_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	lhip_non_predicate(P),
	\+(P = @_),
	format("ERROR: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

%				MUST HAVE AT LEAST ONE HEAD/NORMAL CLAUSE

lhip_constrain(!,_K,_NK,_S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule~n",[]),
	!,fail.

lhip_constrain({P},_K,_NK,_S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule ~q~n",[{P}]),
	!,fail.

lhip_constrain([],_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule~n",[]),
    !,fail.

lhip_constrain(-P,_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule ~q~n",[-P]),
    !,fail.

lhip_constrain(~P,_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule ~q~n",[~P]),
    !,fail.

lhip_constrain((?P?),_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("ERROR: rule body must contain a normal or head rule ~q~n",[(?P?)]),
    !,fail.

%                    -ERROR conditions-

%                    +Mult. Cls+

%                    +{}/!+

lhip_constrain((!,P),K,[{!}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl)
:-
	!,lhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl).

lhip_constrain(({E},P),K,[{E}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl)
:-
	!,lhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl).

lhip_constrain((!:P),K,[{!}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl)
:-
	!,lhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl).

lhip_constrain(({E}:P),K,[{E}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl)
:-
	!,lhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf/Rl).

%                    -{}/!-

%                    +FCls+

%                    +Adjacency+

lhip_constrain((*P:Q),[Px|K],NK,S,CvS,CvE,End,EPK,EPK,CvS,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,CvS,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,De,[Px,Px],Cov,Rnf/Rl).

% adjacency does not transmit backwards through a delete or optional

lhip_constrain(([]:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[_,S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	Px = (Pdel,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

lhip_constrain((-P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[P,S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	Px = (Pdel,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

lhip_constrain(((?P?):Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPK,SNK,CvS,De,C,Rf/Rnf),!,
	Px = (Popt,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

% a negation followed by an adjacency means that the following thing
% must be treated as a head
lhip_constrain((~P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPK,SNK,CvS,SNI,C,_),!,
	Px = (\+Popt,CvS=SNI,C=0),
	(Q=(*Q1),Qn=Q;
	 Q=(*Q1,Qr),Qn=Q;
	 Q=(Q1,Qr),Qn=(*Q1,Qr);
	 Q=(*Q1 : Qr), Qn=Q;
	 Q=(Q1 : Qr), Qn=(*Q1 : Qr);
	 Qn= *Q),
	lhip_constrain(Qn,K,NK,S,CvS,CvE,End,EPK,EPK,SNK,SNI,[Px,[]],Cov,Rf/Rl).

lhip_constrain((P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,De,[Px,Px],Cov,Rnf/Rl).

%                    -Adjacency-

lhip_constrain((*P,Q),[Px|K],NK,S,CvS,CvE,End,EPK,EPK,CvS,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,CvS,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,_SNI,[Px,Px],Cov,Rnf/Rl).

lhip_constrain(([],Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[_,S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	Px = (Pdel;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

lhip_constrain((-P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[P,S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	Px = (Pdel;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

lhip_constrain(((?P?),Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPK,SNK,CvS,De,C,Rf/Rnf),!,
	Px = (Popt;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf/Rl).

lhip_constrain((~P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPK,SNK,CvS,De,C,_),!,
	Px = (\+Popt,CvS=SNI,De=EPK,C=0),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rf/Rl).

lhip_constrain((P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,SNK,CvS,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,_SNI,[Px,Px],Cov,Rnf/Rl).

%                    -FCls-

%                    +Adjacency+

lhip_constrain((*P:Q),[Px|K],NK,S,CvS,CvE,End,EPK,_EPI,Ds,Ds,
			_PC,[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,Ds,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,De,[Px,Px],Cov,Rnf/Rl).

lhip_constrain(([]:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[_,S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	Px = (Pdel,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain((-P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[P,S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	Px = (Pdel,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain(((?P?):Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPI,SNK,Ds,De,C,Rf/Rnf),!,
	Px = (Popt,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain((~P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPI,SNK,Ds,SNI,C,_),!,
	Px = (\+Popt,Ds=SNI,C=0),
	(Q=(*Q1),Qn=Q;
	 Q=(*Q1,Qr),Qn=Q;
	 Q=(Q1,Qr),Qn=(*Q1,Qr);
	 Q=(*Q1 : Qr), Qn=Q;
	 Q=(Q1 : Qr), Qn=(*Q1 : Qr);
	 Qn= *Q),
	lhip_constrain(Qn,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,[Px,PdC],Cov,Rf/Rl).

lhip_constrain((P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			_PC,[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,De,[Px,Px],Cov,Rnf/Rl).

%                    -Adjacency-

lhip_constrain((*P,Q),[Px|K],NK,S,CvS,CvE,End,EPK,_EPI,Ds,Ds,
			_PC,[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,Ds,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,_SNI,[Px,Px],Cov,Rnf/Rl).

lhip_constrain(([],Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[_,S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	Px = (Pdel;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain((-P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[P,S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	Px = (Pdel;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain(((?P?),Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPI,SNK,Ds,De,C,Rf/Rnf),!,
	Px = (Popt;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf/Rl).

lhip_constrain((~P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,
			[_PC,PdC],[C|Cov],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPI,SNK,Ds,De,C,_),!,
	Px = (\+Popt,Ds=SNI,De=EPI,C=0),
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rf/Rl).

lhip_constrain((P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			_PC,[C|Cov],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPI,SNK,Ds,De,C,Rf/Rnf,_]),!,
	lhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,_SNI,[Px,Px],Cov,Rnf/Rl).

%                    -Mult. Cls-

%                    +Last Cls+

%                    +CUT+

% last clause is cut, but there were other clauses, so previous clause
% is last clause
lhip_constrain(!,[],[{!}],_S,_CvS,CvE,End,_EPK,CvE,End,CvE,_PC,[],Rl/Rl).

%                    -CUT-

%                    +{P}+

% last clause is embedded code, but there were other clauses
lhip_constrain({P},[],[{P}],_S,_CvS,CvE,End,_EPK,CvE,End,CvE,_PC,[],Rl/Rl).

%                    -{P}-

%                    +Lc=Fc+

% last clause is first clause
lhip_constrain(*P,[Px],[],S,CvS,CvE,End,EPK,EPK,End,CvE,[[],[]],[C],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,CvS,CvE,C,Rf/Rl,_]),!.

lhip_constrain(P,[],[Px],S,CvS,CvE,End,EPK,EPK,End,CvE,[[],[]],[C],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,CvS,CvE,C,Rf/Rl,_]),!.

%                    -Lc=Fc-

% any other last clause
lhip_constrain(*P,[Px],[],S,_CvS,CvE,End,EPK,_EPI,Ds,Ds,_PC,[C],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPK,End,Ds,CvE,C,Rf/Rl,_]),!.

lhip_constrain([],[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[_,S,EPI,End,Ds,CvE,C,Rf/Rl,_]),!,
	Px = (Pdel;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

lhip_constrain(-P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf/Rl)
:-
	lhip_clause(lhip_delete,Pdel,[P,S,EPI,End,Ds,CvE,C,Rf/Rl,_]),!,
	Px = (Pdel;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

lhip_constrain((?P?),[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf/Rl)
:-
	lhip_body(P,Popt,S,EPI,End,Ds,CvE,C,Rf/Rl),
	Px = (Popt;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

lhip_constrain(~P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rl/Rl)
:-
	lhip_body(P,Popt,S,EPI,End,Ds,CvE,C,_),
	Px = (\+Popt,Ds=EPI,CvE=EPI,C=0).

lhip_constrain(P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf/Rl)
:-
	lhip_clause(P,Px,[S,EPI,End,Ds,CvE,C,Rf/Rl,_]),!.

%                    -Last Cls-


/*===========================================================
 * Predicate: lhip_unravel/5
 * Modes: lhip_unravel(+K,+N,-Cls,+C,+Covers)
 * 
 * Purpose: unravels the heads and non-heads to give the body
 * 
 * Licensed Use: 
 * 
 * Created: 20-Nov-93
 * Last Update: 
 *=========================================================*/

% there was only one clause
lhip_unravel([],[{P}],P,C,[C]) :- !.
lhip_unravel([],[P],P,C,[C]) :- !.
lhip_unravel([P],[],P,C,[C]) :- !.

% more than one clause, unravel
lhip_unravel([P|K],N,(P,Cls),C,Covers)
:-
	!,lhip_unravel(K,N,Cls,C,Covers).

lhip_unravel([],[],(C is Cls),C,Covers)
:-
	!,lhip_make_cover_cntr(Covers,Cls).

lhip_unravel([],[{P}],(P,C is Cls),C,Covers)
:-
	!,lhip_make_cover_cntr(Covers,Cls).

lhip_unravel([],[P],(P,C is Cls),C,Covers)
:-
	!,lhip_make_cover_cntr(Covers,Cls).

lhip_unravel([],[{P}|N],(P,Cls),C,Covers)
:-
	!,lhip_unravel([],N,Cls,C,Covers).

lhip_unravel([],[P|N],(P,Cls),C,Covers)
:-
	!,lhip_unravel([],N,Cls,C,Covers).

/*===========================================================
 * Predicate: lhip_make_cover_cntr/2
 * Modes: lhip_make_cover_cntr(+List,-Counter)
 * 
 * Purpose: Given a list of terms, make a term which "adds"
 *			them together
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/
lhip_make_cover_cntr([],_) :- !,fail.
lhip_make_cover_cntr([C],C).
lhip_make_cover_cntr([C|Cs],Counter+C)
:-
	lhip_make_cover_cntr(Cs,Counter).


/*===========================================================
 * Predicate: lhip_clause
 * Modes: lhip_clause(+LC,-PC,+XArgs)
 * 
 * Purpose: convert a lhip clause into a prolog one.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 18-Nov-93
 *=========================================================*/

lhip_clause(!,_,_)
:-
	format("ERROR: ! is not a clause, check that it does not",[]),
	format(" precede ! or embedded code~n",[]),
	!,fail.

lhip_clause([],_,_)
:-
	format("ERROR: [] is not a clause~n",[]),
	!,fail.

% disjunction, heads are local to the disjunction
lhip_clause((P;Q),
	(Px;Qx),
	[Fs,S,E,Ds,De,C,Rf/Rl,_])
:-
	!,lhip_body(P,Px,Fs,S,E,Ds,De,C,Rf/Rl),
	lhip_body(Q,Qx,Fs,S,E,Ds,De,C,Rf/Rl).

% terminal
lhip_clause((@X),X1,Args)
:-
	!,lhip_clause(lhip_terminal,X1,[X|Args]).		

% Generic error handling.
lhip_clause(P,_,_)
:-
	lhip_non_predicate(P),
	format("ERROR: ~q is not a clause~n",[P]),
	!,fail.

lhip_clause(X,Y,Args) 
:-
	!,X =.. Xl,
	append(Xl,Args,Yl),
	Y =.. Yl.

/*===========================================================
 * Predicate: lhip_non_predicate/1
 * Modes: 
 * 
 * Purpose: succeeds if the argument is a prolog connective,
 *			such as ';' or ','.
 * 
 * Licensed Use: 
 * 
 * Created: 23-Nov-93
 * Last Update: 
 *=========================================================*/

lhip_non_predicate(P)
:-
	functor(P,F,_),
	(F=',';
	 F=';';
	 F=':';
	 F='{}';
	 F='~';
	 F='-';
	 F='.';
	 F='*';
	 F='?';
	 F='@').

/*===========================================================
 * Predicate: lhip_terminal
 * Modes: 
 * 
 * Purpose: handle a terminal, allowing for alternation in
 *			the input signalled by embedded lists.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/

% if start has reached end, then fail as end is a far boundary
lhip_terminal(_Term,_Sent,Start,End,_F,_E,_C,_,_)
:-
	Start >= End,
	!,fail.
% note: a terminal can ONLY cover 1 token
lhip_terminal(Term,Sent,Start,_End,Start,End0,1,X/X,_)
:-
	arg(Start,Sent,Term),
	End0 is Start+1.
lhip_terminal(Term,Sent,Start,_End,Start,End0,1,X/X,_)
:-
	arg(Start,Sent,TermS),
	member(Term,TermS),
	End0 is Start+1.
lhip_terminal(Term,Sent,Start,End,Found,End0,C,X/Y,_)
:-
	NewStart is Start+1,
	lhip_terminal(Term,Sent,NewStart,End,Found,End0,C,X/Y,_).

/*===========================================================
 * Predicate: lhip_success_ref/1
 * Modes: lhip_success_ref(?T)
 * 
 * Purpose: the global success_counter value is T
 * 
 * Licensed Use: 
 *		lhip_success_ref(+T)	
 *			set success_counter value to T
 *		lhip_success_ref(-T)	
 *			T is the current success_counter value
 * 
 * Created: 25-Nov-93
 * Last Update:  6-Dec-93
 *=========================================================*/

lhip_success_ref(T)
:-
	var(T),
	lhip_success_counter(T).

lhip_success_ref(T)
:-
	nonvar(T),
	retract(lhip_success_counter(_)),
	assert(lhip_success_counter(T)).

/*===========================================================
 * Predicate: lhip_asserts/2
 * Modes: lhip_asserts(+Cls,-Ref)
 * 
 * Purpose: performs an assertz of Cls, having first bound
 *			Ref to the value of the success counter (which
 *			is then augmented by 1).
 * 
 * Licensed Use: 
 * 
 * Created:  6-Dec-93
 * Last Update: 
 *=========================================================*/

lhip_asserts(Cls,Ref)
:-
	lhip_success_ref(Ref),
	assertz(Cls),
	NewRef is Ref+1,
	lhip_success_ref(NewRef).
