% TRANSITIVE.PL

% transitive(+RelSymbol,+NewRelSymbol)
% computes transitive closure of binary relation. 
% terminates iff number of pairs is finite.

% transitive(+RelSymbol,+NewRelSymbol,+RestrictionRelSymbol)
% computes transitive closure of binary relation. 
% always terminates
 

:- module(transitive,[ transitive/2,
	               transitive/3,
		       transitive_ground/2]).

:- use_module(library(database),[clean_up_database/1,
                        assertz_most_general/2]). 

:- use_module(library(lists), [member/2, append/3 ]).


% if we have
% p(1,2)
% p(2,3)
% p(3,4)
% then we obtain, upon  ?- transitive(p,trans_p)
% the clauses:
%
%
% trans_p(1,2).
% trans_p(1,3).
%  ..
% trans_p(3,4).

% :- meta_predicate transitive(:,:).

add_module(Pred0,M:Pred) :-
	( Pred0 = M:Pred -> true ; Pred0 = Pred, M=user ).

transitive(A1,A2):-
	add_module(A1,B1),
	add_module(A2,B2),
	transitive2(B1,B2,user:'=').

transitive(A1,A2,R1):-
	add_module(A1,B1),
	add_module(A2,B2),
	add_module(R1,R2),
	transitive2(B1,B2,R2).

transitive2(M0:Rel,M:NewRel,Res):-
%	abolish(M:NewRel,2),
	NewRelC =.. [NewRel,_,_],
	clean_up_database(M:NewRelC),
	foundation(M0:Rel,M:NewRel,Res,P0),
	transitive_n(P0,M:NewRel,Res).

foundation(RelSym,NewRelSym,Res,P0):-
	findall(Pair,t0(RelSym,NewRelSym,Res,Pair),P0).

t0(M0:RelSym,M:NewRelSym,Mr:Res,Pair):-
	Rel =.. [RelSym,A1,A2],
	Pair1 =.. [NewRelSym,A1,A2],
	CalRes =.. [Res,Pair1,Pair],
	M0:Rel,
	Mr:CalRes,
	assertz_most_general(M:Pair,no).   %=> database.pl, fails if already exists

transitive_n([],_,_Res).
transitive_n([H|T],Rel,Res):-
	new_pairs(Rel,H,Res,P),
	append(T,P,T2),
	transitive_n(T2,Rel,Res).

new_pairs(NewRel,Pair1,Res,Pn):-
	findall(Pair,new_pair(NewRel,Pair1,Res,Pair),Pn).

new_pair(M:Rel,Sel1,Mr:Res,Pair):-
	Pair1 =.. [Rel,Arg1,Arg3],
	Sel1 =.. [Rel,Arg1,Arg2],
	Sel2 =.. [Rel,Arg2,Arg3],
	ResCall =.. [Res,Pair1,Pair],
	M:Sel2,
	Mr:ResCall,
	assertz_most_general(M:Pair,no). %=> database.pl, fails if already exists


%%%%%%%%%%%%%%%%%%%%%%%
%% transitive_ground %%
%%%%%%%%%%%%%%%%%%%%%%%

%% transitive_ground(ListOfPairs,NewListOfPairs)
%% where a pair is X/Y.
transitive_ground(Pairs,TrPairs) :-
	process(Pairs,[],TrPairs,Pairs).

%% process(Agenda,DatabaseIn,DatabaseOut,BasicCase)
process([],T,T,_).
process([Pair|Agenda0],New0,New,Basic) :-
	new_pairs(Pair,Basic,Pairs),
	add_pairs(Pairs,Agenda0,Agenda,New0,New1),
	process(Agenda,New1,New,Basic).

new_pairs(Pair,Basic,Pairs) :-
	findall(NPair,new_pair(Pair,Basic,NPair),Pairs).

new_pair(A/B,Basic,A/C) :-
	member(B/C,Basic).

add_pairs([],A,A,N,N).
add_pairs([H|T],A0,A,N0,N) :-
	add_pair(H,A0,A1,N0,N1),
	add_pairs(T,A1,A,N1,N).

add_pair(Pair,A,A,N,N) :-
	( member(Pair,A)
        ; member(Pair,N) ),
	!.
add_pair(Pair,A,[Pair|A],N,[Pair|N]).

