:- module(tk_menu, [
	           x/0, start_x/0, restart_x/0, restart_x/1, halt_x/0,

		   send_flag/1, send_flag/2, receive_flag/2,
		   send_boolean_flag/1, send_boolean_flag/2,
		   receive_boolean_flag/2, send_pid/0,

		   send_spy/2,
		   
		   warning/1
	         ]).

:- use_module( library(tk) ).

:- use_module( library(concat),  [ concat/3,
	                           concat_all/2,
				   concat_all/3,
				   term_atom/2,
				   atom_term/2 
			         ]).

:- use_module( library(lists),   [ member/2 ]).

:- use_module( library(flags) ).

%% starting and halting tk interface

:- initialize_flag(tk_started,off).

% x starts up Tk environment. If tksteal is one of the arguments
% with which Prolog has been called, then tksteal support is
% enabled.

x :-
	unix(argv(List)),
	x(List).

x([]) :-
	start_x(0).

x(List) :-
	member(notk,List),!.

x(List) :-
	member(tksteal,List),!,
	start_x(1).


% restart_x: halt and start again. Only possible if there is
% currently no tksteal support. But that is checked in halt_x.
% Use restart_x(1) to kill start_x(0) and start start_x(1).
restart_x :-
	restart_x(0).  % default

restart_x(Bool) :-
	halt_x,
	start_x(Bool).

% start_x(0) starts Tk environment without tksteal support
% start_x(1) starts Tk environment with    tksteal support
start_x(Bool) :-
	flag(tk_started,off),
	tk:tk_init('',[]),              % protcl built-in
	concat('set steal ',Bool,Cmd),  % enables tksteal support
	tcl_eval(Cmd),
	send_dir,
	tk_menu_tcl_files,
	(  user:tk_startup_hook 
	-> true
	;  true
	),
	tcl_eval('wm deiconify .'),
 	tcl_eval(update),
	flag(tk_started,off,on(Bool)).

halt_x :-
	flag(tk_started,on(0)),
	!,
        tcl_eval('destroy .'),
	flag(tk_started,on(0),off).

halt_x :-
	warning('"Cannot terminate Tk session without terminating Prolog.\
Terminating Prolog will also terminate Tk"'),
	fail.

tk_menu_tcl_file('SP_menu.tcl').
tk_menu_tcl_file('question.tcl').
tk_menu_tcl_file('qbox.tcl').

tk_dir_slash(Path) :-
	source_file(tk_dir(_),File),
	concat(Path,'tk_menu.pl',File).

tk_dir(Path) :-
	source_file(tk_dir(_),File),
	concat(Path,'/tk_menu.pl',File).

send_dir :-
	tk_dir(Path),
	concat('set dir ',Path,Cmd),
	tcl_eval(Cmd).

tk_menu_tcl_files :-
	% find out directory we're in
	tk_dir_slash(Path),
	( tk_menu_tcl_file(F),
	  concat(Path,F,PathF),
	  concat('source ',PathF,Cmd),
	  tcl_eval(Cmd),
	  fail
	; true
	).


%% many of the following predicates are messy, mostly because
%% from tk we can only send commands to prolog with atomic arguments
%% and similarly we can only send commands to tk by putting the command
%% into one big atom

%%%%%%%%%%%%%%%%%%%%%%%
%%%%%% utilities %%%%%%
%%%%%%%%%%%%%%%%%%%%%%%

%%% send_flag(Flag)
%%% sets the tk variable flag(Flag) to the current value of the
%%% Prolog variable Flag 
%%%
%%% receive_flag(Flag,Val)
%%% is used by TK to change the value of Flag to Val

% send_flag(Flag)
send_flag(Flag,Val) :-
	flag(Flag,_Old,Val),
	concat_all([set,' flag(',Flag,') ',Val],Cmd),
	tcl_eval(Cmd).

% send_flag(Flag)
send_flag(Flag) :-
	flag(Flag,Val),
	concat_all([set,' flag(',Flag,') ',Val],Cmd),
	tcl_eval(Cmd).

% receive_flag(Flag,Val)
receive_flag(Flag,Val) :-
	flag(Flag,_,Val).

% special case for boolean flags, since the values used
% in tk are '0', '1' whereas we have on and off in Prolog
send_boolean_flag(Flag) :-
	flag(Flag,Val0),
	tr_val(Val0,Val),
	concat_all(['set ','flag(',Flag,') ',Val],Atom),
	tcl_eval(Atom).

send_boolean_flag(Flag,Val0) :-
	flag(Flag,_,Val0),
	tr_val(Val0,Val),
	concat_all(['set ','flag(',Flag,') ',Val],Atom),
	tcl_eval(Atom).

receive_boolean_flag(Val,Flag) :-
	tr_val(Val0,Val),
	flag(Flag,_,Val0).

tr_val(on,'1').
tr_val(off,'0').
tr_val(undefined,undefined).

warning(Atom) :-
	concat_all([tk_dialog,'.d','Warning',Atom,warning,0,ok],At,' '),
	tcl_eval(At).

send_pid :-
	unix(mktemp('XXXXXX',F)),
	name(F,[_|Tail]),
	number_chars(Pid,Tail),
	concat('set SP_pid ',Pid,Cmd),
	tcl_eval(Cmd).

%%%%%%%%%%%%%%%%%%%%%
%%%% menu(DEBUG) %%%%
%%%%%%%%%%%%%%%%%%%%%

% (un)setting spy-points from tk
% we need to parse the predicate specification
% send_spy(SpyOrNoSpy,AtomOfPredSpec)
%
%
send_spy(_Cmd,0) :- 
	!, fail.  % this case should be handled in .tcl file

send_spy(Cmd,Atom) :-
	atom_term(Atom,Spec),
	functor(Term,Cmd,1),
	arg(1,Term,Spec),
	call(Term).


