%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1989                                                %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%						          %
% Printer of terms representing feature structures, as    %
% defined in package feature.pl / c_feature.pl            %
%                                                         %
% added: trees of such animals for ftag                   %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- use_module(library(prettyvars)).
:- use_module(library(decons), [ prolog_conjunction/2 ] ).
:- use_module(library(lists),  [ member/2,
	                         append/3 ]).
% tty_fs/1
% tty_fs_clause/1
% tty_fs_listing/1

tty_fs(FS) :-
	print_tty_fs(fs(FS)).

tty_fs_clause((H:-B0)) :-
	!,
	ensure_prolog_conjunction(B0,B),          % lazyness
	print_tty_fs(clause(H,B)).

tty_fs_clause(Unit) :-
	print_tty_fs(clause(Unit)).

tty_fs_listing([H|T]) :-
	print_tty_fs(listing([H|T])).

print_tty_fs(listing(L)) :-
	!,
	( member(Thing,L),
	  (   Thing = (H:-B0) 
          ->  ensure_prolog_conjunction(B0,B),
	      print_tty_fs(clause(H,B))
          ;   print_tty_fs(clause(Thing))
          ),
	  fail
        ; true ).

print_tty_fs(Thing0) :-
	call_residue(copy_term(Thing0,Thing1),Cons0),
	user:rewrite_body(Cons0,[],Cons,[]),            % note: from link_clause.pl!!
	add_cons(Thing1,Cons,Thing2),
	prettyvars(Thing2,0,No),
	change_it_fs(Thing2,Thing,No),
	write_it_fs(Thing).

add_cons(clause(H,B0),C,clause(H,B)) :-
	append(B0,C,B).
add_cons(clause(H),[],clause(H)).
add_cons(clause(H),[C0|C],clause(H,[C0|C])).
add_cons(fs(X),[H|T],fs(X,[H|T])).
add_cons(fs(X),[],fs(X)).
add_cons(fs(X,C0),C1,fs(X,C)) :-
	append(C0,C1,C).

change_it_fs(clause(H0),clause(H),No) :-
	change_pretty_constraints([H0],[H],[],No).

change_it_fs(clause(H0,B0),clause(H,B),No) :-
	change_pretty_constraints([H0|B0],[H|B],[],No).

change_it_fs(fs(FS0,C0),fs(FS,C),No) :-
	change(FS0,FS,[],Out,No,OutNo),
	change_pretty_constraints(C0,C,Out,OutNo).

change_it_fs(fs(FS0),fs(FS),No) :-
	change(FS0,FS,[],_Out,No,_OutNo).

write_it_fs(fs(FS,C)) :-
	pp(FS,[]),
	write_pretty_constraints(C),
	nl.

write_it_fs(fs(FS)) :-
	pp(FS,[]),
	write('.'),nl.

write_it_fs(clause(H)) :-
	write_pretty_constraint(H),
	write('.'),nl.

write_it_fs(clause(H,B)) :-
	write_pretty_constraint(H), 
	write_pretty_constraints(B),
	nl.

pretty(FS) :-
	tty_fs(FS).

write_pretty_constraints([]) :-
	write('.'),nl.
write_pretty_constraints([H|T]) :-
	write(':-'),nl,
	write_pretty_constraint(H),
	write_pretty_constraints0(T).

write_pretty_constraints0([]) :- 
	write('.').
write_pretty_constraints0([H|T]) :-
	write(','),nl,
	write_pretty_constraint(H),
	write_pretty_constraints0(T).

write_pretty_constraint(H) :-
	H =.. [F|Args],
	write(F),
	write_begin_functor(Args),
	write_pretty_arguments0(Args),
	write_end_functor(Args).

write_begin_functor([]).
write_begin_functor([_|_]):-
	write('(').

write_end_functor([]).
write_end_functor([_|_]) :-
	write(')').

write_pretty_arguments0([H|T]):-
	pp(H,[]),
	write_pretty_arguments(T).

write_pretty_arguments([]).
write_pretty_arguments([H|T]):-
	write(','),
	pp(H,[]),
	write_pretty_arguments(T).

pp(_Var/n=FS,Tab):-
	!,
        ppl(FS,Tab).

pp(Var/_='R',_Tab):-
        !,
        write(' <'),
        write_var(Var),
        write('>').  

pp(Var/y=[],_Tab):-
        !,
        write(' <'),
        write_var(Var),
        write('>').

pp(Var/y='$VAR'(_),_Tab):-
	!,
        write(' <'),
        write_var(Var),
        write('>').

pp(Var/y=FS,Tab):-
        write(' <'),
        write_var(Var),
        write('>'), % what happened here?
	ppl(FS,Tab).

pp(lex(_:Lex),_Tab):-
	!,
	write(' "'),
	write(Lex),
	write('"').

pp(lex(Lex),_Tab):-
	!,
	write(' "'),
	write(Lex),
	write('"').

do_not_print(_Var/n='$VAR'(_)).

ppl([a(_Att,Thing)|Rest],Tab):-
	do_not_print(Thing),!,
	ppl(Rest,Tab).

ppl([a(type,Types)|T],Tab):-
	!,
	write(' {'),
	write_as_conj(Types,PTypes),
	write(PTypes),
	write('}'),
	ppl2(T,Tab).

ppl([a('BOOLEAN',_Type,Val)|T],Tab):-
	give_boolean_type(Val,Exp),
	write(' {'),write(Exp),write('}'),
	ppl2(T,Tab).

ppl([a('UNTYPED',Att,Val)|T],Tab):-
        append(Tab,[' |'],Tab2),
	write(' {U}'),nl,
	write_list(Tab2), 
	user:catch_print_error(Att,Val,Tab2),!,
	ppl2(T,Tab).

ppl([a('UNTYPED',_,_Val)|T],Tab):-
	!,      % should have been catched by catch_print_error
	write(' (error....)'),
	ppl2(T,Tab).

ppl([a(Att,FS)|T],Tab):-
        !,
        write(Att),
        name(Att,AttStr),
        length(AttStr,Length),
        append(Tab,[' |',tab(Length)],Tab2),
        pp(FS,Tab2),
        ppl2(T,Tab).

ppl([a('TREE',Mark,Cat,Ds)|T],Tab):-
	append(Tab,['/'],Tabt),
	mrk_t(Tabt,Mark,Tab2),
	pp(Cat,Tab2),
	pplist(Ds,1,Tab2),
	ppl2(T,Tab).

ppl([],_).

ppl('$VAR'(No),_) :- write('$VAR'(No)).  % changed gj 21/7/93 5/11/93

mrk_t(Tab,foot,Tab2):-
	!,
	append(Tab,['='],Tab2).
mrk_t(Tab,subs,Tab2):-
	!,
	append(Tab,['*'],Tab2).
mrk_t(Tab,_,Tab).


%pplist(lex(X),I,Tab):-
%	nl,write_list(Tab),write_list([I]),write(X).
pplist([],_,_).
pplist([H|T],I,Tab):-
	append(Tab,[I],Tab2),
	nl,write_list(Tab2),
	pp(H,Tab2),
	I2 is I + 1,
	pplist(T,I2,Tab).
pplist('$VAR'(_),_,_).

ppl2([a(_Att,Thing)|Rest],Tab):-
	do_not_print(Thing),!,
	ppl2(Rest,Tab).

ppl2([a(Att,FS)|T],Tab):-
        !,
	nl,write_list(Tab),write(' |'),
        write(Att),
        name(Att,AttStr),
        length(AttStr,Length),
        append(Tab,[' |',tab(Length)],Tab2),
        pp(FS,Tab2),
        ppl2(T,Tab).

ppl2([],_).

write_var(No):-
	write('$VAR'(No)).





% convert only if not already converted..
ensure_prolog_conjunction(A,B) :-
	prolog_conjunction(A,B),
	!.
ensure_prolog_conjunction(A,A).

write_list([]).
write_list([H|T]):-
	write_term(H),
	write_list(T).

% write_term(Exp)
% write an expression, which is either of a special form
% or otherwise simply written using `write'
% specials include:
%   variables
%   tab(I)
%   nl
%   msg(L)
%   A-B
%   msg2(L)
%
% write_term2(Exp) is similar, but writes an extra space after each
% written term

write_term(X):-
	special_write(X),
	!.

write_term(X):-
	print(X).

special_write(tab(X)):-
	tab(X),
	!.

special_write(nl) :-
	!,
	nl.




