%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              %
% written by Gertjan van Noord %
% (C) 1989-1994                %
%                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- use_module([ library(link_clause),
		library(flags)
	      ]).

:- use_module( library(concat), [ concat/3,
	                          concat_all/3 ]).

:- use_module( library(lists),  [ member/2,
	                          append/3 ]).

:- use_module([ 
		library(charsio)
	      ] ).


% general command interpreter
% ( inspired by Dominique Petitpierre's version
%   for Eurotra 1.2 - but much more powerful )
%
% r/0
%
% a general command interpreter package
% with the following niceties:
%
% history mechanism
% alias mechanism
% redirection of output to some file
% escape to unix shell
% easily extendible for task-specific commands
%
% There are two types of commands: commands defined
% here with dcg clauses built_in_cmd(Call). The argument of the
% built_in_cmd is the command eventually to be executed.
% The second type of command is a command defined
% by the user with dcg clauses of the form cmd(Call).
% Note that user defined commands have precedence over built-in
% commands; all built-in commands can be redefined!
% (however, don't: users may be used to the version defined here..)
% 
% A command is typed in by the user as one line of text
%
% the following meta-devices apply:
% all occurences of $word are replaced by the definition of
% the alias word. The alias command itself can be used to
% define aliases:
%
% 19 |: alias hallo ! cat hallo
% 20 |: $hallo
%
% so command number 20 will have the same effect as typing
%
% 33 |: ! cat hallo
%
% and if this command HAS been typed as command number 33 then
% typing
%
% 35 |: $33
%
% gives also the same result (history mechanism works only for last
% 20 commands)
%
% Moreover, if no alias has been defined, then it will apply the last
% command that started with the name of the alias:
%
% 66 |: parse john kisses mary
% 67 |: $parse
%
% will have the same meaning (in this order) if the macro parse 
% is not defined.
%
%
% Each command may be ended by > File to indicate that output
% has to be redirected to a file. A message will be issued if file
% already exist, and the user will be prompted for an optional 
% other file name.
% Note that commands try to eat their arguments maximally, so:
% 40 |: ! cat hallo > hallo2
%
% is interpreted so that the output redirection is part of the 
% shell command (and hence there will be no warning if the file
% hallo2 exists!)
%
% It is also possible to issue Prolog commands; however the following
% restrictions apply: only functor notation and/or list notation (thus
% no operators). It works by ending some prolog term with a dot! Usually
% however there must be a space BEFORE the dot (because the dot is not
% a syntax character to be able to read file names easily).
%
% 39 |: member(X,[X|T]).
%
% Note that this may succeed, but 'yes' or 'no' and variable bindings 
% will NOT be printed. The interpreter reads reentrencies correctly though.
% (there are some problems reading these prolog-terms and sometimes
%  the system will probably fail to understand you; you can always enter
%  prolog (by typing p) and entering the commands you want to prolog
%  directly)
%



% To use this package, simply define some dcg clauses for cmd(Call).
% Suppose you have written a small parser for which the command
% parse(ListOfWords-[]) will print the possible parses. One way of
% using this package then is to define:
%
% cmd(parse(ListOfWords-[])) -->
%        [*],
%       word_list(ListOfWords,+).
%
% Note that word_list/4 is a built-in predicate to simply parse a list
% of words, the second argument, + or * indicates whether or not the
% list can be empty.
% Now, to use this you can type 
%
% 4 |: * john kisses mary
%
% and parse([john,kisses,mary]-[]) will be executed. To repeat the
% same parse you can type $* or $4. You can also define aliases for
% (parts) of strings you want to parse several times (eg. during
% debugging of your programmetje)
%
%
% For an overview of built-in commands, issue the command bip
%
%
% You can define the predicate 'header/0'. Each time the command interpreter
% is started it will optionally call this predicate. 

add_command(P) :-
	add_linking_clause(P,ud_cmd,3).

del_command(P) :-
	del_linking_clause(P,ud_cmd,3).

r :- on_exception(Exception,
	          restart,
		  handle_exception(Exception)
	         ).

handle_exception(restart) :-
	nl, write('*** execution restarted ***'),nl,nl,
	r.

handle_exception(reserved(3)) :-
	nl, write('*** execution aborted ***'),nl, nl.

handle_exception(reserved(4)) :-
	nl, write('*** execution halted ***'),nl, nl,
	my_halt.

handle_exception(Exception) :-
	( print_uncaught_exception(Exception) -> true ; write(Exception),nl ),
	nl, write('*** exception occurred ***'),nl,nl,
	r.

:- assertz((print_uncaught_exception(permission_error(_,open,file,File,_)) :-
	format("the file ~w does not exist~n",[File])
          )).

try_header :-
	(  predicate_property(header,_)
        -> header 
        ;  true 
        ).

restart :-
        try_header,
        repeat,                     % repeat loop
        succc(N,Prompt),            % incorporate number in prompt
        prompt(_,Prompt),           % and redefine prompt
        read_line(Input_line,End_input,user),
        replace_aliases(Input_line,Input_line2),
        ( 
        \+ Input_line2 = [] 
        ; 
        End_input = end_of_file
        ),
        usr_cmd(N,Input_line2,End_input),
        !. % repeat cut
        


% usr_cmd/1 not normally used; might be used to execute commands
% from some prolog predicate (i.e. not directly by the user)
usr_cmd(Input_line):-
        parse_cmd(Cmd,user,Input_line,[]),
        user:Cmd.



usr_cmd(_N,_Input_line,end_of_file):-
        my_halt,
        !.

usr_cmd(N,[H|T],end_of_line) :-          % fail if Input line is empty
        (  
        parse_cmd(Cmd,File,[H|T],[]),
        succing,                  % increase counter if succesfully parsed
        alias([N,H|T])            % define alias for command counter
        -> 
        exec_cmd(Cmd,File)        % execute it
        ;  
        format(user_output,"*** Sorry I do not understand ***~n",[]),
        fail
        ).

exec_cmd(prolog,user) :-          % escape to prolog
	!,
	format(user_output,"*** execution interrupted ***~n",[]),
	nl(user_output).

exec_cmd(Cmd,File) :-             % the normal case
        switch_to(tell,File,Old), 
	( Cmd = M:Cmd2 -> true ; Cmd=Cmd2, M=user),
        M:Cmd2,
        switch_back_to(tell,Old),
        !,
        fail.        %failure gets back to read a new command

exec_cmd(Cmd,_) :-                % command was succesfully parsed,
                                  % but can't be executed:
	format(user_output,"~n*** error: cannot execute command:  \c
***~n~w~n",[Cmd]),
        fail.


% succc/0
% build prompt
succc(N,Prompt):-                 
        (
        succer(N);
        N=1
        ),
        concat(N,' |: ',Prompt),
        !.

% succing/0
% increase command counter and remove 
% commands from aliases that are too long ago
% (to save space; 20 can easily be changed)


succing:-                         
        (  
        retract_succer(N),
        N2 is N + 1,
        assert_succer(N2)
        ; 
        N = 2,
        assert_succer(2) 
        ),
        !,
        (  
        N > 20 
        -> 
        N3 is N - 20, 
        ( retract_user_alias(N3,_) -> true ; true )
        ;  
        true
        ).

succer(N) :-
        recorded(succer,succer(N),_).
assert_succer(N) :-
        recordz(succer,succer(N),_).
retract_succer(N) :-
        recorded(succer,succer(N),Ref),erase(Ref).

user_alias(A,B):-
        recorded(user_alias,user_alias(A,B),_).
assert_user_alias(A,B):-
        recorda(user_alias,user_alias(A,B),_).
retract_user_alias(A,B):-
        recorded(user_alias,user_alias(A,B),Ref),erase(Ref).

% replace_aliases(+ListOfWords,-ListOfWords)
% all words in the first list that start with
% a dollar sign are replaced by appropriate
% lists

% 1: finished:
replace_aliases([],[]).

% 2: a proper alias
replace_aliases([$,Word|T],Res ):-
        user_alias(Word,Y),
        !,
        replace_aliases(T,T2),
        append(Y,T2,Res).

% 3: find command that started with that word
replace_aliases([$,Word|T],Res ):-
        user_alias(_,[Word|Y]),
        !,
        replace_aliases(T,T2),
        append([Word|Y],T2,Res).

% 4: not a dollar sign, continue
replace_aliases([H|T],[H|T2]):-
        replace_aliases(T,T2).

% parse_cmd(-Cmd,-File)
% use dcg to parse command
% first cmd/3 is used, i.e. user defined command; 
% note the cuts

parse_cmd(X,File,P0,P) :-
        cmd(X,P0,P1),
        on_file(File,P1,P),!.

% new: user defined cmd cf. link_clause
parse_cmd(X,File,P0,P) :-
        ud_cmd(X,P0,P1),          % this is what is added by add_command/1
        on_file(File,P1,P),!.

parse_cmd(X,File) -->
        built_in_cmd(X),
        on_file(File),!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                          %
% DCG grammar for user commands            %
%                                          %
% should be defined, with top_node         %
% cmd(X)                                   %
% where X is something to be executed      %
% if not, then some built-in commands are  %
% used such as reconsult, shell, etc.      %
%                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% first try to parse it as one command; if not
% try a conjunction (there are some problems...)
built_in_cmd(X) -->
        b_cmd(X),
        !.

built_in_cmd((X,Y)) -->
        b_cmd(X),
        [(',')],
        built_in_cmd(Y).

% for comments:
b_cmd(true) -->
        ['%'],list_of_atoms(_,*),!.

% to set a flag
b_cmd((flag(Tag,_,New),wr_flag(Tag))) -->
        [flag],
        [Tag],
        prolog_term(New).

% to see a flag
b_cmd(wr_flag(Tag)) -->
        [flag],
        [Tag].

% fcompile list of files, sicstus only
b_cmd(fcompile(File_list)) -->
        abbrev(fcompile,fc),
        file_list(see,File_list,+).

b_cmd(use_module(File_list)) -->
        [um],
        file_list(see,File_list,+).

% compile a list of files
b_cmd(ensure_loaded(File_list)) -->
        [el],
        file_list(see,File_list,+).

% compile a list of files
b_cmd(compile(File_list)) -->
        abbrev(compile,c),
        file_list(see,File_list,+).

% reconsult a list of files
b_cmd(reconsult(File_list)) -->
        abbrev(reconsult,rc),
        file_list(see,File_list,+).

% load a list of files
b_cmd(load(File_list)) -->
        [ld],
	file_list(see,File_list,+).

% fcompile list of library files, sicstus only
b_cmd(use_module(File_list)) -->
        [libum],
        l_file_list(see,File_list,+).

b_cmd(fcompile(File_list)) -->
        [libfc],
        l_file_list(see,File_list,+).

% ensure_loaded a list of library files
b_cmd(ensure_loaded(File_list)) -->
        abbrev(libel,lib),
        l_file_list(see,File_list,+).

% compile a list of library files
b_cmd(compile(File_list)) -->
        [libc],
        l_file_list(see,File_list,+).

% reconsult a list of library files
b_cmd(reconsult(File_list)) -->
        [librc],
        l_file_list(see,File_list,+).

% load a list of library files
b_cmd(load(File_list)) -->
        [libld],
        l_file_list(see,File_list,+).

b_cmd(save(File)) -->
        [freeze],
        file(tell,File).

% show version
b_cmd(header) -->
        [version].

% prolog command
b_cmd(X) -->
        [p],
        prolog_term(X).

% escape to prolog
b_cmd(prolog) -->
        [p].

% halt
b_cmd(my_halt) -->
        [quit].

% enter prolog break
b_cmd(break) -->
        [b].

% help
b_cmd(help) -->
        [help].

% prolog debug
b_cmd(debug) -->
        [d].

% escape to unix shell
b_cmd((concat:concat_all(X,Cmd,' '),
       unix(shell(Cmd))
     )) -->
        [!],
        list_of_atoms(X,+).

% define/see aliases
b_cmd(alias(List)) -->
        [alias],
        list_of_atoms(List,*).

% show built-in commands
b_cmd(bip(C)) -->
        abbrev(bip,?),
	[C],
	 { bip(C,_) }.

b_cmd(bip) -->
	abbrev(bip,?).

% set spy point

b_cmd(spy(Module:Pred)) -->
	[spy],
	[Module],
	{current_module(Module)},
	pred(Pred).

b_cmd(spy(Pred)) -->
        [spy],
        pred(Pred).

b_cmd(unix(cd(Dir))) -->
	[cd],
	[Dir].
b_cmd(unix(cd('~'))) -->
	[cd].

b_cmd(unix(shell(pwd))) -->
	[pwd].

b_cmd(unix_shell(ls)) -->
	[ls].

% finally try to parse it as a prolog query (if ended by full stop)
% this doesn't always work; p Cmd is a bit safer
b_cmd(Term) --> prolog_term(Term),['.'].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                        %
% auxiliary predicates for built_in_cmd  %
%                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% for spy
pred(X/Y) -->
        [X],{atom(X)},
        ['/'],
        [Y],{integer(Y)}.
pred(X) -->
        [X],{atom(X)}.

% several predicates for files

file_list(SeeOrTell,[File|Words],_) -->
        file(SeeOrTell,File),
        file_list(SeeOrTell,Words,*).
file_list(_,[],*) -->
        [].

l_file_list(SeeOrTell,[library(File)|Files],_) -->
	file(see,File),
        l_file_list(SeeOrTell,Files,*).
l_file_list(_,[],*) -->
        [].

file(see,File) --> 
        [File]. 
%        {which(File,'.pl',_)},!.
%
%file(see,File) --> 
%        [File], 
%        { concat(File,'.pl',File2),
%          exists(File2)
%        }.

file(tell,File) --> 
        [File], 
        {\+ unix(access(File,4))},!.

file(tell,File2) --> 
        [File],
        { check_file(tell,File,File2) }.


list_of_atoms([H|T],_Mrk) -->
    [H],
    {atomic(H)},
    list_of_atoms(T,_).

list_of_atoms([],*) --> [].


%%%%%%%%%%
%        %
% ALIAS  %
%        %
%%%%%%%%%%


% alias
% no arguments: list of all aliases
alias([]) :-
      ( user_alias(X,Z),
        print(X=Z),
        nl,
	fail
      ; true
      ).


% one argument: show value of that particular alias
alias([H]) :-
        !,
        user_alias(H,Z),
	print(H=Z),
        nl.

% several arguments: set value of head of list to tail
alias([H|T]):-
	( retract_user_alias(H,_),
          fail
	; true 
        ), % remove old values
        assert_user_alias(H,T). % assertA for taking the last one easily


% show built-in commands:
%
%
bip :-  write('List of commands: '),nl,
      ( bip(_),
        fail
      ; true
      ).

bip(Cmd):-
	bip(Cmd,Msg),
	write(Cmd),write(:),nl,
	format(Msg,[]).

bip(redirect,     
"Cmd > File         to redirect output to File
Cmd >> File        to redirect output to File, 
                   overwriting File if File already exists").
bip(conjuction,   
"CmdA , CmdB        for a conjunction of commands").
bip(flag,         
"flag Flag          to execute flag predicate").
bip(reconsult,    
"rc File            to reconsult a file").
bip(fcompile,     
"fc File            to fcompile a file (Sicstus only)").
bip(compile,      
"c File             to compile a file (For C, YAP compile = reconsult)").
bip(load,         
"ld File            to load a compiled file (Sicstus only)").
bip(lib,          
"lib File           to compile lib/File if neccesary").
bip(lib-compile,  
"libc File          to compile lib/File").
bip(lib-reconsult,
"librc File         to consult lib/File").
bip(prolog,       
"p                  to enter prolog").
bip(quit,         
"quit               to quit cmd-int and prolog").
bip(break,        
"break              for a prolog break (type end_of_file to come back)").
bip(spy,          
"spy Pred/Ar
spy Pred           to spy a predicate").
bip(version,      
"version            to see the current version").
bip(unix,         
"! Command          for a shell command (under UNIX only)").
bip(cd,           
"cd Dir             to change directory to Dir").
bip(alias,        
"alias              to see all aliases
alias Name         to see a specific alias
alias Name Val     to set an alias").
bip(bip,          
"bip Bip            to get info on built-in predicate").
bip(comment,      
"% Words            don't do anything (comment, useful for batch jobs)").
bip(freeze,       
"freeze File        to freeze current program as a saved state in file File").
bip(prolog_call,  
"Call .             will execute that call").

bip(Cmd,Msg):-
	user_help(Cmd,Msg).

% for redirection of output:
% 1 : overwriting on a file
on_file(overwrite(File)) -->
	['>'],['>'],[File].    % file name is not checked.

% 2: a file
on_file(File) -->
        ['>'],[File].   % file name is not checked here, it
                        % be checked later..

% 3: user
on_file(user) --> 
        [].


% not a real abbreviation (too expensive) but one of
% two possible forms
abbrev(Full,Abb) -->
        ( 
        [Full]
        ; 
        [Abb]),
        !.


% list of words (can be anything) 
% a word is defined in read_line
% the second argument indicates whether the list
% can be empty (*) or not (something else).
word_list([Word|Words],_) -->
        [Word],
	{ word(Word)},
        word_list(Words,*).

word_list([],*) -->
        [].

word('>') :-
	!,
	fail.
word(_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                      %
% parse as a prolog term, works only for functor and list notation!    %
% keeps track of reentrencies of variables                             %
%                                                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

prolog_term(Term) -->
    prolog_term(Term,_In,_Out).  %to keep track of co_instantiated variables

prolog_term(Term,In,Out) -->
    prolog_var(Term,In,Out).

prolog_term(Term,In,Out) -->
    prolog_atom(Functor,In,In2),
    ['('],
    prolog_conj(List,In2,Out),
    [')'],
    { Term =.. [Functor|List] }.


prolog_term(Term,In,Out) -->
    prolog_atom(Term,In,Out).

prolog_term(Term,In,Out) -->
	prolog_integer(Term,In,Out).

prolog_term(List,In,Out) -->
    ['['],
    prolog_list(List,In,Out),
    [']'].

prolog_atom(Word,In,In) -->
    [Word],
    { name(Word,[H|String]),
      ( 128 > H,H > 96 ;
        H = 42, String = [])}.

prolog_atom([],In,In) -->
	['['],
	[']'].

prolog_integer(Integer,I,I) -->
	[Integer],
	{ integer(Integer) }.

prolog_var(Something,In,Out) -->
    [Word],
    { name(Word,[H|_String]),
      91 > H,H > 64 ,
      subst_member(Word,Something,In,Out)}.

prolog_var(Something,In,Out) -->
    [Word],
    { name(Word,[95|String]),
      ( String = [] 
      ->In = Out
      ; subst_member(Word,Something,In,Out))}.


prolog_conj([H|T],In,Out) -->
    prolog_term(H,In,In2),
    prolog_conj2(T,In2,Out).


prolog_conj2([H|T],In,Out) -->
    [','],
    prolog_term(H,In,In2),
    prolog_conj2(T,In2,Out).

prolog_conj2([],In,In) --> [].
    

prolog_list([H|T],In,Out) -->
    prolog_term(H,In,In2),
    prolog_list2(T,In2,Out).


prolog_list2([H|T],In,Out) -->
    [','],
    prolog_term(H,In,In2),
    prolog_list2(T,In2,Out).

prolog_list2(T,In,Out) -->
    ['|'],
    prolog_term(T,In,Out).

prolog_list2([],In,In) --> [].
    


subst_member(A,B,[],[A=B]):-!.
subst_member(A,B,[A=B|T],[A=B|T]):-!.
subst_member(A,B,[H|T],[H|T2]):-
    subst_member(A,B,T,T2).




% a list of prolog terms, with possibly coinstantiated
% variables:
sequence_of_terms(T,X) -->
	sequence_of_terms(T,X,[],_).
sequence_of_terms([H|T],_,In,Out) -->
    prolog_term(H,In,In2),
    sequence_of_terms(T,_,In2,Out).
sequence_of_terms([],*,X,X) --> [].




%%% ASK %%%%%

ask(List):-
	on_exception(stop,
        (   flag(ask,off)
        ->  nl
        ;
	repeat, write('? '),
	read_line(Line,_,user),
        replace_aliases(Line,Line2),
	find_answer(Line2,List,Command),
	call(Command),
	!
        ),
	true
        ).

find_answer([Line],List,Command):-
	member(Line=Command,List),
	!.
find_answer(Line,_,Cmd):-
	find_answer(Line,Cmd),
	!.
find_answer(_,Rest,fail):-
	findall(X,member(X=_,Rest),Rest2),
	concat_all(['<CR>',all,break,debug,trace,stop,restart|Rest2],Atom,' '),
	write(user_output,Atom),
	nl(user_output).

find_answer([],true).
find_answer([ask,off],flag(ask,_,off)).
find_answer([all],flag(ask,_,off)).
find_answer([break],break).
find_answer([b],break).
find_answer([restart],raise_exception(restart)).
find_answer([debug],debug).
find_answer([trace],trace).
find_answer([stop],raise_exception(stop)).
	
%%%%%%%%%%%%%
% ASK_LOOPS %
%%%%%%%%%%%%%
%
% ask_parsed_loop(Object,Stop,Call)
% Object is a result of some non-deterministic call to Prolog,
% eg. parser. User is prompted whether to inspect result,
% backtrack to new results, etc.
ask_parsed_loop(Object,List):-
	(  flag(ask,off)
        -> nl, fail
        ;  
	flag(ask,on),
	read_loop(Object,List,Stop),
	stop = Stop
        ).

read_loop(Object,List,Stop):-
        repeat,
	write('? '),
        read_line(Line,_,user),
	replace_aliases(Line,Line2),
        msg_parse(Line2,Call,Stop,Object,List),
	user:Call,
	!.

msg_parse_list(L,(C,fail),nostop,Object,List):-
	member(l(L,Object)=C,List).

msg_parse(L,C,S,List,O):-
	msg_parse_list(L,C,S,List,O),
	!.

msg_parse([],true,nostop,_,_):-
	!.
msg_parse([L],C,S,_List,_):-
	msg_parse(L,C,S),
	!.
msg_parse(_,true,nostop,_,Rest):-
	findall(X,member(l([X|_],_)=_,Rest),Rest2),
	concat_all(['one of ','<CR>',no,all,break,trace,debug,restart|Rest2],Atom,' '),
	write(user_output,Atom),
	nl(user_output),
	fail.

msg_parse(y,true,nostop).
msg_parse(yes,true,nostop).
msg_parse(n,true,stop).
msg_parse(no,true,stop).
msg_parse(all,true,nostop):-
	flag(ask,_,off).
msg_parse(a,true,nostop):-
	flag(ask,_,off).
msg_parse(break,(break,fail),nostop).
msg_parse(restart,(raise_exception(restart),fail),nostop).
msg_parse(trace,trace,nostop).

% hack to end gmlib's interface program
% ispgm --- otherwise keeps alive ad inf
my_halt :-
	(  predicate_property(started,_),
	   started
        -> end
        ;  true
        ),
	halt.


prompting(Generator,Name,Action) :-
	Generator,
	write(Name),
	read_stop(Action),
	!.
prompting(_,_,_).

read_stop(Action) :-
	flag(ask,off),!,
	nl, 
	call(Action),
	fail. 

read_stop(Action) :-
	write(' ? '),
	ttyflush,
	read_line(Terms),
	( Terms = [] -> Term = [] ; Terms = [Term|_] ),
	read_stop(Term,Action).

read_stop(stop,_).
read_stop(s,_).
read_stop(no,_) :-
	fail.
read_stop(n,_) :-
	fail.
read_stop(y,Action) :-
	Action,
	fail.
read_stop(yes,Action) :-
	Action,
	fail.
read_stop([],Action) :-
	Action,
	fail.


%%%% special read_line for this file. This is all very messy.
%%%% code should be rewritten by using standard tokeniser...


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1989                                                %
% Adapted from Dominique Petitpierre                      %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% read_line
%
%
% read_line(-Word_list,?End_input,+File)
% read_line(-Word_list)
% read_line(-Word_list,?End_input)
%
% read_line will read a line (ended with end of line)
% and yields a list of words - a word is something between
% spaces or between other 'syntax_char's

/*
:- module(read_l,[read_line/1,
                  read_line/2,
		  read_line/3,          %obsolete
		  read_line_from_file/2,
		  read_line_from_file/3,
		  read_line_from_stream/2,
		  read_line_from_stream/3,
		  read_line_from_chars/2,
		  read_line_from_chars/3
	         ]).
*/

read_line_from_file(File,Line) :-
	read_line_from_file(File,Line,_).

read_line_from_file(File,Line,End) :-
	seeing(Old),
	force_on_fail( ( see(File),
                         read_line(Line,End)
                       ),
                       ( seen,
	                 see(Old)
		       )
	             ).


read_line_from_stream(Stream,Line) :-
	read_line_from_stream(Stream,Line,_End).

read_line_from_stream(Stream,Line,End) :-
	current_input(Old),
	force_on_fail( ( set_input(Stream),
                         read_line(Line,End)
		       ),
                       ( set_input(Old),
                         close(Stream)
		       )).

read_line_from_chars(Chars,Line):-
	read_line_from_chars(Chars,Line,_End).

read_line_from_chars(Chars,Line,End) :-
	open_chars_stream(Chars,read,Stream),   % lib(charsio)
	read_line_from_stream(Stream,Line,End).

read_line(List):-
	read_line(List,_).

read_line(List,Char,File):-
	ttyflush,  %added for sicstus on HP
	switch_to(see,File,Old),
	read_line(List,Char),
	switch_back_to(see,Old).

read_line(Word_list,End_input) :-
        get_next(C_in),
        skip_space(C_in,C_out),
        read_words(C_out,Word_list,End_input).

read_words(8'32,[],end_of_file) :-
        !.

read_words(8'12,[],end_of_line) :-
        !.

read_words(-1,[],end_of_file):-
        !.

read_words(C_in,[Word|Words],End_input) :-
        read_word(C_in,Str,C_mid),
        name(Word,Str),
        skip_space(C_mid,C_out),
        read_words(C_out,Words,End_input).

read_word(Word):-
        get_next(C_in),
        read_word(C_in,Wordstr,_),
        name(Word,Wordstr).

read_word(C_in,[],C_in) :- 
        (
        is_space(C_in) 
        ; 
        is_end_input(C_in)),
        !.

read_word(C_in,[C_in],C_out):-
        syntax_char(C_in),
        get_next(C_out),
        !.

read_word(C_in,[C_in|Chars],C_out) :-
        get_next(C_mid),
        (  
        syntax_char(C_mid) 
        -> 
        Chars = [],
        C_out = C_mid
        ;  
        read_word(C_mid,Chars,C_out)
        ).


skip_space(C_in,C_out) :-
        is_space(C_in),
        !,
        get_next(C_mid),
        skip_space(C_mid,C_out).

skip_space(C_in,C_in).

get_next(-1) :-
	at_end_of_stream,!.   % sicstus 2.1.#8

get_next(Char) :-
	get0(Char).


is_space(8'40). % space

is_space(8'11). % tab


is_end_input(8'32).  % end of file = ^Z into which Unix's ^D is converted 
is_end_input(-1).

is_end_input(8'12).  % end of line = newline = line feed 



syntax_char(40). % (

syntax_char(41). % )

syntax_char(91). % [

syntax_char(93). % ]

syntax_char(123). % {

syntax_char(125). % }

syntax_char(44). % ,

syntax_char(59). % ;

syntax_char(124). % |

syntax_char(60). % <

syntax_char(62). % >

syntax_char(45). % -

syntax_char(43). % +

syntax_char(33). % !

syntax_char(42). % *

syntax_char(61). % =

syntax_char(63). % ?

syntax_char(36). % $

syntax_char(58). % :

syntax_char(92). % \

%syntax_char(46). % .  these two for filenames...

%syntax_char(47). % /                 ,,
  
% meta.pl
% meta predicates that are better not put in a module

% makes sure that TidyUp is called after Call, even if an error
% occurs. The error is simply passed on after tidying up. Don't
% do difficult things here, otherwise we might loose contact with
% Prolog...
%
% note: if Call simply fails, then TidyUp is NOT called (this is a
% feature, not a bug).
% use force_on_fail for the other possibility

force(Call,TidyUp) :-
	on_exception(Error,
                     call(Call),
                     ((  call(TidyUp) 
                      -> true 
                      ;  true 
                      ),
		      raise_exception(Error)
                     )
		    ),
	call(TidyUp).

force_on_fail(Call,TidyUp) :-
	force(Call,TidyUp),
	!.
force_on_fail(_Call,TidyUp) :-
	call(TidyUp),
	fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% adapted from Dominique Petitpierre                      %
% (C) 1989                                                %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% defines switch_to(+SeeOrTell,+New,?Old),
%         switch_back_to(+SeeOrTell,?New,+Old).

/* 
:- module(files,[switch_to/3,
	         switch_back_to/3,
		 switch_back_to/2,
		 which/3,
		 change_file/2,
		 check_file/3]).
*/



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                      %
%         FILE HANDLING predicates     %
%                                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% switch_to(+SeeOrTell,+New,?Old)
% switch_back_to(+SeeOrTell,?New,+Old)
switch_to(See_or_tell,New_file,Old_file) :-
        check_file(See_or_tell,New_file,New_file2),
        ing(See_or_tell,Old_file), %  fetch current stream file
        change_file(See_or_tell,New_file2),!,
        switch_to_hook(See_or_tell,Old_file).



% in case one tries to resatisfy switch_to, it does in fact a switch_back_to   %
% and fail (this prevent to loose contact with prolog!!).                       %

switch_to_hook(_,_).
switch_to_hook(See_or_tell,Old_file) :-
        switch_back_to(See_or_tell,Old_file),
        fail.


switch_back_to(SeeOrTell,_,Old):-
        switch_back_to(SeeOrTell,Old).


switch_back_to(see,Old_file) :-
        seen,
        change_file(see,Old_file),
	!. /*  restore old stream */

switch_back_to(tell,Old_file) :-
        told,
        change_file(tell,Old_file),
	!. /*  restore old stream */



check_file(_,user,user):-!.
check_file(tell,overwrite(File),File):-!.
check_file(tell,File,File) :-
        \+ exists(File),!.
check_file(tell,File,File2) :-
%        msg(['the file ',File,' already exists.',nl,
%              'Please type new name or <CR> to clobber: ']),
	format(user_output,
"the file ~w already exists.
Please type new name or <CR> to clobber: ",[File]),
        read_line(X,_,user),
        ( X = [] 
        -> File = File2 
        ; X = [H], check_file(tell,H,File2)
        ).

check_file(see,File,File) :-
        exists(File),!.

check_file(see,File,File2) :-
%        msg(['the file ',File,' does not exist',nl,
%             'please type another file name (or <cr> to stop)']),
	format(user_output,
"the file ~w does not exist.
Please type another file name (or <cr> to stop)",[File]),
        read_line([H],_,user),  %thus fails if []
        check_file(see,H,File2).



change_file(see,File) :-
        see(File).

change_file(tell,File) :-
        tell(File).

ing(see,File):-
        seeing(File).

ing(tell,File):-
        telling(File).






which(A,B,C):-
    which(A,B,C,_).
which(H,Ext,File,File2):-
    findall(F2,(library_directory(F),
                concat(F,'/',F2)),List),
    name(H,Hstr),
    name(Ext,ExtStr),
    member(Dir,[''|List]),  %current directory is searched first
    name(Dir,DirStr),
    append(DirStr,Hstr,FileStr),
    name(File,FileStr),
    ( exists(File),
      File = File2
    ;
      append(FileStr,ExtStr,FileQlStr),
      name(File2,FileQlStr),
      exists(File2)
    ),!.

exists(F):-
	unix(access(F,4)).  % file exists if you can read it.
