:- module(call_tree,
	[ 
          call_tree_bu/0,
	  call_tree_bu/1,
	  call_tree_bu_g/0,
	  call_tree_bu_g/1,
	  call_tree_bu_tk/0,
	  call_tree_bu_tk/1,
	  call_tree_bu_latex/0,
	  call_tree_bu_latex/1
        ]).

:- use_module( library(lists), [ nth/3,
	                         select/3 ]).

%% call_tree_bu(Functor)
%% pretty prints in a tree format the hierarchy related to the predicate
%% Fuctor/1 as follows. Functor/1 dominates all predicates that call Functor/1
%% in their body.
%%
%% e.g.
%% p(X) :- q(X). q(X) :- r(X). s(X) :- r(X).
%%
%% gives      r
%%           / \
%%          q   s
%%          |
%%          p
%%
%% other calls in the body are attached to the label, eg:
%% p(X) :- q(X). q(X) :- r(X). s(X) :- r(X), t(X).
%%
%% gives      r
%%           / \
%%          q   s [t]
%%          |
%%          p
%%
%% leaves of the tree can be defined by the user (e.g. to stop the tree at
%% interesting point, and to give interesting info in the label, cf.
%% user:call_leaf(Call,Label)
%%
%% It is assumed that the predicates are not recursive.

:- user:del_graphic(call_tree:gpx,call_tree:glx,call_tree:gdx).

gpx(t,X,X).
gdx(t,No,tree(_,_,Ds),D) :-
	nth(No,Ds,D).
glx(t,tree(L,_,_),L).

:- user:add_graphic(call_tree:gpx,call_tree:glx,call_tree:gdx).

call_tree_bu :-
	user:call_default(F),
	call_tree_bu(F).

call_tree_bu_g :-
	user:call_default(F),
	call_tree_bu_g(F).

call_tree_bu_tk :-
	user:call_default(F),
	call_tree_bu_tk(F).

call_tree_bu_latex :-
	user:call_default(F),
	call_tree_bu_latex(F).

call_tree_td :-
	user:call_default(F),
	call_tree_td(F).

call_tree_td_g :-
	user:call_default(F),
	call_tree_td_g(F).

call_tree_bu(F) :-
	Head =.. [F,_],
	user:call_clause(Head,Body),
	l_tree(Head+Body,Tree),
	p_tree:pretty_graphic(t,Tree).

call_tree_bu_g(F) :-
	Head =.. [F,_],
	user:call_clause(Head,Body),
	l_tree(Head+Body,Tree),
	gm_tree:gm_tree(t,Tree).

call_tree_bu_tk(F) :-
	Head =.. [F,_],
	user:call_clause(Head,Body),
	l_tree(Head+Body,Tree),
	tk_tree:tk_tree(t,Tree).

call_tree_bu_latex(F) :-
	Head =.. [F,_],
	user:call_clause(Head,Body),
	l_tree(Head+Body,Tree),
	latex:latex_tree(t,Tree).

l_tree(Def+Cons,tree(Label,_,[])) :-
	user:call_leaf(Def,L),
	!,
	build_lab(L+Cons,Label).

l_tree(Call+Body,tree(Label,_,Ds)) :-
	build_lab(CallF+Body,Label),
	functor(Call,CallF,_),
	findall(D0,a_d(Call,D0),Ds0),
	l_tree_ds(Ds0,Ds).

build_lab(F+Body,L):-
	functors(Body,Fs),
	( Fs = [] -> L = F ; L = F+Fs ).

functors([],[]).
functors([Term|T0],[F|T]) :-
	functor(Term,F,_),
	functors(T0,T).
	

a_d(Call,D+Body) :-
	user:call_clause(D,Body0),
	select(Call,Body0,Body).
	
l_tree_ds([],[]).
l_tree_ds([H0|T0],[H|T]):-
	l_tree(H0,H),
	l_tree_ds(T0,T).

:- user:del_command(call_tree:grammar_cmd).

grammar_cmd(call_tree:call_tree_bu(F),[lt,F|P],P).
grammar_cmd(call_tree:call_tree_bu_g(F),[glt,F|P],P).
grammar_cmd(call_tree:call_tree_bu,[lt|P],P).
grammar_cmd(call_tree:call_tree_bu_g,[glt|P],P).

:- user:add_command(call_tree:grammar_cmd).

