%% (c) Gertjan van Noord 1994 
%% control Sicstus with X (menus etc.)
%% using TK/TCL interface by M.Meier

:- use_module( library(tk)).

:- use_module( library(tk_menu) ).

:- use_module( library(lists),   [ member/2,
	                           append/3,
	                           reverse/2 ] ).

:- use_module( library(concat),  [ concat/3,
	                           concat_all/2,
				   concat_all/3,
				   term_atom/2,
				   atom_term/2
			         ]).

:- use_module( library(flags) ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%% HDRUG interface to TK %%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

tk_startup_hook :-
	source_file(tk_startup_hook,ThisFile),
	concat(Path,'tk.pl',ThisFile),
	concat(Path,'hdrug.tcl',TclFile),
	concat('source ',TclFile,Cmd),
	tcl_eval(Cmd).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%% checks which modules are used %%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

parser_exists :-
	flag(parser_mode,L),
	parser_exists(L).

parser_exists([]) :-
	tcl_eval('set parser_exists 0').

parser_exists([_|_]) :-
	tcl_eval('set parser_exists 1').


generator_exists :-
	flag(generator_mode,L),
	generator_exists(L).

generator_exists([]) :-
	tcl_eval('set generator_exists 0').

generator_exists([_|_]) :-
	tcl_eval('set generator_exists 1').

test_suite_exists :-
	(   a_sentence(_,_,_) 
	->  tcl_eval('set test_suite_exists 1')
	;   tcl_eval('set test_suite_exists 0')
	).

test_result_exists :-
	(   table_entry(_,_,_,_,Parser,_),
            flag(parser_mode,List),
	    member(Parser,List)              % check these are parse results
	->  tcl_eval('set test_result_exists 1')
	;   tcl_eval('set test_result_exists 0')
	).

object_exists :-
	(   object(_,_)
	->  tcl_eval('set object_exists 1')
	;   tcl_eval('set object_exists 0')
	).

type_exists :-
	(   feature:define_type(_,_,_,_,_)
	->  tcl_eval('set type_exists 1')
	;   tcl_eval('set type_exists 0')
	).

predicate_exists :-
	(   user_clause(_,_)
	->  tcl_eval('set predicate_exists 1')
	;   tcl_eval('set predicate_exists 0')
	).


%%%%%%%%%%%%%%%%%%%
%%%% MENU(run) %%%%
%%%%%%%%%%%%%%%%%%%

%% parsing / generation related
%%
%% to ask for parsing a sentence from tk we do
%%    prolog start_sent
%%    foreach i $sent {
%%       prolog "next_word $i"

start_sent :-
	flag(cur_sent,_,[]).

next_word(W) :-
	flag(cur_sent,S,[W|S]).

parse_sent :-
	flag(cur_sent,S0),
	reverse(S0,S),
	parse(S).

parse_sent_comp :-
	flag(cur_sent,S0),
	reverse(S0,S),
	parse_compare(S).

generate_atom(Atom) :-
	atom_term(Atom,Term),
	gen([Term]).

generate_atom_comp(Atom) :-
	atom_term(Atom,Term),
	generate_compare([Term]).

generate_object(No0) :-
	atom_chars(No0,Ch),
	number_chars(No,Ch),
	object(No,Obj),
	translate_obj(generate,Obj).

generate_object_comp(No0) :-
	atom_chars(No0,Ch),
	number_chars(No,Ch),
	generate_compare_object(No).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%% MENU(options) %%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

send_topcats :-
	findall(TopName,top(TopName,_),TopCats),
	concat_all(TopCats,Atom,' '),
	concat_all([set,topcats,'"',Atom,'"'],Cmd,' '),
	tcl_eval(Cmd).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%% MENU(view) %%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- use_module([ library(show),
	        library(tk_tree) ]).

:- initialize_flag(canvas,'.cv.canvas').

%% sh(Filter,Output,Things)
%% output: { user, latex, tk }
%% filter: { fs([]), sem, term(print), tree(Kind)

% show types
show_type_prolog_tree(Type0) :-
	atom_term(Type0,Type),
	feature:pretty_type_graphic(Type).
show_type_prolog_text(Type0) :-
	atom_term(Type0,Type),
	feature:pretty_type(Type).
show_type_canvas(Type0) :-
	atom_term(Type0,Type),
	feature:pretty_type_tk(Type).
show_type_latex(Type0) :-
	atom_term(Type0,Type),
	feature:pretty_type_latex(Type).

% show predicates
show_predicate_prolog_text(Pred0) :-
	atom_term(Pred0,F/A),
	show:all_defs(F,A,Clauses),
	sh(term(print),user,Clauses).

show_predicate_prolog_matrix(Pred0) :-
	atom_term(Pred0,F/A),
	show:all_defs(F,A,Clauses),
	sh(fs([]),user,Clauses).

show_predicate_latex_text(Pred0) :-
	atom_term(Pred0,F/A),
	show:all_defs(F,A,Clauses),
	sh(term(print),latex,Clauses).

show_predicate_latex_matrix(Pred0) :-
	atom_term(Pred0,F/A),
	show:all_defs(F,A,Clauses),
	sh(fs([]),latex,Clauses).

show_object_canvas_tree(Type,Obj) :-
	trans_object(Obj,Thing),
	sh(tree(Type),tk,Thing).

show_object_prolog_tree(Type,Obj) :-
	trans_object(Obj,Thing),
	sh(tree(Type),user,Thing).

show_object_prolog_semantics(Obj) :-
	trans_object(Obj,Thing),
	sh(sem,user,Thing).

show_object_prolog_matrix(Obj) :-
	trans_object(Obj,Thing),
	sh(fs([]),user,Thing).

show_object_prolog_text(Obj) :-
	trans_object(Obj,Thing),
	sh(term(print),user,Thing).

show_object_latex_tree(Type,Obj) :-
	trans_object(Obj,Thing),
	sh(tree(Type),latex,Thing).

show_object_latex_matrix(Obj) :-
	trans_object(Obj,Thing),
	sh(fs([]),latex,Thing).

show_object_latex_text(Obj) :-
	trans_object(Obj,Thing),
	sh(term(print),latex,Thing).

show_object_latex_semantics(Obj) :-
	trans_object(Obj,Thing),
	sh(sem,latex,Thing).

trans_object(NoAtom,[object(No,Obj)]) :-
	atom_chars(NoAtom,Chars),
	number_chars(No,Chars),
	object(No,Obj).

send_treedefs :-
	findall(G,(graphic_path(G,_,_),\+G=type),Gs),
	concat_all(Gs,GsAtom,' '),
	concat_all([set,treedefs,'"',GsAtom,'"'],Cmd,' '),
	tcl_eval(Cmd).

:- initialize_flag(show_object,No^show_object_prolog_semantics(No)).

show_object(No) :-
	flag(show_object,No^Pred),
	call(Pred).

%%%%%%%%%%%%%%%%%%%%%%%%
%%% MENU(test_suite) %%% 
%%%%%%%%%%%%%%%%%%%%%%%%

%%% information which parsers should be on/off 
send_parsers :-
	send_on(parser_mode,parser),
	send_off(parser_mode,parser),
	send_list(parser_mode,parsers).

send_generators :-
	send_on(generator_mode,generator),
	send_off(generator_mode,generator),
	send_list(generator_mode,generators).

% for each parser P that is on, set parser(P) to 1
% for each generator G that is on, set generator(G) to 1
send_on(PM,V) :-
	( flag(PM,List),
          member(P,List),
	  concat_all(['set ',V,'(',P,') 1'],Atom),
	  tcl_eval(Atom),
	  fail
        ; true).

% for each parser P that is off, set parser(P) to 0
% for each generator G that is off, set generator(G) to 0
send_off(PM,V):-
	( flag(off(PM),List),
	  member(P,List),
	  concat_all(['set ',V,'(',P,') 0'],Atom),
	  tcl_eval(Atom)
	; true ).

% set parsers to the list of existing (on/off) parsers 
% set generators to the list of existing (on/off) generators
send_list(PM,Vs) :-
	findall(P,apm(PM,P),L),
	append(L,['}'],L1),
	concat_all([set,Vs,'{'|L1],Atom,' '),
	tcl_eval(Atom).

apm(PM,P) :-
	flag(PM,L), 
	member(P,L).
apm(PM,P) :-
	flag(off(PM),L),
	member(P,L).

change_pm('0',Parser,PM) :-
	del_flag(PM,Parser),
	add_flag(off(PM),Parser).

change_pm('1',Parser,PM) :-
	del_flag(off(PM),Parser),
	add_flag(PM,Parser).

update_types :-
	write('Updating types...'),ttyflush,
	tcl_eval('set types {}'),
	( setof(Type,A^B^C^D^(feature:define_type(Type,A,B,C,D)),All),
	  ( member(T,All),
	    \+ T = top,     % send top as the last one seperately
	    send_a_type(T),
	    fail
	  ; send_a_type(top)
          )
        ; true
        ),
	write('Done.'),nl.

send_a_type(Type0) :-
	term_atom(Type0,Type),
	concat_all([set,' types',' [linsert $types 0 ',Type,' ]'],Atom),
	tcl_eval(Atom).

update_preds :-
	write('Updating predicates...'),ttyflush,
	tcl_eval('set preds {}'),
	( setof(X/Y,H^B^(user_clause(H,B),functor(H,X,Y)),All),
	  ( member(F/A,All),
	    send_a_pred(F,A),
	    fail
	  ; true 
          )
        ; true),
	write('Done.'),nl.

send_a_pred(F,A) :-
	concat_all([F,'/',A],Spec),
	concat_all([set,' preds',' [linsert $preds 0 ',Spec,' ]'],Atom),
	tcl_eval(Atom).


update_sents :-
	write('Updating sentences...'),ttyflush,
	tcl_eval('set sents {}'),
	( ( sentence(_,S)
          ; sentence(_,_,S)
          ),
          send_a_sentence(on(_),S),
	  fail
        ; true ),
	write('Done.'),nl.

send_a_sentence(off,_).
send_a_sentence(on(_),S) :-
	concat_all(S,Sent,' '),
	concat_all(['global sents ; set sents [eval {linsert $sents 0 ',
                                 '{',Sent,'}',' }',']'],Atom),
	tcl_eval(Atom).

send_a_lf(off,_).
send_a_lf(on(_),L) :-
	extern_sem([Ex],L),
	term_atom(Ex,Lf),
	concat_all(['global lfs ; set lfs [eval {linsert $lfs 0 ','{',Lf,'} }', ']'],Atom),
	tcl_eval(Atom).

update_objs :-
	find_current_no(No),
	concat('set objs ',No,Cmd),
	tcl_eval(Cmd).

hdrug_startup_hook_begin :-             % is called upon completion of
                                        % source hdrug.tcl

	send_flag(parser),
	send_flag(generator),
	send_flag(top_features),
	parser_exists,
        generator_exists,
        type_exists,
	update_types,
	update_preds,
	update_objs,
	update_sents,
	(  gram_startup_hook_begin   % allow for hooks in specific grammars
	-> true 
	;  true
	).

hdrug_startup_hook_end :-
	(  gram_startup_hook_end
        -> true
        ;  true
        ).


%% predicates called from Prolog
%%
%% i.e. these should succeed if tk interface is not started.

%% these predicates are used in go.pl
%% tk_retract_objects/1
%% tk_assert_object/1
%% tk_send_a_sentence/1
%% tk_send_a_lf/1

tk_retract_objects(No) :-
	flag(tk_started,Tk),
	tk_retract_objects(Tk,No).

tk_retract_objects(off,_).
tk_retract_objects(on(_),No) :-
	No > 1,
	concat('SP_delete_objects ',No,Cmd),
	tcl_eval(Cmd).
tk_retract_objects(on(_),1).

tk_assert_object(No) :-
	flag(tk_started,TK),
	tk_assert_object(TK,No).

tk_assert_object(off,_).
tk_assert_object(on(_),No) :-
	concat('SP_object ',No,Cmd),
	tcl_eval(Cmd).

tk_send_a_lf(L) :-
	flag(tk_started,Tk),
	send_a_lf(Tk,L).

tk_send_a_sentence(S) :-
	flag(tk_started,TK),
	send_a_sentence(TK,S).

