%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                         %
% written by Gertjan van Noord                            %
% (C) 1989 1990                                           %
%                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% GO.PL: calls the parsers/translaters/generators, defines
% path loops, create objects etc.

:- use_module([library(equi)          % for create_new_object
	       ]).

:- use_module( library(concat),  [ concat_all/3 ]).
:- use_module( library(flags)).
:- use_module( library(between), [ between/3 ]).
:- use_module( library(gen_sym), [ gen_sym/2 ]).
:- use_module( library(count_edges), [ count_edges/2]).
:- use_module( library(lists),       [ member/2 ]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   PARSE, GENERATE          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

gen(X):-
	extern_sem(X,Sem),
	semantics(Node,Sem),
        translate(generate,o(Node,_Phon,Sem)).

fparse :-
	flag(c_string,_,on),
	compile(user),
	flag(c_string,_,off),
	string(String),
	parse(String).

parse(X):-
	extern_phon(X,Phon),
	phonology(Node,Phon),
        translate(parse,o(Node,Phon,_Sem)).

translate_obj(PG,ExObj):-
	( extract(PG,ExObj,Obj),
	  translate(PG,Obj),
	  fail
	; true).

translate(A,O):-
	init_object(O),
	statistics(runtime,[Start,_]),
        translate2(A,O,Start),
	statistics(runtime,[End,_]),
	Total is (End - Start),
        format("cputime total ~w msec~n",[Total]).

translate2(PG,Obj,Time) :-
	(  flag(object_saving,semi)
	-> flag(current_no,No,1),
	   retract_objects(No)
	;  true
	),
        find_current_no(CNo),
        instantiate_top(Obj),
	format("~w: ",[PG]),
	short_before(PG,Obj),
        prove(PG,Obj),
	ask_message(PG,Obj,Time,CNo).

translate2(_,_,_).

init_object(o(Node,Phon,Sem)):-
	semantics(Node,Sem),
	phonology(Node,Phon).

extract(parse,o(_,Str,_),o(_,Str,_)).
extract(generate,o(_,_,Sem),o(_,_,Sem)).

other(parse,generate).
other(generate,parse).

%% this is were parser/generator are called proper:
prove(parse,Obj):-
	start_parse(Obj).

prove(generate,Obj):-
	start_generate(Obj).

%%%%%%%%%
% parse %
%%%%%%%%%

rgo :-
	rgo(_).

rgo(X) :-
	rt,
	go(X).

rt :-
	retractall(table_entry(_Name,_Length,_Amb,_Time,_Parser,_Edges)).	

go_l([]).
go_l([H|T]):-
	go(H),
	go_l(T).

rgo_l(L):-
	rt,
	go_l(L).

go :-
	statistics_run(_).

go(X) :-
	statistics_run(X).

% sentences are given as
% sentence(Key,MaxMilliSec,Sentence) or
% sentence(Key,Sentence) 
% in the latter case the maximum is computed by number of words
% times maximum_per_word

a_sentence(Ref,Max,Sentence):-
	( sentence(Ref,Max,Sentence)
        ; dyn_sentence(Ref,Max,Sentence)
        ).

a_sentence(Ref,Max,Sentence):-
	( sentence(Ref,Sentence)
	; dyn_sentence(Ref,Sentence)
        ),
	length(Sentence,Length),
	maximum_per_word(Length,Max).

sentences :-
	( a_sentence(A,B,C),
          write(sentence(A,B,C)),nl,
	  fail
        ; true
        ).

% this e.g. gives:
%  3 words   12700
%  8         29200
% 16         86800
% 25        197500
maximum_per_word(L,Max) :-
	user_max(L,Max),!.
maximum_per_word(L,Max) :-
	Max is 10000 + (L * L * 300).

print_table :-
	setof(table_entry(A,B,C,D,E,F),table_entry(A,B,C,D,E,F),Table),
	( member(El,Table),
	  write(El),nl,
	  fail
        ; true).

print_table_add :-
	setof(T,add_info(T),Table),
	( member(El,Table),
	  write(El),nl,
	  fail
        ; true).

add_info(t(Length,Av,Parser)):-
	parser_mode(Parser),
	setof(Time,Nm^Am^Ed^(table_entry(Nm,Length,Am,Time,Parser,Ed),\+Am=space_out),Times),
	sum_list(Times,0,Sum),
	length(Times,Len),
	Av is Sum/Len.

% fails if a parse with same length or longer already failed to terminate
% within max

:- flag(useful_try_check,_,on).

useful_try(Parser,Ref,_Sentence):-
	table_entry(Ref,_,_,_,Parser,_),
	!,
	format("already done ~w for ~w~n",[Ref,Parser]),
	fail.
useful_try(Parser,Ref,Sentence):-
	flag(useful_try_check,on),
	length(Sentence,Length),
	table_entry(Name,Length2,time_out,_,Parser,_),
	Length >= Length2,
	!,
        format("not doing ~w for ~w because of time-out of ~w",[Ref,Parser,Name]),
	fail.
useful_try(_,_,_).

statistics_run(Ref) :-
	a_sentence(Ref,Max,Sentence),
	parse_compare_st(Ref,Max,Sentence),
	fail;
	true.

parse_compare(Sentence):-
	gen_sym(Sym,'$'),
	assertz(dyn_sentence(Sym,Sentence)),
	statistics_run(Sym).

parse_compare(Max,Sentence):-
	gen_sym(Sym,'$'),
	assertz(dyn_sentence(Sym,Max,Sentence)),
	statistics_run(Sym).

parse_compare_st(Ref,Max,Sentence):-
	length(Sentence,Len),
	flag(ask,_,off),
	flag(object_saving,_,semi),
	( parser_mode(Parser),
	  flag(parser,_,Parser),
	  useful_try(Parser,Ref,Sentence),
	  time_out(parse(Ref,Sentence),Max,Succ),
	  (  Succ=time_out 
          -> format("timed out after ~w msec~n",[Max]),
	     flag(parser,P),
	     P:count(NoItems),
	     P:clean,
	     assertz(table_entry(Ref,Len,time_out,Max,Parser,NoItems))
          ;  true),
	  fail
        ; true
        ).

parse(Ref,Sentence):-
	length(Sentence,Len),
	extern_phon(Sentence,Phon),
	phonology(Node,Phon),
	init_object(o(Node,Phon,_Sem)),
	statistics(runtime,[Start,_]),
        translate2(parse,o(Node,Phon,_Sem),Start),
	statistics(runtime,[End,_]),
	Total is (End - Start),
        format("cputime total ~w msec~n",[Total]),
	flag(parser,Parser),
	Parser:count(NoEdges),
	Parser:clean,
	count_edges(object(_,_),NoAnalyses),
	assertz(table_entry(Ref,Len,NoAnalyses,Total,Parser,NoEdges)).


%%%%%%%%%%%%%%
% generation %
%%%%%%%%%%%%%%

generate_compare_object(No) :-
	object(No,o(_,_,Sem)),
	semantics(Node,Sem),
	extern_sem(X,Sem),
	init_object(o(Node,Phon,Sem)),
	generate_compare1(o(Node,Phon,Sem),X).

generate_compare(X):-
	extern_sem(X,Sem),
	semantics(Node,Sem),
	init_object(o(Node,Phon,Sem)),
	generate_compare1(o(Node,Phon,Sem),X).

generate_compare1(Obj,Ex) :-
	gen_sym(Ref,'g'),
	flag(ask,_,off),
	flag(object_saving,_,semi),
	( generator_mode(M),
	  flag(generator,_,M),
	  statistics(runtime,[Start,_]),
	  translate2(generate,Obj,Start),
	  statistics(runtime,[End,_]),
	  Total is (End - Start),
	  format("cputime total ~w msec~n",[Total]),
	  flag(generator,Generator),
	  Generator:count(NoEdges),
	  Generator:clean,
	  count_edges(object(_,_),NoAnalyses),
	  assertz(table_entry(Ref,Ex,NoAnalyses,Total,Generator,NoEdges)),
	  fail
	; true
        ).

% SHORT/2
% short(ParseGenerate,Obj)

show_time(Time):-
	statistics(runtime,[Now,_]),
	Total is (Now - Time),
        format("cputime passed ~w msec~n",[Total]).

short_after(parse,o(_,_,Sem)):-
	short_sem(Sem).

short_after(generate,o(_,Phon,_)):-
	short_phon(Phon).

short_before(parse,o(_,Phon,_)):-
	short_phon(Phon).

short_before(generate,o(_,_,Sem)):-
	short_sem(Sem).

short_sem(Sem):-
	extern_sem([ExtSem],Sem),
	write(ExtSem),nl.

short_phon(Phon):-
	extern_phon(ExtPhon,Phon),
	concat_all(ExtPhon,Atom,' '),
	write(Atom),nl.

%%%%%%%%%%%%%%%%%%%%%%
%                    %
% CREATE NEW OBJECT  %
%                    %
%%%%%%%%%%%%%%%%%%%%%%


%test for existence of object only if object exist in same 
%'found loop'!!

create_new_object(_No,_Val):-
	flag(object_saving,off),!.

create_new_object(No,Obj_val):-
        new_no(No),
        assert_object(No,Obj_val),
	format("created object: ~w~n",[No]).

create_new_object(_No,_Val,_MinNo):-
	flag(object_saving,off),!.

create_new_object(No,Obj_val,MinNo) :-
        object_does_not_exist(Obj_val,MinNo),
        new_no(No),
        assert_object(No,Obj_val),
	format("created object: ~w~n",[No]).

object_does_not_exist(Obj,MinNo):-
        find_current_no(Cur),
        between(MinNo,Cur,No),
        object(No,Obj2),
        equal(Obj,Obj2),
        !,
	format("(Found object identical to: ~w)~n",[No]),
        fail.

object_does_not_exist(_,_).

new_no(No) :-
	find_current_no(No),
	No2 is No + 1,
	flag(current_no,_,No2).

find_current_no(No):-
	flag(current_no,N),
	(   integer(N) 
        ->  No = N
	;   No is 1
	).

retract_objects(No) :-
	tk_retract_objects(No),
	( recorded(object,_,Ref),
	  erase(Ref),
	fail
        ; true).

assert_object(No,Val):-
	tk_assert_object(No),
	recordz(object,object(No,Val),_).

object(No,Val):-
	recorded(object,object(No,Val),_).

correct_id(I):-
	find_current_no(I2),
	I1 is I2 - 1,
	between(1,I1,I).

% ask_message(GP,Object)
% shortly shows object, saves object if os = on, asks for other commands concerning
% this object if ask=on

ask_message(GP,Object,Time,MinNo):-
	show_time(Time),
	create_new_object(_No,Object,MinNo),
	short_after(GP,Object),
	call_list(Object,List),
	ask_parsed_loop(Object,List).

call_list(O,[l([show],O) = show_pretty(long,O),
             l([s],O)    = show_pretty(long,O),
             l([g,N],O)  = p_graphic(N,O),
	     l([i],O)    = show_pretty(internal,O)
            ]).

p_graphic(N,o(O,_,_)):-
	pretty_graphic(N,O).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% definition of some smaller commands    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

i_top(undefined,_Cat):-
	!.
i_top(Name,Cat):-
	top(Name,Cat).

instantiate_top(o(Obj,_,_)):-
	flag(top_features,Name),
	i_top(Name,Obj),
	wr_flag(top_features,Name).

