:- module(right_chart_pack,[]).

:- use_module([ 
	        library(database),
	        library(flags)
	      ]).
:- use_module( library(wlists), [ wreverse/2 ] ).

:- add_flag(parser_mode,right_chart_pack).

% right_chart.pl + packing.

% this is an inactive chart parser with packing.
% The idea is that the items used for packing
% forget about delayed constraints. This makes
% administration of those items cheap, but 
% recovering of parse trees very expensive.

clean :-
	clean_up_database(right_chart_pack:inactive(_,_,_,_)),
	clean_up_database(right_chart_pack:tt(_,_,_,_,_,_)).

parse(o(Node,String,_)) :-
	length(String,Max),
	bu_lr(0,Max),
	write('recover parse trees'), nl,
	apply_semantics(Node,0,Max).

bu_lr(I,I) :- !.

bu_lr(I0,I):-
	lexical_lookup(I0,Agenda),
	process(Agenda),
	I1 is I0 + 1,
	bu_lr(I1,I).

lexical_lookup(I0,Agenda) :-
	findall(Edge,lookup(I0,Edge),Agenda).

lookup(I0,inactive(M,I0,I)):-
	user:ign_lex(I0,I,M,_Name).

% process(+ListOfEdges)
% process a list of edges
process([]).
process([Edge|OldAgenda]):-
	process_one(Edge,SubAgenda),         %% process each one in turn
	append(SubAgenda,OldAgenda,Agenda),     %% depth first!
	process(Agenda).                     %% process the rest

% process_one(+Edge,-Agenda)
% depending on the form of Edge, builds all new edges from Edge 
% by prediction or completion
process_one(inactive(Cat,P0,P1),Agenda):-
	findall(Edge,completer(Cat,P0,P1,Edge),Agenda).

% completer(+Cat,+P1,+P,-Edge)
% selects active edge whose rightmost symbol matches Cat
completer(Cat,P1,P,Edge):-
	apply_rule(Cat,P1,P,M,P0,P,Tree),
	store(M,P0,P,Tree,Edge).   % store in database

% rightmost is not the head
apply_rule(Cat,P3,P,Mother,P0,P,tree(Rule,t(P1,P2,Head),Lds,Rds)) :-
	user:ign_h_rule(Head,Mother,Ls,Rs,Rule),
	wreverse(Rs,[Cat|RestRs]),
	select_lefties(RestRs,P2,P3,[t(P3,P,Cat)],Rds),
	an_inactive(P2,P1,Head),
	select_lefties_nr(Ls,P0,P1,Lds).

% rightmost is the head
apply_rule(Head,P1,P,Mother,P0,P,tree(Rule,t(P1,P,Head),Lds,[])) :-
	user:ign_h_rule(Head,Mother,Ls,[],Rule),
	select_lefties_nr(Ls,P0,P1,Lds).

select_lefties([],P,P,His,His).
select_lefties([H|T],P0,P,HisIn,HisOut):-
	an_inactive(P,P1,H),
	select_lefties(T,P0,P1,[t(P1,P,H)|HisIn],HisOut).

select_lefties_nr([],P,P,[]).
select_lefties_nr([H|T],P0,P,[t(P1,P,H)|HisOut]):-
	an_inactive(P,P1,H),
	select_lefties_nr(T,P0,P1,HisOut).

an_inactive(P,P0,Cat) :-
	inactive(P,P0,Cat,Cons),
	call(user:Cons).
an_inactive(P,P0,Cat) :-
	user:ign_lex(P0,P,Cat,_).

% store(+M,+P0,+P,+Tree,-Edge)
% stores an edge, and depending on form gives different representation
% of the same edge back. If the edge already exists, then failure.
store(M,P0,P,tree(Rule,Head,Lds,Rds),inactive(M,P0,P)):-
	assert_tt(tt(P0,P,Rule,Head,Lds,Rds)),
	assert_if(P,P0,M).

assert_if(P,P0,Term0) :-
	call_residue(user:copy_term(Term0,Term),Constraints),
	user:rewrite_body(Constraints,[],true,NConstraints),
	assertz_most_general(right_chart_pack:inactive(P,P0,Term,NConstraints),no).

% for tt forget about constraints, these will be
% evaluated at unpack time
assert_tt(Pred):-
	call_residue(user:copy_term(Pred,NPred),_),
	assertz_most_general(right_chart_pack:NPred,_).

list :-
	listing(inactive/4),
	listing(tt/6).

count :-
	count_edges(right_chart_pack:inactive(_,_,_,_),B),
	write(B),write(' inactive edges'),nl,
	count_edges(right_chart_pack:tt(_,_,_,_,_,_),D),
	write(D),write(' tt edges'),nl.

count(B) :-
	count_edges(right_chart_pack:inactive(_,_,_,_),B).

% apply_semantics(Cat,P0,P,Module)
% builds results on the basis of packed representation,
% using the rules WITH the semantics.
% It is assumed that in Module the predicate tt represents
% the packed forest, such that tt(P0,P,Name,Daughters), where
% P0, P start/end positions, Name is rulename, and Daughters is
% a list of t(P0,P,Cat) triples, or


apply_semantics(Mother,P0,P) :-
	user:lex(P0,P,Mother,_).
apply_semantics(Mother,P0,P):-
	tt(P0,P,Name,t(Q0,Q,Head),Ls,Rs),
	user:h_rule(Head,Mother,Lds,Rds,Name),
	apply_semantics(Head,Q0,Q),
	apply_ds(Rs,Rds),
	apply_ds(Ls,Lds).

apply_ds([],[]).
apply_ds([t(P0,P,Cat)|Tail],[Cat|DT]):-
	apply_semantics(Cat,P0,P),
	apply_ds(Tail,DT).


