:- module(earley,[]).

:- use_module(library(lists), [ append/3 ]).


:- use_module([ library(flags),
	        library(database)
	      ]).

:- use_module( library(count_edges), 
	[ count_edges/2,
	  report_count_edges/1 ]).

:- add_flag(parser_mode,earley).

clean :-
	clean_up_database(earley:inactive(_,_,_)),
	clean_up_database(earley:active(_,_,_,_,_)),
	clean_up_database(earley:tt(_,_,_,_)),
	retractall(pr(_,_)).

parse(o(Node,String,_)) :-
	flag(ticks,_,0),
	initialize(Node,Agenda),
	process(Agenda),
	left_right(String,0,Max),
	packing:apply_semantics(Node,0,Max,earley).

% left_right(+String,+TopCat,+Max)
left_right([],Max,Max).
left_right([_|T],I1,Max) :-
	scanner(I1,Agenda),   % scan next one
	process(Agenda),         % process resulting edges
	I2 is I1 + 1,            
	left_right(T,I2,Max).   % move to the right in sentence

% process(+ListOfEdges)
% process a list of edges
process([]).
process([Edge|OldAgenda]):-
	process_one(Edge,SubAgenda),         %% process each one in turn
	append(SubAgenda,OldAgenda,Agenda),  %% depth first!
	process(Agenda).                     %% process the rest

% initialize(+TopCat,-Agenda)
% builds edges for the root category S of the form: S --> .X 
initialize(TopCat,Agenda) :-
	findall(Edge,init(TopCat,Edge),Agenda).

init(TopCat,Edge):-
	user:ign_rule(TopCat,Body,Name),
	store(Body,TopCat,0,0,tree(Name,[]),Edge).

% process_one(+Edge,-Agenda)
% depending on the form of Edge, builds all new edges from Edge 
% by prediction or completion
%
% for inactive edge: completion
% for active edge: prediction, but also completion with
% respect to gaps!
%
% for lexical edges, do nothing

process_one(inactive(Cat,P0,P1),Agenda):-
	findall(Edge,completer(Cat,P0,P1,Edge),Agenda).

process_one(active(M,[H|T],P0,P,His),Agenda):-
	user:restriction(H,HR),
	findall(Edge,predictor(HR,P,Edge),Agenda1),
	findall(Edge,gap_completer_e(M,[H|T],P0,P,His,Edge),Agenda2),
	append(Agenda1,Agenda2,Agenda).

gap_completer_e(M,[H|T],P0,P,tree(R,His),Edge):-
	inactive(P,P,H),
	append(His,[t(P,P,H)],His2),
	store(T,M,P0,P,tree(R,His2),Edge).

% scanner: scans the next lexical entry
scanner(I1,Agenda) :-
	findall(Edge,scanner_one(I1,Edge),Agenda).

scanner_one(I0,Edge) :-
	user:ign_lex(I0,I,M,Name),
	store([],M,I0,I,tree(Name,[]),Edge).

% completer(+Cat,+P1,+P,-Edge)
% selects active edge whose leftmost symbol matches Cat
% ('move-dot')
completer(Cat,P1,P,Edge):-
	active(P0,P1,M,[Cat|Rest],tree(MR,His2)),       % select active edge
	append(His2,[t(P1,P,Cat)],His3),
	store(Rest,M,P0,P,tree(MR,His3),Edge).   % store in database

% if the completer is called with an empty edge, then it may be
% the case that not all edges are already around, for which completion
% is possible...

% predictor(Cat,P,Edge)
% predicts a rule with mother Cat, which might start at P
predictor(CatR,P,_):-
	numbervars(pr(CatR,P),0,_),
	pr(CatR,P),
	!,
	fail.  % hence already predicted
predictor(CatR,P,Edge):-
	assertz(pr(CatR,P)),
	user:ign_rule(CatR,Body,Name),
	store(Body,CatR,P,P,tree(Name,[]),Edge).

% store(+Edge,-Edge)
% stores an edge, and depending on form gives different representation
% of the same edge back. If the edge already exists, then failure.
store(_,_,_,_,_,_):-
	flag(ticks,V),
	V2 is V+1,
	flag(ticks,_,V2),
	fail.
store([],M,P0,P,tree(Rule,Ds),inactive(M,P0,P)):-
	assertz_most_general(earley:tt(P0,P,Rule,Ds),_),
	assertz_most_general(earley:inactive(P0,P,M),no).

store([H|T],M,P0,P,His,active(M,[H|T],P0,P,His)):-
	assertz_most_general(earley:active(P0,P,M,[H|T],His),no).

count(No):-
	count_edges(earley:active(_,_,_,_,_),Act),
	count_edges(earley:inactive(_,_,_),InAct),
	No is Act+InAct.

count :-
	count_edges(earley:active(_,_,_,_,_),Act),
	count_edges(earley:inactive(_,_,_),InAct),
	count_edges(earley:tt(_,_,_,_),Tt),
	flag(ticks,V),
	format("active: ~w~n inactive: ~w~n tt: ~w~n ticks: ~w~n",
                [Act,InAct,Tt,V]).

list :-
	listing(earley:active),
	listing(earley:inactive),
	listing(earley:tt).




pack_rule(Name,Mother,Ds) :-
	user:rule(Mother,Ds,Name).



