(eval-when (eval compile)
  (require 'records)
  (require 'traps))

;;;  A few utilities to begin...

(defun average (&rest num-list)
  (round (apply '+ num-list)
         (length num-list)))

(defun lastelt (list)
  (car (last list)))

;;;  Now down to work!

(defobject *graphic-view* *scroller-view*)
(defobject *graphic-window* *scrolling-window*)

(ask *graphic-view*
  (have 'border-x 10)
  (have 'border-y 10)
  (have 'top-object nil)
  (have 'selected-object nil))

(defobfun (exist *graphic-window*) (init-list) 
  (usual-exist (init-list-default init-list 
                                  :window-font '("monaco" 9)
                                  :view-class *graphic-view*)))

(defobfun (view-close *graphic-view*) ()
  (when (objvar selected-object)
    (setf (objvar selected-object) nil))
  (usual-view-close))

(defobfun (view-draw-contents *graphic-view*) ()
  (usual-view-draw-contents)
  (when (objvar selected-object)
    (ask (objvar selected-object)
      (highlight))))

(defobfun (view-click-event-handler *graphic-view*) (where)
  (when (objvar selected-object)
    (ask (objvar selected-object)
      (deselect))
    (setq selected-object nil))
  (if (objvar top-object)
    (let ((result 
           (ask (objvar top-object)
             (click-event-handler (point-h where) (point-v where)))))
      (when result
        (have 'selected-object result)
        (ask result (select))))))

(defobfun (draw *graphic-view*) (object)
  (when (objvar selected-object)
    (setf (objvar selected-object) nil))
  (let ((view (self))
        (x (objvar border-x))
        (y (objvar border-y)))
    (ask object 
      (have 'view (objvar view))
      (layout x y)))
  (have 'right (+ (objvar border-x) (ask object (objvar right))))
  (have 'bottom (+ (objvar border-y) (ask object (objvar bottom))))
  (set-field-size (objvar right) (objvar bottom))
  (install-picture #'(lambda ()
                       (ask object
                         (draw)))))

(defobfun (draw *graphic-window*) (object)
  (ask (objvar scroll-view)
    (draw object)))

(setq *graphics-obj* (kindof nil))

(defobfun (exist *graphics-obj*) (init-list) 
  (usual-exist init-list))

(defobfun (mid-x *graphics-obj*) ()
  (round (+ left right) 2))

(defobfun (mid-y *graphics-obj*) ()
  (round (+ top bottom) 2))

(defobfun (head-x *graphics-obj*) ()
  (mid-x))

(defobfun (head-y *graphics-obj*) ()
  (- (mid-y) (round (ask view (font-info)) 2)))

(defobfun (descendants *graphics-obj*) ()
  '())

(defobfun (click-event-handler *graphics-obj*) (x y)
  (if (and (<= left x right)
           (<= top y bottom))
    (or (some #'(lambda (d) (ask d (click-event-handler x y)))
              (descendants))
        (self))
    nil))

(defobfun (highlight *graphics-obj*) ()   ; draws/erases the famous 6 dots
  (let ((top (objvar top))
        (left (objvar left))
        (bottom (objvar bottom))
        (right (objvar right))
        (mid (average (objvar left) (objvar right))))
    (ask (objvar view)
      (with-focused-view (self)
        (set-pen-size #@(3 3))
        (set-pen-mode :patXor)
        (move-to left top) (line-to (+ left 1) top)
        (move-to mid top) (line-to (+ mid 1) top)
        (move-to right top) (line-to (- right 1) top)
        (move-to left bottom) (line-to (+ left 1) bottom)
        (move-to mid bottom) (line-to (+ mid 1) bottom)
        (move-to right bottom) (line-to (- right 1) bottom)
        (pen-normal)))))
      
(defobfun (select *graphics-obj*) ()
  (highlight))

(defobfun (deselect *graphics-obj*) ()
  (highlight))
          
          
(setq *string-obj* (kindof *graphics-obj*))

(defobfun (exist *string-obj*) (init-list)
  (usual-exist init-list)
  (have 'string 
        (princ-to-string (getf init-list :string "*unspecified string*"))))

(defobfun (size-x *string-obj*) ()
  (let ((s (objvar string)))
    (ask (objvar view) (string-width s))))

(defobfun (size-y *string-obj*) ()
  (multiple-value-bind (ascent descent widmax leading) (ask (objvar view) (font-info))
    (+ ascent descent)))

(defobfun (head-y *string-obj*) ()
  (objvar top))

(defobfun (layout *string-obj*) (left top)
  (have 'top top)
  (have 'left left)
  (have 'bottom (+ top (size-y)))
  (have 'right (+ left (size-x))))

(defobfun (draw *string-obj*) ()
  (let ((left left)
        (top top)
        (bottom bottom)
        (right right))
    (ask view
      (erase-rect left top right (1+ bottom))
      (move-to left (+ top (font-info))))
    (with-port (ask view wptr)
      (with-pstrs ((s string))
        (_DrawString :ptr s)))))

(setq *tree-obj* (kindof *graphics-obj*))

(ask *tree-obj*
  (have 'dY 8)
  (have 'dX 12))

(defobfun (exist *tree-obj*) (init-list)
  (usual-exist init-list)
  (have 'label (getf init-list :label nil))
  (have 'daughters (getf init-list :daughters nil)))

(defobfun (descendants *tree-obj*) ()
  daughters)

(defobfun (layout *tree-obj*) (x y)
  (let ((dX dX)                  ; make object variables into lexical bindings
        (dY dY)
        (daughters daughters)
        (view view)
        (xx x) 
        (ymax y)
        yy diff label-size)
    (labels ((layout-daughters ()
               (dolist (d daughters)
                 (ask d
                   (have 'view view)
                   (layout xx yy)
                   (setq xx (+ right dX))
                   (setq ymax (max ymax bottom))))))
      (have 'top y)
      (have 'left x)
      (ask label (have 'view view))
      (setq yy (+ y dY (ask label (size-y))))
      (layout-daughters)
      (have 'bottom ymax)
      (have 'right (- xx dX))
      (setq label-size (ask label (size-x)))
      (setq diff (- label-size (- right left)))
      (when (> diff 0)
        (setq xx (+ x (round diff 2)))
        (have 'right (+ x label-size))
        (layout-daughters))
        
    (if daughters
      (progn
        (ask label
          (layout (- (round (+ (ask (first daughters) (head-x))
                               (ask (first (last daughters)) (head-x)))
                            2)
                     (round (size-x) 2))
                  y)))
      (progn
        (ask label
          (layout x y))
        (have 'bottom (ask label bottom))
        (have 'right (ask label right)))))))

(defobfun (head-x *tree-obj*) ()
  (ask label (head-x)))

(defobfun (head-y *tree-obj*) ()
  (ask label (head-y)))

(defobfun (draw *tree-obj*) ()
  (let ((start-x (ask label (head-x)))
        (start-y (- (ask label bottom) 2)))   ; move line up 
    (dolist (d daughters)
      (let ((end-x (ask d (head-x)))
            (end-y (ask d top))) 
        (ask view
          (move-to start-x start-y)
          (line-to end-x end-y))
        (ask d
          (draw))))
    (ask label
      (draw))))

(defun list-to-tree (list)
  (if (consp list)
    (oneof *tree-obj* :label (oneof *string-obj* :string (first list))
                      :daughters (mapcar #'list-to-tree (rest list)))
    (oneof *string-obj* :string list)))

;;;  This function draws a tree representation in a tree window

(defun drawTree (tree &key (selectable nil))
  "draws a tree in a specially created Tree Window window"
  (if (listp tree)
    (setq tree (list-to-tree tree)))
  (let ((front (front-window)))
    (if (and (boundp '*draw-tree-window*) 
             (typep *draw-tree-window* *graphic-window*))
      (ask *draw-tree-window*
        (unless (and (ownp 'wptr)
                     wptr)
          (exist '(:window-title "Tree Window"
                   :window-size #@(160 160)))))
      (setq *draw-tree-window* (oneof *graphic-window* 
                                      :window-title "Tree Window"
                                      :window-size #@(160 160))))
    (ask *draw-tree-window* 
      (window-show)
      (draw tree)
      (if selectable
        (ask scroll-view
          (have 'top-object tree)))
      (window-select))
    (ask front (window-select))))

(setq *avm-tree-obj* (kindof *tree-obj*))

(defobfun (select *avm-tree-obj*) ()
  (usual-select)
  (drawAvm (avs-to-avm avm)))
          
(defobfun (exist *avm-tree-obj*) (init-list)
  (usual-exist init-list)
  (have 'avm (getf init-list :avm nil)))


;;;  Attribute-value matrices

(setq *avpair-obj* (kindof *graphics-obj*))

(ask *avpair-obj*
  (have 'dX 5)
  (have 'attribute-indent 5))

(setq *avm-obj* (kindof *graphics-obj*))

(ask *avm-obj*
  (have 'dY 3)
  (have 'dX 5)
  (have 'xhandle 4)
  (have 'empty-height 3/2)
  (have 'empty-width 15))

(defobfun (exist *avpair-obj*) (init-list)
  (usual-exist init-list)
  (have 'attribute (getf init-list :attribute nil))
  (have 'value (getf init-list :value nil)))

(defobfun (exist *avm-obj*) (init-list)
  (usual-exist init-list)
  (have 'avpairs (getf init-list :avpairs nil)))

(defobfun (head-y *avpair-obj*) ()
  (ask attribute (head-y)))

(defobfun (head-y *avm-obj*) ()
  (if (= 1 (length avpairs))
    (ask (first avpairs) (head-y))
    (if (= 0 (length avpairs))
      top
      (usual-head-y))))

(defobfun (size-x *avpair-obj*) ()
  (let ((v view))
    (ask attribute (have 'view v))
    (ask value (have 'view v)))
  (+ dX attribute-indent (ask attribute (size-x)) (ask value (size-x))))

(defobfun (size-y *avpair-obj*) ()
  (let ((v view))
    (ask attribute (have 'view v))
    (ask value (have 'view v)))
  (max (ask attribute (size-y)) (ask value (size-y))))

(defobfun (size-x *avm-obj*) ()
  (let ((v view))
    (if avpairs
      (+ (apply #'max (mapcar #'(lambda (avp) 
                                  (ask avp (have 'view v))
                                  (ask avp (size-x))) avpairs))
         dX)
      empty-width)))

(defobfun (size-y *avm-obj*) ()
  (let ((v view))
    (if avpairs
      (+ (apply #'+ (mapcar #'(lambda (avp) 
                                (ask avp (have 'view v))
                                (ask avp (size-y))) avpairs))
         (* (1+ (length avpairs)) dY))
      (round (* (ask v (font-info)) empty-height)))))
  

(defobfun (layout *avpair-obj*) (x y)
  (let ((view view)
        (attribute attribute)
        (value value)
        (dX dX)
        (ai attribute-indent))
    (ask attribute               ; pass views on to descendants
      (have 'view view))
    (ask value
      (have 'view view)
      (layout (+ x dX ai (ask attribute (size-x))) y))
    (ask attribute
      (layout (+ x ai) (ask value (head-y))))
    (have 'top y)
    (have 'left x)
    (have 'bottom (ask value bottom))
    (have 'right (ask value right))))
    
(defobfun (layout *avm-obj*) (x y)
  (have 'top y)
  (have 'left x)
  (if avpairs
    (let ((view view)
          (lastelt (lastelt avpairs))
          (dY dY)
          (dX dX))
      (labels ((layout-list (ps xx yy)
                            (when ps
                              (ask (first ps)
                                (have 'view view)
                                (layout xx yy)
                                (layout-list (rest ps) xx (+ bottom dY))))))
        (layout-list avpairs x (+ y dY)))
      (have 'bottom (+ dY (ask lastelt bottom)))
      (have 'right (+ dX (reduce #'(lambda (r o) (max r (ask o right)))
                                 avpairs :initial-value 0))))
    (let ((h (ask view (font-info)))) 
      (have 'bottom (+ top (round (* h empty-height))))
      (have 'right (+ left empty-width)))))

(defobfun (draw *avpair-obj*) ()
  (ask value (draw))
  (ask attribute (draw)))

(defobfun (draw *avm-obj*) ()
  (let ((top top)
        (left left)
        (bottom bottom)
        (right right)
        (xhandle xhandle))
    (ask view
      (erase-rect left top right bottom)
      (_PicComment :word 0 :word 0 :long 0)   ;  Put in a left pic parenthesis
      (move-to (+ left xhandle) top)
      (line-to left top)
      (line-to left bottom)
      (line-to (+ left xhandle) bottom)
      (move-to (- right xhandle) top)
      (line-to right top)
      (line-to right bottom)
      (line-to (- right xhandle) bottom)
      (_PicComment :word 1 :word 0 :long 0))   ;  Put in a right pic parenthesis
    (dolist (p avpairs)
      (ask p
        (draw)))))

(defun list-to-avm (list)
  (if (listp (first list))
    (oneof *avm-obj* 
           :avpairs (mapcar #'(lambda (p)
                                (oneof *avpair-obj* 
                                       :attribute (oneof *string-obj*
                                                         :string (car p))
                                       :value (list-to-avm (cdr p))))
                            list))
    (oneof *string-obj* :string (first list))))

(defun drawAvm (avm)
  "draws an avm in a specially created Avm view view"
  (if (listp avm)
    (setq avm (list-to-avm avm)))
  (let ((front (front-window)))
    (if (and (boundp '*draw-avm-window*) 
             (typep *draw-avm-window* *graphic-window*))
      (ask *draw-avm-window*
        (unless (and (ownp 'wptr)
                     wptr)
          (exist '(:window-title "Avm Window"
                   :window-position #@(166 44)
                   :window-size #@(160 160)))))
      (setq *draw-avm-window* (oneof *graphic-window* 
                                      :window-title "Avm Window"
                                      :window-position #@(166 44)
                                      :window-size #@(160 160))))
    (ask *draw-avm-window* 
      (window-show)
      (draw avm)
      (window-select))
    (ask front (window-select))))

;;;  These routines translate the representation of AVMs into graphic
;;;  objects


(setq *indexed-avm-obj* (kindof *avm-obj*))

(ask *indexed-avm-obj*
  (have 'index nil)
  (have 'index-gap 2))

(defobfun (size-x *indexed-avm-obj*) ()
  (if index
    (let ((w view))
      (ask index (have 'view w))
      (+ index-gap (usual-size-x) (ask index (size-x))))
    (usual-size-x)))

(defobfun (layout *indexed-avm-obj*) (x y)
  (if index
    (let ((view view))
      (ask index (have 'view view))
      (usual-layout (+ (ask index (size-x)) index-gap x) y)
      (let ((bottom bottom))
        (ask index
          (layout x (- bottom (size-y))))))
    (usual-layout x y)))

(defobfun (draw *indexed-avm-obj*) ()
  (usual-draw)
  (if index
    (ask index
      (draw))))

;;;  These routines draw the boxed strings

(setq *boxed-string-obj* (kindof *string-obj*))

(defobfun (draw *boxed-string-obj*) ()
  (let ((left left)
        (top top)
        (bottom bottom)
        (right right)
        (size (find-if #'numberp (ask view (window-font))))
        (descent (second (multiple-value-list (font-info)))))
    (ask view
      (set-window-font (max 9 (floor (* 3 size) 4)))
      (erase-rect left top right (1+ bottom))
      (move-to (+ left (floor size 5)) 
               (- bottom descent)))
    (with-port (ask view wptr)
      (with-pstrs ((s string))
        (_DrawString :ptr s)))
    (ask view
      (set-window-font size)
      (frame-rect left top right bottom))))

(defobfun (size-x *boxed-string-obj*) ()
  (+ (usual-size-x) (floor (find-if #'numberp (ask view (window-font))) 5)))
