;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: FUG5 -*-
;;; -----------------------------------------------------------------------
;;; File:         fd-to-graph4.l
;;; Description:  Convert an FD from Lisp to the Grapher input format
;;;               Relativize-fd and Optimize-fd.
;;;               Version with hash-table for path-table.
;;;               New version of gdp - keep cycles.
;;; Author:       Michael Elhadad
;;; Created:       5 Nov 1993
;;; Modified:     19 Nov 1993: Fixed 2 bugs - deal with special-att
;;;                            In (a {b}) register both {a} and {b}.
;;;               26 Nov 1993: Moved to structure representation.
;;;                            Cleaned up usage of special vars.
;;;                            Fixed bug of (a {x y z}) to also register
;;;                            {x} and {x y}.
;;;               28 Nov 1993: Add *check-ambiguities* flag.
;;;               06 Jan 1994: Fixed bugs in build-id-table.
;;;               16 Jan 1994: More bugs in build-id-table
;;;                            Used different labels for visited to locate.
;;;               21 Feb 1994: Still working on it...
;;; Package:      FUG5
;;; -----------------------------------------------------------------------

;;; TODO: (optimize-fd fd) which replaces all paths with the shortest
;;; equivalent path possible.
;;; This could become the first draft of the next version of gdp...


(in-package "FUG5")

;;; The grapher input format is a file with the following specs:
;;; ### list_nodes
;;; ### list_edges
;;; ### list_fuf_nodes
;;; <optional label> <unique id>   # Repeated for each node in the fd
;;; ### list_fuf_edges
;;; <id1> <id2> label              # Repeated for each arc in the fd

;;; The following example illustrates the transformation:
;;; ((a 1)
;;;  (b ((b1 1) (b2 2)))
;;;  (c {b b1}))
;;;
;;; ### list_nodes
;;; ### list_edges
;;; ### list_fuf_nodes
;;; A a1
;;; B a2
;;; 1 a3
;;; 1 a4
;;; 2 a5
;;; ### list_fuf_edges
;;; a1 a2 a
;;; a1 a3 b
;;; a1 a4 c
;;; a3 a4 b1
;;; a3 a5 b2
;;;
;;; Note that each node correspond to an instantiated path in the FD.

;; ==================================================
;; Data Structures
;; ==================================================
;; Eventually, there is one id for each node in the graph
;; but there can be more ids than nodes at time - for equations like:
;; (a {x y z}) when {x y z} is not physically present in the graph.
;; Therefore, I cannot use the eq pointer to the pairs in the fd as ids.
;; This means it is hard to go from the result of gdp to the id.

(defparameter *check-ambiguities* nil
  "Does relativize-fd check that the output does not contain ambiguous
relative paths (value t), or does it rely on the grammar disambiguation
convention (ie, do we assume that the output of relativize-fd will be used
as a grammar) (value nil).")

;; Key is path
(defvar *path-table* (make-hash-table :test #'equal :size 500)
  "A hashtable of all triplets (path id value) identified in the fd during
  fd-to-graph conversion indexed by path. (value is non nil only for
  leaves.)")  

(defvar *edge-table* nil
  "A list of triplets (id1 id2 label) recording every edge during
   fd-to-graph conversion")

(defvar *new-physical-path* (make-hash-table :test #'eq)
  "A hash table to remember relocation of fds during relocate and optimize.")

(defparameter *unbound* (gensym) 
  "Mark that a path-entry has not received a value.")

(defstruct path-entry
  path
  (id (gentemp "N"))
  (visited nil)
  (value *unbound*))

(defun path-entry-bound (pe)
  (not (eq (path-entry-value pe) *unbound*)))

(defun path-entry-unbound (pe)
  (eq (path-entry-value pe) *unbound*))

(defstruct edge-entry
  id1
  id2
  label)

(defun find-entry (path)
  "Return the path entry for a given path"
  (gethash (path-l path) *path-table*))

(defun add-entry (&key path (id (gentemp "N")) visited (value *unbound*))
  "Add an entry to the hash-table."
  (setf (gethash (path-l path) *path-table*)
	(make-path-entry :path path :id id :visited visited :value value)))

(defun path-to-id (path &optional (*path-table* *path-table*))
  "Convert a path to an id.  Assume that path-table has already been built."
  (let ((entry (find-entry path)))
    (assert (path-entry-p entry) (path)
	    "Internal problem in fd-to-graph: *path-table* not complete.~%~
             Path ~s missing." path)
    (path-entry-id entry)))

(defun path-to-id2 (path *path-table*)
  "Convert a path to an id if found, return nil otw."
  (let ((entry (find-entry path)))
    (when entry (path-entry-id entry))))

(defun find-id (id &optional (*path-table* *path-table*))
  (maphash #'(lambda (pathl pe) 
	       (declare (ignore pathl))
	       (if (eq id (path-entry-id pe)) (return-from find-id pe)))
	   *path-table*))

(defun id-to-path (id &optional (*path-table* *path-table*))
  "Convert id to path in the graph representation of an FD."
  (let ((entry (find-id id)))
    (if (path-entry-p entry) (path-entry-path entry) nil)))

(defun parents (id &optional (*edge-table* *edge-table*))
  "Return a list of all parents of a node in an FD."
  (loop for e in *edge-table*
	do if (eq (edge-entry-id2 e) id)
	      collect (edge-entry-id1 e)))


;; ==================================================
;; Equivalence class management
;; ==================================================

;; First build a quotient set. In a quotient class, every entry has the same
;; id and only one entry has a non-nil value (the one where the value is
;; physically stored).  All the values are nil in case of a cycle or a
;; non-instantiated class.

(defun quotient-set (path-table)
  "Get a hash-table of path entries and return an array of lists
   where each list is an equivalence class with shortest path first
   in every class.  Indexed by id."
  (let ((q (make-hash-table :size 500)))
    (maphash #'(lambda (pathl pe)
		 (let* ((id (path-entry-id pe))
			(current (gethash id q)))
		   (setf
		    (gethash id q)
		    (if (and current
			     (< (path-len (path-entry-path (first current)))
				(length pathl)))
		       (cons (first current) (cons pe (rest current)))
		      (cons pe current)))))
	     path-table)
    q))

(defun rep (path quotient &optional pe)
  "Find the representative path of a path given the quotient set.
   If the path's path entry is known, use it."
  (let ((class (get-class path quotient pe)))
    (if class (path-entry-path (first class)))))

(defun get-class (path quotient &optional pe)
  "Get the equivalence class of a path given the quotient set.
   If path's path-entry is known use it, else find it."
  (let ((pe (or pe (find-entry path))))
    (if (path-entry-p pe)
	(gethash (path-entry-id pe) quotient))))


;; ==================================================
;; Build the tables for paths and edges
;; ==================================================

(defun build-id-table (fd path)
  (let ((*input* fd))
    (clrhash *path-table*)
    (build-id-table-aux fd path path)))

;; phys-path: where does fd appear physical in the total fd.
;; path: where we come from visiting fd.
;; Example: fd = ((a {b c}) (b ((c ((d 1))))))
;; can call (build ((d 1)) {a} {b c}) to add ids for paths {a d}.
;; Assumption: fd is in canonical form (list of pairs with no path on
;; left). 
;; Associate a unique id to each existing path in the fd
;; Return a table of path/id/optional label
;; This is a depth-first search of the physical FD graph repeated as many
;; times as necessary for structure shared subgraphs.  Ie, if an arc is
;; shared 3 times (it belongs to 3 distinct paths), it is traversed 3
;; times. 
;; Conflations are computed here so that 2 paths leading to the same
;; node receive the same id.

(defun build-id-table-aux (fd path phys-path)
  ;; (format t "~&p = ~s / pp = ~s" path phys-path)
  (let ((pe1 (find-entry path))
	(pe2 (find-entry phys-path)))
    (cond 
     ;; ((and pe1 (path-entry-visited pe1)) *path-table*)
     ((leaf-p fd)
      (multiple-value-bind (pe1 pe2) (merge-ids phys-path path fd pe1 pe2)
	  (setf (path-entry-visited pe2) t))
      *path-table*)
     ((path-p fd)
      (setf fd (absolute-path fd phys-path));; phys-path to interpret ^.
      ;; We now have 4 paths playing a role here (not necessarily distinct):
      ;; path and phys-path: where we are.
      ;; fd and phys-to: where we are told to go.
      ;; All 4 have to receive the same id.
      ;; ****** Check cycles up: ((a ((b ((c {a}))))))
      (multiple-value-bind (val phys-to missing cycle) (gdp *input* fd)
	(unless (path-null missing)
	  ;; Uninstantiated path: make sure all arcs of path get an id 
	  ;; from below the physical leaf (phys-to) to the end of fd.
	  (add-chain phys-to missing)
	  (setf phys-to (path-append phys-to missing)))
	;; Same path as 1st argument each time, so that at the end all
	;; receive the id of phys-to.
	(let ((pe3 (find-entry phys-to))
	      (pe4 (find-entry fd)))
	  (multiple-value-setq (pe3 pe1) 
	      (merge-ids phys-to path val pe3 pe1))
	  (multiple-value-setq (pe3 pe2) 
	      (merge-ids phys-to phys-path val pe3 pe2))
	  (multiple-value-setq (pe3 pe4)
	      (merge-ids phys-to fd val pe3 pe4))
	  ;; Compute all extensions of path in val.
	  ;; The extensions of fd, phys-to and phys-path will be computed when
	  ;; they will be traversed physically (in the regular depth first
	  ;; traversal). 
	  (setf (path-entry-visited pe2) t)
	  (unless (or cycle (path-entry-visited pe3))
	    (build-id-table-aux val path phys-to)))
	*path-table*))
     (t
      (multiple-value-setq (pe2 pe1) (merge-ids phys-path path fd pe2 pe1))
      (setf (path-entry-visited pe2) t)
      (mapc #'(lambda (pair) 
		(assert (consp pair) (pair fd)
			"Ill-formed fd in fd-to-graph: ~s" pair)
		(if (special-p (first pair))
		    (add-entry 
		     :path (path-extend path (first pair))
		     :visited :v8
		     :value (second pair))
		  (build-id-table-aux (second pair)
				      (path-extend path (first pair))
				      (path-extend phys-path (first pair)))))
	    fd)
      *path-table*))))


;; Conflate ids for paths p1 and p2
;; Make sure both have an id.
;; If ids already exist and are different, replace p2 with p1 everywhere.
;; If only one exists, use its id.
(defun merge-ids (p1 p2 val pe1 pe2)
  (cond
   ((and (null pe1) (null pe2))
    (setf pe1 (add-entry :path p1 :value val))
    (setf pe2 (add-entry :path p2 :value val :id (path-entry-id pe1))))
   ((and pe1 pe2)
    (unless (eq (path-entry-id pe1) (path-entry-id pe2))
      (maphash #'(lambda (l p)
		   (declare (ignore l))
		   (if (eq (path-entry-id p) (path-entry-id pe2))
		       (setf (path-entry-id p) (path-entry-id pe1))))
	       *path-table*)))
   ((null pe1)
    (setf pe1 (add-entry :path p1 :value val :id (path-entry-id pe2))))
   ((null pe2)
    (setf pe2 (add-entry :path p2 :value val :id (path-entry-id pe1)))))
  (values pe1 pe2))


(defun add-chain (phys-leaf extension)
  "Phys-leaf points to a leaf in the physical fd.
  extension is a path that must be defined under this leaf.
  For each arc in extension, make sure an id exists.
  Ex: fd = ((a {x y z}) (x ((b 1))))
  (add-chain {x} {y z}) creates ids for {x y} and {x z}."
  (loop for arc in (path-l extension)
	initially (setf p phys-leaf)
	for p = (path-extend p arc)
	do (unless (find-entry p)
	     (add-entry :path p :value nil))))



;; ==================================================-
;; OPTIMIZE-FD
;; ==================================================

(defun optimize-fd (fd)
  (let* ((*input* fd)
	 (*path-table* (make-hash-table :test #'equal :size 500))
	 (*edge-table* nil)
	 (*path-table* (build-id-table fd {}))
	 (root-entry (find-entry {}))
	 (*edge-table* (build-edge-table fd {} root-entry))
	 (quotient (quotient-set *path-table*)))
    (maphash #'(lambda (pathl pe) 
		 (declare (ignore pathl))
		 (setf (path-entry-visited pe) nil))
	     *path-table*)
    ;; First sweep through fd, then patch missing uninstantiated paths
    ;; Those remain unvisited after the sweep. Add them as equations:
    ;; ({uninstantiated-path} {rep})
    (let ((res (optimize-fd-aux fd quotient root-entry)))
      (maphash 
       #'(lambda (pathl pe)
	   (unless (or (path-entry-visited pe)
		       (special-p (car (path-last (path-entry-path pe)))))
	     (let* ((val (rep (path-entry-path pe) quotient pe)))
	       (unless (equal pathl (path-l val))
		 (nconc res (list (list (path-entry-path pe) val)))))))
       *path-table*)
      res)))

(defun optimize-fd-aux (fd quotient pe)
  "Get an FD with absolute paths, and optimize it to make gdp processing
   more efficient.  This includes 2 steps:
   1/ If (a {b}) (b leaf) and leaf is a true leaf (cannot be specialized),
      then optimize (a {b}) into (a leaf).
   2/ If there are indirections or conflations, replace all conflations
      with the shortest possible path equivalent:
      ((a {b}) (b {c}) (c {e1 e2}) (e1 ((e2 ((e3 3))))))
      -> ((a ((e3 3))) (b {a}) (c {a}) (e1 ((e2 {a}))))"
  (cond 
   ((leaf-p fd) 
    (setf (path-entry-visited pe) :v1)
    fd)
   ;; If have a path and the value is a real leaf, replace conflation by
   ;; value, else replace conflation with shortest path, making sure not
   ;; to create a cycle at the representative site.
   ((path-p fd) 
    (setf fd (absolute-path fd (path-entry-path pe)))
    (setf (path-entry-visited pe) :v2)
    (let ((class (get-class fd quotient)))
      (if (every #'real-leaf-p class)
	  (path-entry-value (first class))
	(let ((rep (path-entry-path (first class))))
	  (if (path-equal (path-entry-path pe) rep)
	      (let ((entry (find-if-not #'null class 
					:key #'path-entry-value)))
		;; If nil, means either there is a cycle, or 
		;; there is nothing instantiated: resolve it with nil.
		(unless entry
		  (format t "~&Cycle at class: ~s" class))
		(when entry
		  ;; First optimize the value, then mark the path as
		  ;; visited
		  (setf (path-entry-value entry)
			(optimize-fd-aux (path-entry-value entry)
					 quotient
					 entry))
		  (setf (path-entry-visited entry) :v3)
		  (path-entry-value entry)))
	    (progn
	      ;; (setf (path-entry-visited (first class)) :v4)
	      rep))))))
   ;; If the visited path has already been visited, return a path to its
   ;; representative (no need to traverse down)
   ((path-entry-visited pe)
    (rep (path-entry-path pe) quotient pe))
   ;; Else recurse down into fd.
   (t 
    (setf (path-entry-visited pe) :v5)
    (let ((path (path-entry-path pe)))
      (mapcar 
       #'(lambda (pair)
	   (list (car pair)
		 (if (special-p (car pair))
		     (second pair)
		   (let ((newpath (path-extend path (car pair))))
		     (optimize-fd-aux (second pair) 
				      quotient 
				      (find-entry2 newpath quotient))))))
       fd)))))


(defun real-leaf-p (object)
  (and object
       (leaf-p object)
       (null (subtype object))))



;; ======================================================================
;; Relocate
;; ======================================================================

;; First find the physical-representant of rpath so that relative paths are
;; correctly interpreted in the sub-fd.
(defun relocate (total rpath)
  (let* ((*input* total)
	 (*path-table* (make-hash-table :test #'equal :size 500))
	 (*path-table* (build-id-table total {}))
	 (q1 (quotient-set *path-table*))
	 (*new-physical-path* (make-hash-table :test #'eq)))
    (multiple-value-bind (const rpath missing cycle) (gdp total rpath)
      (cond 
       (cycle nil)
       ((path-null missing)
	(relocpairs q1 rpath rpath (make-path) const (copy-tree const)))
       (t nil)))))


;; q1 = the quotient set for conflation for total
;; rpath = from where do we relocate
;; tpath = where are we within total
;; cpath = where are we within the relocated fd (result)
;; pairs = the current fd being processed
;; result = accumulator for the relocated constituent
(defun relocpairs (q1 rpath tpath cpath pairs result)
  (if (leaf-p pairs)
    pairs
    (loop for pair in pairs
          for pair-ind = 0 then (+ pair-ind 1)
	  do (relocpair q1 rpath tpath cpath pair-ind pairs result)
	  finally (return result))))

;; pair-ind = index of pair within the fd
(defun relocpair (q1 rpath tpath cpath pair-ind pairs result &aux p)
  (let* ((pair (nth pair-ind pairs))
	 (feature (first pair))
	 (value (second pair)) 
	 (new-cpath (path-extend cpath feature))
	 (new-tpath (path-extend tpath feature)))
    ;; (format t "~&new-tpath = ~s - new-cpath = ~s" new-tpath new-cpath)
    (cond ((setf p (get-new-physical-path new-tpath))
	   (setf (second (nth pair-ind (top-gdp result cpath)))
		 (make-relative-path new-cpath p)))
	  ((leaf-p value) result)
	  ;; Problem with pattern and cset having paths in their values.
	  ((member feature *special-attributes*)
	   (setf (second (nth pair-ind (top-gdp result cpath)))
		 (copy-special value feature cpath)))
	  ((not (path-p value))
	   (relocpairs q1 rpath new-tpath new-cpath value result)
	   (record-new-physical-path new-tpath new-cpath))
	  (t
	   (multiple-value-bind (pointed-val point-to missing cycle)
	       (gdp *input* (absolute-path value new-tpath))
	     (cond
	      (cycle
	       ;; (format t "~&Cycle - ~s" point-to)
	       (setf (second (nth pair-ind (top-gdp result cpath)))
		     nil))
	      ;; Truncate and relativize in-scope paths
	      ((setf p (in-scope q1 point-to rpath))
	       ;; Keep at least one copy of the out-of-scope value
	       ;; after relocation
	       (let ((new-phys (get-new-physical-path point-to)))
		 (cond 
		  ((null new-phys)
		   ;; (format t "~&Keep value ~s" pointed-val)
		   (record-new-physical-path point-to new-cpath)
		   (setf (second (nth pair-ind (top-gdp result cpath)))
			 (copy-tree pointed-val))
		   (relocpairs q1 rpath point-to new-cpath
			       pointed-val result))
		  (t 
		   (setf (second (nth pair-ind (top-gdp result cpath)))
			 (make-relative-path new-cpath new-phys))))))
	      ;; Resolve out-of-scope paths
	      (T 
	       (record-new-physical-path point-to new-cpath)
	       (setf (second (nth pair-ind (top-gdp result cpath)))
		     (copy-tree pointed-val))
	       (relocpairs q1 rpath point-to new-cpath
			   pointed-val result))))))))


(defun in-scope (q1 path rpath)
  "Does path point to a location that is under the scope of rpath?
  ie, is there a representant of path that has rpath as a prefix?
  Return the prefixed path if exists or else nil.
  Assumption: path is the physical representant.
  If path itself is in-scope, return it (to avoid breaking the conflation)."
  (if (path-prefix path rpath)
      path
    (let ((class (get-class path q1)))
      (loop for pe in class
	    do (if (path-prefix (path-entry-path pe) rpath)
		   (return (path-entry-path pe)))))))


(defun get-new-physical-path (path)
  (gethash (path-to-id path) *new-physical-path*))

(defun record-new-physical-path (tpath npath)
  (setf (gethash (path-to-id tpath) *new-physical-path*) npath))

(defun longest-common-prefix (l1 l2 &optional accu)
  "Return the lcp of 2 lists.
  Ex: (longest-common-prefix '(1 2 3 4) '(1 2 a)) --> (1 2)"
  (cond ((or (null l1) (null l2)) (nreverse accu))
	((equalp (car l1) (car l2))
	 (longest-common-prefix (cdr l1) (cdr l2) (cons (car l1) accu)))
	(t (nreverse accu))))

(defun make-relative-path (p1 p2)
  "Transform p2 into a path relative to p1.
  Return 3 vals: the relative path, how many ^ are necessary, and the
  length of the shared prefix between p1 and p2."
  (let* ((l1 (path-l p1))
	 (l2 (path-l p2))
	 (lcp (longest-common-prefix l1 l2))
	 (llcp (length lcp))
	 (uplevel (- (length l1) llcp)))
    (values 
     (make-path :l (append 
		    (make-sequence 'list uplevel :initial-element '^)
		    (subseq l2 llcp)))
     uplevel
     llcp)))

;; ------------------------------------------------------------
;; INSERT-FD: reverse of relocate, insert a total fd within a larger total
;; fd under path subfd-path.
;; ------------------------------------------------------------
;; Example: 
;; (insert-fd '((a {b}) (b 1) (c {^ b}))
;;            '((b 2))
;;            {c})
;; =>
;; ((b 2)
;;  (c ((a {c b})  <------ NOTE updated path.
;;      (b 1)
;;      (c {c b}))))  <--- NOTE relative path is resolved
;;

;; Local flag means: if T interpret unbound absolute paths to be local to
;; the inserted fd, if NIL interpret them to be defined in the new total
;; FD. 
(defun insert-fd (fd total subfd-path &key local)
  (filter-flags (u total (insert-empty-fd fd subfd-path local))))

;; Just put an fd under a path 
;; Example: (insert-empty-fd '((a {b}) (c {^ a})) {x y} T)
;; =>
;; ((x ((y ((a {x y b})     <--- NOTE updated path
;;          (c {^ a}))))))  <--- NOTE preserve relative path
;;
;; Example: (insert-empty-fd '((a {b}) (c {^ a})) {x y} T)
;; =>
;; ((x ((y ((a {b})         <--- NOTE NOT updated path
;;          (c {^ a}))))))  <--- NOTE preserve relative path
;;
(defun insert-empty-fd (fd path local)
  (if (path-null path)
    fd
    (let* ((total (build-fd-from-path path))
	   (pair (the-last-arc-of-path total path)))
      ;; Get rid of all flags after second
      (if (or (leaf-p fd) (path-p fd))
	(setf (cdr pair) (list fd))
	(setf (cdr pair) (list (insert-patch fd path path local))))
      total)))

;; Copy the fd appearing at level path and patches the paths according to
;; relocation under path rpath in total fd.
(defun insert-patch (fd path rpath local)
  (cond ((null fd) fd)
	(t (cons (insert-patch-pair (car fd) path rpath local)
		 (insert-patch (cdr fd) path rpath local)))))

(defun insert-patch-pair (pair path rpath local)
  (let* ((attr (car pair))
	 (value (second pair)))
    (cond ((leaf-p value) (list attr value))
	  ((and (path-p value) (or (not local) (path-relative-p value))) 
	   (list attr value))
	  ((path-p value) ;; an absolute path and local is T
	   (list attr (path-append rpath value)))
	  ((member attr *special-attributes*)
	   (copy-special-pair pair path))
	  (t (list attr
		   (insert-patch value (path-extend path attr) 
				 rpath local))))))


;; ======================================================================
;; Extensional To Intensional Conversion
;; ======================================================================

(defun compare-features (f1 f2)
  (string< (format nil "~s" (car f1)) (format nil "~s" (car f2))))

(defun outgoing-arcs (fd)
  (mapcar #'first fd))

(defun ext-equal (fd1 fd2) 
  (cond
   ((and (leaf-p fd1) (leaf-p fd2)) (equalp fd1 fd2))
   ((or (leaf-p fd1) (leaf-p fd2)) nil)
   (t (and (= (length fd1) (length fd2))
	   (set-equal (outgoing-arcs fd1) (outgoing-arcs fd2))
	   (equal (sort (unfold fd1) #'compare-features)
		  (sort (unfold fd2) #'compare-features))))))

(defun ext-equal2 (fd1 fd2)
  "Check that 2 UNFOLDED fds are equal modulo feature movement"
  (cond 
   ((and (leaf-p fd1) (leaf-p fd2)) (equalp fd1 fd2))
   ((or (leaf-p fd1) (leaf-p fd2)) nil)
   (t (and (= (length fd1) (length fd2))
	   (set-equal (outgoing-arcs fd1) (outgoing-arcs fd2))
	   (equal (sort fd1 #'compare-features)
		  (sort fd2 #'compare-features))))))
  
(defun unfold (fd)
  "Replace all paths in an fd with the value they point to.
  In case of cycles, keep a path (only case where a path remains in output)."
  (let ((*input* fd))
    (unfold-aux fd {})))

(defun unfold-aux (fd path) 
  "Get an FD with absolute paths in equations (e.g., (a {x y a b})), and
  replace them with the value they point to."
  (cond 
   ((leaf-p fd) fd)
   ((path-p fd) 
    (setf fd (physical-representant 
	      *input*
	      (if (path-relative-p fd) (absolute-path fd path) fd)))
    (if (path-prefix path fd)     ;; Cycle condition
	fd
      (unfold-aux (gdp *input* fd) fd)))
   (t (mapcar
       #'(lambda (pair)
	   (list
	    (car pair)
	    (if (special-p (car pair))
		(second pair)
	      (let ((newpath (path-extend path (car pair))))
		(unfold-aux (second pair) newpath)))))
       fd))))

(defun ext2int (fd path)
  ;; Check to see if the fd@path is extensionally identical to another path
  ;; in fd, and if yes, return the path where they match (not path itself).
  ;; Traverse fd and check at each node if (ext-equal fd@path node)
  (let* ((unfolded (unfold fd))
	 (const (top-gdp unfolded path)))
    (ext2int-aux unfolded const path {})))

(defun ext2int-aux (fd const origpath path)
  (cond ((path-equal origpath path) nil)    ;; don't want original place.
	((leaf-p fd) (if (equalp const fd) path))
	((path-p fd) ;; remains because of cycles.
	 nil)
	((ext-equal2 fd const) path)
	(t (some #'(lambda (pair)
		     (let ((newpath (path-extend path (car pair))))
		       (ext2int-aux (second pair) const origpath newpath)))
		 fd))))

(defun s-ext2int (fd path)
  "Check to see if the fd appearing under path is extensionally equivalent
to another path, if yes replace fd@path with a pointer to that other path,
if not, try again for all defined extensions of path to find a match in the
rest of the fd."
  (let* ((unfolded (unfold fd))
	 (const (top-gdp unfolded path))
	 (res (copy-tree fd)))
    (s-ext2int-aux res path unfolded const)
    res))

(defun s-ext2int-aux (fd path unfolded const)
  (let ((match (ext2int-aux unfolded const path {})))
    (if match
	(setf (second (top-gdpp fd path)) match)
      (mapc #'(lambda (pair)
		(let ((newpath (path-extend path (first pair))))
		  (s-ext2int-aux fd newpath unfolded (second pair))))
	    const))))



;; ======================================================================
;; Checking / Verification of table building functions
;; ======================================================================

(defun get-value (path quotient &optional pe)
  "Like gdp given a quotient set."
  (let ((pe (if (path-entry-p pe) pe (find-entry path))))
    (if (path-entry-p pe) 
	(if (path-entry-value pe)
	    (path-entry-value pe)
	  (let ((class (get-class path quotient pe)))
	    (loop for pe in class
		  for val = (path-entry-value pe) 
		  do (if val (return val)))))
      (gdp *input* path))))

(defun make-tables (fd)
  (setf *input* fd)
  (setf *path-table* (make-hash-table :test #'equal :size 500)
	*edge-table* nil)
  (setf *path-table* (build-id-table fd {}))
  (values (hash-table-count *path-table*)))


;; ======================================================================
;; GDP: New version of gdp that returns value AND physical path.
;; ======================================================================

(defun gdp (fd path)
  "given an fd (representing a connected graph) and a path, 
  return 4 values: 
  1. value pointed by path
  2. physical path where this value occurs in fd.
  3. what's missing from path in the physical rep.
  4. whether a loop was found."
  ;; Check for cycles in the way. Mark our way in already-tried.
  ;; For fds in canonical form, all paths point to the physical
  ;; representant directly, and already-tried is not necessary.  For other
  ;; fds, you need it as shown by example: ((a {o}) (o ((m {a m}))))

  ;; (clrhash *already-tried*)
  (do* ((fd fd) 
	(path path)
	(already-tried (list path))
	(cpath (make-path))
	(current-fset (find-fset fd))
	(arc-tried (list :top fd))
	(node-tried (list arc-tried)))
      ((and (not (path-p fd)) (path-null path)) (values fd cpath path))
    ;; (format t "~&DO path = ~s - cpath = ~s - fd = ~s" path cpath fd)
    (cond 
     ;; Do we stop at any or do we go below it: below is unspecified.
     ((eq fd 'any) 
      (if (path-null path) 
	  (return (values 'any cpath path))
	(return (values nil cpath path))))
     ((eq fd 'given) 
      (if (path-null path) 
	  (return (values 'given cpath path))
	(return (values nil cpath path))))
     ((eq fd nil) (return (values nil cpath path)))

     ;; Below any other atom is impossible: none
     ((leaf-p fd) (return (values 'none cpath path)))

     ;; Need to go up to the root and go down again to the indirection.
     ((path-p fd)
      ;; A non-special cannot point to a special: check validity here.
      (let ((from (car arc-tried))
	    (to   (car (last (path-l fd)))))
	(cond 
	 ((eq from to))  ;; fine - type equality
	 ((or (member from *special-attributes*)
	      (member to *special-attributes*))
	  (return (values 'none cpath path)))))
      ;; ok proceed...
      (let* ((path-prefix (absolute-path fd cpath))
	     (npath (path-append path-prefix path))
	     (tried node-tried))
	;; This is a loop: ({a b c} {a b c d e})
	;; (format t "~&already tried = ~s - pp = ~s" already-tried path-prefix)
	;; (maphash #'(lambda (p k) (print k)) *already-tried*)
	(when (or (path-prefix path-prefix cpath)
		  (path-prefix cpath path-prefix)
		  (member path-prefix already-tried :test #'path-equal))
	  ;; (member npath already-tried :test #'path-equal)
	  (return (values node-tried cpath path t)))
	;; (record-already-tried cpath node-tried)
	(setf path npath)
	(push path already-tried)
	(setf cpath (make-path))
	(setf fd *input*)
	(setf arc-tried (list :top fd))
	(setf node-tried (list arc-tried))))

     ;; Special attributes are also atomic: cannot go below them.
     ;; but can have a path as value.
     ((member (path-car path) *special-attributes*)
      (cond ((cdr (path-l path)) (return (values 'none cpath path)))
	    ((or (null current-fset) 
		 (member (path-car path) current-fset)
		 (eq (path-car path) 'fset))
	     (setf arc-tried (safe-assoc (path-car path) fd))
	     (setf node-tried fd)
	     (if (consp arc-tried)
		 (setf fd (second arc-tried))
	       (return (values nil cpath path)))
	     (setf cpath (path-extend cpath (path-car path)))
	     (setf path (path-cdr path)))
	    (t (return (values 'none cpath path)))))

     ;; Are we within the authorized FSET? If so, go down one level.
     ((or (null current-fset)
	  (member (path-car path) current-fset))
      (setf arc-tried (safe-assoc (path-car path) fd))
      (setf node-tried fd)
      (if (consp arc-tried)
	  (setf fd (second arc-tried))
	(return (values nil cpath path)))
      (setf cpath (path-extend cpath (path-car path)))
      (setf path (path-cdr path))
      (setf current-fset (find-fset fd)))

     ;; Not within the authorized FSET: value is none.
     (t (return (values 'none cpath path))))))


(defvar *already-tried* (make-hash-table :test #'equal))
  
(defun already-tried (path)
  (gethash (path-l path) *already-tried*))

(defun record-already-tried (path fd)
  (setf (gethash (path-l path) *already-tried*) fd))

