;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:FUG5 -*-
;;; -----------------------------------------------------------------------
;;; File:         DETERMINE.L 
;;; Description:  "Determination" stage of FUGs with SUCCESS.
;;; Author:       Michael Elhadad
;;; Created:      16-Oct-88
;;; Modified:     27-Oct-88
;;;               29 Apr 90 - added *special-attributes*
;;;               16 May 90 - path-p and leaf-p.
;;;               11 Jun 90 - added success arg to determine.
;;;               20 Jun 90 - cleaned up det1 and filters.
;;;               25 Feb 91 - do not depend on throw anymore!
;;;                           Changed determine, det-tests and det1.
;;;               28 Jul 91 - added FREEZE and IGNORE and AGENDA-POLICY
;;;               19 Sep 91 - added *use-any*
;;;               13 Nov 91 - updated calls to *fail*
;;;               26 Nov 91 - added filter-flags and relocate
;;;               11 Dec 91 - added force-constituent-agenda call
;;;               13 Dec 91 - added det-constituents to deal with
;;;               constituents appearing as a result of a delayed alt. 
;;;               18 Dec 91 - added *use-wait*
;;;               23 Dec 91 - added change of failure address in det-const.
;;;               24 Dec 91 - remove call to add-constituent-agenda
;;;               06 Feb 92 - added :no-cset arg to determine to avoid
;;;                           going thru the det-const for non-structural fct.
;;;               11 Feb 92 - added *added-cset* in determine to ensure
;;;                           complete traversal of cset structure.
;;;               18 Feb 92 - fixed relocate - added insert-fd.
;;;               24 Feb 92 - added remove-duplicates in det-constituents.
;;;               02 Aug 92 - fixed insert-empty-fd to accept leafs/paths JR
;;;               20 Oct 92: Added level to trace-format
;;;               26 Oct 93: Update call to call-linearizer in det-any
;;;                          (add parameter cat-att).
;;;               04 Jan 94: Some anti-looping code in relocate.
;;;               09 Jan 94: Major bug fix in relocate: added tpath arg.
;;; Package:      FUG5
;;; Status:       Experimental
;;; -----------------------------------------------------------------------
;;;
;;; FUF - a functional unification-based text generation system. (Ver. 5.2)
;;;  
;;; Copyright (c) 19{87-94} by Michael Elhadad. all rights reserved.
;;;  
;;; Permission to use, copy, and/or distribute for any purpose and
;;; without fee is hereby granted, provided that both the above copyright
;;; notice and this permission notice appear in all copies and derived works.
;;; Fees for distribution or use of this software or derived works may only
;;; be charged with express written permission of the copyright holder.
;;; THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT EXPRESS OR IMPLIED WARRANTY.
;;; -----------------------------------------------------------------------


(in-package "FUG5")
(format t "Determiner...~%")

;; ------------------------------------------------------------
;; DETERMINE
;; ------------------------------------------------------------


;; Here is a version without the det-constituent fix
#+ignore(defun determine (fd fail frame success grammar cat-att cset-att)
  "Check AGENDAS, TESTs and ANYs at the end of unification"
  ;; Agenda-policy determines what to do with frozen alts at the end of
  ;; unif: keep them undeveloped or force them (:keep or :force).
  (det-agenda 
   fd fail frame
   #'(lambda (fd fail frame)
       (if (empty-agenda)
	 (det-tests 
	  fd (reverse (frame-tests frame)) fail frame
	  #'(lambda (fd fail frame)
	      (if *use-any*
		(det-any fd fd (make-path) frame fail success)
		(funcall success fd fail frame))))
	 ;; The delayed stuff has added more delays... deal with them.
	 (determine fd fail frame success
		    grammar cat-att cset-att)))))

;; Finally, it looks like det-constituents makes constituent-agenda useless...
#+ignore(defun determine (fd fail frame success grammar cat-att cset-att)
  "Check AGENDAS, TESTs and ANYs at the end of unification"
  ;; Agenda-policy determines what to do with frozen alts at the end of
  ;; unif: keep them undeveloped or force them (:keep or :force).
  (det-agenda 
   fd fail frame
   #'(lambda (fd fail frame)
       (force-constituent-agenda 
        fd fail frame
	#'(lambda (fd fail frame)
	    (det-constituents 
	     (list (make-path)) grammar cat-att cset-att
	     fd fail frame 
	     #'(lambda (fd fail frame)
		 (if (and (empty-agenda) (empty-constituent-agenda))
		   (det-tests 
		    fd (reverse (frame-tests frame)) fail frame
		    #'(lambda (fd fail frame)
			(if *use-any*
			  (det-any fd fd (make-path) frame fail success cat-att)
			  (funcall success fd fail frame))))
		   ;; The delayed stuff has added more delays... deal with them.
		   (determine fd fail frame success
			      grammar cat-att cset-att)))))))))


;; This is the good version: with just what we need!
(defun determine (fd fail frame success grammar cat-att cset-att
		     &optional use-cset)
  "Check AGENDAS, TESTs and ANYs at the end of unification"
  ;; Agenda-policy determines what to do with frozen alts at the end of
  ;; unif: keep them undeveloped or force them (:keep or :force).
  (setf *added-cset* nil)
  (if (and *use-wait* (not use-cset))
    (det-agenda 
     fd fail frame
     #'(lambda (fd fail frame)
	 (det-constituents 
	  (list (make-path)) grammar cat-att cset-att
	  fd fail frame 
	  #'(lambda (fd fail frame)
	      (filter-agenda frame)
	      (if (and (empty-agenda) (not *added-cset*))
		(det-tests 
		 fd (reverse (frame-tests frame)) fail frame
		 #'(lambda (fd fail frame)
		     (if *use-any*
		       (det-any fd fd (make-path) frame fail success cat-att)
		       (funcall success fd fail frame))))
		;; The delayed stuff has added more delays... deal with them.
		;; Or there has been an added cset at some level.
		(determine fd fail frame success
			   grammar cat-att cset-att))))))

    ;; Don't bother about wait: just checks tests and any
    (det-tests 
     fd (reverse (frame-tests frame)) fail frame
     #'(lambda (fd fail frame)
	 (if *use-any*
	   (det-any fd fd (make-path) frame fail success cat-att)
	   (funcall success fd fail frame))))))


;; ------------------------------------------------------------
;; DET-ANY: is there an any somewhere in fd
;; ------------------------------------------------------------
(defun det-any (fd total-fd path frame fail success cat-att)
  "Check for any ANYs left."
  (declare (special *from-top*))
  (cond ((null fd) (funcall success total-fd fail frame))
	((eq fd 'any) 
	 (trace-format *trace-determine* frame 30
		       "Fail in Determine: found an ANY at level ~s" path)
	 (when (and *global-tracing* *local-tracing* 
		    *trace-determine* *from-top*)
	   (trace-indent ">" frame)
	   (format t "CURRENT SENTENCE:~%")
	   (trace-indent ">" frame)
	   (print-sentence (call-linearizer *input* :cat-attribute cat-att)))
	 (*fail* fail frame path path :e))
	((leaf-p fd) (funcall success total-fd fail frame))
	((path-p fd) (funcall success total-fd fail frame))
	((leaf-p (car fd)) (error "Ill-formed fd in det-any"))
	((or (member (caar fd) *special-attributes*)
	     (path-p (cadar fd)))
	 (det-any (cdr fd) total-fd path frame fail success cat-att))
	(t 
	  (let ((new-path (path-extend path (caar fd)))
		(sub-fd (cadar fd)))
	    (det-any 
	     sub-fd total-fd new-path frame fail
	     #'(lambda (another-fd fail frame)
		 (declare (ignore another-fd))
		 (det-any (cdr fd) total-fd path frame fail success cat-att))
	     cat-att)))))


;; ------------------------------------------------------------
;; DET-TESTS: evaluation of all delayed tests.
;; ------------------------------------------------------------
(defun det-tests (fd tests fail frame success)
  (if (null tests)
    (funcall success fd fail frame)
    (let ((test (test-test (car tests)))
	  (path (test-path (car tests))))
      (if (eval `(let ((path ',path) (%frame% ,frame)) ,test))
	(progn
	  (trace-format *trace-determine* frame 5
			"TEST succeeds: ~s at level ~s"
			(filter-macro-char test) path)
	  (det-tests fd (cdr tests) fail frame success))
	(*fail* fail frame path path :e
		"Fail in testing ~s at level ~s"
		(filter-macro-char test) path)))))
    

;; ------------------------------------------------------------
;; DET-AGENDA: forces evaluation of all delayed stuff.
;; ------------------------------------------------------------

;; Agenda-policy determines what to do with frozen alts at the end of
;; unif: keep them undeveloped or force them (:keep or :force).

(defun det-agenda (fd fail frame success)
  (filter-agenda frame)   ;; remove the entries whose ignore clauses now match
  (cond 
   ((empty-agenda) (funcall success fd fail frame))
   ((eq *agenda-policy* :keep)
    ;; put all the alts into the result fd at the level they come from
    ;; (path2). 
    (mapc #'(lambda (ag-item)
	      ;; if path points to a leaf, force it to become part of the
	      ;; alt. fd-adjoin is defined in wait.l
	      ;; Note that fd-adjoin removes undoably the ag-item
	      (fd-adjoin fd ag-item frame))
	  *agenda*)
    (funcall success fd fail frame))
   ((eq *agenda-policy* :force)
    ;; Need to force the alt-unify of all entries successively.
    ;; force-agenda is defined in wait.l
    (force-agenda fd fail frame success))))


;; ------------------------------------------------------------
;; DET-CONSTITUENTS : check that all constituents have been visited
;;                    after all delayed stuff has been evaluated.
;;                    So if a delayed part creates a constituent,
;;                    it will be visited now.
;; Works as unify-breadth-first except that children of an already visited
;; constituent are still visited.  This forces a re-evaluation of all the
;; csets at all levels of *input*.
;; ------------------------------------------------------------

(defun det-constituents (lpath grammar cat-att cset-att fd fail frame success)
  (declare (special *failure-address* *changes-made*))
  (cond 
   ((null lpath) (funcall success *input* fail frame))
   (t (let* ((arc (gdpp *input* (car lpath) frame))
	     (cset (find-cset (safe-second arc) (car lpath) 
			      cat-att cset-att))) 
	(cond 
	 ((or (arc-is-marked arc) (arc-is-marked-after-wait arc)
	      (path-null (car lpath)))
	  (det-constituents 
	   (remove-duplicates (append (cdr lpath) cset)
			      :test #'path-equal :from-end t)
	   grammar cat-att cset-att
	   fd fail frame success))
	 (t
	  ;; Change failure address so that bk-class is not confused.
	  (trace-format 
	   *trace-bk-class* frame 5
	   "BKd: Switch from ~s to ~s" *failure-address* (car lpath))
	  (setf *failure-address* (car lpath))
	  (setf *changes-made* t)
	  (unify-cat 
	   (safe-second arc) grammar (car lpath) frame fail
	   #'(lambda (fd fail frame)
	       (cond 
		((eq fd :frozen)  ;; delay traversal of this constituent
		 #+ignore(add-constituent-agenda 
		  (car lpath) frame grammar cat-att cset-att)
		 (det-constituents
		  (cdr lpath) grammar cat-att cset-att
		  fd fail frame success ))
		(t
		 (let ((cset (find-cset fd (car lpath) cat-att cset-att)))
		   (when *trace-cset*
		     (if cset
		       (trace-format 
			*trace-cset* frame 20
			"Expanding constituent ~s into cset ~s." 
			(car lpath) cset)
		       (trace-format
			*trace-cset* frame 20
			"Constituent ~s is a leaf." (car lpath))))
		   (det-constituents
		    (append (cdr lpath) cset) grammar cat-att cset-att
		    fd fail frame success)))))
	   arc
	   cat-att
	   cset-att
	   )))))))


;; ------------------------------------------------------------
;; Some utilities to clean-up fds
;; ------------------------------------------------------------

(defun filter-nones (fd)
  "Remove all pairs (att none) from an fd at all levels."
  (cond ((null fd) fd)
	((leaf-p fd) fd)
	((path-p fd) fd)
	((leaf-p (car fd)) (error "Ill-formed fd in filter-nones"))
	((eq 'none (cadar fd)) (filter-nones (cdr fd)))
	((or (leaf-p (cadar fd)) 
	     (member (caar fd) *special-attributes*)
	     (path-p (cadar fd)))
	 (cons (car fd) (filter-nones (cdr fd))))
	(t 
	 (cons (list (caar fd)
		     (filter-nones (cadar fd)))
	       (filter-nones (cdr fd))))))

(defun filter-nils (fd)
  "Remove all pairs (att nil) from an fd at all levels."
  (cond ((null fd) fd)
	((leaf-p fd) fd)
	((path-p fd) fd)
	((leaf-p (car fd)) (error "Ill-formed fd in filter-nils"))
	((eq nil (cadar fd)) (filter-nils (cdr fd)))
	((or (leaf-p (cadar fd)) 
	     (member (caar fd) *special-attributes*)
	     (path-p (cadar fd)))
	 (cons (car fd) (filter-nils (cdr fd))))
	(t 
	 (let ((sub-fd (filter-nils (cadar fd)))
	       (rest-fd (filter-nils (cdr fd))))
	   (if sub-fd
	       (cons (list (caar fd) sub-fd) rest-fd)
	     rest-fd)))))

(defun filter-flags (fd)
  "Remove all :i and :e from an fd at all levels."
  (cond ((null fd) fd)
	((leaf-p fd) fd)
	((path-p fd) fd)
	((leaf-p (car fd)) (error "Ill-formed fd in filter-nils"))
	((eq nil (cadar fd)) (filter-flags (cdr fd)))
	((or (leaf-p (cadar fd)) 
	     (member (caar fd) *special-attributes*)
	     (path-p (cadar fd)))
	 (cons (list (caar fd) (cadar fd)) (filter-flags (cdr fd))))
	(t 
	 (let ((sub-fd (filter-flags (cadar fd)))
	       (rest-fd (filter-flags (cdr fd))))
	   (if sub-fd
	       (cons (list (caar fd) sub-fd) rest-fd)
	     rest-fd)))))


;; ------------------------------------------------------------
;; RELOCATE: grab a sub-fd from a total fd and make it a 
;; stand-alone total fd (resolve paths outside the sub-fd).
;; NOTE: Relocate is NOT smart about csets and patterns (it does not follow
;; the paths to resolve them and update them).
;; ------------------------------------------------------------
;; Example:
;; (relocate '((a ((a1 {^ a2})
;;                 (a2 2)
;;                 (a3 {a a1})
;;                 (a4 {c})
;;                 (a4 {b})))
;;             (b {a a1})
;;             (c ((c1 1))))
;;           {a})
;; =>
;; ((a1 {^ a2})     <--- NOTE keep relative path
;;  (a2 2)
;;  (a3 {a1})       <--- NOTE updated path
;;  (a4 ((c 1)))    <--- NOTE resolved path
;;  (a5 2))         <--- NOTE loose conflation a5/a1 because went out of a
;;                            scope.

(defun relocate (total rpath)
  (let* ((total (relativize-fd total))
	 (const (top-gdp total rpath)))
    (relocpairs total rpath rpath (make-path) const (copy-tree const))))

;; total = the total fd
;; rpath = from where do we relocate
;; tpath = where are we within total
;; cpath = where are we within the relocated fd (result)
;; pair-ind = index of pair within the fd
;; pairs = the currend fd being processed
;; result = accumulator for the relocated constituent
(defun relocpair (total rpath tpath cpath pair-ind pairs result)
  (let* ((pair (nth pair-ind pairs))
	 (feature (first pair))
	 (value (second pair)) 
	 (new-cpath (path-extend cpath feature))
	 (new-tpath (path-extend tpath feature))
	 (point-to (if (path-p value)
		       ;; (absolute-path value (path-append rpath new-path))
		       (absolute-path value new-tpath)
		     nil)))
    (format t "~&tpath = ~s; cpath = ~s; pair = ~s" tpath cpath pair)
    (cond ((leaf-p value) 
	   result)
	  ;; Problem with pattern and cset having paths in their values.
	  ;; Copy special deals with it.
	  ((member feature *special-attributes*)
	   (setf (second (nth pair-ind (top-gdp result cpath)))
		 (copy-special value feature cpath)))
	  ((not (path-p value))
	   (relocpairs total rpath new-tpath new-cpath value result))
	  ;; Preserve relative paths
	  ((and (path-relative-p value) (path-prefix point-to rpath))
	   result)
	  ;; Explicit check for cycles
	  ((path-equal value {^})
	   result)
	  ;; Truncate in-scope paths: could relativize as well
	  ((path-prefix point-to rpath)
	   (setf (second (nth pair-ind (top-gdp result cpath)))
		 (path-nthcdr (path-len rpath) value))
	   result)
	  ;; Resolve out-of-scope paths
	  (T 
	   (let ((pointed-value (top-gdp total point-to)))
	     (setf (second (nth pair-ind (top-gdp result cpath)))
		   (copy-tree pointed-value))
	     (relocpairs total rpath point-to new-cpath
			 pointed-value result))))))

(defun relocpairs (total rpath tpath cpath pairs result)
  (if (leaf-p pairs)
    pairs
    (loop for pair in pairs
          for pair-ind = 0 then (+ pair-ind 1)
	  do (relocpair total rpath tpath cpath pair-ind pairs result)
	  finally (return result))))


;; ------------------------------------------------------------
;; INSERT-FD: reverse of relocate, insert a total fd within a larger total
;; fd under path subfd-path.
;; ------------------------------------------------------------
;; Example: 
;; (insert-fd '((a {b}) (b 1) (c {^ b}))
;;            '((b 2))
;;            {c})
;; =>
;; ((b 2)
;;  (c ((a {c b})  <------ NOTE updated path.
;;      (b 1)
;;      (c {c b}))))  <--- NOTE relative path is resolved
;;

(defun insert-fd (fd total subfd-path)
  (filter-flags (u total (insert-empty-fd fd subfd-path))))

;; Just put an fd under a path 
;; Example: (insert-empty-fd '((a {b}) (c {^ a})) {x y})
;; =>
;; ((x ((y ((a {x y b})     <--- NOTE updated path
;;          (c {^ a}))))))  <--- NOTE preserve relative path
;;
(defun insert-empty-fd (fd path)
  (if (path-null path)
    fd
    (let* ((total (build-fd-from-path path))
	   (pair (the-last-arc-of-path total path)))
      ;; Get rid of all flags after second
      (if (or (leaf-p fd) (path-p fd))
	(setf (cdr pair) (list fd))
	(setf (cdr pair) (list (insert-patch fd path path))))
      total)))

;; Copy the fd appearing at level path and patches the paths according to
;; relocation under path rpath in total fd.
(defun insert-patch (fd path rpath)
  (cond ((null fd) fd)
	(t (cons (insert-patch-pair (car fd) path rpath)
		 (insert-patch (cdr fd) path rpath)))))

(defun insert-patch-pair (pair path rpath)
  (let* ((attr (car pair))
	 (value (second pair)))
    (cond ((leaf-p value) (list attr value))
	  ((and (path-p value) (path-relative-p value)) (list attr value))
	  ((path-p value) ;; an absolute path
	   (list attr (path-append rpath value)))
	  ((member attr *special-attributes*)
	   (copy-special-pair pair path))
	  (t (list attr
		   (insert-patch value (path-extend path attr) rpath))))))


;; ------------------------------------------------------------
;; ANY-P : is there something at this location
;; ------------------------------------------------------------

(defun any-p (path)
  (if *use-any*
    (not (member (gdp *input* path) '(none nil)))
    t))


;; -----------------------------------------------------------------------
(provide "$fug5/determine")
;; -----------------------------------------------------------------------

