;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:FUG5 -*-
;;; -----------------------------------------------------------------------
;;; File :        GENERATOR.L
;;; Description:  Generator manipulation functions
;;; Author :	  Michael Elhadad
;;; Created :	  22-Jul-87
;;; Modified :	  02-Nov-87
;;;               17 May 90: moved macros to macros.l
;;;               17 Dec 91: renamed first to gen-first
;;; Tag file:     TAGS
;;; Package :	  FUG5
;;; Macros :      empty, freeze, cons-gen
;;; -----------------------------------------------------------------------
;;;
;;; FUF - a functional unification-based text generation system. (Ver. 5.2)
;;;  
;;; Copyright (c) 19{87-93} by Michael Elhadad. all rights reserved.
;;;  
;;; Permission to use, copy, and/or distribute for any purpose and
;;; without fee is hereby granted, provided that both the above copyright
;;; notice and this permission notice appear in all copies and derived works.
;;; Fees for distribution or use of this software or derived works may only
;;; be charged with express written permission of the copyright holder.
;;; THIS SOFTWARE IS PROVIDED ``AS IS'' WITHOUT EXPRESS OR IMPLIED WARRANTY.
;;; -----------------------------------------------------------------------


(in-package "FUG5")
(format t "Generator...~%")

; --------------------------------------------------------------------------
; Comments :
; --------------------------------------------------------------------------
; A generator is a pair : (VALUE . CONTINUATION) or nil.                   
; nil is the empty generator.                                              
; where VALUE is any lisp object, and CONTINUATION is a closure            
; CONT. should require no argument (something like #'(lambda () ...))      
; (that is what Scheme calls a thunk).					   
; The value of CONT. must be a new generator.				   
; Patterned after SCHEME streams.					   
; --------------------------------------------------------------------------


(defvar *the-empty-generator* nil "Used by module GEN")
(defvar *no-first* '*no-first* "Used by module GEN for first-if and filter")

(defun displace (l1 l2)
  (when (consp l1)
	(rplaca l1 (car l2))
	(rplacd l1 (cdr l2))))


(defun force (thunk)
  "Forces the evaluation of a frozen expression (by FREEZE)
  Implements memoization"
  (let ((result (funcall thunk)))
    (displace thunk `(lambda () ',result))
    result))

(defun list->gen (list)
  "Converts a list to an equivalent generator.
  Inverse function is gen->list (or exhaust)"
  (if (null list)
      *the-empty-generator*
      ; cons-gen
      (cons (car list) #'(lambda nil (list->gen (cdr list))))))
  
(defun gen-first (generator)
  "Returns the first value of a generator"
  (if (empty generator)
      ;; (format t "generator exhausted")
    nil
    (car generator)))

(defun next (generator)
  "Returns a new generator which starts with the next value"
  (if (empty generator)
      (format t "generator exhausted")
      (force (cdr generator))))

(defun first-if (pred generator)
  (if (empty generator)
      *no-first*
      (if (funcall pred (gen-first generator))
	  (gen-first generator)
	  (first-if pred (next generator)))))

(defun next-if (pred generator)
  (cond ((empty generator) *the-empty-generator*)
	((funcall pred (gen-first generator)) (next generator))
	(t (next-if pred (next generator)))))

(defun filter (pred generator)
  (if (empty generator)
      generator
      (let ((first (first-if pred generator)))
	(if (eq first *no-first*)
	    *the-empty-generator*
	    (cons-gen (first-if pred generator)
		      (filter pred (next-if  pred generator)))))))

(defun exhaust (generator)
  "Returns a list of all the values of a generator.
  May run forever if generator is an implicit infinite computation"
  (do ((g generator (next g))
       (result nil (cons (gen-first g)  result)))
      ((empty g) (reverse result))))

(defun set-of (generator)
  "Returns all the values generated by a generator with no repetition"
  (do ((g generator (next g))
       (result nil (pushnew (gen-first g) result)))
      ((empty g) result)))


(defun mapgen (f gen)
  "Returns a generator whose values are those of gen modified by the 
  application of f"
  (if (empty gen)
      gen
      (cons (funcall f (gen-first gen))
	    #'(lambda nil (mapgen f (next gen))))))


(defun concatgen (gen1 gen2)
  "Returns a generator generating all the values of gen1, then all the
  values of gen2. Gen2 must be frozen."
  (if (empty gen1)
      (force gen2)
      (cons (gen-first gen1)
	    #'(lambda nil (concatgen (next gen1) gen2)))))

(defun multiplygen (f gen1 gen2)
  "Generic function to combine generators. Returns a generator creating
  all the elements of the form f(ei1,ej2), for ei1 generated by gen1,
  and ej2 generated by gen2. f must be a function of 2 arguments.
  The order of traversal of the 2 generators is j varying the faster."
  (if (or (empty gen1) (empty gen2))
      *the-empty-generator*
      (concatgen 
       (mapgen #'(lambda (first2) (funcall f (gen-first gen1) first2)) gen2)
       #'(lambda () (multiplygen f (next gen1) gen2)))))


(defun appendgen (gen1 gen2)
  (multiplygen #'append gen1 gen2))


; --------------------------------------------------------------------------
; Some examples of streams
; --------------------------------------------------------------------------

; This one is infinite
(defun integers (i)
  (cons i #'(lambda () (integers (1+ i)))))

; This one is useful
(defun powerset (set)
  "Returns a generator producing all 2^n subsets of a set"
  (if (null set)
      (cons-gen set *the-empty-generator*)
      (concatgen
       (mapgen #'(lambda (e) (cons (car set) e)) (powerset (cdr set)))
       (freeze (powerset (cdr set))))))


;; -----------------------------------------------------------------------
(provide "$fug5/generator")
;; -----------------------------------------------------------------------
