;;
;;
;;      Title : D-Unify and D-IsExtensionOf
;;
;;      Function :  finds the unification of two syntactic
;;            categories.  Also finds out if two categories are extensions
;;
;;      Author :   Alan W Black  17th March 1986
;;     
;;      Copyright : Graeme Ritchie, Alan Black,
;;                  Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;         >>>   TERM UNIFICATION   <<<
;;
;;      Description :
;;          These functions find the unifcation and extension of two
;;          categories that can contain variables
;;
;;          category A is an extension of category B iff
;;           1. all atomic valued features B are in A with the same
;;              values.
;;           2. for any category-valued feature f in A, the value of f in A
;;              is an extension of the value of f in B.
;;
;;          The unification of two categories is the smallest category 
;;          that is an extension of both.  It can be thought of as
;;          the union of categories (and values of category valued features.
;;
;;          These routines are pretty fundemental to the ideas of feature
;;          grammars in general (and GPSG in particular).  I have tried
;;          to make them as efficient as possible but this is not easy
;;          as they are pretty difficult tasks.
;;
;;          If there was not any variables it would be a lot easier,
;;          but not nearly as useful.
;;
;;      Parameters :
;;      Returns :
;;      Side Effect :
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;
;;
(declare
   (localf
      D-Unify 
      D-UnifyEachValue 
      D-UnifyFPair 
      D-IsExtensionOf 
      D-FindType 
      D-UniquifyVariables 
      D-UniquifyCategory 
      D-UniquifyFeature 
      D-DereferenceVariables 
      D-AddDefault
   )
)

(defmacro D-MakeBinding (var value bindings)
;;
;;   Adds a new binding to the bindings list
;;
   `(cons
      (list ,var ,value)
      ,bindings
   )
)

(defun D-Unify (cat1 bind cat2)
;;
;;   Term unification.  This returns a set of bindings (always a non null
;;   list) if unfication is successful or the atom 'FAILED.
;;
   (cond
      ((eq (D-GetCategoryType cat1) (D-GetCategoryType cat2))
	 (catch 
	    (D-UnifyEachValue
	       (D-GetCategoryValues cat1)
	       bind
	       (D-GetCategoryValues cat2)))
      )
      (t  'FAILED)
   )
)

(defun D-UnifyEachValue (values1 bind values2)
;;
;;  unifies each value of two categories
;;
   (let ( (localbind bind) )
      (mapc   
	 #'(lambda (fv1 fv2)
	    (cond
	       ((neq fv1 fv2) 
	          (setq localbind
		     (D-UnifyFPair
			fv1 localbind fv2)))))
	 values1
	 values2
      )
      localbind
   )
)

(defun D-UnifyFPair (fvg1 bind fvg2)
;;
;;   unifies the two values making necessary changes to the bindings
;;   
   (let ( (fv1 (D-FindType fvg1 bind))
	  (fv2 (D-FindType fvg2 bind)) )
      (cond
	 ((and (eq (car fv1) 'LITERAL) (eq (car fv2) 'LITERAL))
	    (cond
	       ((listp (cadr fv1))    ;; a category valued feature
		  (D-Unify 
		     (cadr fv1)
		     bind
		     (cadr fv2)))
	       ((eq (cadr fv1) (cadr fv2))
		  bind)
	       (t (throw 'FAILED))))   ;; different literals
	 ((eq (car fv1) 'LITERAL)  ;; fv2 a variable
	    (cond
	       ((or (eq (caddr fv2) (DK-category))  ;; category valued var
		 (memq (cadr fv1) (caddr fv2))) ;; is it in range of var
		     (D-MakeBinding   ;; may get used later
			(cadr fv2)
			(cadr fv1)
			bind))
	       (t (throw 'FAILED))  ;; lit not in range of variable
	   )
	 )
	 ((eq (car fv2) 'LITERAL)  ;; fv1 a variable
	    (cond
	       ((or (eq (caddr fv1) (DK-category))  ;; category valued var
		 (memq (cadr fv2) (caddr fv1))) ;; is it in range of var
		    (D-MakeBinding
		       (cadr fv1)  ;; Variable name
		       (cadr fv2)  ;; variable value
		       bind))
	       (t (throw 'FAILED))  ;; lit not in range of variable
	   )
	 )
	 (t        ;; both are variables
	    (let ( (newrange (D-IntersectVar (caddr fv1) (caddr fv2))) )
	       (cond
		  ((null newrange) (throw 'FAILED))  ;; vars incompatible
		  (t
		     (let ( (newvar (D-MakeNewVariable newrange)) )
			(D-MakeBinding
			   (cadr fv1)  ;; variable name
			   newvar      ;; new variable is value
			   (D-MakeBinding
			      (cadr fv2)
			      newvar
			      bind))))))
	 )
      )
   )
)

(defun D-IsExtensionOf (cat1 bind1 cat2)
;;
;;  In term unification this becomes a degenerate case of UNification
;;  (well almost)
;;  because I am lazy I will make this basically Unification
;;
;;  This is wrong as it should be cat1 is less specified than cat2
;;
   (let  ( (unified (D-Unify cat1 bind1 cat2)) )
      (cond
	 ((eq unified 'FAILED)
	    nil
         )
	 (t
	    unified    ;; the bindings
         ))
   )
)
 
(defun D-FindType (term bindings)
;;
;;   Finds out what a term is, either a variable or a literal.
;;   If Term is directly a variable it looks for any binding of
;;   it (and recursively)
;;   returns (LITERAL <value>)
;;        or (VARIABLE <name> <range>)
;;
;;   note that the original term passed maybe a variable but the
;;   bindings are searched until the end of the isbound to relationship
;;   is reached
;;
   (cond
      ((D-VariableP term)
         (let ( (binding (D-PattBinding term bindings)) )
	    (cond
	       ((eq binding 'D-UNBOUND)
		  (list 'VARIABLE term (D-GetVarRange term)))
	       (t (D-FindType binding bindings))
            )
         )
      )
      (t
	 (list 'LITERAL term)
      )
   )
)

(defun D-UniquifyVariables (rule)
;;
;;  Substitutes all variables in the rule for new unique variable
;;  names.  This is to avoid name clashes
;;
   (cond
     ((D-GetGRuleVarFlag rule)
	 (let  ( (bindings (ncons (list nil nil))) )
	    (cons
	       (D-GetGRuleName rule)       ;; name of rule
	       (mapcar
		  #'(lambda (category)
		     (D-UniquifyCategory category bindings))
		  (D-GetGRuleCategories rule)
	       )
	    )
         )
      )
      (t (cdr rule))   ;; return simple rules without flags
   )
)

(defun D-UniquifyCategory (cat bindings)
;;
;;   substitute all variable in a category for normalise ones
;;
   (cons
      (D-GetCategoryType cat)
      (mapcar
         #'(lambda (fvalue)
            (D-UniquifyFeature fvalue bindings)
         )
         (D-GetCategoryValues cat)
      )
   )
)

(defun D-UniquifyFeature (fvalue bindings)
;;
;;  substitute all variables for unique ones
;;
   (cond
      ((D-VariableP fvalue)
	 (let ( (value (assq fvalue bindings)) )
	    (cond       ;; variable unchanged   
	       ((null value) ;; variable is unbound
		  (let ( (newvar (D-MakeNewVariable 
				       (D-GetVarRange fvalue))) )
		     (attach (list fvalue newvar) bindings)
                     newvar))
               (t
		  (cadr value))))
      )
      ((listp fvalue)    ;; a category value
	 (D-UniquifyCategory fvalue bindings))
      (t    ;; simple literal
	 fvalue
      )
   )
)

(defun D-DereferenceVariables (category bindings)
;;
;;   This checks for any occurrences of the given variables and instantiates
;;   them in category.  The new category is returned
;;
   (cond
      ((equal bindings '((t t)))  ;; if no bindings nothing to do
	 category  ;; this is not required but is for efficiency
      )
      (t
	 (cons
	    (D-GetCategoryType category)
	    (mapcar
	       #'(lambda (feat)           ;; for each feature in category
		  (let ( (fv (D-FindType feat bindings)) )
		     (cond
			((eq (car fv) 'LITERAL)
			   (cond
			      ((listp (cadr fv))   ;; a category valued feat
				 (D-DereferenceVariables
				    (cadr fv) bindings))
			      (t
				 (cadr fv))))
			(t      ;; a variable
			   (cadr fv)
			)
                     )
                  ))
               (D-GetCategoryValues category)
            )
         )
      )
   )
)
 
(defun D-AddDefault (fsd cat bindings)
;;
;;  checks to see if the given category has the fsd if not adds it
;;  if it has then no change
;;
;;  returns a set of bindings
;;
   (cond
      ((memq (car fsd)      ;; does this category have this feature ?
	     (cdr (assq (D-GetCategoryType cat) D-CATEGORYDEFINITIONS)))
         (let ( (result 
            (catch
	       (D-UnifyFPair 
		  (catch
		     (mapc                      ;; find that value
			#'(lambda (name value)
			   (cond
			      ((eq name (car fsd)) (throw value))
			      (t nil))) 
                        (cdr (assq (D-GetCategoryType cat)
			     D-CATEGORYDEFINITIONS))
                        (D-GetCategoryValues cat)))
                     bindings
                     (cadr fsd)))) )    ;; the default value
            (cond
	       ((eq result 'FAILED)   ;; already has a value
		  bindings    ; no change to bindings
               )
	       (t             ;; made binding
		  result      ;; return new set
               )
            )
         )
      )
      (t      ;; category does not contain value
	 bindings
      )
   )
)

