;;
;;      Name: catrouts
;;
;;      Functions:  basic functions for dealing with categories
;;
;;      Author: Alan W Black  May 1987
;;              Dept of A.I.
;;              University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;             >>>  TERM UNIFICATION VERSION <<<
;;
;;      The major functions are
;;       D-MakeCategory  - takes a printable category and
;;           returns a category structure
;;       D-MakePrintCategory - takes a category structure
;;           and returns a printable one.
;;
;;      These two functions are used to interface to the
;;      different types of category implementation 
;;      mainly Bay Area and Term Unification.
;;

(declare 
   (special
      D-VARIABLENUMBER
      D-VARIABLES)
   (localf
      D-MakeCategory 
      D-MakeCategoryList
      D-MakePCategory 
      D-FindCategoryType 
      D-SortAndAddFeatures 
      D-CheckCategory 
      D-CheckForNoDuplicates 
      D-CheckFeatures 
      D-CheckFeaturePair 
      D-MakeVarRangesNum 
      D-HeadFeatures 
      D-WDaughterFeatures 
      D-ProcessGRules
      D-SetUpDistCat
   )
)

(defmacro DK-VERSION-STAMP () 
   `(append (DK-LISP-VERSION) (ncons 'TU)))

(defmacro D-GetCategoryType (category)
   `(car ,category))

(defmacro D-GetCategoryValues (category)
   `(cdr ,category))

(defmacro D-GetCategorySkeleton (cattype)
   `(cdr (assq ,cattype D-CATEGORYDEFINITIONS)))

(defun D-MakeCategory (pcategory)
;;
;;   Takes a printable category and produces a category structure in
;;   return.
;;
;;   for TERM UNIFICATION
;;
;;   Following Steve's suggestions.  A category becomes
;;   (
;;     type       ;;  a symbol
;;     fvalue1    
;;     fvalue2
;;     ...
;;   )
;;
   (let ( (cattype (D-FindCategoryType pcategory)) )
      (cond
	 ((not (eq (length cattype) 1))
	    (princ "Unknown category type: ")
	    (terpri)
	    (princ pcategory) (terpri)
	    (D-PList cattype) (terpri)
            (cons nil nil))
         (t
	    (cons 
	       (car cattype)   ;; category type
	       (car            ;; drop the bindings
	          (D-SortAndAddFeatures     ;; give it the cat type skeleton
	             (D-GetCategorySkeleton (car cattype))
	             pcategory nil nil))))
      )
   )
)

(defun D-MakeCategoryList (pcategories)
;;
;;  to cope with sets of categories that may have shared variables.
;;  This is used to generate the categories that in one rule.  The
;;  difference is that the binds are passed on to the next category
;;
   (let ( binds )
      (mapcar
         #'(lambda (pcat)
            (let ( (cattype (D-FindCategoryType pcat)) )
	    (cond
	       ((not (eq (length cattype) 1))
		  (princ "Unknown category type: ")
		  (terpri)
		  (princ pcat) (terpri)
		  (D-PList cattype) (terpri)
		  (cons nil nil))
	       (t
		  (let ( (rest (D-SortAndAddFeatures
			  (cdr (assq (car cattype) D-CATEGORYDEFINITIONS))
			  pcat binds nil)) )
                  (setq binds (cdr rest))
		  (cons 
		     (car cattype)   ;; category type
		     (car            ;; drop the bindings
			rest)))))))
        pcategories)
   )
)

(defun D-MakePCategory (category)
;;
;;  returns an alist that is nice and printable
;;
  ; (append
      ;(D-GetCategoryFixed (D-GetCategoryType category))
      (mapcan
         #'(lambda (fname fvalue)
	    (cond
	       ((D-VariableP fvalue)  
		  ;; not really sure what to do here so leave variable in
                  (ncons (list fname fvalue)) 
               )
	       ((D-CatValFeatP fname)
	          (ncons
		     (list fname (D-MakePCategory fvalue)))
               )
	       (t  ;; simple case
	          (ncons (list fname fvalue)))))
       (D-GetCategorySkeleton (D-GetCategoryType category))
       (D-GetCategoryValues category)
      )
   ;)
)

(defun D-FindCategoryType (minimalcat)
;;
;; returns a list of possible type of the given minial category
;;
   (let ( (feats (mapcar #'car minimalcat)) )
      ;(setq TOTAL (+ 1 TOTAL))
      (mapcan
         #'(lambda (type)
	    (cond
	       ((D-Subset feats (cdr type))
		  (ncons (car type)))
               (t   ;; not of this type
		  nil))
         )
         D-CATEGORYDEFINITIONS))
)

(defun D-SortAndAddFeatures (skeleton mincat binds newcat)
;;
;;  returns a full category from the given skeleton and the minimal
;;  list of fpairs. i.e. a list of values
;;
;;
   (cond
      ((null skeleton)
	 (cons
	    (nreverse newcat)
	    binds))
      (t
	 (let ((fpair (assq (car skeleton) mincat)))
	    (cond
               ((null fpair)      ;; unspecified category
		  (D-SortAndAddFeatures 
		     (cdr skeleton)
		     mincat binds
		     (cons
			(D-MakeNewVariable (D-GetFeatRange (car skeleton)))
			newcat))
               )
               ((and (D-DeclaredVariableP (cadr fpair)) 
		     (assq (cadr fpair) binds))     ;; a bound user variable
		  (D-SortAndAddFeatures
		     (cdr skeleton)
		     mincat binds
		     (cons
			(cadr (assq (cadr fpair) binds))
                        newcat))
               )
               ((D-DeclaredVariableP (cadr fpair))       ;; an unbound user var
		  (let ( (newvar      
			   (D-MakeNewVariable 
			     (D-GetDeclVarRange (cadr fpair)))) )
		  (D-SortAndAddFeatures
		     (cdr skeleton)
		     mincat
		     (cons (list (cadr fpair) newvar) binds)
		     (cons newvar newcat))))
               ((D-CatValFeatP (car skeleton))   ;; a real category value
		  (D-SortAndAddFeatures
		     (cdr skeleton)
		     mincat binds
		     (cons
			(D-MakeCategory (cadr fpair))
                        newcat)))
               (t               ;; specified atomic value
		  (D-SortAndAddFeatures
		     (cdr skeleton)
		     mincat binds
		     (cons (cadr fpair) newcat)))
            )
         )
      )
   )
)
		  
(defun D-CheckCategory (pcategory)
;;
;;   This checks the consistancy of a category.  All values of
;;   features must be consistant with their declarations.  If they
;;   are not a warning is given, but processing continues.  No
;;   guarantee is given for running grammars that give inconsistancy
;;   warnings, but they will probably work.
;;
   (cond
      ((listp pcategory)    ;; catgegory must be a list
	 (cond
	    ((D-CheckForNoDuplicates pcategory nil (length pcategory))
	       (D-CheckFeatures pcategory t)
            )
	    (t nil)    ;; must have been duplicates
         )
      )
      (t
	 (D-errmsg " invalid category " pcategory)
      )
   )
)

(defun D-CheckForNoDuplicates (pcategory catsfound numcats)
;;
;;  checks the category to find any feature names that are used more
;;  than once
;;  duplications are reported to the terminal, nil is always returned
;;
   (cond
      ((null pcategory)
	 (cond
	    ((eq numcats (length catsfound)) t) ;; check ok
	    (t nil)         ;; must have been some duplicates
         )
      )
      ((memq (caar pcategory) catsfound)
	 (D-errmsg "duplicate feature name " (caar pcategory))
	 (D-CheckForNoDuplicates (cdr pcategory) catsfound numcats)
      )
      (t
	 (D-CheckForNoDuplicates 
	    (cdr pcategory)
	    (cons (caar pcategory) catsfound)
	    numcats
         )
      )
   )
)

(defun D-CheckFeatures (pcategory resultflag)
;;
;;   checks the features in a category and returns nil if at
;;   least one error, t otherwise
;;
   (cond
      ((null pcategory) resultflag)
      ((atom (car pcategory))
	 (D-errmsg "not feature pair " (car pcategory))
      )
      ((D-CheckFeaturePair (car pcategory))
         (D-CheckFeatures (cdr pcategory) resultflag)
      )
      (t     ;; there was an error here
	 (D-CheckFeatures (cdr pcategory) nil)   ;; pass on error flag
      )
   )
)

(defun D-CheckFeaturePair (featurepair)
;;
;;  This checks the validity of a feature pair, The feature must be
;;  declared and have an apropriate value
;;
   (cond
      ((D-CatValFeatP (car featurepair))  ;; category valued feature
	 (cond
	    ((listp (cadr featurepair))   ;; must be a list
	       (D-CheckCategory
	          (cadr featurepair)
               )
            )
	    ((and (D-DeclaredVariableP (cadr featurepair))
		  (eq (D-GetDeclVarRange (cadr featurepair)) (DK-category)))
               t
            )
	    (t
	       (D-errmsg "Not a valid category-value for feature "
			  (car featurepair))
            )
         )
      )
      ((D-FeatureP (car featurepair))    ;; atomic valued feature
	 (cond
	    ((or (D-FeatureValueP (car featurepair)
				  (cadr featurepair))
                 (and (D-DeclaredVariableP (cadr featurepair))
		      (D-Subset (D-GetDeclVarRange (cadr featurepair))
				(D-GetFeatRange (car featurepair)))))
               t                ;; feature pair ok
            )
	    (t
	       (D-errmsg "Invalid feature value" featurepair)
            )
         )
      )
      (t
	 (D-errmsg "Invalid feature name" featurepair)
      )
   )
)

(defun D-MakeVarRangesNum (category)
;;
;; replaces teh var ranges with number so as the entryies are actually
;; smaller
;;
   t
)

(defun D-HeadFeatures (category)
;;
;;    Returns a special form of category with the non head features marked
;;    with names of (ncons nil)
;;
   (mapcar
      #'(lambda (fpair)
	    (cond
	       ((D-HeadFeatureP (car fpair)) fpair)
	       (t (list (ncons nil) nil))))
      (D-GetCategoryFPairs category)
   )
)

(defun D-WDaughterFeatures (category)
;;
;;    Returns the list of wdaughter features as declared in the dcls file
;;
   (mapcar
      #'(lambda (fpair)
	    (cond
	       ((D-DaughterFeatureP (car fpair)) fpair)
	       (t (list (ncons nil) nil))))
      (D-GetCategoryFPairs category)
   )
)

(defun D-ProcessGRules (rules)
;;
;;  Normalises all the variables in the rules to standard cons cells
;;  building real categories in the rules
;;
   (mapcar
      #'(lambda (rule)
	 (cons t
	    (cons
	       (cadr rule)   ;;the rule name
	       (D-MakeCategoryList (cddr rule))))
      )
      rules)
)

(defun D-SetUpDistCat (catnames)
;;
;;   This does different things dependent on term or bay area
;;   unification.  This one (term) does nothing
;;
   catnames
)

