/* dbmtry.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* If you have problems with the dbm interface, try this program.
 * If it fails with ALERT messages when given an argument of 300 or so,
 * you almost certainly have a faulty dbm.
 *
 * On SysV, by the way, check for delitem() calling bcopy() with
 * overlapping arguments...
 *
 * This version of the test program prints messages even when things are OK.
 *
 * $Id: dbmtry.c,v 1.7 92/05/10 23:33:03 lee Exp $
 *
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h>
#include <fcntl.h>
#ifdef bsdhash
# include <sys/types.h>
#endif
#include "smalldb.h"

char *progname = "dbmtry";
int AsciiTrace = 0;

/** Unix system calls: **/
extern void exit();

/** Unix Library Functions: **/
extern int atoi();
extern int strlen();
extern void perror();

/** liblqtext Library Functions: **/
extern void lqWriteAccess();
extern void lqReadOnlyAccess();

/** Routines in this file which need declaring: **/
void printvalues();
static int FindMax();
static void SetMax();

/** **/

static char *TestFile = "/tmp/trydbm";
static int ErrorCount = 0;

int
main(ac, av)
    int ac;
    char *av[];
{
    DBM *db;
    int max = atoi(av[1]);
    int i;
    datum key, data;
    char dbuf[30];
    int min;

    if (ac <= 1) {
	fprintf(stderr, "Usage: %s maxkey\n", av[0]);
	exit(1);
    }

    lqWriteAccess();

    fprintf(stderr, "%s: ** Using test database \"%s\"\n", progname, TestFile);

    if ((db = startdb(TestFile)) == (DBM *) 0) {
	Error(E_FATAL|E_SYS, "Couldn't open test database %s", TestFile);
    }

    if ((min = FindMax(db)) < 0) {
	min = 0;
    } else {
	if (min + 1 >= max) {
	    Error(E_WARN, "%s previously had a stored max of %d, using %d",
		TestFile,
		min,
		max + min
	    );
	    max += min;
	}
	printvalues(min, "previously stored");
	++min; /* start one above the last max */
    }

    fprintf(stderr, "%s: ** writing from %d up to %d.\n", progname, min, max);

    for (i = min; i <= max; i++) {
	char buf[20];
	register int s_val;

	sprintf(buf, "%d", i);
	sprintf(dbuf, "%d data item here", i);
	    /* Note: the number is at the start to help speed the
	     * strcmp, as it is most likely to differ
	     */
	key.dsize = strlen(buf) + 1; /* include the \0 so we can strcmp() */
	key.dptr = buf;
	data.dptr = dbuf;
	data.dsize = strlen(dbuf) + 1;
	s_val = dbm_store(db, key, data, DBM_INSERT);
	if (s_val != 0) {
	    Error(E_WARN, "ALERT! dbm_store %d returned %d, not 0", i, s_val);
	    ++ErrorCount;
	}
    }

    fprintf(stderr, "%s: ** write test complete: %d error%s\n",
	progname,
	ErrorCount,
	(ErrorCount == 1) ? "" : "s"
    );

    SetMax(db, max);

    enddb(db);
    cleanupdb();

    printvalues(max, "all");

    if (ErrorCount) {
	Error(E_FATAL, "**** ALERT **** Total of %d errors, should be 0",
	    ErrorCount
	);
    } else {
	fprintf(stderr, "%s: test passed.\n", progname);
    }
    exit(0);
}

void
printvalues(max, note)
    int max;
    char *note;
{
    DBM *db;
    int i;
    char buf[20];
    datum key, data;

    lqReadOnlyAccess();

    db = startdb(TestFile);

    if (!db) {
	Error(E_FATAL|E_SYS, "Unable to open database %s", TestFile);
    }

    i = FindMax(db);
    if (i != max) {
	Error(E_WARN, "FindMax() returned %d, but %d was expected", i, max);
    }

    fprintf(stderr, "%s: ** Checking %s stored data from 0 up to %d.\n",
	progname, note, max
    );

    /* Note: always start at zero */
    for (i = 0; i <= max; i++) {

	sprintf(buf, "%d", i);
	key.dsize = strlen(buf) + 1;
	key.dptr = buf;
	data = dbm_fetch(db, key);

	if (data.dsize == 0) {
	    Error(E_WARN, "ALERT! Item %d has been lost! ALERT!", i);
	    ++ErrorCount;
	} else {
	    char *Buf[100];
	    (void) sprintf(Buf, "%d data item here", i);
	    if (strncmp(Buf, data.dptr, data.dsize) != 0) {
		Error(E_WARN, "ALERT! %d: Corrupt: \"%s\" != \"%s\" ALERT!",
				i, data.dptr, Buf);
		++ErrorCount;
	    }
	}
    }
    enddb(db);
}

static char *MaxValueString = "MAX VALUE";

static int
FindMax(db)
    DBM *db;
{
    int i;
    datum key, data;

    key.dptr = MaxValueString;
    key.dsize = strlen(MaxValueString);

    data = dbm_fetch(db, key);

    if (data.dsize == 0 || !data.dptr) return -1;
    i = atoi(data.dptr);
    if (i <= 0) {
	Error(E_WARN, "Max Value stored in %s was as %d, strange", TestFile, i);
	return -1;
    }

    return i;
    
}

static void
SetMax(db, max)
    DBM *db;
    int max;
{
    int i;
    char buf[20];
    datum key, data;

    key.dptr = MaxValueString;
    key.dsize = strlen(MaxValueString);

    (void) sprintf(buf, "%d", max);
    data.dptr = buf;
    data.dsize = strlen(buf) + 1; /* include the \0 so atoi() will work */

    i = dbm_store(db, key, data, DBM_REPLACE);
    if (i < 0) {
	Error(E_WARN|E_SYS,
	    "Failed to insert Max marker (%d) into %s",
	    max,
	    TestFile
	);
    }
}
