/* numbers.c -- Copyright 1989, 1992 Liam R. E. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* Routines to read and write numbers in a compressed format, preserving
 * block boundaries.
 * The actual compression seems to be about 50%, as most numbers turn
 * out to fit in 16 bits.  Interestingly, there is room for another one
 * or two bits, I think, that could be used for something else, in the
 * main pblock index.  For example, it could mark whether words were
 * plural/-"ing"/"un"-/ with 2 bits.
 *
 * $Id: numbers.c,v 1.12 92/08/02 09:22:49 lee Exp $
 */

#include "globals.h"
#include "error.h"
#include "numbers.h"

/* ReadNumber and WriteNumber take/return a long, using a compression
 * algorithm to reduce the amount of data taken.
 * A 1 in the top bit means another byte follows, hence fitting
 * 7 bits into each bytes.
 * Speed is very important.  These functions are the backbone of lq-text.
 *
 * The routines use (char *) pointers instead of files prefixes with an s.
 * see numbers.h for some related macros.
 *
 * There used to be FILE ** versions, but I never used them.
 */


#ifndef HAVE_INLINE

#define PutC(ch, S)  (*((*S)++) = (char) (ch))

void
sWriteNumber(s, Number)
    char **s;
    unsigned long Number;
{
    /* Compressed numbers:
     * 7 bit numbers --> single byte;
     * 8...14 bits --> 2 bytes
     * 15...21 bits --> 3 bytes
     * 22..28 bits --> 4 bytes
     * 29..32 bits --> 5 bytes
     */
    while (Number > 0177) {
	PutC((Number & 0177) | 0200, s);
	Number >>= 7;
    }
    PutC(Number & 0177, s);
}

#define GetC(S) \
    ( (unsigned int) * (unsigned char *) ((* (unsigned char **)S)++) )

unsigned long
sReadNumber(s)
    char **s;
{
    unsigned long Result = 0L;
    int ThereIsMore;
    int Shift = 0;

    /* Read a number, 7 bits at a time, lsb first, until there is
     * a byte without the top bit set -- that's the most significant
     * byte, and there is no more of this number.
     */

    do {
	Result |= ((ThereIsMore = GetC(s)) & 0177) << Shift;
	ThereIsMore &= 0200;
	Shift += 7;
    } while (ThereIsMore);

    return Result;
}

#endif /* HAVE_INLINE */
