/* Error.c $Id: error.c,v 1.9 92/02/06 22:48:45 lee Exp $
 *
 * Liam Quin
 *
 */

#include <stdio.h>

#define _ERROR_C /* so that we don't say "extern int errno" */

#include "error.h"

#ifdef NEEDERRNO
  int errno = 0;
#else
# include <errno.h>
  extern int sys_nerr;
  extern char *sys_errlist[];
    /* Sys_errlist contains the error messages that correspond to the
     * various values of errno defined in <errno.h>, and possibly others.
     *
     * Unix defines sys_errlist to have at least sys_nerr entries. 
     * errno.h also defines 
     *     extern int errno;
     * so we don't do it here.  On Unix systems, errno holds the number
     * of the last error encountered in a system call or library function.
     * (it contains rubbish if no error occurred).
     */
#endif

#define E_NOMALLOC 4096
    /* This is a fake Severity, that works like E_MEMORY but without
     * printing "out of memory"
     */

extern char *progname;
    /* You should set "progname" in main() to be the last component
     * of argv[0] if the command is one intended to be invoked by
     * humans directly (e.g. /bin/sh ---> sh), or the entire of
     * argv[0] if this is a library program (e.g. /usr/lib/sendmail),
     * since in the latter case the user might not have invoked this
     * command directly, and thus might not know where it lives!
     */


/* Error() should use varargs and vprintf(), but I am not convinced that
 * vprintf() is sufficiently widely available.
 */

/*VARARGS2*/
void
Error(Severity, str, a, b, c, d, e, f, g, h)
    int Severity;
    char *str;
    int a, b, c, d, e, f, g, h;
{
    extern char *getenv();
    register char *p;
    static char *cmdname = NULL;
    int esav = errno;
	/* Esav holds the value of errno on entry, since library functions
	 * such as printf() will (in general) alter errno.
	 */

    if (!str || !*str) {
	static char PanicBuf[100 + sizeof(__FILE__)];

	/* Someone called Error() without giving a string, or (more likely)
	 * forgot the Severity argument and did
	 *	Error("oh dear");
	 * so we have to invent a message that the user can report to
	 * technical report (or that will help the programmer!)
	 */
	(void) sprintf(PanicBuf,
		"%s: Error(%d, 0) called with %s error message",
		__FILE__, Severity, (str == (char *) 0) ? "NULL" : "empty");
	str = PanicBuf;
	Severity |= E_FATAL | E_INTERNAL | E_NOMALLOC;
	/* We also forbid malloc(), in case the programmer actually did
	 * Error(E_WARN, strcpy(malloc(......)...));
	 */
    } else if (Severity == 0) {
	/* You have to use one of the E_* values from error.h, none of
	 * which are ever zero.  This might mark a programmer doing
	 * Error(!E_FATAL, ....)
	 * but this is a mistake...
	 * Since we have an actual error message (because we are in the
	 * "else" branch), we'll print two error messages...
	 */
	 Error(E_INTERNAL|E_WARN, "Error() called with Severity (arg 1) 0");
	 Severity |= (E_FATAL|E_INTERNAL|E_NOMALLOC);
    }

    if (!cmdname) {
	/* Shell scripts can do
	 * CMDNAME="henry"
	 * in order to make error messages print as
	 * henry: thisprog: error: ....
	 *
	 * We only ever do the getenv() once, so later changes made with
	 * putenv("CMDNAME", "new value") will not be reflected here.
	 * This is intentional, as it means that error messages are less
	 * likely to change diring a single run of the program!
	 */
	cmdname = getenv("CMDNAME");
	if (cmdname && cmdname[0] == '\0') {
	    cmdname = NULL;
	    /* This can happen if someone does
	     * CMDNAME=""
	     * in the shell.
	     */
	}
    }

    /** Should really do this for each newline in the error message: **/

	    /*  Note: in order to do this properly, we'd have to parse the
	     *  format string, handling each "%" individually, coping with "*"
	     *  (as in "%*.*d) correctly.  We can't allocate a buffer with
	     *  malloc() if there was a MEMORY error, of course, and we have
	     *  no idea how large a fixed-size buffer to use.
	     *  This is because you could do
	     *      Error(E_WARN, "A %s b", "\n\n\n");
	     *  if you wanted (or if a filename contained a newline).
	     */

    /** Print the error message! **/

    /* First the command name */
    if (cmdname) {
	(void) fprintf(stderr, "%s: ", cmdname);
    }

    /* Now the program name */
    if (progname) {
	(void) fprintf(stderr, "%s: ", progname);
    } else {
	(void) fprintf(stderr, "[progname unset]: ");
	/* You could call this obnoxious, but the assumption is that
	 * the programmer will catch this error (of not setting
	 * progname) as soon as the first diagnostic is printed!
	 */
    }
    /** end of per-line stuff */

    if (Severity & E_INTERNAL) {
	(void) fprintf(stderr, "internal ");
    }

    if (Severity & E_FATAL) {
	(void) fprintf(stderr, "fatal error: ");
    } else {
	if (Severity & E_INTERNAL) {
	    (void) fprintf(stderr, "error: ");
	    /* Hence E_INTERNAL|E_WARN turns into  "internal error: " */
	} else {
	    (void) fprintf(stderr, "warning: ");
	}
    }
    if (Severity & E_USAGE) {
	(void) fprintf(stderr, "usage: ");
    }

    if (Severity & E_MEMORY) {
	/* Apart from making programs that check malloc() a lot smaller
	 * (they don't each have to say "out of memory"...), the E_MEMORY
	 * flag can be used to inhibit a pop-up window on systems where
	 * the window system calls malloc()... otherwise we would get
	 * into a mess whilst trying to open the new window!
	 *
	 * This implementation does not use pop-up dialogue boxes, but
	 * I have others that do, and that are compatible with this one
	 * from the caller's point of view.
	 */
	(void) fprintf(stderr, "out of memory: ");
    }

    /* Ensure that there is no newline at the end of the format passed to
     * fprintf, so that we can append a system error message if we so
     * choose:
     */
    p = &str[strlen(str) - 1];

    if (*p == '\n') {
	*p = '\0';
    } else if (*(p - 1) == '\n' && *p == '\r') {
	*--p = '\0';
    }

    /* Now call printf() to put out the actual message.
     */
    (void) fprintf(stderr, str, a, b, c, d, e, f, g, h);
	/* NOTE: we can't report an error if stderr is broken... so
	 * don't check the return value...
	 */

    if (Severity & E_SYS) {
	/* Print a Unix perror()-style error message if asked so to do.
	 * We use the value of errno that we saved right at the start in
	 * esav, in case fprintf() or getenv() or something clobbered errno.
	 */
#ifdef NEEDERRNO
	switch (errno) {
	case 0: /* no error */
	    break;
	case ENOENT:
	    (void) fprintf(stderr, " no such file or directory");
	    break;
	/** Add more values here if you implement them **/
	default:
	    (void) fprintf(stderr, " unknown system error %d", esav);
	    errno = 0;
	    break;
	}
#else
	if (esav > 0 && esav < sys_nerr) {
	    (void) fprintf(stderr, ": %s", sys_errlist[esav]);
	} else if (esav != 0) {
	    (void) fprintf(stderr, " unknown system error %d", esav);
	}
#endif
    }

    (void) fputc('\n', stderr);
	/* Finally, terminate the message.
	 * TODO: check that the tty is in "onlcr" mode, and, if not,
	 * put a \n\r at the start of the message and a \r at the end
	 * NOTDONE FIXME
	 */

    if (Severity & E_XHINT) {
	if (cmdname) {
	    (void) fprintf(stderr, "%s: ", cmdname);
	}
	if (progname) {
	    (void) fprintf(stderr,
		"%s: use the -x option for an explanation\n", progname);
	} else {
	    (void) fprintf(stderr, "use the -x option for an explanation\n");
	}
    }

    if ((Severity & E_ABORT) == E_ABORT) {
	if ((Severity & E_BUG) == E_BUG) {
	    extern int AsciiTrace;

	    if (AsciiTrace >= 3) {
		(void) fflush(stderr);
		fprintf(stderr, "%s: generating core dump for debugging.\n",
				progname);
		abort();
	    } else {
		exit(1);
	    }
	} else {
	    abort();
	}
    }

    if (Severity & E_FATAL) {
	exit(1);
    }
}

#ifdef NEEDERRNO

/* Define fake open and fopen routines that set errno for a better error
 * message.  Another way would be to have them call Error() directly, but
 * that would break code that tested return values.
 *
 * Note that open and fopen are defined in error.h to be _e_open and _e_fopen,
 * so we are not really defining open() and fopen() here at all.  If you
 * get a message from the loader about __e_open being multiply defined,
 * or being defined more than once, you should edit error.h and change those
 * names to something else (perhaps Prisilla and Gertrude) and recompile
 * everyhithing that uses error.h yourself.
 */

int open(filename, mode, flags)
    char *filename;
    int mode;
    int flags;
{
    register int result;

#undef open /* so we can call the real one */
    if ((result = open(filename, mode, flags)) >= 0) {
	errno = 0;
	    /* Unix does not set errno in this case, but we might as well */
	return result;
    } else {
	/* if there is some other way of determining the error, it would
	 * be worth doing here...
	 */
	errno = ENOENT;
	    /* No entry in the directory -- i.e., file does not exist */
	return result;
    }
}

FILE *
fopen(filename, mode)
    char *filename;
    char *mode;
{
    register FILE *Result;

    errno = 0;
#undef fopen /* so we can call the real one */
    if ((Result = fopen(filename, mode)) != (char *) 0) {
	return Result;
    }
    /* if there is some other way of determining the error, it would
     * be worth doing here...
     */
    errno = ENOENT; /* No entry in the directory: i.e., file does not exist */
    return result;
}
#endif /*NEEDERRNO*/

/*
 * $Log:	error.c,v $
 * Revision 1.9  92/02/06  22:48:45  lee
 * Made E_ABORT fatal.
 * 
 * Revision 1.8  92/02/06  22:47:39  lee
 * Added declaration for AsciiTrace (sorry!)
 * 
 * Revision 1.7  92/02/06  22:46:21  lee
 * Added code for E_BUG and E_ABORT.
 * 
 * Revision 1.6  90/08/28  16:06:23  lee
 * added XHINT to provide the "use %s -x for an explanation" message.
 * 
 * Revision 1.1  90/08/25  21:37:30  lee
 * Initial revision
 * 
 *
 */
