/* lqkwik.c -- Copyright 1991 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* lqkwik -- produce a keyword-in-context list of matches...
 * Liam R. Quin, February 1991 and later...
 *
 * $Id: lqkwik.c,v 1.1 91/03/02 20:37:47 lee Rel1-10 $
 */

#define COLS 65   /* the width of kwik    word index */
#define WORDCOL 25 /* where to put the    word in the index */
#define GAPWIDTH 2 /* space before the    word itself */
#define SCREENWIDTH 79

int Cols = COLS;
int WordCol = WORDCOL;
int GapWidth = GAPWIDTH;
int ScreenWidth = SCREENWIDTH;

static int LinesAbove = 5;
static int linesBelow = 5;

#include "globals.h" /* defines and declarations for database filenames */

#include <malloc.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h> /* for fileinfo.h */
#include <sys/stat.h>

#include <stdio.h>

#include "fileinfo.h"
#include "wordinfo.h"
#include "wordrules.h"
#include "pblock.h"
#include "emalloc.h"

/** Unix system calls that need declaring: **/
extern long lseek();
extern int open(), close();
extern int read();
extern void exit();
extern int stat();

/** Unix/C Library Functions that need declaring: **/
#ifndef tolower
 extern int tolower();
#endif
extern int strlen();
extern int strcmp();
extern unsigned sleep();
extern int atoi();
extern long atol();
extern void perror();

/** lqtext library functions that need declaring: **/
extern int MySystem();
extern int TooCommon();
extern void SetDefault();
extern void DefaultUsage();

/** Functions within this file that are used before being defined: **/
int ReadMatchFile();
int ShowFile();
void Output();

/** **/

/** some useful macros: **/
#define max(choir,boy) (((choir)>(boy))?(choir):(boy))
#define min(choir,boy) (((choir)<(boy))?(choir):(boy))

/** **/

int AsciiTrace = 0;

extern int errno;

char *progname = "lqkwik"; /* set from argv[] in main() */

int SelectedNames = -1;
FILE *InfoStream = 0;

static char *Revision = "@(#) showfile.c 2.2";

int
main(argc, argv)
    int argc;
    char *argv[];
{
    extern int optind, getopt();
    extern char *optarg; /* for getopt */
    int ch; /* for getopt */
    int ErrFlag = 0; /* see how getopt makes programs cleaner? */
    int NumberOfFiles;
    char *FileWithMatches = (char *) 0;
    char **MatchList;
    int MatchCount = 0;
    int Origc;
    int Right = Cols - (WordCol + GapWidth);
    int Left = WordCol - GapWidth;

    progname = argv[0];

    SetDefaults(argc, argv);

    /* All lq-text programs must call SetDefaults() before getopt, and
     * must then be prepared to ignore options z with arg and Z without.
     */
    while ((ch = getopt(argc, argv, "a:b:f:l:r:g:w:o:z:ZVvx")) != EOF) {
	switch (ch) {
	case 'z':
	    break; /* done by SetDefaults(); */
	case 'V':
	    fprintf(stderr, "%s version %s\n", progname, Revision);
	    break;
	case 'v':
	    AsciiTrace = 1;
	    break;
	case 'f':
	    FileWithMatches = optarg;
	    break;
	case 'g':
	    GapWidth = atoi(optarg);
	    break;
	case 'l':
	    Left = atoi(optarg);
	    break;
	case 'r':
	    Right = atoi(optarg);
	    break;
	case 'w':
	    ScreenWidth = atoi(optarg);
	    break;
	case 'x':
	    ErrFlag = (-1);
	    break;
	case '?':
	default:
	    ErrFlag = 1;
	}
    }

    if (ErrFlag < 0) { /* -x or -xv was used */
	fprintf(stderr, "usage: %s [-xv] [options] [matches...]\n", progname);
	fprintf(stderr,
	"use %s -x, -xv or -xvv for more detailed explanations.\n", progname);

	if (AsciiTrace) {
	    DefaultUsage();
	    fprintf(stderr, "\n\
	-f file -- \"file\" contains a list of matches, one per line\n");
	    fprintf(stderr, "\
	-g n    -- set gap between text and matched phrase to n [%d]\n\
	-l n    -- display n characters to the left of each phrase [%d]\n\
	-r n    -- display r chars to the right of each phrase's start [%d]\n\
	-w n    -- truncate the line after n characters [default: %d]\n",
		    GapWidth,
		    Left,
		    Right,
		    ScreenWidth
	    );
	}
	if (AsciiTrace > 1) {
	    fputs("\
	Matches should be in the form of\n\
		BlockNumber  WordInBlock  FileName\n\
	where BlockBumber and WordInBlock are positive numbers.\n\
	(This is the format produced by the lqword -l command.)\n\
", stderr);
	}
	exit(0);
    } else if (ErrFlag > 0) {
	fprintf(stderr, "use %s -x for an explanation.\n", progname);
	exit(1);
    }

    Cols = Left + Right + GapWidth;
    WordCol = Left + GapWidth;

    if (AsciiTrace) {
	fprintf(stderr,"Left:%d  Right:%d  Cols:%d  Gap:%d  WC:%d  SW:%d\n",
			Left, Right,	  Cols, GapWidth, WordCol,ScreenWidth);
    }

    if (ScreenWidth <= Cols + 2) {
	fprintf(stderr,
	    "%s: ScreenWidth %d, %d text cols -- no room for file names!\n",
	    progname, ScreenWidth, Cols);
	exit(1);
    }

    /* open the file for the selected output */
    if (SelectedNames > 0) {
	if ((InfoStream = fdopen(SelectedNames, "w")) == (FILE *) 0) {
	    int e = errno;

	    fprintf(stderr, "%s: -o %d: can't open stream ",
	    					progname, SelectedNames);
	    errno = e;
	    perror("for writing");
	    exit(1);
	}
    }

    /* check that we can get at the file containing the matches, if one
     * was supplied.
     */
    if (FileWithMatches) {
	struct stat StatBuf;
	char *msg = 0;

	if (stat(FileWithMatches, &StatBuf) < 0) {
	    int e = errno; /* on many systems, fprintf() changes errno! */
	    fprintf(stderr, "%s: can't open match-list file ", FileWithMatches);
	    errno = e;
	    perror(progname);
	    exit(1);
	} else if (AsciiTrace) {
	    switch (StatBuf.st_mode & S_IFMT) {
	    case S_IFDIR:
		fprintf(stderr,
		"%s: ca't read matches from \"%s\" -- it's a directory!\n",
						progname, FileWithMatches);
		exit(1);
	    case S_IFREG:
		break;
#ifdef S_IFIFO
	    case S_IFIFO:
		msg = "named pipe or fifo";
		/* fall through */
#endif
	    case S_IFCHR:
		if (!msg) msg = "raw special device";
		/* fall through */
	    case S_IFBLK:
		if (!msg) msg = "block special device";
		/* fall through */
#ifdef S_IFNAM
	    case S_IFNAM:
		if (!msg) msg = "named special file"; /* wot dat? */
		/* fall through */
#endif
	    default:
		if (!msg) msg = "special file";

		fprintf(stderr,
		    "%s: warning: file \"%s\" containing matches is a %s\n",
		    progname, FileWithMatches, msg);
		
		/* but continue anyway... */

	    }
	}
	/* Now read the file, and make an array of matches... */
	if (ReadMatchFile(FileWithMatches, StatBuf.st_size, &MatchCount, &MatchList) < 0) {
	    fprintf(stderr, "%s: couldn't read matches from \"%s\"\n",
						progname, FileWithMatches);
	    exit(1);
	}
    }

    argv += optind;
    argc -= optind;

    if (MatchCount) {
	argc = MatchCount;
	argv = MatchList;
    }

    if (argc < 3) {
	fprintf(stderr,
	"%s: matches must have at least 3 parts; use -xv for an explanation\n",
								progname);
	exit(1);
    } else if (argc % 3) {
	/* Note: I could detect lqword output here (i.e., without -l) */
	fprintf(stderr, "%s: can't understand match format;\n", progname);
	fprintf(stderr, "%s: use -xv for more explanation.\n", progname);
	exit(1);
    }

    Origc = argc;

    NumberOfFiles = argc / 3;

    while (argc > 0) {
	int Where;

	if (ShowFile(argv[2], atol(*argv), (unsigned) atoi(argv[1])) < 0) {
	    int i;

	    /* This avoids repeated messages about the same file */
	    for (i = argc - 3; i > 0; i -= 3) {
		if (STREQ(argv[2], argv[2 + 3])) {
		    argv += 3;
		} else {
		    break;
		}
	    }
	    argc = i + 3; /* so we can subtract 3 ... */

	}
	argv += 3;
	argc -= 3;
    }

    return 0;
}

int
ReadMatchFile(FileWithMatches, FileSize, MatchCount, MatchList)
    char *FileWithMatches;
    off_t FileSize;
    int *MatchCount;
    char ** *MatchList;
{
    extern char *strtok();

    int fd;
    char **Result;
    char *StorageArea;
    char *p;
    unsigned int n_matches;
    int BytesRead;
    int i;
    char *NextStr;

    if (!FileWithMatches || !*FileWithMatches) {
	fprintf(stderr, "%s: match-list file (from -f) has empty name!\n",
								progname);
	exit(1);
    }

    if ((fd = open(FileWithMatches, O_RDONLY)) == 0) {
	int e = errno;
	fprintf(stderr, "%s: can't open match-list file ", progname);
	errno = e;
	perror(FileWithMatches);
	exit(1);
    }

    /* We know the number of bytes, and each space or newline will get
     * turned into a null, so here goes...
     * The +1 below is to ensure that there is space for a \0, even if a
     * pesky user didn't put a \n at * the end of the file...
     * Sometimes I hate emacs...
     */
    if ((StorageArea = malloc((unsigned) FileSize + 1)) == (char *) 0) {
	fprintf(stderr, "%s: not enough memory to read match-list \"%s\"\n",
						    progname, FileWithMatches);
	exit(1);
    }

    /* now read the list... */
    if ((BytesRead = read(fd, StorageArea, FileSize)) != FileSize) {
	if (BytesRead < 0) {
	    int e = errno;

	    fprintf(stderr, "%s: couldn't read %u bytes from ",
							progname, FileSize);
	    errno = e;
	    perror(FileWithMatches);
	    exit(1);
	} else {
	    int e = errno;

	    fprintf(stderr, "%s: ", progname);
	    if (BytesRead > 4) { /* minimum plausible for 3 items */
		fprintf(stderr, "warning: ");
	    }
	    fprintf(stderr, "only read %u bytes, not %u, from file ",
						progname, BytesRead, FileSize);
	    errno = e;
	    if (errno) perror(FileWithMatches);
	    else fprintf(stderr, "\"%s\"\n", FileWithMatches);

	    if (BytesRead <= 4) exit(1);

	    StorageArea = realloc(StorageArea, (unsigned) (BytesRead + 1));
	    if (StorageArea == (char *) 0) { /* unlikely, it got smaller  */
		fprintf(stderr, "%s: can't realloc for \"%s\"\n",
						progname, FileWithMatches);
		exit(1);
	    }
	    FileSize = BytesRead;
	}
    }

    /* null-terminate it */
    StorageArea[FileSize] = '\0';

    /* got the data, now make an array... first, count the matches */
    for (n_matches = 1, p = StorageArea; p - StorageArea < FileSize; p++) {
	if isspace(*p) ++n_matches;
    }

    /* If there *was* trailing new-line, we overestimated by one.
     * This doesn't matter.  If memory is that tight, initscr() will fail.
     * In any case, allow extra space for a trailing null entry.
     */
    ++n_matches;
    if (n_matches < 3) n_matches = 3;

    Result = (char **) malloc((unsigned) n_matches * sizeof(char *));
    if (Result == (char **) 0) {
	fprintf(stderr, "%s: out of memory reading match file \"%s\"\n",
						progname, FileWithMatches);
	exit(1);
    }

    /* Now step through the Storage Area filling in the pointers to the args */
    ;

    NextStr = (char *) 0;
    i = -1;
    for (p = StorageArea; p - StorageArea <= BytesRead; p++) {
	if (!NextStr) NextStr = p;
	if (isspace(*p) || p - StorageArea == BytesRead) {
	    if (p - StorageArea != BytesRead) *p = '\0';
	    while (isspace(*p)) { /* eat multiple blanks */
		p++;
	    }
	    if (++i >= n_matches) {
		n_matches += 20;
		if ((Result = (char **)
		    realloc((char *) Result, n_matches * sizeof(char *))) ==
								(char **) 0) {
		    fprintf(stderr,
			"%s: out of memory [%u] in match-file \"%s\"\n",
			progname, n_matches * sizeof(char *), FileWithMatches);
		    /* TODO -- return with fewer matches -- NOTDONE */
		    exit(1);
		}
	    }
	    *p = '\0'; /* OK at the very end cos of the extra byte! */
	    Result[i] = NextStr;
	    NextStr = (char *) 0;
	}
    }

    if (i + 2 < n_matches) {
	Result = (char **) realloc((char *)Result,
					(unsigned) (i+2) * sizeof(char **));
	if (Result == (char **) 0) {
	    fprintf(stderr, "%s: no memory for match-list from \"%s\"\n",
						progname, FileWithMatches);
	    exit(1);
	}
    }

    if (close(fd) < 0) {
	fprintf(stderr, "%s: warning: obscure problem closing %d (\"%s\")\n",
						progname, fd, FileWithMatches);
	sleep(5);
    }

    (*MatchList) = Result;
    return (*MatchCount = i);
}

int
ShowFile(FileName, BlockInFile, WordInBlock)
    char *FileName;
    unsigned long BlockInFile;
    unsigned int WordInBlock;
{
    static char *Buffer = 0;
    int fd;
    static unsigned int BufLen;
    int AmountRead;
    register char *p;
    register char *q;
    int InTargetWord = 0;
    char *StartOfMyWord;
    int ThisWord = 0;
    char *Start;
    char *ThisLine = emalloc(Cols + 1); /* +1 for trailing \0 */
    char *FirstBit = emalloc(WordCol - GapWidth + 1); /* +1 for trailing \0 */
    char *LastBit = emalloc(Cols - WordCol + 1); /* +1 for trailing \0 */
    char *FirstStart;

    if (Buffer == (char *) 0) {
	BufLen = Cols * 10;
	if (BufLen < FileBlockSize * 3) BufLen = FileBlockSize * 3;
	Buffer = emalloc(BufLen);
    }

    errno = 0;

    if ((fd = open(FileName, O_RDONLY, 0)) < 0) {
	int e = errno;
	char *doc;

	if ((doc = FindFile(FileName)) == (char *) 0) {
	    fprintf(stderr, "%s: %s: ", progname, FileName);
	    errno = e;
	    perror(FileName);
	    efree(ThisLine); efree(FirstBit); efree(LastBit);
	    return -1;
	} else if ((fd = open(doc, O_RDONLY, 0)) < 0) {
	    fprintf(stderr, "%s: %s: ", progname, FileName);
	    errno = e;
	    perror(doc);
	    efree(ThisLine); efree(FirstBit); efree(LastBit);
	    return -1;
	}
	FileName = doc;
    }

    errno = 0;
    if (lseek(fd, BlockInFile? (long) ((BlockInFile - 1) * FileBlockSize) : 0L,
								    0) < 0) {
	int e = errno;
	fprintf(stderr, "%s: %s: ", progname, FileName);
	errno = e;
	perror("lseek");
	efree(ThisLine); efree(FirstBit); efree(LastBit);
	return;
    }

    errno = 0;
    if ((AmountRead = read(fd, Buffer, BufLen)) < MinWordLength) {
	int e = errno;
	fprintf(stderr, "%s: %s: ", progname, FileName);
	errno = e;
	perror("read");
	efree(ThisLine); efree(FirstBit); efree(LastBit);
	return -1;
    }


    /** Find the required word */
    if (BlockInFile) {
	/* start 1 char before the end of the previous block */
	StartOfMyWord = &Buffer[FileBlockSize - 1];
	/* perhaps the last word of the previous block spans the block
	 * boundary?
	 */
	while (WithinWord(*StartOfMyWord)) StartOfMyWord++;
	if (StartOfMyWord < &Buffer[FileBlockSize]) {
	    StartOfMyWord = &Buffer[FileBlockSize];
	}
    } else {
	StartOfMyWord = Buffer;
    }

    (void) close(fd);

    for (ThisWord = 0; ThisWord <= WordInBlock + 1; ThisWord++) {
bored:
	/* skip to the start of a word */
	while (!StartsWord(*StartOfMyWord)) {
	    ++StartOfMyWord;
	}

	Start = StartOfMyWord;

	/* find the end of the word */
	while (WithinWord(*StartOfMyWord)) {
	    if (*StartOfMyWord == '\'' && !EndsWord(StartOfMyWord[1])) break;
	    StartOfMyWord++;
	}

	/* Assert: StartOfMyWord points 1 character beyond the end of the
	 * word pointed to by Start
	 */
	/* see if it's long enough */
	if (StartOfMyWord - Start < MinWordLength) {
	    goto bored;
	}

	/** See if it's the right one */
	if (ThisWord == WordInBlock) {
	    StartOfMyWord = Start;
	    break;
	}
    }


    /* Find context before the keyword */

    q = &FirstBit[WordCol - GapWidth];
    *q-- = '\0';

    for (p = StartOfMyWord - 1; p >= Buffer; --p, --q) {
	*q = (isspace(*p)) ? ' ' : *p;
	if (q == FirstBit) break;
    }

    FirstStart = q;

    /* now build up the rest of the buffer */

    q = LastBit;
    *q = '\0';

    InTargetWord = 0;

    for (p = StartOfMyWord; p - Buffer < AmountRead; p++) {
	if (q >= &LastBit[Cols - WordCol]) break;

	switch (InTargetWord) {
	case 0:
	    if (StartsWord(*p)) {
		InTargetWord = 1;
	    }
	    break;
	case 1:
	    if (!WithinWord(*p)) {
		InTargetWord = 2;
	    }
	}
	if (isspace(*p)) {
	    *q = ' ';
	} else {
	    *q = *p;
	}
	*++q = '\0';
	if (q >= &LastBit[Cols - WordCol]) break;
    }

    printf("%*.*s", WordCol - GapWidth, WordCol - GapWidth, FirstStart);

    /* do the gap */
    {

	register int i;

	for (i = GapWidth; i > 0; i--) {
	    putchar(' ');
	}
    }

    printf("%-*.*s", Cols - WordCol, Cols - WordCol, LastBit);
	
    printf(":");
    {
	int OverShoot = Cols + 2 + strlen(FileName) - ScreenWidth; /* +2 is ": " */

	if (OverShoot > 0) {
	    FileName += OverShoot + 2;
	    printf("...");
	} else {
	    putchar(' ');
	}

    }
    printf("%s\n", FileName);

    efree(ThisLine); efree(FirstBit); efree(LastBit);
    return 0;
}


/*
 * $Log:	lqkwik.c,v $
 * Revision 1.1  91/03/02  20:37:47  lee
 * Initial revision
 * 
 *
 */
